/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.security.DigestInputStream;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.util.Arrays;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.permission.PermissionStatus;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.LayoutVersion;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.common.InconsistentFSStateException;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.FSImage;
import org.apache.hadoop.hdfs.server.namenode.FSImageCompression;
import org.apache.hadoop.hdfs.server.namenode.FSImageSerialization;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodeFileUnderConstruction;
import org.apache.hadoop.hdfs.server.namenode.SaveNamespaceContext;
import org.apache.hadoop.io.MD5Hash;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
@InterfaceStability.Evolving
class FSImageFormat {
    private static final Log LOG = FSImage.LOG;

    private FSImageFormat() {
    }

    static class Saver {
        private final SaveNamespaceContext context;
        private boolean saved = false;
        private MD5Hash savedDigest;
        private static final byte[] PATH_SEPARATOR = DFSUtil.string2Bytes("/");

        private void checkSaved() {
            if (!this.saved) {
                throw new IllegalStateException("FSImageSaver has not saved an image");
            }
        }

        private void checkNotSaved() {
            if (this.saved) {
                throw new IllegalStateException("FSImageSaver has already saved an image");
            }
        }

        Saver(SaveNamespaceContext context) {
            this.context = context;
        }

        MD5Hash getSavedDigest() {
            this.checkSaved();
            return this.savedDigest;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void save(File newFile, FSImageCompression compression) throws IOException {
            this.checkNotSaved();
            FSNamesystem sourceNamesystem = this.context.getSourceNamesystem();
            FSDirectory fsDir = sourceNamesystem.dir;
            long startTime = Time.now();
            MessageDigest digester = MD5Hash.getDigester();
            FileOutputStream fout = new FileOutputStream(newFile);
            DigestOutputStream fos = new DigestOutputStream(fout, digester);
            DataOutputStream out = new DataOutputStream(fos);
            try {
                out.writeInt(HdfsConstants.LAYOUT_VERSION);
                out.writeInt(sourceNamesystem.unprotectedGetNamespaceInfo().getNamespaceID());
                out.writeLong(fsDir.rootDir.numItemsInTree());
                out.writeLong(sourceNamesystem.getGenerationStamp());
                out.writeLong(this.context.getTxId());
                out = compression.writeHeaderAndWrapStream(fos);
                LOG.info((Object)("Saving image file " + newFile + " using " + compression));
                byte[] byteStore = new byte[4 * HdfsConstants.MAX_PATH_LENGTH];
                ByteBuffer strbuf = ByteBuffer.wrap(byteStore);
                FSImageSerialization.saveINode2Image(fsDir.rootDir, out);
                this.saveImage(strbuf, fsDir.rootDir, out);
                sourceNamesystem.saveFilesUnderConstruction(out);
                this.context.checkCancelled();
                sourceNamesystem.saveSecretManagerState(out);
                strbuf = null;
                this.context.checkCancelled();
                out.flush();
                this.context.checkCancelled();
                fout.getChannel().force(true);
            }
            finally {
                out.close();
            }
            this.saved = true;
            this.savedDigest = new MD5Hash(digester.digest());
            LOG.info((Object)("Image file of size " + newFile.length() + " saved in " + (Time.now() - startTime) / 1000L + " seconds."));
        }

        private void saveImage(ByteBuffer currentDirName, INodeDirectory current, DataOutputStream out) throws IOException {
            List<INode> children = current.getChildrenRaw();
            if (children == null || children.isEmpty()) {
                return;
            }
            int prefixLen = currentDirName.position();
            if (prefixLen == 0) {
                out.writeShort(PATH_SEPARATOR.length);
                out.write(PATH_SEPARATOR);
            } else {
                out.writeShort(prefixLen);
                out.write(currentDirName.array(), 0, prefixLen);
            }
            out.writeInt(children.size());
            int i = 0;
            for (INode child : children) {
                FSImageSerialization.saveINode2Image(child, out);
                if (i++ % 50 != 0) continue;
                this.context.checkCancelled();
            }
            for (INode child : children) {
                if (!child.isDirectory()) continue;
                currentDirName.put(PATH_SEPARATOR).put(child.getLocalNameBytes());
                this.saveImage(currentDirName, (INodeDirectory)child, out);
                currentDirName.position(prefixLen);
            }
        }
    }

    static class Loader {
        private final Configuration conf;
        private final FSNamesystem namesystem;
        private boolean loaded = false;
        private long imgTxId;
        private MD5Hash imgDigest;

        Loader(Configuration conf, FSNamesystem namesystem) {
            this.conf = conf;
            this.namesystem = namesystem;
        }

        MD5Hash getLoadedImageMd5() {
            this.checkLoaded();
            return this.imgDigest;
        }

        long getLoadedImageTxId() {
            this.checkLoaded();
            return this.imgTxId;
        }

        private void checkLoaded() {
            if (!this.loaded) {
                throw new IllegalStateException("Image not yet loaded!");
            }
        }

        private void checkNotLoaded() {
            if (this.loaded) {
                throw new IllegalStateException("Image already loaded!");
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void load(File curFile) throws IOException {
            this.checkNotLoaded();
            assert (curFile != null) : "curFile is null";
            long startTime = Time.now();
            MessageDigest digester = MD5Hash.getDigester();
            DigestInputStream fin = new DigestInputStream(new FileInputStream(curFile), digester);
            DataInputStream in = new DataInputStream(fin);
            try {
                int imgVersion = in.readInt();
                if (this.getLayoutVersion() != imgVersion) {
                    throw new InconsistentFSStateException(curFile, "imgVersion " + imgVersion + " expected to be " + this.getLayoutVersion());
                }
                in.readInt();
                long numFiles = in.readLong();
                long genstamp = in.readLong();
                this.namesystem.setGenerationStamp(genstamp);
                this.imgTxId = LayoutVersion.supports(LayoutVersion.Feature.STORED_TXIDS, imgVersion) ? in.readLong() : 0L;
                FSImageCompression compression = LayoutVersion.supports(LayoutVersion.Feature.FSIMAGE_COMPRESSION, imgVersion) ? FSImageCompression.readCompressionHeader(this.conf, in) : FSImageCompression.createNoopCompression();
                in = compression.unwrapInputStream(fin);
                LOG.info((Object)("Loading image file " + curFile + " using " + compression));
                LOG.info((Object)("Number of files = " + numFiles));
                if (LayoutVersion.supports(LayoutVersion.Feature.FSIMAGE_NAME_OPTIMIZATION, imgVersion)) {
                    this.loadLocalNameINodes(numFiles, in);
                } else {
                    this.loadFullNameINodes(numFiles, in);
                }
                this.loadFilesUnderConstruction(in);
                this.loadSecretManagerState(in);
                int eof = in.read();
                assert (eof == -1) : "Should have reached the end of image file " + curFile;
            }
            finally {
                in.close();
            }
            this.imgDigest = new MD5Hash(digester.digest());
            this.loaded = true;
            LOG.info((Object)("Image file of size " + curFile.length() + " loaded in " + (Time.now() - startTime) / 1000L + " seconds."));
        }

        private void updateRootAttr(INode root) {
            long nsQuota = root.getNsQuota();
            long dsQuota = root.getDsQuota();
            FSDirectory fsDir = this.namesystem.dir;
            if (nsQuota != -1L || dsQuota != -1L) {
                fsDir.rootDir.setQuota(nsQuota, dsQuota);
            }
            fsDir.rootDir.setModificationTime(root.getModificationTime());
            fsDir.rootDir.setPermissionStatus(root.getPermissionStatus());
        }

        private void loadLocalNameINodes(long numFiles, DataInputStream in) throws IOException {
            assert (LayoutVersion.supports(LayoutVersion.Feature.FSIMAGE_NAME_OPTIMIZATION, this.getLayoutVersion()));
            assert (numFiles > 0L);
            if (in.readShort() != 0) {
                throw new IOException("First node is not root");
            }
            INode root = this.loadINode(in);
            this.updateRootAttr(root);
            --numFiles;
            while (numFiles > 0L) {
                numFiles -= (long)this.loadDirectory(in);
            }
            if (numFiles != 0L) {
                throw new IOException("Read unexpect number of files: " + -numFiles);
            }
        }

        private int loadDirectory(DataInputStream in) throws IOException {
            String parentPath = FSImageSerialization.readString(in);
            FSDirectory fsDir = this.namesystem.dir;
            INode parent = fsDir.rootDir.getNode(parentPath, true);
            if (parent == null || !parent.isDirectory()) {
                throw new IOException("Path " + parentPath + "is not a directory.");
            }
            int numChildren = in.readInt();
            for (int i = 0; i < numChildren; ++i) {
                byte[] localName = new byte[in.readShort()];
                in.readFully(localName);
                INode newNode = this.loadINode(in);
                this.namesystem.dir.addToParent(localName, (INodeDirectory)parent, newNode, false);
            }
            return numChildren;
        }

        private void loadFullNameINodes(long numFiles, DataInputStream in) throws IOException {
            Object parentPath = new byte[][]{new byte[0]};
            FSDirectory fsDir = this.namesystem.dir;
            INodeDirectory parentINode = fsDir.rootDir;
            for (long i = 0L; i < numFiles; ++i) {
                byte[][] pathComponents = FSImageSerialization.readPathComponents(in);
                INode newNode = this.loadINode(in);
                if (this.isRoot(pathComponents)) {
                    this.updateRootAttr(newNode);
                    continue;
                }
                if (!this.isParent(pathComponents, (byte[][])parentPath)) {
                    parentINode = fsDir.getParent(pathComponents);
                    parentPath = this.getParent(pathComponents);
                }
                parentINode = fsDir.addToParent(pathComponents[pathComponents.length - 1], parentINode, newNode, false);
            }
        }

        private INode loadINode(DataInputStream in) throws IOException {
            long modificationTime = 0L;
            long atime = 0L;
            long blockSize = 0L;
            int imgVersion = this.getLayoutVersion();
            short replication = in.readShort();
            replication = this.namesystem.getBlockManager().adjustReplication(replication);
            modificationTime = in.readLong();
            if (LayoutVersion.supports(LayoutVersion.Feature.FILE_ACCESS_TIME, imgVersion)) {
                atime = in.readLong();
            }
            blockSize = in.readLong();
            int numBlocks = in.readInt();
            BlockInfo[] blocks = null;
            if (numBlocks >= 0) {
                blocks = new BlockInfo[numBlocks];
                for (int j = 0; j < numBlocks; ++j) {
                    blocks[j] = new BlockInfo(replication);
                    blocks[j].readFields(in);
                }
            }
            long nsQuota = -1L;
            if (blocks == null && numBlocks == -1) {
                nsQuota = in.readLong();
            }
            long dsQuota = -1L;
            if (LayoutVersion.supports(LayoutVersion.Feature.DISKSPACE_QUOTA, imgVersion) && blocks == null && numBlocks == -1) {
                dsQuota = in.readLong();
            }
            String symlink = "";
            if (numBlocks == -2) {
                symlink = Text.readString((DataInput)in);
            }
            PermissionStatus permissions = PermissionStatus.read((DataInput)in);
            return INode.newINode(permissions, blocks, symlink, replication, modificationTime, atime, nsQuota, dsQuota, blockSize);
        }

        private void loadFilesUnderConstruction(DataInputStream in) throws IOException {
            FSDirectory fsDir = this.namesystem.dir;
            int size = in.readInt();
            LOG.info((Object)("Number of files under construction = " + size));
            for (int i = 0; i < size; ++i) {
                INodeFileUnderConstruction cons = FSImageSerialization.readINodeUnderConstruction(in);
                String path = cons.getLocalName();
                INodeFile old = fsDir.getFileINode(path);
                if (old == null) {
                    throw new IOException("Found lease for non-existent file " + path);
                }
                if (((INode)old).isDirectory()) {
                    throw new IOException("Found lease for directory " + path);
                }
                INodeFile oldnode = old;
                fsDir.replaceNode(path, oldnode, cons);
                this.namesystem.leaseManager.addLease(cons.getClientName(), path);
            }
        }

        private void loadSecretManagerState(DataInputStream in) throws IOException {
            int imgVersion = this.getLayoutVersion();
            if (!LayoutVersion.supports(LayoutVersion.Feature.DELEGATION_TOKEN, imgVersion)) {
                return;
            }
            this.namesystem.loadSecretManagerState(in);
        }

        private int getLayoutVersion() {
            return this.namesystem.getFSImage().getStorage().getLayoutVersion();
        }

        private boolean isRoot(byte[][] path) {
            return path.length == 1 && path[0] == null;
        }

        private boolean isParent(byte[][] path, byte[][] parent) {
            if (path == null || parent == null) {
                return false;
            }
            if (parent.length == 0 || path.length != parent.length + 1) {
                return false;
            }
            boolean isParent = true;
            for (int i = 0; i < parent.length; ++i) {
                isParent = isParent && Arrays.equals(path[i], parent[i]);
            }
            return isParent;
        }

        String getParent(String path) {
            return path.substring(0, path.lastIndexOf("/"));
        }

        byte[][] getParent(byte[][] path) {
            byte[][] result = new byte[path.length - 1][];
            for (int i = 0; i < result.length; ++i) {
                result[i] = new byte[path[i].length];
                System.arraycopy(path[i], 0, result[i], 0, path[i].length);
            }
            return result;
        }
    }
}

