/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.blockmanagement;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.FutureTask;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicLong;
import javax.management.ObjectName;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileEncryptionInfo;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.AddBlockFlag;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.HAUtil;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.BlockListAsLongs;
import org.apache.hadoop.hdfs.protocol.BlockStoragePolicy;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.protocol.UnregisteredNodeException;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenIdentifier;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenSecretManager;
import org.apache.hadoop.hdfs.security.token.block.DataEncryptionKey;
import org.apache.hadoop.hdfs.security.token.block.ExportedBlockKeys;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockCollection;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManagerFaultInjector;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockPlacementPolicy;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockReportLeaseManager;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockStatsMXBean;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockStoragePolicySuite;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockToMarkCorrupt;
import org.apache.hadoop.hdfs.server.blockmanagement.BlocksMap;
import org.apache.hadoop.hdfs.server.blockmanagement.CorruptReplicasMap;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeManager;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeStorageInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.HeartbeatManager;
import org.apache.hadoop.hdfs.server.blockmanagement.InvalidateBlocks;
import org.apache.hadoop.hdfs.server.blockmanagement.NumberReplicas;
import org.apache.hadoop.hdfs.server.blockmanagement.PendingDataNodeMessages;
import org.apache.hadoop.hdfs.server.blockmanagement.PendingReplicationBlocks;
import org.apache.hadoop.hdfs.server.blockmanagement.ReplicaUnderConstruction;
import org.apache.hadoop.hdfs.server.blockmanagement.ReplicationWork;
import org.apache.hadoop.hdfs.server.blockmanagement.StorageTypeStats;
import org.apache.hadoop.hdfs.server.blockmanagement.UnderReplicatedBlocks;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.namenode.CacheManager;
import org.apache.hadoop.hdfs.server.namenode.CachedBlock;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.Namesystem;
import org.apache.hadoop.hdfs.server.namenode.ha.HAContext;
import org.apache.hadoop.hdfs.server.namenode.metrics.NameNodeMetrics;
import org.apache.hadoop.hdfs.server.protocol.BlockReportContext;
import org.apache.hadoop.hdfs.server.protocol.BlocksWithLocations;
import org.apache.hadoop.hdfs.server.protocol.DatanodeCommand;
import org.apache.hadoop.hdfs.server.protocol.DatanodeRegistration;
import org.apache.hadoop.hdfs.server.protocol.DatanodeStorage;
import org.apache.hadoop.hdfs.server.protocol.KeyUpdateCommand;
import org.apache.hadoop.hdfs.server.protocol.ReceivedDeletedBlockInfo;
import org.apache.hadoop.hdfs.server.protocol.StorageReceivedDeletedBlocks;
import org.apache.hadoop.hdfs.util.LightWeightHashSet;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.net.Node;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.ExitUtil;
import org.apache.hadoop.util.LightWeightGSet;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class BlockManager
implements BlockStatsMXBean {
    public static final Logger LOG = LoggerFactory.getLogger(BlockManager.class);
    public static final Logger blockLog = NameNode.blockStateChangeLog;
    private static final String QUEUE_REASON_CORRUPT_STATE = "it has the wrong state or generation stamp";
    private static final String QUEUE_REASON_FUTURE_GENSTAMP = "generation stamp is in the future";
    private final Namesystem namesystem;
    private final DatanodeManager datanodeManager;
    private final HeartbeatManager heartbeatManager;
    private final BlockTokenSecretManager blockTokenSecretManager;
    private final PendingDataNodeMessages pendingDNMessages = new PendingDataNodeMessages();
    private volatile long pendingReplicationBlocksCount = 0L;
    private volatile long corruptReplicaBlocksCount = 0L;
    private volatile long underReplicatedBlocksCount = 0L;
    private volatile long scheduledReplicationBlocksCount = 0L;
    private boolean initializedReplQueues;
    private final AtomicLong excessBlocksCount = new AtomicLong(0L);
    private final long startupDelayBlockDeletionInMs;
    private final BlockReportLeaseManager blockReportLeaseManager;
    private ObjectName mxBeanName;
    private final long replicationRecheckInterval;
    final BlocksMap blocksMap;
    final Daemon replicationThread = new Daemon((Runnable)new ReplicationMonitor());
    private final BlockReportProcessingThread blockReportThread = new BlockReportProcessingThread();
    final CorruptReplicasMap corruptReplicas = new CorruptReplicasMap();
    private final InvalidateBlocks invalidateBlocks;
    private final LinkedHashSet<Block> postponedMisreplicatedBlocks = new LinkedHashSet();
    private final int blocksPerPostpondedRescan;
    private final ArrayList<Block> rescannedMisreplicatedBlocks;
    public final Map<String, LightWeightHashSet<Block>> excessReplicateMap = new HashMap<String, LightWeightHashSet<Block>>();
    public final UnderReplicatedBlocks neededReplications = new UnderReplicatedBlocks();
    @VisibleForTesting
    final PendingReplicationBlocks pendingReplications;
    public final short maxReplication;
    int maxReplicationStreams;
    int replicationStreamsHardLimit;
    public final short minReplication;
    public final int defaultReplication;
    final int maxCorruptFilesReturned;
    final float blocksInvalidateWorkPct;
    final int blocksReplWorkMultiplier;
    final boolean encryptDataTransfer;
    private final long maxNumBlocksToLog;
    private boolean shouldPostponeBlocksFromFuture = false;
    private Daemon replicationQueuesInitializer = null;
    private int numBlocksPerIteration;
    private long getBlocksMinBlockSize = -1L;
    private double replicationQueuesInitProgress = 0.0;
    private BlockPlacementPolicy blockplacement;
    private final BlockStoragePolicySuite storagePolicySuite;
    private boolean checkNSRunning = true;
    private AtomicLong numberOfBytesInFutureBlocks;
    private boolean inRollBack = false;

    public long getPendingReplicationBlocksCount() {
        return this.pendingReplicationBlocksCount;
    }

    public long getUnderReplicatedBlocksCount() {
        return this.underReplicatedBlocksCount;
    }

    public long getCorruptReplicaBlocksCount() {
        return this.corruptReplicaBlocksCount;
    }

    public long getScheduledReplicationBlocksCount() {
        return this.scheduledReplicationBlocksCount;
    }

    public long getPendingDeletionBlocksCount() {
        return this.invalidateBlocks.numBlocks();
    }

    public long getStartupDelayBlockDeletionInMs() {
        return this.startupDelayBlockDeletionInMs;
    }

    public long getExcessBlocksCount() {
        return this.excessBlocksCount.get();
    }

    public long getPostponedMisreplicatedBlocksCount() {
        return this.postponedMisreplicatedBlocks.size();
    }

    public int getPendingDataNodeMessageCount() {
        return this.pendingDNMessages.count();
    }

    public long getNumTimedOutPendingReplications() {
        return this.pendingReplications.getNumTimedOuts();
    }

    public BlockManager(Namesystem namesystem, Configuration conf) throws IOException {
        this.namesystem = namesystem;
        this.datanodeManager = new DatanodeManager(this, namesystem, conf);
        this.heartbeatManager = this.datanodeManager.getHeartbeatManager();
        this.blocksPerPostpondedRescan = (int)Math.min(Integer.MAX_VALUE, this.datanodeManager.getBlocksPerPostponedMisreplicatedBlocksRescan());
        this.rescannedMisreplicatedBlocks = new ArrayList(this.blocksPerPostpondedRescan);
        this.startupDelayBlockDeletionInMs = conf.getLong("dfs.namenode.startup.delay.block.deletion.sec", 0L) * 1000L;
        this.invalidateBlocks = new InvalidateBlocks(this.datanodeManager.blockInvalidateLimit, this.startupDelayBlockDeletionInMs);
        this.blocksMap = new BlocksMap(LightWeightGSet.computeCapacity((double)2.0, (String)"BlocksMap"));
        this.blockplacement = BlockPlacementPolicy.getInstance(conf, this.datanodeManager.getFSClusterStats(), this.datanodeManager.getNetworkTopology(), this.datanodeManager.getHost2DatanodeMap());
        this.storagePolicySuite = BlockStoragePolicySuite.createDefaultSuite();
        this.pendingReplications = new PendingReplicationBlocks((long)conf.getInt("dfs.namenode.replication.pending.timeout-sec", -1) * 1000L);
        this.blockTokenSecretManager = BlockManager.createBlockTokenSecretManager(conf);
        this.maxCorruptFilesReturned = conf.getInt("dfs.corruptfilesreturned.max", 500);
        this.defaultReplication = conf.getInt("dfs.replication", 3);
        int maxR = conf.getInt("dfs.replication.max", 512);
        int minR = conf.getInt("dfs.namenode.replication.min", 1);
        if (minR <= 0) {
            throw new IOException("Unexpected configuration parameters: dfs.namenode.replication.min = " + minR + " <= 0");
        }
        if (maxR > Short.MAX_VALUE) {
            throw new IOException("Unexpected configuration parameters: dfs.replication.max = " + maxR + " > " + Short.MAX_VALUE);
        }
        if (minR > maxR) {
            throw new IOException("Unexpected configuration parameters: dfs.namenode.replication.min = " + minR + " > " + "dfs.replication.max" + " = " + maxR);
        }
        this.minReplication = (short)minR;
        this.maxReplication = (short)maxR;
        this.maxReplicationStreams = conf.getInt("dfs.namenode.replication.max-streams", 2);
        this.replicationStreamsHardLimit = conf.getInt("dfs.namenode.replication.max-streams-hard-limit", 4);
        this.blocksInvalidateWorkPct = DFSUtil.getInvalidateWorkPctPerIteration(conf);
        this.blocksReplWorkMultiplier = DFSUtil.getReplWorkMultiplier(conf);
        this.replicationRecheckInterval = (long)conf.getInt("dfs.namenode.replication.interval", 3) * 1000L;
        this.encryptDataTransfer = conf.getBoolean("dfs.encrypt.data.transfer", false);
        this.maxNumBlocksToLog = conf.getLong("dfs.namenode.max-num-blocks-to-log", 1000L);
        this.numBlocksPerIteration = conf.getInt("dfs.block.misreplication.processing.limit", 10000);
        this.getBlocksMinBlockSize = conf.getLongBytes("dfs.balancer.getBlocks.min-block-size", 0xA00000L);
        this.blockReportLeaseManager = new BlockReportLeaseManager(conf);
        this.numberOfBytesInFutureBlocks = new AtomicLong();
        this.inRollBack = this.isInRollBackMode(NameNode.getStartupOption(conf));
        LOG.info("defaultReplication         = " + this.defaultReplication);
        LOG.info("maxReplication             = " + this.maxReplication);
        LOG.info("minReplication             = " + this.minReplication);
        LOG.info("maxReplicationStreams      = " + this.maxReplicationStreams);
        LOG.info("replicationRecheckInterval = " + this.replicationRecheckInterval);
        LOG.info("encryptDataTransfer        = " + this.encryptDataTransfer);
        LOG.info("maxNumBlocksToLog          = " + this.maxNumBlocksToLog);
    }

    private static BlockTokenSecretManager createBlockTokenSecretManager(Configuration conf) throws IOException {
        boolean isEnabled = conf.getBoolean("dfs.block.access.token.enable", false);
        LOG.info("dfs.block.access.token.enable=" + isEnabled);
        if (!isEnabled) {
            if (UserGroupInformation.isSecurityEnabled()) {
                String errMessage = "Security is enabled but block access tokens (via dfs.block.access.token.enable) aren't enabled. This may cause issues when clients attempt to connect to a DataNode. Aborting NameNode";
                throw new IOException(errMessage);
            }
            return null;
        }
        long updateMin = conf.getLong("dfs.block.access.key.update.interval", 600L);
        long lifetimeMin = conf.getLong("dfs.block.access.token.lifetime", 600L);
        String encryptionAlgorithm = conf.get("dfs.encrypt.data.transfer.algorithm");
        LOG.info("dfs.block.access.key.update.interval=" + updateMin + " min(s), " + "dfs.block.access.token.lifetime" + "=" + lifetimeMin + " min(s), " + "dfs.encrypt.data.transfer.algorithm" + "=" + encryptionAlgorithm);
        String nsId = DFSUtil.getNamenodeNameServiceId(conf);
        boolean isHaEnabled = HAUtil.isHAEnabled(conf, nsId);
        if (isHaEnabled) {
            String otherNnId;
            String thisNnId = HAUtil.getNameNodeId(conf, nsId);
            return new BlockTokenSecretManager(updateMin * 60L * 1000L, lifetimeMin * 60L * 1000L, thisNnId.compareTo(otherNnId = HAUtil.getNameNodeIdOfOtherNode(conf, nsId)) < 0 ? 0 : 1, null, encryptionAlgorithm);
        }
        return new BlockTokenSecretManager(updateMin * 60L * 1000L, lifetimeMin * 60L * 1000L, 0, null, encryptionAlgorithm);
    }

    public BlockStoragePolicy getStoragePolicy(String policyName) {
        return this.storagePolicySuite.getPolicy(policyName);
    }

    public BlockStoragePolicy getStoragePolicy(byte policyId) {
        return this.storagePolicySuite.getPolicy(policyId);
    }

    public BlockStoragePolicy[] getStoragePolicies() {
        return this.storagePolicySuite.getAllPolicies();
    }

    public void setBlockPoolId(String blockPoolId) {
        if (this.isBlockTokenEnabled()) {
            this.blockTokenSecretManager.setBlockPoolId(blockPoolId);
        }
    }

    public BlockStoragePolicySuite getStoragePolicySuite() {
        return this.storagePolicySuite;
    }

    @VisibleForTesting
    public BlockTokenSecretManager getBlockTokenSecretManager() {
        return this.blockTokenSecretManager;
    }

    @VisibleForTesting
    void enableRMTerminationForTesting() {
        this.checkNSRunning = false;
    }

    private boolean isBlockTokenEnabled() {
        return this.blockTokenSecretManager != null;
    }

    boolean shouldUpdateBlockKey(long updateTime) throws IOException {
        return this.isBlockTokenEnabled() ? this.blockTokenSecretManager.updateKeys(updateTime) : false;
    }

    public void activate(Configuration conf) {
        this.pendingReplications.start();
        this.datanodeManager.activate(conf);
        this.replicationThread.setName("ReplicationMonitor");
        this.replicationThread.start();
        this.blockReportThread.start();
        this.mxBeanName = MBeans.register((String)"NameNode", (String)"BlockStats", (Object)this);
    }

    public void close() {
        try {
            this.replicationThread.interrupt();
            this.blockReportThread.interrupt();
            this.replicationThread.join(3000L);
            this.blockReportThread.join(3000L);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        this.datanodeManager.close();
        this.pendingReplications.stop();
        this.blocksMap.close();
    }

    public DatanodeManager getDatanodeManager() {
        return this.datanodeManager;
    }

    @VisibleForTesting
    public BlockPlacementPolicy getBlockPlacementPolicy() {
        return this.blockplacement;
    }

    public void setBlockPlacementPolicy(BlockPlacementPolicy newpolicy) {
        if (newpolicy == null) {
            throw new HadoopIllegalArgumentException("newpolicy == null");
        }
        this.blockplacement = newpolicy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void metaSave(PrintWriter out) {
        assert (this.namesystem.hasWriteLock());
        ArrayList<DatanodeDescriptor> live = new ArrayList<DatanodeDescriptor>();
        ArrayList<DatanodeDescriptor> dead = new ArrayList<DatanodeDescriptor>();
        this.datanodeManager.fetchDatanodes(live, dead, false);
        out.println("Live Datanodes: " + live.size());
        out.println("Dead Datanodes: " + dead.size());
        UnderReplicatedBlocks underReplicatedBlocks = this.neededReplications;
        synchronized (underReplicatedBlocks) {
            out.println("Metasave: Blocks waiting for replication: " + this.neededReplications.size());
            for (Block block : this.neededReplications) {
                this.dumpBlockMeta(block, out);
            }
        }
        out.println("Mis-replicated blocks that have been postponed:");
        for (Block block : this.postponedMisreplicatedBlocks) {
            this.dumpBlockMeta(block, out);
        }
        this.pendingReplications.metaSave(out);
        this.invalidateBlocks.dump(out);
        Set<Block> corruptBlocks = this.corruptReplicas.getCorruptBlocks();
        out.println("Corrupt Blocks:");
        for (Block block : corruptBlocks) {
            Collection<DatanodeDescriptor> corruptNodes = this.corruptReplicas.getNodes(block);
            if (corruptNodes == null) {
                LOG.warn(block.getBlockId() + " is corrupt but has no associated node.");
                continue;
            }
            int numNodesToFind = corruptNodes.size();
            for (DatanodeStorageInfo storage : this.blocksMap.getStorages(block)) {
                DatanodeDescriptor node = storage.getDatanodeDescriptor();
                if (!corruptNodes.contains((Object)node)) continue;
                String storageId = storage.getStorageID();
                DatanodeStorageInfo storageInfo = node.getStorageInfo(storageId);
                DatanodeStorage.State state = storageInfo == null ? null : storageInfo.getState();
                out.println("Block=" + block.getBlockId() + "\tNode=" + node.getName() + "\tStorageID=" + storageId + "\tStorageState=" + state + "\tTotalReplicas=" + this.blocksMap.numNodes(block) + "\tReason=" + this.corruptReplicas.getCorruptReason(block, node));
                if (--numNodesToFind != 0) continue;
                break;
            }
            if (numNodesToFind <= 0) continue;
            String[] corruptNodesList = new String[corruptNodes.size()];
            int i = 0;
            for (DatanodeDescriptor d : corruptNodes) {
                corruptNodesList[i] = d.getHostName();
                ++i;
            }
            out.println(block.getBlockId() + " corrupt on " + StringUtils.join((CharSequence)",", (String[])corruptNodesList) + " but not all nodes are" + "found in its block locations");
        }
        this.getDatanodeManager().datanodeDump(out);
    }

    private void dumpBlockMeta(Block block, PrintWriter out) {
        ArrayList<DatanodeDescriptor> containingNodes = new ArrayList<DatanodeDescriptor>();
        ArrayList<DatanodeStorageInfo> containingLiveReplicasNodes = new ArrayList<DatanodeStorageInfo>();
        NumberReplicas numReplicas = new NumberReplicas();
        this.chooseSourceDatanode(block, containingNodes, containingLiveReplicasNodes, numReplicas, 5);
        assert (containingLiveReplicasNodes.size() >= numReplicas.liveReplicas());
        int usableReplicas = numReplicas.liveReplicas() + numReplicas.decommissionedAndDecommissioning();
        if (block instanceof BlockInfo) {
            BlockCollection bc = this.getBlockCollection((BlockInfo)block);
            String fileName = bc == null ? "[orphaned]" : bc.getName();
            out.print(fileName + ": ");
        }
        out.print(block + (usableReplicas > 0 ? "" : " MISSING") + " (replicas:" + " l: " + numReplicas.liveReplicas() + " d: " + numReplicas.decommissionedAndDecommissioning() + " c: " + numReplicas.corruptReplicas() + " e: " + numReplicas.excessReplicas() + ") ");
        Collection<DatanodeDescriptor> corruptNodes = this.corruptReplicas.getNodes(block);
        for (DatanodeStorageInfo storage : this.blocksMap.getStorages(block)) {
            DatanodeDescriptor node = storage.getDatanodeDescriptor();
            String state = "";
            if (corruptNodes != null && corruptNodes.contains((Object)node)) {
                state = "(corrupt)";
            } else if (node.isDecommissioned() || node.isDecommissionInProgress()) {
                state = "(decommissioned)";
            }
            if (storage.areBlockContentsStale()) {
                state = state + " (block deletions maybe out of date)";
            }
            out.print(" " + (Object)((Object)node) + state + " : ");
        }
        out.println("");
    }

    public int getMaxReplicationStreams() {
        return this.maxReplicationStreams;
    }

    public boolean checkMinReplication(BlockInfo block) {
        return this.countNodes(block).liveReplicas() >= this.minReplication;
    }

    public short getMinReplication() {
        return this.minReplication;
    }

    private boolean commitBlock(BlockInfo block, Block commitBlock) throws IOException {
        if (block.getBlockUCState() == HdfsServerConstants.BlockUCState.COMMITTED) {
            return false;
        }
        assert (block.getNumBytes() <= commitBlock.getNumBytes()) : "commitBlock length is less than the stored one " + commitBlock.getNumBytes() + " vs. " + block.getNumBytes();
        if (block.getGenerationStamp() != commitBlock.getGenerationStamp()) {
            throw new IOException("Commit block with mismatching GS. NN has " + (Object)((Object)block) + ", client submits " + commitBlock);
        }
        List<ReplicaUnderConstruction> staleReplicas = block.commitBlock(commitBlock);
        this.removeStaleReplicas(staleReplicas, block);
        return true;
    }

    public boolean commitOrCompleteLastBlock(BlockCollection bc, Block commitBlock, INodesInPath iip) throws IOException {
        if (commitBlock == null) {
            return false;
        }
        BlockInfo lastBlock = bc.getLastBlock();
        if (lastBlock == null) {
            return false;
        }
        if (lastBlock.isComplete()) {
            return false;
        }
        boolean committed = this.commitBlock(lastBlock, commitBlock);
        NumberReplicas numReplicas = this.countNodes(lastBlock);
        if (numReplicas.liveReplicas() + numReplicas.decommissioning() >= this.minReplication) {
            if (committed) {
                this.addExpectedReplicasToPending(lastBlock);
            }
            this.completeBlock(lastBlock, iip, false);
        }
        return committed;
    }

    public void addExpectedReplicasToPending(BlockInfo blk) {
        DatanodeStorageInfo[] expectedStorages = blk.getUnderConstructionFeature().getExpectedStorageLocations();
        if (expectedStorages.length - blk.numNodes() > 0) {
            ArrayList<DatanodeDescriptor> pendingNodes = new ArrayList<DatanodeDescriptor>();
            for (DatanodeStorageInfo storage : expectedStorages) {
                DatanodeDescriptor dnd = storage.getDatanodeDescriptor();
                if (blk.findStorageInfo(dnd) == null) {
                    pendingNodes.add(dnd);
                }
                this.pendingReplications.increment(blk, pendingNodes.toArray(new DatanodeDescriptor[pendingNodes.size()]));
            }
        }
    }

    private void completeBlock(BlockInfo curBlock, INodesInPath iip, boolean force) throws IOException {
        if (curBlock.isComplete()) {
            return;
        }
        int numNodes = curBlock.numNodes();
        if (!force && numNodes < this.minReplication) {
            throw new IOException("Cannot complete block: block does not satisfy minimal replication requirement.");
        }
        if (!force && curBlock.getBlockUCState() != HdfsServerConstants.BlockUCState.COMMITTED) {
            throw new IOException("Cannot complete block: block has not been COMMITTED by the client");
        }
        this.convertToCompleteBlock(curBlock, iip);
        this.namesystem.adjustSafeModeBlockTotals(0, 1);
        this.namesystem.incrementSafeBlockCount(Math.min(numNodes, this.minReplication));
    }

    private void convertToCompleteBlock(BlockInfo curBlock, INodesInPath iip) throws IOException {
        curBlock.convertToCompleteBlock();
        this.namesystem.getFSDirectory().updateSpaceForCompleteBlock(curBlock, iip);
    }

    public void forceCompleteBlock(BlockInfo block) throws IOException {
        List<ReplicaUnderConstruction> staleReplicas = block.commitBlock(block);
        this.removeStaleReplicas(staleReplicas, block);
        this.completeBlock(block, null, true);
    }

    public LocatedBlock convertLastBlockToUnderConstruction(BlockCollection bc, long bytesToRemove) throws IOException {
        BlockInfo lastBlock = bc.getLastBlock();
        if (lastBlock == null || bc.getPreferredBlockSize() == lastBlock.getNumBytes() - bytesToRemove) {
            return null;
        }
        assert (lastBlock == this.getStoredBlock(lastBlock)) : "last block of the file is not in blocksMap";
        DatanodeStorageInfo[] targets = this.getStorages(lastBlock);
        bc.convertLastBlockToUC(lastBlock, targets);
        NumberReplicas replicas = this.countNodes(lastBlock);
        this.neededReplications.remove(lastBlock, replicas.liveReplicas(), replicas.readOnlyReplicas(), replicas.decommissionedAndDecommissioning(), this.getReplication(lastBlock));
        this.pendingReplications.remove(lastBlock);
        for (DatanodeStorageInfo storage : targets) {
            this.invalidateBlocks.remove(storage.getDatanodeDescriptor(), lastBlock);
        }
        this.namesystem.adjustSafeModeBlockTotals(targets.length >= this.minReplication ? -1 : 0, -1);
        long fileLength = bc.computeContentSummary(this.getStoragePolicySuite()).getLength();
        long pos = fileLength - lastBlock.getNumBytes();
        return this.createLocatedBlock(lastBlock, pos, BlockTokenIdentifier.AccessMode.WRITE);
    }

    private List<DatanodeStorageInfo> getValidLocations(Block block) {
        ArrayList<DatanodeStorageInfo> locations = new ArrayList<DatanodeStorageInfo>(this.blocksMap.numNodes(block));
        for (DatanodeStorageInfo storage : this.blocksMap.getStorages(block)) {
            if (this.invalidateBlocks.contains(storage.getDatanodeDescriptor(), block)) continue;
            locations.add(storage);
        }
        return locations;
    }

    private List<LocatedBlock> createLocatedBlockList(BlockInfo[] blocks, long offset, long length, int nrBlocksToReturn, BlockTokenIdentifier.AccessMode mode) throws IOException {
        int curBlk = 0;
        long curPos = 0L;
        long blkSize = 0L;
        int nrBlocks = blocks[0].getNumBytes() == 0L ? 0 : blocks.length;
        for (curBlk = 0; curBlk < nrBlocks; ++curBlk) {
            blkSize = blocks[curBlk].getNumBytes();
            assert (blkSize > 0L) : "Block of size 0";
            if (curPos + blkSize > offset) break;
            curPos += blkSize;
        }
        if (nrBlocks > 0 && curBlk == nrBlocks) {
            return Collections.emptyList();
        }
        long endOff = offset + length;
        ArrayList<LocatedBlock> results = new ArrayList<LocatedBlock>(blocks.length);
        do {
            results.add(this.createLocatedBlock(blocks[curBlk], curPos, mode));
        } while ((curPos += blocks[++curBlk].getNumBytes()) < endOff && curBlk < blocks.length && results.size() < nrBlocksToReturn);
        return results;
    }

    private LocatedBlock createLocatedBlock(BlockInfo[] blocks, long endPos, BlockTokenIdentifier.AccessMode mode) throws IOException {
        long blkSize;
        int curBlk = 0;
        long curPos = 0L;
        int nrBlocks = blocks[0].getNumBytes() == 0L ? 0 : blocks.length;
        for (curBlk = 0; curBlk < nrBlocks && curPos + (blkSize = blocks[curBlk].getNumBytes()) < endPos; ++curBlk) {
            curPos += blkSize;
        }
        return this.createLocatedBlock(blocks[curBlk], curPos, mode);
    }

    private LocatedBlock createLocatedBlock(BlockInfo blk, long pos, BlockTokenIdentifier.AccessMode mode) throws IOException {
        LocatedBlock lb = this.createLocatedBlock(blk, pos);
        if (mode != null) {
            this.setBlockToken(lb, mode);
        }
        return lb;
    }

    private LocatedBlock createLocatedBlock(BlockInfo blk, long pos) throws IOException {
        int numNodes;
        int numCorruptReplicas;
        if (!blk.isComplete()) {
            DatanodeStorageInfo[] storages = blk.getUnderConstructionFeature().getExpectedStorageLocations();
            ExtendedBlock eb = new ExtendedBlock(this.namesystem.getBlockPoolId(), (Block)blk);
            return BlockManager.newLocatedBlock(eb, storages, pos, false);
        }
        int numCorruptNodes = this.countNodes(blk).corruptReplicas();
        if (numCorruptNodes != (numCorruptReplicas = this.corruptReplicas.numCorruptReplicas(blk))) {
            LOG.warn("Inconsistent number of corrupt replicas for " + (Object)((Object)blk) + " blockMap has " + numCorruptNodes + " but corrupt replicas map has " + numCorruptReplicas);
        }
        boolean isCorrupt = numCorruptReplicas == (numNodes = this.blocksMap.numNodes(blk));
        int numMachines = isCorrupt ? numNodes : numNodes - numCorruptReplicas;
        DatanodeStorageInfo[] machines = new DatanodeStorageInfo[numMachines];
        int j = 0;
        if (numMachines > 0) {
            boolean noCorrupt = numCorruptReplicas == 0;
            for (DatanodeStorageInfo storage : this.blocksMap.getStorages(blk)) {
                if (storage.getState() == DatanodeStorage.State.FAILED) continue;
                if (noCorrupt) {
                    machines[j++] = storage;
                    continue;
                }
                DatanodeDescriptor d = storage.getDatanodeDescriptor();
                boolean replicaCorrupt = this.isReplicaCorrupt(blk, d);
                if (!isCorrupt && replicaCorrupt) continue;
                machines[j++] = storage;
            }
        }
        if (j < machines.length) {
            machines = Arrays.copyOf(machines, j);
        }
        assert (j == machines.length) : "isCorrupt: " + isCorrupt + " numMachines: " + numMachines + " numNodes: " + numNodes + " numCorrupt: " + numCorruptNodes + " numCorruptRepls: " + numCorruptReplicas;
        ExtendedBlock eb = new ExtendedBlock(this.namesystem.getBlockPoolId(), (Block)blk);
        return BlockManager.newLocatedBlock(eb, machines, pos, isCorrupt);
    }

    public LocatedBlocks createLocatedBlocks(BlockInfo[] blocks, long fileSizeExcludeBlocksUnderConstruction, boolean isFileUnderConstruction, long offset, long length, boolean needBlockToken, boolean inSnapshot, FileEncryptionInfo feInfo) throws IOException {
        boolean isComplete;
        LocatedBlock lastlb;
        assert (this.namesystem.hasReadLock());
        if (blocks == null) {
            return null;
        }
        if (blocks.length == 0) {
            return new LocatedBlocks(0L, isFileUnderConstruction, Collections.emptyList(), null, false, feInfo);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("blocks = " + Arrays.asList(blocks));
        }
        BlockTokenIdentifier.AccessMode mode = needBlockToken ? BlockTokenIdentifier.AccessMode.READ : null;
        List<LocatedBlock> locatedblocks = this.createLocatedBlockList(blocks, offset, length, Integer.MAX_VALUE, mode);
        if (!inSnapshot) {
            BlockInfo last = blocks[blocks.length - 1];
            long lastPos = last.isComplete() ? fileSizeExcludeBlocksUnderConstruction - last.getNumBytes() : fileSizeExcludeBlocksUnderConstruction;
            lastlb = this.createLocatedBlock(last, lastPos, mode);
            isComplete = last.isComplete();
        } else {
            lastlb = this.createLocatedBlock(blocks, fileSizeExcludeBlocksUnderConstruction, mode);
            isComplete = true;
        }
        LocatedBlocks locations = new LocatedBlocks(fileSizeExcludeBlocksUnderConstruction, isFileUnderConstruction, locatedblocks, lastlb, isComplete, feInfo);
        CacheManager cm = this.namesystem.getCacheManager();
        if (cm != null) {
            cm.setCachedLocations(locations);
        }
        return locations;
    }

    public ExportedBlockKeys getBlockKeys() {
        return this.isBlockTokenEnabled() ? this.blockTokenSecretManager.exportKeys() : ExportedBlockKeys.DUMMY_KEYS;
    }

    public void setBlockToken(LocatedBlock b, BlockTokenIdentifier.AccessMode mode) throws IOException {
        if (this.isBlockTokenEnabled()) {
            b.setBlockToken(this.blockTokenSecretManager.generateToken(NameNode.getRemoteUser().getShortUserName(), b.getBlock(), EnumSet.of(mode)));
        }
    }

    void addKeyUpdateCommand(List<DatanodeCommand> cmds, DatanodeDescriptor nodeinfo) {
        if (this.isBlockTokenEnabled() && nodeinfo.needKeyUpdate()) {
            cmds.add(new KeyUpdateCommand(this.blockTokenSecretManager.exportKeys()));
            nodeinfo.setNeedKeyUpdate(false);
        }
    }

    public DataEncryptionKey generateDataEncryptionKey() {
        if (this.isBlockTokenEnabled() && this.encryptDataTransfer) {
            return this.blockTokenSecretManager.generateDataEncryptionKey();
        }
        return null;
    }

    public short adjustReplication(short replication) {
        return replication < this.minReplication ? this.minReplication : (replication > this.maxReplication ? this.maxReplication : replication);
    }

    public void verifyReplication(String src, short replication, String clientName) throws IOException {
        String err = null;
        if (replication > this.maxReplication) {
            err = " exceeds maximum of " + this.maxReplication;
        } else if (replication < this.minReplication) {
            err = " is less than the required minimum of " + this.minReplication;
        }
        if (err != null) {
            throw new IOException("Requested replication factor of " + replication + err + " for " + src + (clientName == null ? "" : ", clientName=" + clientName));
        }
    }

    public boolean isSufficientlyReplicated(BlockInfo b) {
        int replication = Math.min(this.minReplication, this.getDatanodeManager().getNumLiveDataNodes());
        return this.countNodes(b).liveReplicas() >= replication;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public BlocksWithLocations getBlocks(DatanodeID datanode, long size) throws IOException {
        this.namesystem.checkOperation(NameNode.OperationCategory.READ);
        this.namesystem.readLock();
        try {
            this.namesystem.checkOperation(NameNode.OperationCategory.READ);
            BlocksWithLocations blocksWithLocations = this.getBlocksWithLocations(datanode, size);
            return blocksWithLocations;
        }
        finally {
            this.namesystem.readUnlock();
        }
    }

    private BlocksWithLocations getBlocksWithLocations(DatanodeID datanode, long size) throws UnregisteredNodeException {
        BlockInfo curBlock;
        DatanodeDescriptor node = this.getDatanodeManager().getDatanode(datanode);
        if (node == null) {
            blockLog.warn("BLOCK* getBlocks: Asking for blocks from an unrecorded node {}", (Object)datanode);
            throw new HadoopIllegalArgumentException("Datanode " + datanode + " not found.");
        }
        int numBlocks = node.numBlocks();
        if (numBlocks == 0) {
            return new BlocksWithLocations(new BlocksWithLocations.BlockWithLocations[0]);
        }
        int startBlock = ThreadLocalRandom.current().nextInt(numBlocks);
        Iterator<BlockInfo> iter = node.getBlockIterator(startBlock);
        ArrayList<BlocksWithLocations.BlockWithLocations> results = new ArrayList<BlocksWithLocations.BlockWithLocations>();
        long totalSize = 0L;
        while (totalSize < size && iter.hasNext()) {
            curBlock = iter.next();
            if (!curBlock.isComplete() || curBlock.getNumBytes() < this.getBlocksMinBlockSize) continue;
            totalSize += this.addBlock(curBlock, results);
        }
        if (totalSize < size) {
            iter = node.getBlockIterator();
            for (int i = 0; i < startBlock && totalSize < size; ++i) {
                curBlock = iter.next();
                if (!curBlock.isComplete() || curBlock.getNumBytes() < this.getBlocksMinBlockSize) continue;
                totalSize += this.addBlock(curBlock, results);
            }
        }
        return new BlocksWithLocations(results.toArray(new BlocksWithLocations.BlockWithLocations[results.size()]));
    }

    void removeBlocksAssociatedTo(DatanodeDescriptor node) {
        Iterator<BlockInfo> it = node.getBlockIterator();
        while (it.hasNext()) {
            this.removeStoredBlock(it.next(), node);
        }
        this.pendingDNMessages.removeAllMessagesForDatanode(node);
        node.resetBlocks();
        this.invalidateBlocks.remove(node);
    }

    void removeBlocksAssociatedTo(DatanodeStorageInfo storageInfo) {
        assert (this.namesystem.hasWriteLock());
        Iterator<BlockInfo> it = storageInfo.getBlockIterator();
        DatanodeDescriptor node = storageInfo.getDatanodeDescriptor();
        while (it.hasNext()) {
            Block block = it.next();
            this.removeStoredBlock(block, node);
            this.invalidateBlocks.remove(node, block);
        }
        this.namesystem.checkSafeMode();
        LOG.info("Removed blocks associated with storage {} from DataNode {}", (Object)storageInfo, (Object)node);
    }

    void addToInvalidates(Block block, DatanodeInfo datanode) {
        if (!this.isPopulatingReplQueues()) {
            return;
        }
        this.invalidateBlocks.add(block, datanode, true);
    }

    private void addToInvalidates(Block b) {
        if (!this.isPopulatingReplQueues()) {
            return;
        }
        StringBuilder datanodes = new StringBuilder();
        for (DatanodeStorageInfo storage : this.blocksMap.getStorages(b)) {
            if (storage.getState() != DatanodeStorage.State.NORMAL) continue;
            DatanodeDescriptor node = storage.getDatanodeDescriptor();
            this.invalidateBlocks.add(b, node, false);
            datanodes.append((Object)node).append(" ");
        }
        if (datanodes.length() != 0) {
            blockLog.debug("BLOCK* addToInvalidates: {} {}", (Object)b, (Object)datanodes);
        }
    }

    void removeFromInvalidates(DatanodeInfo datanode) {
        if (!this.isPopulatingReplQueues()) {
            return;
        }
        this.invalidateBlocks.remove(datanode);
    }

    public void findAndMarkBlockAsCorrupt(ExtendedBlock blk, DatanodeInfo dn, String storageID, String reason) throws IOException {
        assert (this.namesystem.hasWriteLock());
        BlockInfo storedBlock = this.getStoredBlock(blk.getLocalBlock());
        if (storedBlock == null) {
            blockLog.debug("BLOCK* findAndMarkBlockAsCorrupt: {} not found", (Object)blk);
            return;
        }
        DatanodeDescriptor node = this.getDatanodeManager().getDatanode((DatanodeID)dn);
        if (node == null) {
            throw new IOException("Cannot mark " + blk + " as corrupt because datanode " + dn + " (" + dn.getDatanodeUuid() + ") does not exist");
        }
        DatanodeStorageInfo storage = null;
        if (storageID != null) {
            storage = node.getStorageInfo(storageID);
        }
        if (storage == null) {
            storage = storedBlock.findStorageInfo(node);
        }
        if (storage == null) {
            blockLog.debug("BLOCK* findAndMarkBlockAsCorrupt: {} not found on {}", (Object)blk, (Object)dn);
            return;
        }
        this.markBlockAsCorrupt(new BlockToMarkCorrupt(storedBlock, blk.getGenerationStamp(), reason, CorruptReplicasMap.Reason.CORRUPTION_REPORTED), storage, node);
    }

    private void markBlockAsCorrupt(BlockToMarkCorrupt b, DatanodeStorageInfo storageInfo, DatanodeDescriptor node) throws IOException {
        boolean corruptedDuringWrite;
        if (b.getCorrupted().isDeleted()) {
            blockLog.debug("BLOCK markBlockAsCorrupt: {} cannot be marked as corrupt as it does not belong to any file", (Object)b);
            this.addToInvalidates(b.getCorrupted(), node);
            return;
        }
        short expectedReplicas = b.getCorrupted().getReplication();
        if (storageInfo != null) {
            storageInfo.addBlock(b.getStored());
        }
        this.corruptReplicas.addToCorruptReplicasMap(b.getCorrupted(), node, b.getReason(), b.getReasonCode());
        NumberReplicas numberOfReplicas = this.countNodes(b.getStored());
        boolean hasEnoughLiveReplicas = numberOfReplicas.liveReplicas() >= expectedReplicas;
        boolean minReplicationSatisfied = numberOfReplicas.liveReplicas() >= this.minReplication;
        boolean hasMoreCorruptReplicas = minReplicationSatisfied && numberOfReplicas.liveReplicas() + numberOfReplicas.corruptReplicas() > expectedReplicas;
        boolean bl = corruptedDuringWrite = minReplicationSatisfied && b.isCorruptedDuringWrite();
        if (hasEnoughLiveReplicas || hasMoreCorruptReplicas || corruptedDuringWrite) {
            this.invalidateBlock(b, node);
        } else if (this.isPopulatingReplQueues()) {
            this.updateNeededReplications(b.getStored(), -1, 0);
        }
    }

    private boolean invalidateBlock(BlockToMarkCorrupt b, DatanodeInfo dn) throws IOException {
        blockLog.debug("BLOCK* invalidateBlock: {} on {}", (Object)b, (Object)dn);
        DatanodeDescriptor node = this.getDatanodeManager().getDatanode((DatanodeID)dn);
        if (node == null) {
            throw new IOException("Cannot invalidate " + b + " because datanode " + dn + " does not exist.");
        }
        NumberReplicas nr = this.countNodes(b.getStored());
        if (nr.replicasOnStaleNodes() > 0) {
            blockLog.debug("BLOCK* invalidateBlocks: postponing invalidation of {} on {} because {} replica(s) are located on nodes with potentially out-of-date block reports", new Object[]{b, dn, nr.replicasOnStaleNodes()});
            this.postponeBlock(b.getCorrupted());
            return false;
        }
        if (nr.liveReplicas() >= 1) {
            this.addToInvalidates(b.getCorrupted(), dn);
            this.removeStoredBlock(b.getStored(), node);
            blockLog.debug("BLOCK* invalidateBlocks: {} on {} listed for deletion.", (Object)b, (Object)dn);
            return true;
        }
        blockLog.debug("BLOCK* invalidateBlocks: {} on {} is the only copy and was not deleted", (Object)b, (Object)dn);
        return false;
    }

    public void setPostponeBlocksFromFuture(boolean postpone) {
        this.shouldPostponeBlocksFromFuture = postpone;
    }

    private void postponeBlock(Block blk) {
        this.postponedMisreplicatedBlocks.add(blk);
    }

    void updateState() {
        this.pendingReplicationBlocksCount = this.pendingReplications.size();
        this.underReplicatedBlocksCount = this.neededReplications.size();
        this.corruptReplicaBlocksCount = this.corruptReplicas.size();
    }

    public int getUnderReplicatedNotMissingBlocks() {
        return this.neededReplications.getUnderReplicatedBlockCount();
    }

    int computeInvalidateWork(int nodesToProcess) {
        List<DatanodeInfo> nodes = this.invalidateBlocks.getDatanodes();
        Collections.shuffle(nodes);
        nodesToProcess = Math.min(nodes.size(), nodesToProcess);
        int blockCnt = 0;
        for (DatanodeInfo dnInfo : nodes) {
            int blocks = this.invalidateWorkForOneNode(dnInfo);
            if (blocks <= 0) continue;
            blockCnt += blocks;
            if (--nodesToProcess != 0) continue;
            break;
        }
        return blockCnt;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int computeReplicationWork(int blocksToProcess) {
        List<List<BlockInfo>> blocksToReplicate = null;
        this.namesystem.writeLock();
        try {
            blocksToReplicate = this.neededReplications.chooseUnderReplicatedBlocks(blocksToProcess);
        }
        finally {
            this.namesystem.writeUnlock();
        }
        return this.computeReplicationWorkForBlocks(blocksToReplicate);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @VisibleForTesting
    int computeReplicationWorkForBlocks(List<List<BlockInfo>> blocksToReplicate) {
        DatanodeStorageInfo[] targets;
        int scheduledWork = 0;
        LinkedList<ReplicationWork> work = new LinkedList<ReplicationWork>();
        this.namesystem.writeLock();
        try {
            UnderReplicatedBlocks underReplicatedBlocks = this.neededReplications;
            synchronized (underReplicatedBlocks) {
                for (int priority = 0; priority < blocksToReplicate.size(); ++priority) {
                    for (BlockInfo block : blocksToReplicate.get(priority)) {
                        ReplicationWork rw = this.scheduleReplication(block, priority);
                        if (rw == null) continue;
                        work.add(rw);
                    }
                }
            }
        }
        finally {
            this.namesystem.writeUnlock();
        }
        HashSet<Node> excludedNodes = new HashSet<Node>();
        for (ReplicationWork rw : work) {
            excludedNodes.clear();
            for (Object dn : rw.getContainingNodes()) {
                excludedNodes.add((Node)dn);
            }
            rw.chooseTargets(this.blockplacement, this.storagePolicySuite, excludedNodes);
        }
        this.namesystem.writeLock();
        try {
            for (ReplicationWork rw : work) {
                Object dn;
                targets = rw.getTargets();
                if (targets == null || targets.length == 0) {
                    rw.resetTargets();
                    continue;
                }
                dn = this.neededReplications;
                synchronized (dn) {
                    if (this.validateReplicationWork(rw)) {
                        ++scheduledWork;
                    }
                }
            }
        }
        finally {
            this.namesystem.writeUnlock();
        }
        if (blockLog.isInfoEnabled()) {
            for (ReplicationWork rw : work) {
                targets = rw.getTargets();
                if (targets == null || targets.length == 0) continue;
                StringBuilder targetList = new StringBuilder("datanode(s)");
                for (DatanodeStorageInfo target : targets) {
                    targetList.append(' ');
                    targetList.append((Object)target.getDatanodeDescriptor());
                }
                blockLog.debug("BLOCK* ask {} to replicate {} to {}", new Object[]{rw.getSrcNode(), rw.getBlock(), targetList});
            }
        }
        if (blockLog.isDebugEnabled()) {
            blockLog.debug("BLOCK* neededReplications = {} pendingReplications = {}", (Object)this.neededReplications.size(), (Object)this.pendingReplications.size());
        }
        return scheduledWork;
    }

    boolean hasEnoughEffectiveReplicas(BlockInfo block, NumberReplicas numReplicas, int pendingReplicaNum, int required) {
        int numEffectiveReplicas = numReplicas.liveReplicas() + pendingReplicaNum;
        return numEffectiveReplicas >= required && (pendingReplicaNum > 0 || this.isPlacementPolicySatisfied(block));
    }

    private ReplicationWork scheduleReplication(BlockInfo block, int priority) {
        if (block.isDeleted() || !block.isCompleteOrCommitted()) {
            this.neededReplications.remove(block, priority);
            return null;
        }
        short requiredReplication = this.getExpectedReplicaNum(block);
        ArrayList<DatanodeDescriptor> containingNodes = new ArrayList<DatanodeDescriptor>();
        ArrayList<DatanodeStorageInfo> liveReplicaNodes = new ArrayList<DatanodeStorageInfo>();
        NumberReplicas numReplicas = new NumberReplicas();
        DatanodeDescriptor srcNode = this.chooseSourceDatanode(block, containingNodes, liveReplicaNodes, numReplicas, priority);
        if (srcNode == null) {
            LOG.debug("Block " + (Object)((Object)block) + " cannot be repl from any node");
            return null;
        }
        assert (liveReplicaNodes.size() >= numReplicas.liveReplicas());
        int pendingNum = this.pendingReplications.getNumReplicas(block);
        if (this.hasEnoughEffectiveReplicas(block, numReplicas, pendingNum, requiredReplication)) {
            this.neededReplications.remove(block, priority);
            blockLog.debug("BLOCK* Removing {} from neededReplications as it has enough replicas", (Object)block);
            return null;
        }
        int additionalReplRequired = numReplicas.liveReplicas() < requiredReplication ? requiredReplication - numReplicas.liveReplicas() - pendingNum : 1;
        BlockCollection bc = this.getBlockCollection(block);
        return new ReplicationWork(block, bc, srcNode, containingNodes, liveReplicaNodes, additionalReplRequired, priority);
    }

    private boolean validateReplicationWork(ReplicationWork rw) {
        int pendingNum;
        BlockInfo block = rw.getBlock();
        int priority = rw.getPriority();
        if (block.isDeleted() || !block.isCompleteOrCommitted()) {
            this.neededReplications.remove(block, priority);
            rw.resetTargets();
            return false;
        }
        short requiredReplication = this.getExpectedReplicaNum(block);
        NumberReplicas numReplicas = this.countNodes(block);
        if (this.hasEnoughEffectiveReplicas(block, numReplicas, pendingNum = this.pendingReplications.getNumReplicas(block), requiredReplication)) {
            this.neededReplications.remove(block, priority);
            rw.resetTargets();
            blockLog.debug("BLOCK* Removing {} from neededReplications as it has enough replicas", (Object)block);
            return false;
        }
        DatanodeStorageInfo[] targets = rw.getTargets();
        if (numReplicas.liveReplicas() >= requiredReplication && !this.isPlacementPolicySatisfied(block) && rw.getSrcNode().getNetworkLocation().equals(targets[0].getDatanodeDescriptor().getNetworkLocation())) {
            return false;
        }
        rw.getSrcNode().addBlockToBeReplicated(block, targets);
        DatanodeStorageInfo.incrementBlocksScheduled(targets);
        this.pendingReplications.increment(block, DatanodeStorageInfo.toDatanodeDescriptors(targets));
        blockLog.debug("BLOCK* block {} is moved from neededReplications to pendingReplications", (Object)block);
        int numEffectiveReplicas = numReplicas.liveReplicas() + pendingNum;
        if (numEffectiveReplicas + targets.length >= requiredReplication) {
            this.neededReplications.remove(block, priority);
        }
        return true;
    }

    public DatanodeStorageInfo[] chooseTarget4WebHDFS(String src, DatanodeDescriptor clientnode, Set<Node> excludes, long blocksize) {
        return this.blockplacement.chooseTarget(src, 1, (Node)clientnode, Collections.emptyList(), false, excludes, blocksize, this.storagePolicySuite.getDefaultPolicy(), null);
    }

    public DatanodeStorageInfo[] chooseTarget4AdditionalDatanode(String src, int numAdditionalNodes, Node clientnode, List<DatanodeStorageInfo> chosen, Set<Node> excludes, long blocksize, byte storagePolicyID) {
        BlockStoragePolicy storagePolicy = this.storagePolicySuite.getPolicy(storagePolicyID);
        return this.blockplacement.chooseTarget(src, numAdditionalNodes, clientnode, chosen, true, excludes, blocksize, storagePolicy, null);
    }

    public DatanodeStorageInfo[] chooseTarget4NewBlock(String src, int numOfReplicas, Node client, Set<Node> excludedNodes, long blocksize, List<String> favoredNodes, byte storagePolicyID, EnumSet<AddBlockFlag> flags) throws IOException {
        BlockStoragePolicy storagePolicy;
        List<DatanodeDescriptor> favoredDatanodeDescriptors = this.getDatanodeDescriptors(favoredNodes);
        DatanodeStorageInfo[] targets = this.blockplacement.chooseTarget(src, numOfReplicas, client, excludedNodes, blocksize, favoredDatanodeDescriptors, storagePolicy = this.storagePolicySuite.getPolicy(storagePolicyID), flags);
        if (targets.length < this.minReplication) {
            throw new IOException("File " + src + " could only be replicated to " + targets.length + " nodes instead of minReplication (=" + this.minReplication + ").  There are " + this.getDatanodeManager().getNetworkTopology().getNumOfLeaves() + " datanode(s) running and " + (excludedNodes == null ? "no" : Integer.valueOf(excludedNodes.size())) + " node(s) are excluded in this operation.");
        }
        return targets;
    }

    List<DatanodeDescriptor> getDatanodeDescriptors(List<String> nodes) {
        ArrayList<DatanodeDescriptor> datanodeDescriptors = null;
        if (nodes != null) {
            datanodeDescriptors = new ArrayList<DatanodeDescriptor>(nodes.size());
            for (int i = 0; i < nodes.size(); ++i) {
                DatanodeDescriptor node = this.datanodeManager.getDatanodeDescriptor(nodes.get(i));
                if (node == null) continue;
                datanodeDescriptors.add(node);
            }
        }
        return datanodeDescriptors;
    }

    @VisibleForTesting
    DatanodeDescriptor chooseSourceDatanode(Block block, List<DatanodeDescriptor> containingNodes, List<DatanodeStorageInfo> nodesContainingLiveReplicas, NumberReplicas numReplicas, int priority) {
        containingNodes.clear();
        nodesContainingLiveReplicas.clear();
        DatanodeDescriptor srcNode = null;
        DatanodeDescriptor decommissionedSrc = null;
        int live = 0;
        int readonly = 0;
        int decommissioned = 0;
        int decommissioning = 0;
        int corrupt = 0;
        int excess = 0;
        Collection<DatanodeDescriptor> nodesCorrupt = this.corruptReplicas.getNodes(block);
        for (DatanodeStorageInfo storage : this.blocksMap.getStorages(block)) {
            int countableReplica;
            DatanodeDescriptor node = storage.getDatanodeDescriptor();
            LightWeightHashSet<Block> excessBlocks = this.excessReplicateMap.get(node.getDatanodeUuid());
            int n = countableReplica = storage.getState() == DatanodeStorage.State.NORMAL ? 1 : 0;
            if (nodesCorrupt != null && nodesCorrupt.contains((Object)node)) {
                corrupt += countableReplica;
            } else if (node.isDecommissionInProgress()) {
                decommissioning += countableReplica;
            } else if (node.isDecommissioned()) {
                decommissioned += countableReplica;
            } else if (excessBlocks != null && excessBlocks.contains(block)) {
                excess += countableReplica;
            } else {
                nodesContainingLiveReplicas.add(storage);
                live += countableReplica;
            }
            if (storage.getState() == DatanodeStorage.State.READ_ONLY_SHARED) {
                ++readonly;
            }
            containingNodes.add(node);
            if (nodesCorrupt != null && nodesCorrupt.contains((Object)node) || priority != 0 && !node.isDecommissionInProgress() && node.getNumberOfBlocksToBeReplicated() >= this.maxReplicationStreams || node.getNumberOfBlocksToBeReplicated() >= this.replicationStreamsHardLimit || excessBlocks != null && excessBlocks.contains(block)) continue;
            if (node.isDecommissioned()) {
                if (decommissionedSrc != null && !ThreadLocalRandom.current().nextBoolean()) continue;
                decommissionedSrc = node;
                continue;
            }
            if (srcNode == null) {
                srcNode = node;
                continue;
            }
            if (!ThreadLocalRandom.current().nextBoolean()) continue;
            srcNode = node;
        }
        if (numReplicas != null) {
            numReplicas.set(live, readonly, decommissioned, decommissioning, corrupt, excess, 0);
        }
        if (live == 0 && srcNode == null && decommissionedSrc != null) {
            return decommissionedSrc;
        }
        return srcNode;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processPendingReplications() {
        BlockInfo[] timedOutItems = this.pendingReplications.getTimedOutBlocks();
        if (timedOutItems != null) {
            this.namesystem.writeLock();
            try {
                for (int i = 0; i < timedOutItems.length; ++i) {
                    NumberReplicas num;
                    BlockInfo bi = this.blocksMap.getStoredBlock(timedOutItems[i]);
                    if (bi == null || !this.isNeededReplication(bi, (num = this.countNodes(timedOutItems[i])).liveReplicas())) continue;
                    this.neededReplications.add(bi, num.liveReplicas(), num.readOnlyReplicas(), num.decommissionedAndDecommissioning(), this.getReplication(bi));
                }
            }
            finally {
                this.namesystem.writeUnlock();
            }
        }
    }

    public long requestBlockReportLeaseId(DatanodeRegistration nodeReg) {
        assert (this.namesystem.hasReadLock());
        DatanodeDescriptor node = null;
        try {
            node = this.datanodeManager.getDatanode(nodeReg);
        }
        catch (UnregisteredNodeException e) {
            LOG.warn("Unregistered datanode {}", (Object)nodeReg);
            return 0L;
        }
        if (node == null) {
            LOG.warn("Failed to find datanode {}", (Object)nodeReg);
            return 0L;
        }
        long leaseId = this.blockReportLeaseManager.requestLease(node);
        BlockManagerFaultInjector.getInstance().requestBlockReportLease(node, leaseId);
        return leaseId;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean processReport(DatanodeID nodeID, DatanodeStorage storage, BlockListAsLongs newReport, BlockReportContext context) throws IOException {
        long endTime;
        DatanodeDescriptor node;
        this.namesystem.writeLock();
        long startTime = Time.monotonicNow();
        Collection<Object> invalidatedBlocks = Collections.emptyList();
        String strBlockReportId = context != null ? Long.toHexString(context.getReportId()) : "";
        try {
            node = this.datanodeManager.getDatanode(nodeID);
            if (node == null || !node.isRegistered()) {
                throw new IOException("ProcessReport from dead or unregistered node: " + nodeID);
            }
            DatanodeStorageInfo storageInfo = node.getStorageInfo(storage.getStorageID());
            if (storageInfo == null) {
                storageInfo = node.updateStorage(storage);
            }
            if (this.namesystem.isInStartupSafeMode() && storageInfo.getBlockReportCount() > 0) {
                blockLog.info("BLOCK* processReport 0x{}: discarded non-initial block report from {} because namenode still in startup phase", (Object)strBlockReportId, (Object)nodeID);
                this.blockReportLeaseManager.removeLease(node);
                boolean bl = !node.hasStaleStorages();
                return bl;
            }
            if (context != null && !this.blockReportLeaseManager.checkLease(node, startTime, context.getLeaseId())) {
                boolean bl = false;
                return bl;
            }
            if (storageInfo.getBlockReportCount() == 0) {
                blockLog.info("BLOCK* processReport 0x{}: Processing first storage report for {} from datanode {}", new Object[]{strBlockReportId, storageInfo.getStorageID(), nodeID.getDatanodeUuid()});
                this.processFirstBlockReport(storageInfo, newReport);
            } else {
                invalidatedBlocks = this.processReport(storageInfo, newReport, context);
            }
            storageInfo.receivedBlockReport();
        }
        finally {
            long endTime2 = Time.monotonicNow();
            this.namesystem.writeUnlock();
        }
        for (Block block : invalidatedBlocks) {
            blockLog.debug("BLOCK* processReport 0x{}: {} on node {} size {} does not belong to any file", new Object[]{strBlockReportId, block, node, block.getNumBytes()});
        }
        NameNodeMetrics metrics = NameNode.getNameNodeMetrics();
        if (metrics != null) {
            metrics.addBlockReport((int)(endTime - startTime));
        }
        blockLog.info("BLOCK* processReport 0x{}: from storage {} node {}, blocks: {}, hasStaleStorage: {}, processing time: {} msecs, invalidatedBlocks: {}", new Object[]{strBlockReportId, storage.getStorageID(), nodeID, newReport.getNumberOfBlocks(), node.hasStaleStorages(), endTime - startTime, invalidatedBlocks.size()});
        return !node.hasStaleStorages();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeBRLeaseIfNeeded(DatanodeID nodeID, BlockReportContext context) throws IOException {
        this.namesystem.writeLock();
        try {
            DatanodeDescriptor node = this.datanodeManager.getDatanode(nodeID);
            if (context != null) {
                if (context.getTotalRpcs() == context.getCurRpc() + 1) {
                    long leaseId = this.getBlockReportLeaseManager().removeLease(node);
                    BlockManagerFaultInjector.getInstance().removeBlockReportLease(node, leaseId);
                }
                LOG.debug("Processing RPC with index {} out of total {} RPCs in processReport 0x{}", new Object[]{context.getCurRpc(), context.getTotalRpcs(), Long.toHexString(context.getReportId())});
            }
        }
        finally {
            this.namesystem.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void rescanPostponedMisreplicatedBlocks() {
        if (this.getPostponedMisreplicatedBlocksCount() == 0L) {
            return;
        }
        this.namesystem.writeLock();
        long startTime = Time.monotonicNow();
        long startSize = this.postponedMisreplicatedBlocks.size();
        try {
            Iterator it = this.postponedMisreplicatedBlocks.iterator();
            for (int i = 0; i < this.blocksPerPostpondedRescan && it.hasNext(); ++i) {
                Block b = (Block)it.next();
                it.remove();
                BlockInfo bi = this.blocksMap.getStoredBlock(b);
                if (bi == null) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug("BLOCK* rescanPostponedMisreplicatedBlocks: Postponed mis-replicated block " + b + " no longer found " + "in block map.");
                    continue;
                }
                MisReplicationResult res = this.processMisReplicatedBlock(bi);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("BLOCK* rescanPostponedMisreplicatedBlocks: Re-scanned block " + b + ", result is " + (Object)((Object)res));
                }
                if (res != MisReplicationResult.POSTPONE) continue;
                this.rescannedMisreplicatedBlocks.add(b);
            }
        }
        finally {
            this.postponedMisreplicatedBlocks.addAll(this.rescannedMisreplicatedBlocks);
            this.rescannedMisreplicatedBlocks.clear();
            long endSize = this.postponedMisreplicatedBlocks.size();
            this.namesystem.writeUnlock();
            LOG.info("Rescan of postponedMisreplicatedBlocks completed in " + (Time.monotonicNow() - startTime) + " msecs. " + endSize + " blocks are left. " + (startSize - endSize) + " blocks were removed.");
        }
    }

    private Collection<Block> processReport(DatanodeStorageInfo storageInfo, BlockListAsLongs report, BlockReportContext context) throws IOException {
        LinkedList<BlockInfo> toAdd = new LinkedList<BlockInfo>();
        TreeSet<Block> toRemove = new TreeSet<Block>();
        LinkedList<Block> toInvalidate = new LinkedList<Block>();
        LinkedList<BlockToMarkCorrupt> toCorrupt = new LinkedList<BlockToMarkCorrupt>();
        LinkedList<StatefulBlockInfo> toUC = new LinkedList<StatefulBlockInfo>();
        this.reportDiff(storageInfo, report, toAdd, toRemove, toInvalidate, toCorrupt, toUC);
        String strBlockReportId = "";
        if (context != null) {
            strBlockReportId = Long.toHexString(context.getReportId());
        }
        DatanodeDescriptor node = storageInfo.getDatanodeDescriptor();
        for (StatefulBlockInfo statefulBlockInfo : toUC) {
            this.addStoredBlockUnderConstruction(statefulBlockInfo, storageInfo);
        }
        for (Block block : toRemove) {
            this.removeStoredBlock(block, node);
        }
        int numBlocksLogged = 0;
        for (BlockInfo blockInfo : toAdd) {
            this.addStoredBlock(blockInfo, storageInfo, null, (long)numBlocksLogged < this.maxNumBlocksToLog);
            ++numBlocksLogged;
        }
        if ((long)numBlocksLogged > this.maxNumBlocksToLog) {
            blockLog.info("BLOCK* processReport 0x{}: logged info for {} of {} reported.", new Object[]{strBlockReportId, this.maxNumBlocksToLog, numBlocksLogged});
        }
        for (Block block : toInvalidate) {
            this.addToInvalidates(block, node);
        }
        for (BlockToMarkCorrupt blockToMarkCorrupt : toCorrupt) {
            this.markBlockAsCorrupt(blockToMarkCorrupt, storageInfo, node);
        }
        return toInvalidate;
    }

    public void markBlockReplicasAsCorrupt(BlockInfo block, long oldGenerationStamp, long oldNumBytes, DatanodeStorageInfo[] newStorages) throws IOException {
        assert (this.namesystem.hasWriteLock());
        BlockToMarkCorrupt b = null;
        if (block.getGenerationStamp() != oldGenerationStamp) {
            b = new BlockToMarkCorrupt(block, oldGenerationStamp, "genstamp does not match " + oldGenerationStamp + " : " + block.getGenerationStamp(), CorruptReplicasMap.Reason.GENSTAMP_MISMATCH);
        } else if (block.getNumBytes() != oldNumBytes) {
            b = new BlockToMarkCorrupt(block, "length does not match " + oldNumBytes + " : " + block.getNumBytes(), CorruptReplicasMap.Reason.SIZE_MISMATCH);
        } else {
            return;
        }
        for (DatanodeStorageInfo storage : this.getStorages(block)) {
            boolean isCorrupt = true;
            if (newStorages != null) {
                for (DatanodeStorageInfo newStorage : newStorages) {
                    if (newStorage == null || !storage.equals(newStorage)) continue;
                    isCorrupt = false;
                    break;
                }
            }
            if (!isCorrupt) continue;
            blockLog.debug("BLOCK* markBlockReplicasAsCorrupt: mark block replica {} on {} as corrupt because the dn is not in the new committed storage list.", (Object)b, (Object)storage.getDatanodeDescriptor());
            this.markBlockAsCorrupt(b, storage, storage.getDatanodeDescriptor());
        }
    }

    private void processFirstBlockReport(DatanodeStorageInfo storageInfo, BlockListAsLongs report) throws IOException {
        if (report == null) {
            return;
        }
        assert (this.namesystem.hasWriteLock());
        assert (storageInfo.getBlockReportCount() == 0);
        for (BlockListAsLongs.BlockReportReplica iblk : report) {
            HdfsServerConstants.ReplicaState reportedState = iblk.getState();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Initial report of block " + iblk.getBlockName() + " on " + (Object)((Object)storageInfo.getDatanodeDescriptor()) + " size " + iblk.getNumBytes() + " replicaState = " + (Object)((Object)reportedState));
            }
            if (this.shouldPostponeBlocksFromFuture && this.namesystem.isGenStampInFuture(iblk)) {
                this.queueReportedBlock(storageInfo, iblk, reportedState, QUEUE_REASON_FUTURE_GENSTAMP);
                continue;
            }
            BlockInfo storedBlock = this.getStoredBlock(iblk);
            if (storedBlock == null) {
                if (!this.namesystem.isInStartupSafeMode() || this.shouldPostponeBlocksFromFuture || this.inRollBack || !this.namesystem.isGenStampInFuture(iblk)) continue;
                this.numberOfBytesInFutureBlocks.addAndGet(iblk.getBytesOnDisk());
                continue;
            }
            HdfsServerConstants.BlockUCState ucState = storedBlock.getBlockUCState();
            BlockToMarkCorrupt c = this.checkReplicaCorrupt(iblk, reportedState, storedBlock, ucState, storageInfo.getDatanodeDescriptor());
            if (c != null) {
                if (this.shouldPostponeBlocksFromFuture) {
                    this.queueReportedBlock(storageInfo, iblk, reportedState, QUEUE_REASON_CORRUPT_STATE);
                    continue;
                }
                this.markBlockAsCorrupt(c, storageInfo, storageInfo.getDatanodeDescriptor());
                continue;
            }
            if (this.isBlockUnderConstruction(storedBlock, ucState, reportedState)) {
                storedBlock.getUnderConstructionFeature().addReplicaIfNotPresent(storageInfo, iblk, reportedState);
                if (this.namesystem.isInSnapshot(storedBlock.getBlockCollectionId())) {
                    int numOfReplicas = storedBlock.getUnderConstructionFeature().getNumExpectedLocations();
                    this.namesystem.incrementSafeBlockCount(numOfReplicas);
                }
            }
            if (reportedState != HdfsServerConstants.ReplicaState.FINALIZED) continue;
            this.addStoredBlockImmediate(storedBlock, storageInfo);
        }
    }

    private void reportDiff(DatanodeStorageInfo storageInfo, BlockListAsLongs newReport, Collection<BlockInfo> toAdd, Collection<Block> toRemove, Collection<Block> toInvalidate, Collection<BlockToMarkCorrupt> toCorrupt, Collection<StatefulBlockInfo> toUC) {
        BlockInfoContiguous delimiter = new BlockInfoContiguous(new Block(), 1);
        DatanodeStorageInfo.AddBlockResult result = storageInfo.addBlock(delimiter);
        assert (result == DatanodeStorageInfo.AddBlockResult.ADDED) : "Delimiting block cannot be present in the node";
        int headIndex = 0;
        if (newReport == null) {
            newReport = BlockListAsLongs.EMPTY;
        }
        for (BlockListAsLongs.BlockReportReplica iblk : newReport) {
            int curIndex;
            HdfsServerConstants.ReplicaState iState;
            BlockInfo storedBlock = this.processReportedBlock(storageInfo, iblk, iState = iblk.getState(), toAdd, toInvalidate, toCorrupt, toUC);
            if (storedBlock == null || (curIndex = storedBlock.findStorageInfo(storageInfo)) < 0) continue;
            headIndex = storageInfo.moveBlockToHead(storedBlock, curIndex, headIndex);
        }
        DatanodeStorageInfo datanodeStorageInfo = storageInfo;
        datanodeStorageInfo.getClass();
        DatanodeStorageInfo.BlockIterator it = new DatanodeStorageInfo.BlockIterator(datanodeStorageInfo, delimiter.getNext(0));
        while (it.hasNext()) {
            toRemove.add((Block)it.next());
        }
        storageInfo.removeBlock(delimiter);
    }

    private BlockInfo processReportedBlock(DatanodeStorageInfo storageInfo, Block block, HdfsServerConstants.ReplicaState reportedState, Collection<BlockInfo> toAdd, Collection<Block> toInvalidate, Collection<BlockToMarkCorrupt> toCorrupt, Collection<StatefulBlockInfo> toUC) {
        DatanodeDescriptor dn = storageInfo.getDatanodeDescriptor();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Reported block " + block + " on " + (Object)((Object)dn) + " size " + block.getNumBytes() + " replicaState = " + (Object)((Object)reportedState));
        }
        if (this.shouldPostponeBlocksFromFuture && this.namesystem.isGenStampInFuture(block)) {
            this.queueReportedBlock(storageInfo, block, reportedState, QUEUE_REASON_FUTURE_GENSTAMP);
            return null;
        }
        BlockInfo storedBlock = this.blocksMap.getStoredBlock(block);
        if (storedBlock == null) {
            toInvalidate.add(new Block(block));
            return null;
        }
        HdfsServerConstants.BlockUCState ucState = storedBlock.getBlockUCState();
        if (LOG.isDebugEnabled()) {
            LOG.debug("In memory blockUCState = " + (Object)((Object)ucState));
        }
        if (this.invalidateBlocks.contains(dn, block)) {
            return storedBlock;
        }
        BlockToMarkCorrupt c = this.checkReplicaCorrupt(block, reportedState, storedBlock, ucState, dn);
        if (c != null) {
            if (this.shouldPostponeBlocksFromFuture) {
                this.queueReportedBlock(storageInfo, storedBlock, reportedState, QUEUE_REASON_CORRUPT_STATE);
            } else {
                toCorrupt.add(c);
            }
            return storedBlock;
        }
        if (this.isBlockUnderConstruction(storedBlock, ucState, reportedState)) {
            toUC.add(new StatefulBlockInfo(storedBlock, new Block(block), reportedState));
            return storedBlock;
        }
        if (reportedState == HdfsServerConstants.ReplicaState.FINALIZED && (storedBlock.findStorageInfo(storageInfo) == -1 || this.corruptReplicas.isReplicaCorrupt(storedBlock, dn))) {
            toAdd.add(storedBlock);
        }
        return storedBlock;
    }

    private void queueReportedBlock(DatanodeStorageInfo storageInfo, Block block, HdfsServerConstants.ReplicaState reportedState, String reason) {
        assert (this.shouldPostponeBlocksFromFuture);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Queueing reported block " + block + " in state " + (Object)((Object)reportedState) + " from datanode " + (Object)((Object)storageInfo.getDatanodeDescriptor()) + " for later processing because " + reason + ".");
        }
        this.pendingDNMessages.enqueueReportedBlock(storageInfo, block, reportedState);
    }

    public void processQueuedMessagesForBlock(Block b) throws IOException {
        Queue<PendingDataNodeMessages.ReportedBlockInfo> queue = this.pendingDNMessages.takeBlockQueue(b);
        if (queue == null) {
            return;
        }
        this.processQueuedMessages(queue);
    }

    private void processQueuedMessages(Iterable<PendingDataNodeMessages.ReportedBlockInfo> rbis) throws IOException {
        for (PendingDataNodeMessages.ReportedBlockInfo rbi : rbis) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Processing previouly queued message " + rbi);
            }
            if (rbi.getReportedState() == null) {
                DatanodeStorageInfo storageInfo = rbi.getStorageInfo();
                this.removeStoredBlock(rbi.getBlock(), storageInfo.getDatanodeDescriptor());
                continue;
            }
            this.processAndHandleReportedBlock(rbi.getStorageInfo(), rbi.getBlock(), rbi.getReportedState(), null);
        }
    }

    public void processAllPendingDNMessages() throws IOException {
        assert (!this.shouldPostponeBlocksFromFuture) : "processAllPendingDNMessages() should be called after disabling block postponement.";
        int count = this.pendingDNMessages.count();
        if (count > 0) {
            LOG.info("Processing " + count + " messages from DataNodes " + "that were previously queued during standby state");
        }
        this.processQueuedMessages(this.pendingDNMessages.takeAll());
        assert (this.pendingDNMessages.count() == 0);
    }

    private BlockToMarkCorrupt checkReplicaCorrupt(Block reported, HdfsServerConstants.ReplicaState reportedState, BlockInfo storedBlock, HdfsServerConstants.BlockUCState ucState, DatanodeDescriptor dn) {
        switch (reportedState) {
            case FINALIZED: {
                switch (ucState) {
                    case COMPLETE: 
                    case COMMITTED: {
                        if (storedBlock.getGenerationStamp() != reported.getGenerationStamp()) {
                            long reportedGS = reported.getGenerationStamp();
                            return new BlockToMarkCorrupt(storedBlock, reportedGS, "block is " + (Object)((Object)ucState) + " and reported genstamp " + reportedGS + " does not match genstamp in block map " + storedBlock.getGenerationStamp(), CorruptReplicasMap.Reason.GENSTAMP_MISMATCH);
                        }
                        if (storedBlock.getNumBytes() != reported.getNumBytes()) {
                            return new BlockToMarkCorrupt(storedBlock, "block is " + (Object)((Object)ucState) + " and reported length " + reported.getNumBytes() + " does not match " + "length in block map " + storedBlock.getNumBytes(), CorruptReplicasMap.Reason.SIZE_MISMATCH);
                        }
                        return null;
                    }
                    case UNDER_CONSTRUCTION: {
                        if (storedBlock.getGenerationStamp() > reported.getGenerationStamp()) {
                            long reportedGS = reported.getGenerationStamp();
                            return new BlockToMarkCorrupt(storedBlock, reportedGS, "block is " + (Object)((Object)ucState) + " and reported state " + (Object)((Object)reportedState) + ", But reported genstamp " + reportedGS + " does not match genstamp in block map " + storedBlock.getGenerationStamp(), CorruptReplicasMap.Reason.GENSTAMP_MISMATCH);
                        }
                        return null;
                    }
                }
                return null;
            }
            case RBW: 
            case RWR: {
                if (!storedBlock.isComplete()) {
                    return null;
                }
                if (storedBlock.getGenerationStamp() != reported.getGenerationStamp()) {
                    long reportedGS = reported.getGenerationStamp();
                    return new BlockToMarkCorrupt(storedBlock, reportedGS, "reported " + (Object)((Object)reportedState) + " replica with genstamp " + reportedGS + " does not match COMPLETE block's genstamp in block map " + storedBlock.getGenerationStamp(), CorruptReplicasMap.Reason.GENSTAMP_MISMATCH);
                }
                if (reportedState == HdfsServerConstants.ReplicaState.RBW) {
                    LOG.info("Received an RBW replica for " + (Object)((Object)storedBlock) + " on " + (Object)((Object)dn) + ": ignoring it, since it is " + "complete with the same genstamp");
                    return null;
                }
                return new BlockToMarkCorrupt(storedBlock, "reported replica has invalid state " + (Object)((Object)reportedState), CorruptReplicasMap.Reason.INVALID_STATE);
            }
        }
        String msg = "Unexpected replica state " + (Object)((Object)reportedState) + " for block: " + (Object)((Object)storedBlock) + " on " + (Object)((Object)dn) + " size " + storedBlock.getNumBytes();
        LOG.warn(msg);
        return new BlockToMarkCorrupt(storedBlock, msg, CorruptReplicasMap.Reason.INVALID_STATE);
    }

    private boolean isBlockUnderConstruction(BlockInfo storedBlock, HdfsServerConstants.BlockUCState ucState, HdfsServerConstants.ReplicaState reportedState) {
        switch (reportedState) {
            case FINALIZED: {
                switch (ucState) {
                    case UNDER_CONSTRUCTION: 
                    case UNDER_RECOVERY: {
                        return true;
                    }
                }
                return false;
            }
            case RBW: 
            case RWR: {
                return !storedBlock.isComplete();
            }
        }
        return false;
    }

    void addStoredBlockUnderConstruction(StatefulBlockInfo ucBlock, DatanodeStorageInfo storageInfo) throws IOException {
        BlockInfo block = ucBlock.storedBlock;
        block.getUnderConstructionFeature().addReplicaIfNotPresent(storageInfo, ucBlock.reportedBlock, ucBlock.reportedState);
        if (ucBlock.reportedState == HdfsServerConstants.ReplicaState.FINALIZED && block.findStorageInfo(storageInfo) < 0) {
            this.addStoredBlock(block, storageInfo, null, true);
        }
    }

    private void addStoredBlockImmediate(BlockInfo storedBlock, DatanodeStorageInfo storageInfo) throws IOException {
        assert (storedBlock != null && this.namesystem.hasWriteLock());
        if (!this.namesystem.isInStartupSafeMode() || this.isPopulatingReplQueues()) {
            this.addStoredBlock(storedBlock, storageInfo, null, false);
            return;
        }
        DatanodeStorageInfo.AddBlockResult result = storageInfo.addBlock(storedBlock);
        int numCurrentReplica = this.countLiveNodes(storedBlock);
        if (storedBlock.getBlockUCState() == HdfsServerConstants.BlockUCState.COMMITTED && numCurrentReplica >= this.minReplication) {
            this.completeBlock(storedBlock, null, false);
        } else if (storedBlock.isComplete() && result == DatanodeStorageInfo.AddBlockResult.ADDED) {
            this.namesystem.incrementSafeBlockCount(numCurrentReplica);
        }
    }

    private Block addStoredBlock(BlockInfo block, DatanodeStorageInfo storageInfo, DatanodeDescriptor delNodeHint, boolean logEveryBlock) throws IOException {
        int curReplicaDelta;
        assert (block != null && this.namesystem.hasWriteLock());
        DatanodeDescriptor node = storageInfo.getDatanodeDescriptor();
        BlockInfo storedBlock = !block.isComplete() ? this.blocksMap.getStoredBlock(block) : block;
        if (storedBlock == null || storedBlock.isDeleted()) {
            blockLog.debug("BLOCK* addStoredBlock: {} on {} size {} but it does not belong to any file", new Object[]{block, node, block.getNumBytes()});
            return block;
        }
        DatanodeStorageInfo.AddBlockResult result = storageInfo.addBlock(storedBlock);
        if (result == DatanodeStorageInfo.AddBlockResult.ADDED) {
            int n = curReplicaDelta = node.isDecommissioned() ? 0 : 1;
            if (logEveryBlock) {
                blockLog.debug("BLOCK* addStoredBlock: {} is added to {} (size={})", new Object[]{node, storedBlock, storedBlock.getNumBytes()});
            }
        } else if (result == DatanodeStorageInfo.AddBlockResult.REPLACED) {
            curReplicaDelta = 0;
            blockLog.warn("BLOCK* addStoredBlock: block {} moved to storageType {} on node {}", new Object[]{storedBlock, storageInfo.getStorageType(), node});
        } else {
            this.corruptReplicas.removeFromCorruptReplicasMap(block, node, CorruptReplicasMap.Reason.GENSTAMP_MISMATCH);
            curReplicaDelta = 0;
            blockLog.debug("BLOCK* addStoredBlock: Redundant addStoredBlock request received for {} on node {} size {}", new Object[]{storedBlock, node, storedBlock.getNumBytes()});
        }
        NumberReplicas num = this.countNodes(storedBlock);
        int numLiveReplicas = num.liveReplicas();
        int numCurrentReplica = numLiveReplicas + this.pendingReplications.getNumReplicas(storedBlock);
        if (storedBlock.getBlockUCState() == HdfsServerConstants.BlockUCState.COMMITTED && numLiveReplicas >= this.minReplication) {
            this.addExpectedReplicasToPending(storedBlock);
            this.completeBlock(storedBlock, null, false);
        } else if (storedBlock.isComplete() && result == DatanodeStorageInfo.AddBlockResult.ADDED) {
            this.namesystem.incrementSafeBlockCount(numCurrentReplica);
        }
        if (!storedBlock.isCompleteOrCommitted()) {
            return storedBlock;
        }
        if (!this.isPopulatingReplQueues()) {
            return storedBlock;
        }
        short fileReplication = this.getExpectedReplicaNum(storedBlock);
        if (!this.isNeededReplication(storedBlock, numCurrentReplica)) {
            this.neededReplications.remove(storedBlock, numCurrentReplica, num.readOnlyReplicas(), num.decommissionedAndDecommissioning(), fileReplication);
        } else {
            this.updateNeededReplications(storedBlock, curReplicaDelta, 0);
        }
        if (numCurrentReplica > fileReplication) {
            this.processOverReplicatedBlock(storedBlock, fileReplication, node, delNodeHint);
        }
        int corruptReplicasCount = this.corruptReplicas.numCorruptReplicas(storedBlock);
        int numCorruptNodes = num.corruptReplicas();
        if (numCorruptNodes != corruptReplicasCount) {
            LOG.warn("Inconsistent number of corrupt replicas for " + (Object)((Object)storedBlock) + "blockMap has " + numCorruptNodes + " but corrupt replicas map has " + corruptReplicasCount);
        }
        if (corruptReplicasCount > 0 && numLiveReplicas >= fileReplication) {
            this.invalidateCorruptReplicas(storedBlock);
        }
        return storedBlock;
    }

    private void invalidateCorruptReplicas(BlockInfo blk) {
        DatanodeDescriptor[] nodesCopy;
        Collection<DatanodeDescriptor> nodes = this.corruptReplicas.getNodes(blk);
        boolean removedFromBlocksMap = true;
        if (nodes == null) {
            return;
        }
        for (DatanodeDescriptor node : nodesCopy = nodes.toArray(new DatanodeDescriptor[0])) {
            try {
                if (this.invalidateBlock(new BlockToMarkCorrupt(blk, null, CorruptReplicasMap.Reason.ANY), node)) continue;
                removedFromBlocksMap = false;
            }
            catch (IOException e) {
                blockLog.debug("invalidateCorruptReplicas error in deleting bad block {} on {}", new Object[]{blk, node, e});
                removedFromBlocksMap = false;
            }
        }
        if (removedFromBlocksMap) {
            this.corruptReplicas.removeFromCorruptReplicasMap(blk);
        }
    }

    public void processMisReplicatedBlocks() {
        assert (this.namesystem.hasWriteLock());
        this.stopReplicationInitializer();
        this.neededReplications.clear();
        this.replicationQueuesInitializer = new Daemon(){

            public void run() {
                try {
                    BlockManager.this.processMisReplicatesAsync();
                }
                catch (InterruptedException ie) {
                    LOG.info("Interrupted while processing replication queues.");
                }
                catch (Exception e) {
                    LOG.error("Error while processing replication queues async", (Throwable)e);
                }
            }
        };
        this.replicationQueuesInitializer.setName("Replication Queue Initializer");
        this.replicationQueuesInitializer.start();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void stopReplicationInitializer() {
        if (this.replicationQueuesInitializer != null) {
            this.replicationQueuesInitializer.interrupt();
            try {
                this.replicationQueuesInitializer.join();
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupted while waiting for replicationQueueInitializer. Returning..");
                return;
            }
            finally {
                this.replicationQueuesInitializer = null;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processMisReplicatesAsync() throws InterruptedException {
        long nrInvalid = 0L;
        long nrOverReplicated = 0L;
        long nrUnderReplicated = 0L;
        long nrPostponed = 0L;
        long nrUnderConstruction = 0L;
        long startTimeMisReplicatedScan = Time.monotonicNow();
        Iterator<BlockInfo> blocksItr = this.blocksMap.getBlocks().iterator();
        long totalBlocks = this.blocksMap.size();
        this.replicationQueuesInitProgress = 0.0;
        long totalProcessed = 0L;
        long sleepDuration = Math.max(1, Math.min(this.numBlocksPerIteration / 1000, 10000));
        while (this.namesystem.isRunning() && !Thread.currentThread().isInterrupted()) {
            this.namesystem.writeLockInterruptibly();
            try {
                int processed;
                block12: for (processed = 0; processed < this.numBlocksPerIteration && blocksItr.hasNext(); ++processed) {
                    BlockInfo block = blocksItr.next();
                    MisReplicationResult res = this.processMisReplicatedBlock(block);
                    switch (res) {
                        case UNDER_REPLICATED: {
                            LOG.trace("under replicated block {}: {}", (Object)block, (Object)res);
                            ++nrUnderReplicated;
                            continue block12;
                        }
                        case OVER_REPLICATED: {
                            LOG.trace("over replicated block {}: {}", (Object)block, (Object)res);
                            ++nrOverReplicated;
                            continue block12;
                        }
                        case INVALID: {
                            LOG.trace("invalid block {}: {}", (Object)block, (Object)res);
                            ++nrInvalid;
                            continue block12;
                        }
                        case POSTPONE: {
                            LOG.trace("postpone block {}: {}", (Object)block, (Object)res);
                            ++nrPostponed;
                            this.postponeBlock(block);
                            continue block12;
                        }
                        case UNDER_CONSTRUCTION: {
                            LOG.trace("under construction block {}: {}", (Object)block, (Object)res);
                            ++nrUnderConstruction;
                            continue block12;
                        }
                        case OK: {
                            continue block12;
                        }
                        default: {
                            throw new AssertionError((Object)("Invalid enum value: " + (Object)((Object)res)));
                        }
                    }
                }
                this.replicationQueuesInitProgress = Math.min((double)(totalProcessed += (long)processed) / (double)totalBlocks, 1.0);
                if (blocksItr.hasNext()) continue;
                LOG.info("Total number of blocks            = " + this.blocksMap.size());
                LOG.info("Number of invalid blocks          = " + nrInvalid);
                LOG.info("Number of under-replicated blocks = " + nrUnderReplicated);
                LOG.info("Number of  over-replicated blocks = " + nrOverReplicated + (nrPostponed > 0L ? " (" + nrPostponed + " postponed)" : ""));
                LOG.info("Number of blocks being written    = " + nrUnderConstruction);
                NameNode.stateChangeLog.info("STATE* Replication Queue initialization scan for invalid, over- and under-replicated blocks completed in " + (Time.monotonicNow() - startTimeMisReplicatedScan) + " msec");
                break;
            }
            finally {
                this.namesystem.writeUnlock();
                Thread.sleep(sleepDuration);
            }
        }
        if (Thread.currentThread().isInterrupted()) {
            LOG.info("Interrupted while processing replication queues.");
        }
    }

    public double getReplicationQueuesInitProgress() {
        return this.replicationQueuesInitProgress;
    }

    private MisReplicationResult processMisReplicatedBlock(BlockInfo block) {
        if (block.isDeleted()) {
            this.addToInvalidates(block);
            return MisReplicationResult.INVALID;
        }
        if (!block.isComplete()) {
            return MisReplicationResult.UNDER_CONSTRUCTION;
        }
        short expectedReplication = this.getExpectedReplicaNum(block);
        NumberReplicas num = this.countNodes(block);
        int numCurrentReplica = num.liveReplicas();
        if (this.isNeededReplication(block, numCurrentReplica) && this.neededReplications.add(block, numCurrentReplica, num.readOnlyReplicas(), num.decommissionedAndDecommissioning(), expectedReplication)) {
            return MisReplicationResult.UNDER_REPLICATED;
        }
        if (numCurrentReplica > expectedReplication) {
            if (num.replicasOnStaleNodes() > 0) {
                return MisReplicationResult.POSTPONE;
            }
            this.processOverReplicatedBlock(block, expectedReplication, null, null);
            return MisReplicationResult.OVER_REPLICATED;
        }
        return MisReplicationResult.OK;
    }

    public void setReplication(short oldRepl, short newRepl, BlockInfo b) {
        if (newRepl == oldRepl) {
            return;
        }
        b.setReplication(newRepl);
        this.updateNeededReplications(b, 0, newRepl - oldRepl);
        if (oldRepl > newRepl) {
            this.processOverReplicatedBlock(b, newRepl, null, null);
        }
    }

    private void processOverReplicatedBlock(BlockInfo block, short replication, DatanodeDescriptor addedNode, DatanodeDescriptor delNodeHint) {
        assert (this.namesystem.hasWriteLock());
        if (addedNode == delNodeHint) {
            delNodeHint = null;
        }
        ArrayList<DatanodeStorageInfo> nonExcess = new ArrayList<DatanodeStorageInfo>();
        Collection<DatanodeDescriptor> corruptNodes = this.corruptReplicas.getNodes(block);
        for (DatanodeStorageInfo storage : this.blocksMap.getStorages(block)) {
            if (storage.getState() != DatanodeStorage.State.NORMAL) continue;
            DatanodeDescriptor cur = storage.getDatanodeDescriptor();
            if (storage.areBlockContentsStale()) {
                LOG.trace("BLOCK* processOverReplicatedBlock: Postponing {} since storage {} does not yet have up-to-date information.", (Object)block, (Object)storage);
                this.postponeBlock(block);
                return;
            }
            LightWeightHashSet<Block> excessBlocks = this.excessReplicateMap.get(cur.getDatanodeUuid());
            if (excessBlocks != null && excessBlocks.contains((Object)block) || cur.isDecommissionInProgress() || cur.isDecommissioned() || corruptNodes != null && corruptNodes.contains((Object)cur)) continue;
            nonExcess.add(storage);
        }
        this.chooseExcessReplicates(nonExcess, block, replication, addedNode, delNodeHint);
    }

    private void chooseExcessReplicates(Collection<DatanodeStorageInfo> nonExcess, BlockInfo storedBlock, short replication, DatanodeDescriptor addedNode, DatanodeDescriptor delNodeHint) {
        assert (this.namesystem.hasWriteLock());
        BlockCollection bc = this.getBlockCollection(storedBlock);
        BlockStoragePolicy storagePolicy = this.storagePolicySuite.getPolicy(bc.getStoragePolicyID());
        List excessTypes = storagePolicy.chooseExcess(replication, DatanodeStorageInfo.toStorageTypes(nonExcess));
        List<DatanodeStorageInfo> replicasToDelete = this.blockplacement.chooseReplicasToDelete(nonExcess, replication, excessTypes, addedNode, delNodeHint);
        for (DatanodeStorageInfo choosenReplica : replicasToDelete) {
            this.processChosenExcessReplica(nonExcess, choosenReplica, storedBlock);
        }
    }

    private void processChosenExcessReplica(Collection<DatanodeStorageInfo> nonExcess, DatanodeStorageInfo chosen, BlockInfo storedBlock) {
        nonExcess.remove(chosen);
        this.addToExcessReplicate(chosen.getDatanodeDescriptor(), storedBlock);
        this.addToInvalidates(storedBlock, chosen.getDatanodeDescriptor());
        blockLog.debug("BLOCK* chooseExcessReplicates: ({}, {}) is added to invalidated blocks set", (Object)chosen, (Object)storedBlock);
    }

    private void addToExcessReplicate(DatanodeInfo dn, Block block) {
        assert (this.namesystem.hasWriteLock());
        LightWeightHashSet<Object> excessBlocks = this.excessReplicateMap.get(dn.getDatanodeUuid());
        if (excessBlocks == null) {
            excessBlocks = new LightWeightHashSet();
            this.excessReplicateMap.put(dn.getDatanodeUuid(), excessBlocks);
        }
        if (excessBlocks.add(block)) {
            this.excessBlocksCount.incrementAndGet();
            blockLog.debug("BLOCK* addToExcessReplicate: ({}, {}) is added to excessReplicateMap", (Object)dn, (Object)block);
        }
    }

    private void removeStoredBlock(DatanodeStorageInfo storageInfo, Block block, DatanodeDescriptor node) {
        if (this.shouldPostponeBlocksFromFuture && this.namesystem.isGenStampInFuture(block)) {
            this.queueReportedBlock(storageInfo, block, null, QUEUE_REASON_FUTURE_GENSTAMP);
            return;
        }
        this.removeStoredBlock(block, node);
    }

    public void removeStoredBlock(Block block, DatanodeDescriptor node) {
        LightWeightHashSet<Block> excessBlocks;
        blockLog.debug("BLOCK* removeStoredBlock: {} from {}", (Object)block, (Object)node);
        assert (this.namesystem.hasWriteLock());
        BlockInfo storedBlock = this.getStoredBlock(block);
        if (storedBlock == null || !this.blocksMap.removeNode(storedBlock, node)) {
            blockLog.debug("BLOCK* removeStoredBlock: {} has already been removed from node {}", (Object)block, (Object)node);
            return;
        }
        CachedBlock cblock = (CachedBlock)this.namesystem.getCacheManager().getCachedBlocks().get((Object)new CachedBlock(block.getBlockId(), 0, false));
        if (cblock != null) {
            boolean removed = false;
            removed |= node.getPendingCached().remove(cblock);
            removed |= node.getCached().remove(cblock);
            if (removed |= node.getPendingUncached().remove(cblock)) {
                blockLog.debug("BLOCK* removeStoredBlock: {} removed from caching related lists on node {}", (Object)block, (Object)node);
            }
        }
        if (!storedBlock.isDeleted()) {
            this.namesystem.decrementSafeBlockCount(storedBlock);
            this.updateNeededReplications(storedBlock, -1, 0);
        }
        if ((excessBlocks = this.excessReplicateMap.get(node.getDatanodeUuid())) != null && excessBlocks.remove(block)) {
            this.excessBlocksCount.decrementAndGet();
            blockLog.debug("BLOCK* removeStoredBlock: {} is removed from excessBlocks", (Object)block);
            if (excessBlocks.size() == 0) {
                this.excessReplicateMap.remove(node.getDatanodeUuid());
            }
        }
        this.corruptReplicas.removeFromCorruptReplicasMap(block, node);
    }

    private void removeStaleReplicas(List<ReplicaUnderConstruction> staleReplicas, BlockInfo block) {
        for (ReplicaUnderConstruction r : staleReplicas) {
            this.removeStoredBlock(block, r.getExpectedStorageLocation().getDatanodeDescriptor());
            NameNode.blockStateChangeLog.debug("BLOCK* Removing stale replica {} of {}", (Object)r, (Object)Block.toString((Block)r));
        }
    }

    private long addBlock(Block block, List<BlocksWithLocations.BlockWithLocations> results) {
        List<DatanodeStorageInfo> locations = this.getValidLocations(block);
        if (locations.size() == 0) {
            return 0L;
        }
        String[] datanodeUuids = new String[locations.size()];
        String[] storageIDs = new String[datanodeUuids.length];
        StorageType[] storageTypes = new StorageType[datanodeUuids.length];
        for (int i = 0; i < locations.size(); ++i) {
            DatanodeStorageInfo s = locations.get(i);
            datanodeUuids[i] = s.getDatanodeDescriptor().getDatanodeUuid();
            storageIDs[i] = s.getStorageID();
            storageTypes[i] = s.getStorageType();
        }
        results.add(new BlocksWithLocations.BlockWithLocations(block, datanodeUuids, storageIDs, storageTypes));
        return block.getNumBytes();
    }

    @VisibleForTesting
    void addBlock(DatanodeStorageInfo storageInfo, Block block, String delHint) throws IOException {
        BlockInfo storedBlock;
        DatanodeDescriptor node = storageInfo.getDatanodeDescriptor();
        node.decrementBlocksScheduled(storageInfo.getStorageType());
        DatanodeDescriptor delHintNode = null;
        if (delHint != null && delHint.length() != 0 && (delHintNode = this.datanodeManager.getDatanode(delHint)) == null) {
            blockLog.warn("BLOCK* blockReceived: {} is expected to be removed from an unrecorded node {}", (Object)block, (Object)delHint);
        }
        if ((storedBlock = this.getStoredBlock(block)) != null && block.getGenerationStamp() == storedBlock.getGenerationStamp()) {
            this.pendingReplications.decrement(storedBlock, node);
        }
        this.processAndHandleReportedBlock(storageInfo, block, HdfsServerConstants.ReplicaState.FINALIZED, delHintNode);
    }

    private void processAndHandleReportedBlock(DatanodeStorageInfo storageInfo, Block block, HdfsServerConstants.ReplicaState reportedState, DatanodeDescriptor delHintNode) throws IOException {
        LinkedList<BlockInfo> toAdd = new LinkedList<BlockInfo>();
        LinkedList<Block> toInvalidate = new LinkedList<Block>();
        LinkedList<BlockToMarkCorrupt> toCorrupt = new LinkedList<BlockToMarkCorrupt>();
        LinkedList<StatefulBlockInfo> toUC = new LinkedList<StatefulBlockInfo>();
        DatanodeDescriptor node = storageInfo.getDatanodeDescriptor();
        this.processReportedBlock(storageInfo, block, reportedState, toAdd, toInvalidate, toCorrupt, toUC);
        assert (toUC.size() + toAdd.size() + toInvalidate.size() + toCorrupt.size() <= 1) : "The block should be only in one of the lists.";
        for (StatefulBlockInfo b : toUC) {
            this.addStoredBlockUnderConstruction(b, storageInfo);
        }
        long numBlocksLogged = 0L;
        for (BlockInfo blockInfo : toAdd) {
            this.addStoredBlock(blockInfo, storageInfo, delHintNode, numBlocksLogged < this.maxNumBlocksToLog);
            ++numBlocksLogged;
        }
        if (numBlocksLogged > this.maxNumBlocksToLog) {
            blockLog.debug("BLOCK* addBlock: logged info for {} of {} reported.", (Object)this.maxNumBlocksToLog, (Object)numBlocksLogged);
        }
        for (Block block2 : toInvalidate) {
            blockLog.debug("BLOCK* addBlock: block {} on node {} size {} does not belong to any file", new Object[]{block2, node, block2.getNumBytes()});
            this.addToInvalidates(block2, node);
        }
        for (BlockToMarkCorrupt blockToMarkCorrupt : toCorrupt) {
            this.markBlockAsCorrupt(blockToMarkCorrupt, storageInfo, node);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void processIncrementalBlockReport(DatanodeID nodeID, StorageReceivedDeletedBlocks srdb) throws IOException {
        assert (this.namesystem.hasWriteLock());
        DatanodeDescriptor node = this.datanodeManager.getDatanode(nodeID);
        if (node == null || !node.isRegistered()) {
            blockLog.warn("BLOCK* processIncrementalBlockReport is received from dead or unregistered node {}", (Object)nodeID);
            throw new IOException("Got incremental block report from unregistered or dead node");
        }
        boolean successful = false;
        try {
            this.processIncrementalBlockReport(node, srdb);
            successful = true;
        }
        finally {
            if (!successful) {
                node.setForceRegistration(true);
            }
        }
    }

    private void processIncrementalBlockReport(DatanodeDescriptor node, StorageReceivedDeletedBlocks srdb) throws IOException {
        DatanodeStorageInfo storageInfo = node.getStorageInfo(srdb.getStorage().getStorageID());
        if (storageInfo == null) {
            storageInfo = node.updateStorage(srdb.getStorage());
        }
        int received = 0;
        int deleted = 0;
        int receiving = 0;
        for (ReceivedDeletedBlockInfo rdbi : srdb.getBlocks()) {
            switch (rdbi.getStatus()) {
                case DELETED_BLOCK: {
                    this.removeStoredBlock(storageInfo, rdbi.getBlock(), node);
                    ++deleted;
                    break;
                }
                case RECEIVED_BLOCK: {
                    this.addBlock(storageInfo, rdbi.getBlock(), rdbi.getDelHints());
                    ++received;
                    break;
                }
                case RECEIVING_BLOCK: {
                    ++receiving;
                    this.processAndHandleReportedBlock(storageInfo, rdbi.getBlock(), HdfsServerConstants.ReplicaState.RBW, null);
                    break;
                }
                default: {
                    String msg = "Unknown block status code reported by " + (Object)((Object)node) + ": " + rdbi;
                    blockLog.warn(msg);
                    assert (false) : msg;
                    break;
                }
            }
            blockLog.debug("BLOCK* block {}: {} is received from {}", new Object[]{rdbi.getStatus(), rdbi.getBlock(), node});
        }
        blockLog.debug("*BLOCK* NameNode.processIncrementalBlockReport: from {} receiving: {}, received: {}, deleted: {}", new Object[]{node, receiving, received, deleted});
    }

    public NumberReplicas countNodes(Block b) {
        int decommissioned = 0;
        int decommissioning = 0;
        int live = 0;
        int readonly = 0;
        int corrupt = 0;
        int excess = 0;
        int stale = 0;
        Collection<DatanodeDescriptor> nodesCorrupt = this.corruptReplicas.getNodes(b);
        for (DatanodeStorageInfo storage : this.blocksMap.getStorages(b)) {
            if (storage.getState() == DatanodeStorage.State.FAILED) continue;
            if (storage.getState() == DatanodeStorage.State.READ_ONLY_SHARED) {
                ++readonly;
                continue;
            }
            DatanodeDescriptor node = storage.getDatanodeDescriptor();
            if (nodesCorrupt != null && nodesCorrupt.contains((Object)node)) {
                ++corrupt;
            } else if (node.isDecommissionInProgress()) {
                ++decommissioning;
            } else if (node.isDecommissioned()) {
                ++decommissioned;
            } else {
                LightWeightHashSet<Block> blocksExcess = this.excessReplicateMap.get(node.getDatanodeUuid());
                if (blocksExcess != null && blocksExcess.contains(b)) {
                    ++excess;
                } else {
                    ++live;
                }
            }
            if (!storage.areBlockContentsStale()) continue;
            ++stale;
        }
        return new NumberReplicas(live, readonly, decommissioned, decommissioning, corrupt, excess, stale);
    }

    int countLiveNodes(BlockInfo b) {
        if (!this.namesystem.isInStartupSafeMode()) {
            return this.countNodes(b).liveReplicas();
        }
        int live = 0;
        Collection<DatanodeDescriptor> nodesCorrupt = this.corruptReplicas.getNodes(b);
        for (DatanodeStorageInfo storage : this.blocksMap.getStorages(b)) {
            if (storage.getState() != DatanodeStorage.State.NORMAL) continue;
            DatanodeDescriptor node = storage.getDatanodeDescriptor();
            if (nodesCorrupt != null && nodesCorrupt.contains((Object)node)) continue;
            ++live;
        }
        return live;
    }

    void processOverReplicatedBlocksOnReCommission(DatanodeDescriptor srcNode) {
        if (!this.isPopulatingReplQueues()) {
            return;
        }
        Iterator<BlockInfo> it = srcNode.getBlockIterator();
        int numOverReplicated = 0;
        while (it.hasNext()) {
            BlockInfo block = it.next();
            short expectedReplication = block.getReplication();
            NumberReplicas num = this.countNodes(block);
            int numCurrentReplica = num.liveReplicas();
            if (numCurrentReplica <= expectedReplication) continue;
            this.processOverReplicatedBlock(block, expectedReplication, null, null);
            ++numOverReplicated;
        }
        LOG.info("Invalidated " + numOverReplicated + " over-replicated blocks on " + (Object)((Object)srcNode) + " during recommissioning");
    }

    boolean isNodeHealthyForDecommission(DatanodeDescriptor node) {
        if (!node.checkBlockReportReceived()) {
            LOG.info("Node {} hasn't sent its first block report.", (Object)node);
            return false;
        }
        if (node.isAlive()) {
            return true;
        }
        this.updateState();
        if (this.pendingReplicationBlocksCount == 0L && this.underReplicatedBlocksCount == 0L) {
            LOG.info("Node {} is dead and there are no under-replicated blocks or blocks pending replication. Safe to decommission.", (Object)node);
            return true;
        }
        LOG.warn("Node {} is dead while decommission is in progress. Cannot be safely decommissioned since there is risk of reduced data durability or data loss. Either restart the failed node or force decommissioning by removing, calling refreshNodes, then re-adding to the excludes files.", (Object)node);
        return false;
    }

    public int getActiveBlockCount() {
        return this.blocksMap.size();
    }

    public DatanodeStorageInfo[] getStorages(BlockInfo block) {
        DatanodeStorageInfo[] storages = new DatanodeStorageInfo[block.numNodes()];
        int i = 0;
        for (DatanodeStorageInfo s : this.blocksMap.getStorages(block)) {
            storages[i++] = s;
        }
        return storages;
    }

    public int getTotalBlocks() {
        return this.blocksMap.size();
    }

    public void removeBlock(BlockInfo block) {
        assert (this.namesystem.hasWriteLock());
        block.setNumBytes(Long.MAX_VALUE);
        this.addToInvalidates(block);
        this.removeBlockFromMap(block);
        this.pendingReplications.remove(block);
        this.neededReplications.remove(block, 5);
        this.postponedMisreplicatedBlocks.remove((Object)block);
    }

    public BlockInfo getStoredBlock(Block block) {
        return this.blocksMap.getStoredBlock(block);
    }

    public void updateLastBlock(BlockInfo lastBlock, ExtendedBlock newBlock) {
        lastBlock.setNumBytes(newBlock.getNumBytes());
        List<ReplicaUnderConstruction> staleReplicas = lastBlock.setGenerationStampAndVerifyReplicas(newBlock.getGenerationStamp());
        this.removeStaleReplicas(staleReplicas, lastBlock);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateNeededReplications(BlockInfo block, int curReplicasDelta, int expectedReplicasDelta) {
        this.namesystem.writeLock();
        try {
            int curExpectedReplicas;
            int pendingNum;
            if (!this.isPopulatingReplQueues() || !block.isComplete()) {
                return;
            }
            NumberReplicas repl = this.countNodes(block);
            if (!this.hasEnoughEffectiveReplicas(block, repl, pendingNum = this.pendingReplications.getNumReplicas(block), curExpectedReplicas = this.getReplication(block))) {
                this.neededReplications.update(block, repl.liveReplicas() + pendingNum, repl.readOnlyReplicas(), repl.decommissionedAndDecommissioning(), curExpectedReplicas, curReplicasDelta, expectedReplicasDelta);
            } else {
                int oldReplicas = repl.liveReplicas() + pendingNum - curReplicasDelta;
                int oldExpectedReplicas = curExpectedReplicas - expectedReplicasDelta;
                this.neededReplications.remove(block, oldReplicas, repl.readOnlyReplicas(), repl.decommissionedAndDecommissioning(), oldExpectedReplicas);
            }
        }
        finally {
            this.namesystem.writeUnlock();
        }
    }

    public void checkReplication(BlockCollection bc) {
        for (BlockInfo block : bc.getBlocks()) {
            int pending;
            short expected = block.getReplication();
            NumberReplicas n = this.countNodes(block);
            if (!this.hasEnoughEffectiveReplicas(block, n, pending = this.pendingReplications.getNumReplicas(block), expected)) {
                this.neededReplications.add(block, n.liveReplicas() + pending, n.readOnlyReplicas(), n.decommissionedAndDecommissioning(), expected);
                continue;
            }
            if (n.liveReplicas() <= expected) continue;
            this.processOverReplicatedBlock(block, expected, null, null);
        }
    }

    private int getReplication(BlockInfo block) {
        return this.getExpectedReplicaNum(block);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int invalidateWorkForOneNode(DatanodeInfo dn) {
        List<Block> toInvalidate;
        this.namesystem.writeLock();
        try {
            if (this.namesystem.isInSafeMode()) {
                LOG.debug("In safemode, not computing replication work");
                int n = 0;
                return n;
            }
            DatanodeDescriptor dnDescriptor = this.datanodeManager.getDatanode((DatanodeID)dn);
            if (dnDescriptor == null) {
                LOG.warn("DataNode " + dn + " cannot be found with UUID " + dn.getDatanodeUuid() + ", removing block invalidation work.");
                this.invalidateBlocks.remove(dn);
                int n = 0;
                return n;
            }
            toInvalidate = this.invalidateBlocks.invalidateWork(dnDescriptor);
            if (toInvalidate == null) {
                int n = 0;
                return n;
            }
        }
        finally {
            this.namesystem.writeUnlock();
        }
        blockLog.debug("BLOCK* {}: ask {} to delete {}", new Object[]{this.getClass().getSimpleName(), dn, toInvalidate});
        return toInvalidate.size();
    }

    boolean isPlacementPolicySatisfied(BlockInfo storedBlock) {
        ArrayList<DatanodeDescriptor> liveNodes = new ArrayList<DatanodeDescriptor>();
        Collection<DatanodeDescriptor> corruptNodes = this.corruptReplicas.getNodes(storedBlock);
        for (DatanodeStorageInfo storage : this.blocksMap.getStorages(storedBlock)) {
            DatanodeDescriptor cur = storage.getDatanodeDescriptor();
            if (cur.isDecommissionInProgress() || cur.isDecommissioned() || corruptNodes != null && corruptNodes.contains((Object)cur)) continue;
            liveNodes.add(cur);
        }
        DatanodeInfo[] locs = liveNodes.toArray(new DatanodeInfo[liveNodes.size()]);
        return this.blockplacement.verifyBlockPlacement(locs, storedBlock.getReplication()).isPlacementPolicySatisfied();
    }

    boolean isNeededReplication(BlockInfo storedBlock, int current) {
        short expected = storedBlock.getReplication();
        return storedBlock.isComplete() && (current < expected || !this.isPlacementPolicySatisfied(storedBlock));
    }

    public short getExpectedReplicaNum(BlockInfo block) {
        return block.getReplication();
    }

    public long getMissingBlocksCount() {
        return this.neededReplications.getCorruptBlockSize();
    }

    public long getMissingReplOneBlocksCount() {
        return this.neededReplications.getCorruptReplOneBlockSize();
    }

    public BlockInfo addBlockCollection(BlockInfo block, BlockCollection bc) {
        return this.blocksMap.addBlockCollection(block, bc);
    }

    BlockCollection getBlockCollection(BlockInfo b) {
        return this.namesystem.getBlockCollection(b.getBlockCollectionId());
    }

    public Iterable<DatanodeStorageInfo> getStorages(Block block) {
        return this.blocksMap.getStorages(block);
    }

    public int numCorruptReplicas(Block block) {
        return this.corruptReplicas.numCorruptReplicas(block);
    }

    public void removeBlockFromMap(BlockInfo block) {
        this.removeFromExcessReplicateMap(block);
        this.blocksMap.removeBlock(block);
        this.corruptReplicas.removeFromCorruptReplicasMap(block);
    }

    private void removeFromExcessReplicateMap(BlockInfo block) {
        for (DatanodeStorageInfo info : this.blocksMap.getStorages(block)) {
            String uuid = info.getDatanodeDescriptor().getDatanodeUuid();
            LightWeightHashSet<Block> excessReplicas = this.excessReplicateMap.get(uuid);
            if (excessReplicas == null || !excessReplicas.remove((Object)block)) continue;
            this.excessBlocksCount.decrementAndGet();
            if (!excessReplicas.isEmpty()) continue;
            this.excessReplicateMap.remove(uuid);
        }
    }

    public int getCapacity() {
        return this.blocksMap.getCapacity();
    }

    public Iterator<BlockInfo> getCorruptReplicaBlockIterator() {
        return this.neededReplications.iterator(4);
    }

    public Collection<DatanodeDescriptor> getCorruptReplicas(Block block) {
        return this.corruptReplicas.getNodes(block);
    }

    public String getCorruptReason(Block block, DatanodeDescriptor node) {
        return this.corruptReplicas.getCorruptReason(block, node);
    }

    public int numOfUnderReplicatedBlocks() {
        return this.neededReplications.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int computeDatanodeWork() {
        if (this.namesystem.isInSafeMode()) {
            return 0;
        }
        int numlive = this.heartbeatManager.getLiveDatanodeCount();
        int blocksToProcess = numlive * this.blocksReplWorkMultiplier;
        int nodesToProcess = (int)Math.ceil((float)numlive * this.blocksInvalidateWorkPct);
        int workFound = this.computeReplicationWork(blocksToProcess);
        this.namesystem.writeLock();
        try {
            this.updateState();
            this.scheduledReplicationBlocksCount = workFound;
        }
        finally {
            this.namesystem.writeUnlock();
        }
        return workFound += this.computeInvalidateWork(nodesToProcess);
    }

    public void clearQueues() {
        this.neededReplications.clear();
        this.pendingReplications.clear();
        this.excessReplicateMap.clear();
        this.invalidateBlocks.clear();
        this.datanodeManager.clearPendingQueues();
        this.postponedMisreplicatedBlocks.clear();
    }

    public static LocatedBlock newLocatedBlock(ExtendedBlock b, DatanodeStorageInfo[] storages, long startOffset, boolean corrupt) {
        return new LocatedBlock(b, DatanodeStorageInfo.toDatanodeInfos(storages), DatanodeStorageInfo.toStorageIDs(storages), DatanodeStorageInfo.toStorageTypes(storages), startOffset, corrupt, null);
    }

    public void shutdown() {
        this.stopReplicationInitializer();
        this.blocksMap.close();
        MBeans.unregister((ObjectName)this.mxBeanName);
        this.mxBeanName = null;
    }

    public void clear() {
        this.clearQueues();
        this.blocksMap.clear();
    }

    public BlockReportLeaseManager getBlockReportLeaseManager() {
        return this.blockReportLeaseManager;
    }

    @Override
    public Map<StorageType, StorageTypeStats> getStorageTypeStats() {
        return this.datanodeManager.getDatanodeStatistics().getStorageTypeStats();
    }

    public void initializeReplQueues() {
        LOG.info("initializing replication queues");
        this.processMisReplicatedBlocks();
        this.initializedReplQueues = true;
    }

    public boolean isPopulatingReplQueues() {
        if (!this.shouldPopulateReplQueues()) {
            return false;
        }
        return this.initializedReplQueues;
    }

    public void setInitializedReplQueues(boolean v) {
        this.initializedReplQueues = v;
    }

    public boolean shouldPopulateReplQueues() {
        HAContext haContext = this.namesystem.getHAContext();
        if (haContext == null || haContext.getState() == null) {
            return false;
        }
        return haContext.getState().shouldPopulateReplQueues();
    }

    public long getBytesInFuture() {
        return this.numberOfBytesInFutureBlocks.get();
    }

    public void clearBytesInFuture() {
        this.numberOfBytesInFutureBlocks.set(0L);
    }

    private boolean isInRollBackMode(HdfsServerConstants.StartupOption option) {
        if (option == HdfsServerConstants.StartupOption.ROLLBACK) {
            return true;
        }
        return option == HdfsServerConstants.StartupOption.ROLLINGUPGRADE && option.getRollingUpgradeStartupOption() == HdfsServerConstants.RollingUpgradeStartupOption.ROLLBACK;
    }

    public void enqueueBlockOp(Runnable action) throws IOException {
        try {
            this.blockReportThread.enqueue(action);
        }
        catch (InterruptedException ie) {
            throw new IOException(ie);
        }
    }

    public <T> T runBlockOp(Callable<T> action) throws IOException {
        FutureTask<T> future = new FutureTask<T>(action);
        this.enqueueBlockOp(future);
        try {
            return future.get();
        }
        catch (ExecutionException ee) {
            Throwable cause = ee.getCause();
            if (cause == null) {
                cause = ee;
            }
            if (!(cause instanceof IOException)) {
                cause = new IOException(cause);
            }
            throw (IOException)cause;
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            throw new IOException(ie);
        }
    }

    @VisibleForTesting
    public void flushBlockOps() throws IOException {
        this.runBlockOp(new Callable<Void>(){

            @Override
            public Void call() {
                return null;
            }
        });
    }

    public int getBlockOpQueueLength() {
        return this.blockReportThread.queue.size();
    }

    boolean isReplicaCorrupt(BlockInfo blk, DatanodeDescriptor d) {
        return this.corruptReplicas.isReplicaCorrupt(blk, d);
    }

    private class BlockReportProcessingThread
    extends Thread {
        private static final long MAX_LOCK_HOLD_MS = 4L;
        private long lastFull;
        private final BlockingQueue<Runnable> queue;

        BlockReportProcessingThread() {
            super("Block report processor");
            this.lastFull = 0L;
            this.queue = new ArrayBlockingQueue<Runnable>(1024);
            this.setDaemon(true);
        }

        @Override
        public void run() {
            try {
                this.processQueue();
            }
            catch (Throwable t) {
                ExitUtil.terminate((int)1, (String)(this.getName() + " encountered fatal exception: " + t));
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void processQueue() {
            while (BlockManager.this.namesystem.isRunning()) {
                NameNodeMetrics metrics = NameNode.getNameNodeMetrics();
                try {
                    Runnable action = this.queue.take();
                    int processed = 0;
                    BlockManager.this.namesystem.writeLock();
                    metrics.setBlockOpsQueued(this.queue.size() + 1);
                    try {
                        long start = Time.monotonicNow();
                        do {
                            ++processed;
                            action.run();
                        } while (Time.monotonicNow() - start <= 4L && (action = (Runnable)this.queue.poll()) != null);
                    }
                    finally {
                        BlockManager.this.namesystem.writeUnlock();
                        metrics.addBlockOpsBatched(processed - 1);
                    }
                }
                catch (InterruptedException e) {
                    if (!Thread.interrupted()) continue;
                    break;
                }
            }
            this.queue.clear();
        }

        void enqueue(Runnable action) throws InterruptedException {
            if (!this.queue.offer(action)) {
                long now;
                if (!this.isAlive() && BlockManager.this.namesystem.isRunning()) {
                    ExitUtil.terminate((int)1, (String)(this.getName() + " is not running"));
                }
                if ((now = Time.monotonicNow()) - this.lastFull > 4000L) {
                    this.lastFull = now;
                    LOG.info("Block report queue is full");
                }
                this.queue.put(action);
            }
        }
    }

    static enum MisReplicationResult {
        INVALID,
        UNDER_REPLICATED,
        OVER_REPLICATED,
        POSTPONE,
        UNDER_CONSTRUCTION,
        OK;

    }

    private class ReplicationMonitor
    implements Runnable {
        private ReplicationMonitor() {
        }

        @Override
        public void run() {
            while (BlockManager.this.namesystem.isRunning()) {
                try {
                    if (BlockManager.this.isPopulatingReplQueues()) {
                        BlockManager.this.computeDatanodeWork();
                        BlockManager.this.processPendingReplications();
                        BlockManager.this.rescanPostponedMisreplicatedBlocks();
                    }
                    Thread.sleep(BlockManager.this.replicationRecheckInterval);
                }
                catch (Throwable t) {
                    if (!BlockManager.this.namesystem.isRunning()) {
                        LOG.info("Stopping ReplicationMonitor.");
                        if (t instanceof InterruptedException) break;
                        LOG.info("ReplicationMonitor received an exception while shutting down.", t);
                        break;
                    }
                    if (!BlockManager.this.checkNSRunning && t instanceof InterruptedException) {
                        LOG.info("Stopping ReplicationMonitor for testing.");
                        break;
                    }
                    LOG.error("ReplicationMonitor thread received Runtime exception. ", t);
                    ExitUtil.terminate((int)1, (Throwable)t);
                }
            }
        }
    }

    static class StatefulBlockInfo {
        final BlockInfo storedBlock;
        final Block reportedBlock;
        final HdfsServerConstants.ReplicaState reportedState;

        StatefulBlockInfo(BlockInfo storedBlock, Block reportedBlock, HdfsServerConstants.ReplicaState reportedState) {
            Preconditions.checkArgument((!storedBlock.isComplete() ? 1 : 0) != 0);
            this.storedBlock = storedBlock;
            this.reportedBlock = reportedBlock;
            this.reportedState = reportedState;
        }
    }
}

