/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicy;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicyState;
import org.apache.hadoop.hdfs.protocol.SystemErasureCodingPolicies;
import org.apache.hadoop.io.erasurecode.CodecUtil;
import org.apache.hadoop.io.erasurecode.ECSchema;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.LimitedPrivate(value={"HDFS"})
public final class ErasureCodingPolicyManager {
    public static Logger LOG = LoggerFactory.getLogger(ErasureCodingPolicyManager.class);
    private int maxCellSize = 0x400000;
    private static final byte[] SUITABLE_STORAGE_POLICIES_FOR_EC_STRIPED_MODE = new byte[]{7, 2, 12};
    private Map<String, ErasureCodingPolicy> policiesByName;
    private Map<Byte, ErasureCodingPolicy> policiesByID;
    private ErasureCodingPolicy[] allPolicies;
    private Map<String, ErasureCodingPolicy> enabledPoliciesByName;
    private ErasureCodingPolicy[] enabledPolicies;
    private static volatile ErasureCodingPolicyManager instance = null;

    public static ErasureCodingPolicyManager getInstance() {
        if (instance == null) {
            instance = new ErasureCodingPolicyManager();
        }
        return instance;
    }

    private ErasureCodingPolicyManager() {
    }

    public void init(Configuration conf) {
        String defaultPolicyName = conf.getTrimmed("dfs.namenode.ec.system.default.policy", "RS-6-3-1024k");
        this.policiesByName = new TreeMap<String, ErasureCodingPolicy>();
        this.policiesByID = new TreeMap<Byte, ErasureCodingPolicy>();
        this.enabledPoliciesByName = new TreeMap<String, ErasureCodingPolicy>();
        for (ErasureCodingPolicy policy : SystemErasureCodingPolicies.getPolicies()) {
            this.policiesByName.put(policy.getName(), policy);
            this.policiesByID.put(policy.getId(), policy);
        }
        if (!defaultPolicyName.trim().isEmpty()) {
            ErasureCodingPolicy ecPolicy = this.policiesByName.get(defaultPolicyName);
            if (ecPolicy == null) {
                String names = this.policiesByName.values().stream().map(ErasureCodingPolicy::getName).collect(Collectors.joining(", "));
                String msg = String.format("EC policy '%s' specified at %s is not a valid policy. Please choose from list of available policies: [%s]", defaultPolicyName, "dfs.namenode.ec.system.default.policy", names);
                throw new HadoopIllegalArgumentException(msg);
            }
            this.enabledPoliciesByName.put(ecPolicy.getName(), ecPolicy);
        }
        this.enabledPolicies = this.enabledPoliciesByName.values().toArray(new ErasureCodingPolicy[0]);
        this.allPolicies = this.policiesByName.values().toArray(new ErasureCodingPolicy[0]);
        this.maxCellSize = conf.getInt("dfs.namenode.ec.policies.max.cellsize", 0x400000);
    }

    public ErasureCodingPolicy[] getEnabledPolicies() {
        return this.enabledPolicies;
    }

    public ErasureCodingPolicy getEnabledPolicyByName(String name) {
        ErasureCodingPolicy ecPolicy = this.enabledPoliciesByName.get(name);
        if (ecPolicy == null && name.equalsIgnoreCase("replication")) {
            ecPolicy = SystemErasureCodingPolicies.getReplicationPolicy();
        }
        return ecPolicy;
    }

    public static boolean checkStoragePolicySuitableForECStripedMode(byte storagePolicyID) {
        boolean isPolicySuitable = false;
        for (byte suitablePolicy : SUITABLE_STORAGE_POLICIES_FOR_EC_STRIPED_MODE) {
            if (storagePolicyID != suitablePolicy) continue;
            isPolicySuitable = true;
            break;
        }
        return isPolicySuitable;
    }

    public ErasureCodingPolicy[] getPolicies() {
        return this.allPolicies;
    }

    public ErasureCodingPolicy getByID(byte id) {
        return this.policiesByID.get(id);
    }

    public ErasureCodingPolicy getByName(String name) {
        return this.policiesByName.get(name);
    }

    public void clear() {
    }

    public synchronized ErasureCodingPolicy addPolicy(ErasureCodingPolicy policy) {
        policy.setState(ErasureCodingPolicyState.DISABLED);
        if (!CodecUtil.hasCodec((String)policy.getCodecName())) {
            throw new HadoopIllegalArgumentException("Codec name " + policy.getCodecName() + " is not supported");
        }
        if (policy.getCellSize() > this.maxCellSize) {
            throw new HadoopIllegalArgumentException("Cell size " + policy.getCellSize() + " should not exceed maximum " + this.maxCellSize + " bytes");
        }
        String assignedNewName = ErasureCodingPolicy.composePolicyName((ECSchema)policy.getSchema(), (int)policy.getCellSize());
        for (ErasureCodingPolicy p : this.getPolicies()) {
            if (p.getName().equals(assignedNewName)) {
                LOG.info("The policy name " + assignedNewName + " already exists");
                return p;
            }
            if (!p.getSchema().equals((Object)policy.getSchema()) || p.getCellSize() != policy.getCellSize()) continue;
            LOG.info("A policy with same schema " + policy.getSchema().toString() + " and cell size " + p.getCellSize() + " already exists");
            return p;
        }
        policy.setName(assignedNewName);
        policy.setId(this.getNextAvailablePolicyID());
        this.policiesByName.put(policy.getName(), policy);
        this.policiesByID.put(policy.getId(), policy);
        this.allPolicies = this.policiesByName.values().toArray(new ErasureCodingPolicy[0]);
        return policy;
    }

    private byte getNextAvailablePolicyID() {
        byte currentId = this.policiesByID.keySet().stream().max(Byte::compareTo).filter(id -> id >= 64).orElse((byte)64);
        return (byte)(currentId + 1);
    }

    public synchronized void removePolicy(String name) {
        ErasureCodingPolicy ecPolicy = this.policiesByName.get(name);
        if (ecPolicy == null) {
            throw new HadoopIllegalArgumentException("The policy name " + name + " does not exist");
        }
        if (ecPolicy.isSystemPolicy()) {
            throw new HadoopIllegalArgumentException("System erasure coding policy " + name + " cannot be removed");
        }
        if (this.enabledPoliciesByName.containsKey(name)) {
            this.enabledPoliciesByName.remove(name);
            this.enabledPolicies = this.enabledPoliciesByName.values().toArray(new ErasureCodingPolicy[0]);
        }
        ecPolicy.setState(ErasureCodingPolicyState.REMOVED);
        LOG.info("Remove erasure coding policy " + name);
    }

    @VisibleForTesting
    public List<ErasureCodingPolicy> getRemovedPolicies() {
        ArrayList<ErasureCodingPolicy> removedPolicies = new ArrayList<ErasureCodingPolicy>();
        for (ErasureCodingPolicy ecPolicy : this.policiesByName.values()) {
            if (!ecPolicy.isRemoved()) continue;
            removedPolicies.add(ecPolicy);
        }
        return removedPolicies;
    }

    public synchronized void disablePolicy(String name) {
        ErasureCodingPolicy ecPolicy = this.policiesByName.get(name);
        if (ecPolicy == null) {
            throw new HadoopIllegalArgumentException("The policy name " + name + " does not exist");
        }
        if (this.enabledPoliciesByName.containsKey(name)) {
            this.enabledPoliciesByName.remove(name);
            this.enabledPolicies = this.enabledPoliciesByName.values().toArray(new ErasureCodingPolicy[0]);
        }
        ecPolicy.setState(ErasureCodingPolicyState.DISABLED);
        LOG.info("Disable the erasure coding policy " + name);
    }

    public synchronized void enablePolicy(String name) {
        ErasureCodingPolicy ecPolicy = this.policiesByName.get(name);
        if (ecPolicy == null) {
            throw new HadoopIllegalArgumentException("The policy name " + name + " does not exist");
        }
        this.enabledPoliciesByName.put(name, ecPolicy);
        ecPolicy.setState(ErasureCodingPolicyState.ENABLED);
        this.enabledPolicies = this.enabledPoliciesByName.values().toArray(new ErasureCodingPolicy[0]);
        LOG.info("Enable the erasure coding policy " + name);
    }

    private void loadPolicy(ErasureCodingPolicy policy) {
        if (!CodecUtil.hasCodec((String)policy.getCodecName()) || policy.getCellSize() > this.maxCellSize) {
            policy.setState(ErasureCodingPolicyState.DISABLED);
        }
        this.policiesByName.put(policy.getName(), policy);
        this.policiesByID.put(policy.getId(), policy);
        if (policy.isEnabled()) {
            this.enablePolicy(policy.getName());
        }
    }

    public synchronized void loadPolicies(List<ErasureCodingPolicy> ecPolicies) {
        Preconditions.checkNotNull(ecPolicies);
        for (ErasureCodingPolicy p : ecPolicies) {
            this.loadPolicy(p);
        }
        this.allPolicies = this.policiesByName.values().toArray(new ErasureCodingPolicy[0]);
    }
}

