/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.StripedFileTestUtil;
import org.apache.hadoop.hdfs.client.HdfsAdmin;
import org.apache.hadoop.hdfs.protocol.AddErasureCodingPolicyResponse;
import org.apache.hadoop.hdfs.protocol.DirectoryListing;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicy;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicyInfo;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.HdfsFileStatus;
import org.apache.hadoop.hdfs.protocol.SystemErasureCodingPolicies;
import org.apache.hadoop.hdfs.server.namenode.ErasureCodingPolicyManager;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.io.erasurecode.ECSchema;
import org.apache.hadoop.io.erasurecode.ErasureCodeConstants;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.test.GenericTestUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

public class TestErasureCodingPolicies {
    private Configuration conf;
    private MiniDFSCluster cluster;
    private DistributedFileSystem fs;
    private static final int BLOCK_SIZE = 16384;
    private ErasureCodingPolicy ecPolicy;
    private FSNamesystem namesystem;
    @Rule
    public Timeout timeout = new Timeout(60000);

    public ErasureCodingPolicy getEcPolicy() {
        return StripedFileTestUtil.getDefaultECPolicy();
    }

    @Before
    public void setupCluster() throws IOException {
        this.ecPolicy = this.getEcPolicy();
        this.conf = new HdfsConfiguration();
        this.conf.setInt("dfs.blocksize", 16384);
        this.cluster = new MiniDFSCluster.Builder(this.conf).numDataNodes(this.ecPolicy.getNumDataUnits() + this.ecPolicy.getNumParityUnits()).build();
        this.cluster.waitActive();
        this.fs = this.cluster.getFileSystem();
        this.namesystem = this.cluster.getNamesystem();
        DFSTestUtil.enableAllECPolicies(this.fs);
    }

    @After
    public void shutdownCluster() throws IOException {
        if (this.cluster != null) {
            this.cluster.shutdown();
            this.cluster = null;
        }
    }

    @Test
    public void testReplicatedFileUnderECDir() throws IOException {
        Path dir = new Path("/ec");
        Path replicatedFile = new Path(dir, "replicatedFile");
        DFSTestUtil.createFile((FileSystem)this.fs, replicatedFile, 0L, (short)3, 0L);
        this.fs.setErasureCodingPolicy(dir, this.ecPolicy.getName());
        Path ecSubDir = new Path(dir, "ecSubDir");
        Path ecFile = new Path(ecSubDir, "ecFile");
        DFSTestUtil.createFile((FileSystem)this.fs, ecFile, 0L, (short)1, 0L);
        Assert.assertNull((Object)this.fs.getClient().getFileInfo(replicatedFile.toString()).getErasureCodingPolicy());
        Assert.assertNotNull((Object)this.fs.getClient().getFileInfo(ecFile.toString()).getErasureCodingPolicy());
        DirectoryListing listing = this.fs.getClient().listPaths(dir.toString(), new byte[0], false);
        HdfsFileStatus[] files = listing.getPartialListing();
        Assert.assertEquals((long)2L, (long)files.length);
        Assert.assertEquals((Object)ecSubDir.getName(), (Object)files[0].getLocalName());
        Assert.assertNotNull((Object)files[0].getErasureCodingPolicy());
        Assert.assertEquals((Object)replicatedFile.getName(), (Object)files[1].getLocalName());
        Assert.assertNull((Object)files[1].getErasureCodingPolicy());
        files = this.fs.getClient().listPaths(ecSubDir.toString(), new byte[0], false).getPartialListing();
        Assert.assertEquals((long)1L, (long)files.length);
        Assert.assertEquals((Object)ecFile.getName(), (Object)files[0].getLocalName());
        Assert.assertNotNull((Object)files[0].getErasureCodingPolicy());
        files = this.fs.getClient().listPaths("/", new byte[0], false).getPartialListing();
        Assert.assertEquals((long)1L, (long)files.length);
        Assert.assertEquals((Object)dir.getName(), (Object)files[0].getLocalName());
        Assert.assertNotNull((Object)files[0].getErasureCodingPolicy());
        Assert.assertTrue((boolean)this.fs.rename(ecFile, new Path("/ecFile")));
        files = this.fs.getClient().listPaths("/", new byte[0], false).getPartialListing();
        Assert.assertEquals((long)2L, (long)files.length);
        Assert.assertEquals((Object)dir.getName(), (Object)files[0].getLocalName());
        Assert.assertNotNull((Object)files[0].getErasureCodingPolicy());
        Assert.assertEquals((Object)ecFile.getName(), (Object)files[1].getLocalName());
        Assert.assertNotNull((Object)files[1].getErasureCodingPolicy());
    }

    @Test
    public void testBasicSetECPolicy() throws IOException, InterruptedException {
        Path testDir = new Path("/ec");
        this.fs.mkdir(testDir, FsPermission.getDirDefault());
        this.fs.setErasureCodingPolicy(testDir, this.ecPolicy.getName());
        Path ECFilePath = new Path(testDir, "foo");
        this.fs.create(ECFilePath);
        INode inode = this.namesystem.getFSDirectory().getINode(ECFilePath.toString());
        Assert.assertTrue((boolean)inode.asFile().isStriped());
        Path notEmpty = new Path("/nonEmpty");
        this.fs.mkdir(notEmpty, FsPermission.getDirDefault());
        Path oldFile = new Path(notEmpty, "old");
        this.fs.create(oldFile);
        this.fs.setErasureCodingPolicy(notEmpty, this.ecPolicy.getName());
        Path newFile = new Path(notEmpty, "new");
        this.fs.create(newFile);
        INode oldInode = this.namesystem.getFSDirectory().getINode(oldFile.toString());
        Assert.assertFalse((boolean)oldInode.asFile().isStriped());
        INode newInode = this.namesystem.getFSDirectory().getINode(newFile.toString());
        Assert.assertTrue((boolean)newInode.asFile().isStriped());
        Path dir1 = new Path("/dir1");
        Path dir2 = new Path(dir1, "dir2");
        this.fs.mkdir(dir1, FsPermission.getDirDefault());
        this.fs.setErasureCodingPolicy(dir1, this.ecPolicy.getName());
        this.fs.mkdir(dir2, FsPermission.getDirDefault());
        try {
            this.fs.setErasureCodingPolicy(dir2, this.ecPolicy.getName());
        }
        catch (IOException e) {
            Assert.fail((String)"Nested erasure coding policies are supported");
        }
        Path fPath = new Path("/file");
        this.fs.create(fPath);
        try {
            this.fs.setErasureCodingPolicy(fPath, this.ecPolicy.getName());
            Assert.fail((String)"Erasure coding policy on file");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains((String)"erasure coding policy for a file", (Throwable)e);
        }
        this.cluster.restartNameNodes();
        this.cluster.waitActive();
        Path disabledPolicy = new Path(dir1, "afterDisabled");
        Assert.assertEquals((String)"Dir does not have policy set", (Object)this.ecPolicy, (Object)this.fs.getErasureCodingPolicy(dir1));
        this.fs.create(disabledPolicy).close();
        Assert.assertEquals((String)"File did not inherit dir's policy", (Object)this.ecPolicy, (Object)this.fs.getErasureCodingPolicy(disabledPolicy));
        this.fs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_ENTER);
        this.fs.saveNamespace();
        this.fs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_LEAVE);
        this.cluster.restartNameNodes();
        Assert.assertEquals((String)"Dir does not have policy set", (Object)this.ecPolicy, (Object)this.fs.getErasureCodingPolicy(dir1));
        Assert.assertEquals((String)"File does not have policy set", (Object)this.ecPolicy, (Object)this.fs.getErasureCodingPolicy(disabledPolicy));
    }

    @Test
    public void testMoveValidity() throws IOException, InterruptedException {
        Path srcECDir = new Path("/srcEC");
        Path dstECDir = new Path("/dstEC");
        this.fs.mkdir(srcECDir, FsPermission.getDirDefault());
        this.fs.mkdir(dstECDir, FsPermission.getDirDefault());
        this.fs.setErasureCodingPolicy(srcECDir, this.ecPolicy.getName());
        this.fs.setErasureCodingPolicy(dstECDir, this.ecPolicy.getName());
        Path srcFile = new Path(srcECDir, "foo");
        this.fs.create(srcFile);
        Path newDir = new Path("/srcEC_new");
        this.fs.rename(srcECDir, newDir);
        this.fs.rename(newDir, srcECDir);
        this.fs.rename(srcECDir, dstECDir);
        this.fs.rename(new Path("/dstEC/srcEC"), srcECDir);
        this.fs.rename(srcFile, dstECDir);
        this.fs.rename(new Path(dstECDir, "foo"), srcECDir);
        Path nonECDir = new Path("/nonEC");
        this.fs.mkdir(nonECDir, FsPermission.getDirDefault());
        this.fs.rename(srcFile, nonECDir);
        Path nonECFile = new Path(nonECDir, "nonECFile");
        this.fs.create(nonECFile);
        this.fs.rename(nonECFile, dstECDir);
    }

    @Test
    public void testReplication() throws IOException {
        Path testDir = new Path("/ec");
        this.fs.mkdir(testDir, FsPermission.getDirDefault());
        this.fs.setErasureCodingPolicy(testDir, this.ecPolicy.getName());
        Path fooFile = new Path(testDir, "foo");
        this.fs.create(fooFile, FsPermission.getFileDefault(), true, this.conf.getInt("io.file.buffer.size", 4096), (short)0, this.fs.getDefaultBlockSize(fooFile), null);
        ErasureCodingPolicy policy = this.fs.getErasureCodingPolicy(fooFile);
        this.fs.setReplication(fooFile, (short)3);
        Assert.assertEquals((Object)policy, (Object)this.fs.getErasureCodingPolicy(fooFile));
    }

    @Test
    public void testGetErasureCodingPolicyWithSystemDefaultECPolicy() throws Exception {
        String src = "/ec";
        Path ecDir = new Path(src);
        this.fs.mkdir(ecDir, FsPermission.getDirDefault());
        Assert.assertNull((Object)this.fs.getClient().getFileInfo(src).getErasureCodingPolicy());
        ErasureCodingPolicy sysDefaultECPolicy = StripedFileTestUtil.getDefaultECPolicy();
        this.fs.getClient().setErasureCodingPolicy(src, sysDefaultECPolicy.getName());
        this.verifyErasureCodingInfo(src, sysDefaultECPolicy);
        this.fs.create(new Path(ecDir, "child1")).close();
        this.verifyErasureCodingInfo(src + "/child1", sysDefaultECPolicy);
    }

    @Test
    public void testErasureCodingPolicyOnReservedDir() throws IOException {
        Path reserveDir = new Path("/.reserved");
        ErasureCodingPolicy policy = this.fs.getErasureCodingPolicy(reserveDir);
        Assert.assertNull((String)"Got unexpected erasure coding policy", (Object)policy);
        Path root = new Path("/");
        Path rawRoot = new Path("/.reserved/raw");
        Path rawRootSlash = new Path("/.reserved/raw/");
        Assert.assertNull((String)"Got unexpected erasure coding policy", (Object)this.fs.getErasureCodingPolicy(root));
        Assert.assertNull((String)"Got unexpected erasure coding policy", (Object)this.fs.getErasureCodingPolicy(rawRoot));
        Assert.assertNull((String)"Got unexpected erasure coding policy", (Object)this.fs.getErasureCodingPolicy(rawRootSlash));
        Path ecDir = new Path("/ec");
        this.fs.mkdirs(ecDir);
        this.fs.setErasureCodingPolicy(ecDir, this.ecPolicy.getName());
        ErasureCodingPolicy policyBase = this.fs.getErasureCodingPolicy(ecDir);
        Assert.assertEquals((String)"Got unexpected erasure coding policy", (Object)this.ecPolicy, (Object)policyBase);
        Path rawRootEc = new Path("/.reserved/raw/ec");
        ErasureCodingPolicy policyMap = this.fs.getErasureCodingPolicy(rawRootEc);
        Assert.assertEquals((String)"Got unexpected erasure coding policy", (Object)this.ecPolicy, (Object)policyMap);
    }

    @Test
    public void testGetErasureCodingPolicy() throws Exception {
        List sysECPolicies = SystemErasureCodingPolicies.getPolicies();
        Assert.assertTrue((String)"System ecPolicies should exist", (sysECPolicies.size() > 0 ? 1 : 0) != 0);
        ErasureCodingPolicy usingECPolicy = (ErasureCodingPolicy)sysECPolicies.get(0);
        String src = "/ec2";
        Path ecDir = new Path(src);
        this.fs.mkdir(ecDir, FsPermission.getDirDefault());
        Assert.assertNull((Object)this.fs.getClient().getFileInfo(src).getErasureCodingPolicy());
        this.fs.getClient().setErasureCodingPolicy(src, usingECPolicy.getName());
        this.verifyErasureCodingInfo(src, usingECPolicy);
        this.fs.create(new Path(ecDir, "child1")).close();
        this.verifyErasureCodingInfo(src + "/child1", usingECPolicy);
    }

    private void verifyErasureCodingInfo(String src, ErasureCodingPolicy usingECPolicy) throws IOException {
        HdfsFileStatus hdfsFileStatus = this.fs.getClient().getFileInfo(src);
        ErasureCodingPolicy actualPolicy = hdfsFileStatus.getErasureCodingPolicy();
        Assert.assertNotNull((Object)actualPolicy);
        Assert.assertEquals((String)"Actually used ecPolicy should be equal with target ecPolicy", (Object)usingECPolicy, (Object)actualPolicy);
    }

    @Test
    public void testSetInvalidPolicy() throws IOException {
        ECSchema rsSchema = new ECSchema("rs", 4, 2);
        String policyName = "RS-4-2-128k";
        int cellSize = 131072;
        ErasureCodingPolicy invalidPolicy = new ErasureCodingPolicy(policyName, rsSchema, cellSize, -1);
        String src = "/ecDir4-2";
        Path ecDir = new Path(src);
        try {
            this.fs.mkdir(ecDir, FsPermission.getDirDefault());
            this.fs.getClient().setErasureCodingPolicy(src, invalidPolicy.getName());
            Assert.fail((String)"HadoopIllegalArgumentException should be thrown forsetting an invalid erasure coding policy");
        }
        catch (Exception e) {
            GenericTestUtils.assertExceptionContains((String)"Policy 'RS-4-2-128k' does not match any enabled erasure coding policies", (Throwable)e);
        }
    }

    @Test
    public void testSetDefaultPolicy() throws IOException {
        String src = "/ecDir";
        Path ecDir = new Path(src);
        try {
            this.fs.mkdir(ecDir, FsPermission.getDirDefault());
            this.fs.getClient().setErasureCodingPolicy(src, null);
            String actualECPolicyName = this.fs.getClient().getErasureCodingPolicy(src).getName();
            String expectedECPolicyName = this.conf.get("dfs.namenode.ec.system.default.policy", "RS-6-3-1024k");
            Assert.assertEquals((Object)expectedECPolicyName, (Object)actualECPolicyName);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    @Test
    public void testGetAllErasureCodingPolicies() throws Exception {
        Collection allECPolicies = this.fs.getAllErasureCodingPolicies();
        ArrayList sysPolicies = new ArrayList(SystemErasureCodingPolicies.getPolicies());
        for (ErasureCodingPolicyInfo ecpi : allECPolicies) {
            if (!ecpi.isEnabled()) continue;
            sysPolicies.remove(ecpi.getPolicy());
        }
        Assert.assertTrue((String)"All system policies should be enabled", (boolean)sysPolicies.isEmpty());
        ECSchema toAddSchema = new ECSchema("rs", 5, 2);
        ErasureCodingPolicy newPolicy = new ErasureCodingPolicy(toAddSchema, 131072);
        ErasureCodingPolicy[] policyArray = new ErasureCodingPolicy[]{newPolicy};
        this.fs.addErasureCodingPolicies(policyArray);
        allECPolicies = this.fs.getAllErasureCodingPolicies();
        Assert.assertEquals((String)"Should return new added policy", (long)(SystemErasureCodingPolicies.getPolicies().size() + 1), (long)allECPolicies.size());
    }

    @Test
    public void testGetErasureCodingPolicyOnANonExistentFile() throws Exception {
        Path path = new Path("/ecDir");
        try {
            this.fs.getErasureCodingPolicy(path);
            Assert.fail((String)"FileNotFoundException should be thrown for a non-existent file path");
        }
        catch (FileNotFoundException e) {
            GenericTestUtils.assertExceptionContains((String)("Path not found: " + path), (Throwable)e);
        }
        HdfsAdmin dfsAdmin = new HdfsAdmin(this.cluster.getURI(), this.conf);
        try {
            dfsAdmin.getErasureCodingPolicy(path);
            Assert.fail((String)"FileNotFoundException should be thrown for a non-existent file path");
        }
        catch (FileNotFoundException e) {
            GenericTestUtils.assertExceptionContains((String)("Path not found: " + path), (Throwable)e);
        }
    }

    @Test
    public void testMultiplePoliciesCoExist() throws Exception {
        List sysPolicies = SystemErasureCodingPolicies.getPolicies();
        if (sysPolicies.size() > 1) {
            for (ErasureCodingPolicy policy : sysPolicies) {
                Path dir = new Path("/policy_" + policy.getId());
                this.fs.mkdir(dir, FsPermission.getDefault());
                this.fs.setErasureCodingPolicy(dir, policy.getName());
                Path file = new Path(dir, "child");
                this.fs.create(file).close();
                Assert.assertEquals((Object)policy, (Object)this.fs.getErasureCodingPolicy(file));
                Assert.assertEquals((Object)policy, (Object)this.fs.getErasureCodingPolicy(dir));
                INode iNode = this.namesystem.getFSDirectory().getINode(file.toString());
                Assert.assertEquals((long)policy.getId(), (long)iNode.asFile().getErasureCodingPolicyID());
                Assert.assertEquals((long)1L, (long)iNode.asFile().getFileReplication());
            }
        }
    }

    @Test
    public void testPermissions() throws Exception {
        UserGroupInformation user = UserGroupInformation.createUserForTesting((String)"ecuser", (String[])new String[]{"ecgroup"});
        final FileSystem userfs = (FileSystem)user.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws Exception {
                return FileSystem.get((Configuration)TestErasureCodingPolicies.this.conf);
            }
        });
        HdfsAdmin useradmin = (HdfsAdmin)user.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<HdfsAdmin>(){

            @Override
            public HdfsAdmin run() throws Exception {
                return new HdfsAdmin(userfs.getUri(), TestErasureCodingPolicies.this.conf);
            }
        });
        Path ecdir = new Path("/ecdir");
        Path ecfile = new Path(ecdir, "ecfile");
        this.fs.setPermission(new Path("/"), new FsPermission(511));
        userfs.mkdirs(ecdir);
        String ecPolicyName = this.ecPolicy.getName();
        useradmin.setErasureCodingPolicy(ecdir, ecPolicyName);
        Assert.assertEquals((String)"Policy not present on dir", (Object)ecPolicyName, (Object)useradmin.getErasureCodingPolicy(ecdir).getName());
        userfs.create(ecfile).close();
        Assert.assertEquals((String)"Policy not present on file", (Object)ecPolicyName, (Object)useradmin.getErasureCodingPolicy(ecfile).getName());
        useradmin.unsetErasureCodingPolicy(ecdir);
        useradmin.setErasureCodingPolicy(ecdir, ecPolicyName);
        userfs.setPermission(ecdir, new FsPermission(365));
        try {
            useradmin.setErasureCodingPolicy(ecdir, ecPolicyName);
            Assert.fail((String)"Should not be able to setECPolicy without write permissions");
        }
        catch (AccessControlException accessControlException) {
            // empty catch block
        }
        try {
            useradmin.unsetErasureCodingPolicy(ecdir);
            Assert.fail((String)"Should not be able to unsetECPolicy without write permissions");
        }
        catch (AccessControlException accessControlException) {
            // empty catch block
        }
        userfs.setPermission(ecdir, new FsPermission(416));
        useradmin.unsetErasureCodingPolicy(ecdir);
        useradmin.setErasureCodingPolicy(ecdir, ecPolicyName);
        UserGroupInformation nobody = UserGroupInformation.createUserForTesting((String)"nobody", (String[])new String[]{"nogroup"});
        HdfsAdmin noadmin = (HdfsAdmin)nobody.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<HdfsAdmin>(){

            @Override
            public HdfsAdmin run() throws Exception {
                return new HdfsAdmin(userfs.getUri(), TestErasureCodingPolicies.this.conf);
            }
        });
        try {
            noadmin.setErasureCodingPolicy(ecdir, ecPolicyName);
            Assert.fail((String)"Should not be able to setECPolicy without write permissions");
        }
        catch (AccessControlException accessControlException) {
            // empty catch block
        }
        try {
            noadmin.unsetErasureCodingPolicy(ecdir);
            Assert.fail((String)"Should not be able to unsetECPolicy without write permissions");
        }
        catch (AccessControlException accessControlException) {
            // empty catch block
        }
        try {
            noadmin.getErasureCodingPolicy(ecdir);
            Assert.fail((String)"Should not be able to getECPolicy without write permissions");
        }
        catch (AccessControlException accessControlException) {
            // empty catch block
        }
        userfs.setPermission(ecdir, new FsPermission(0));
        HdfsAdmin superadmin = new HdfsAdmin(this.fs.getUri(), this.conf);
        superadmin.unsetErasureCodingPolicy(ecdir);
        superadmin.setErasureCodingPolicy(ecdir, ecPolicyName);
        superadmin.getErasureCodingPolicy(ecdir);
        try {
            useradmin.getErasureCodingPolicy(ecdir);
            Assert.fail((String)"Normal user should not have access");
        }
        catch (AccessControlException accessControlException) {
            // empty catch block
        }
        try {
            useradmin.setErasureCodingPolicy(ecfile, ecPolicyName);
            Assert.fail((String)"Normal user should not have access");
        }
        catch (AccessControlException accessControlException) {
            // empty catch block
        }
        try {
            useradmin.unsetErasureCodingPolicy(ecfile);
            Assert.fail((String)"Normal user should not have access");
        }
        catch (AccessControlException accessControlException) {
            // empty catch block
        }
        useradmin.getErasureCodingPolicies();
        noadmin.getErasureCodingPolicies();
        superadmin.getErasureCodingPolicies();
    }

    @Test
    public void testFileLevelECPolicy() throws Exception {
        Path dirPath = new Path("/striped");
        Path filePath0 = new Path(dirPath, "file0");
        Path filePath1 = new Path(dirPath, "file1");
        this.fs.mkdirs(dirPath);
        this.fs.setErasureCodingPolicy(dirPath, this.ecPolicy.getName());
        this.fs.createFile(filePath0).build().close();
        ErasureCodingPolicy ecPolicyOnFile = this.fs.getErasureCodingPolicy(filePath0);
        Assert.assertEquals((Object)this.ecPolicy, (Object)ecPolicyOnFile);
        String illegalPolicyName = "RS-DEFAULT-1-2-64k";
        try {
            this.fs.createFile(filePath1).ecPolicyName("RS-DEFAULT-1-2-64k").build().close();
            Assert.fail((String)"illegal erasure coding policy should not be found");
        }
        catch (Exception e) {
            GenericTestUtils.assertExceptionContains((String)"Policy 'RS-DEFAULT-1-2-64k' does not match any enabled erasure coding policies", (Throwable)e);
        }
        this.fs.delete(dirPath, true);
        this.fs.mkdirs(dirPath);
        ErasureCodingPolicy ecPolicyOnDir = SystemErasureCodingPolicies.getByID((byte)2);
        ecPolicyOnFile = SystemErasureCodingPolicies.getByID((byte)1);
        this.fs.setErasureCodingPolicy(dirPath, ecPolicyOnDir.getName());
        this.fs.createFile(filePath0).ecPolicyName(ecPolicyOnFile.getName()).build().close();
        Assert.assertEquals((Object)ecPolicyOnFile, (Object)this.fs.getErasureCodingPolicy(filePath0));
        Assert.assertEquals((Object)ecPolicyOnDir, (Object)this.fs.getErasureCodingPolicy(dirPath));
        this.fs.delete(dirPath, true);
    }

    @Test
    public void testEnforceAsReplicatedFile() throws Exception {
        Path dirPath = new Path("/striped");
        Path filePath = new Path(dirPath, "file");
        this.fs.mkdirs(dirPath);
        this.fs.setErasureCodingPolicy(dirPath, this.ecPolicy.getName());
        String ecPolicyName = null;
        Collection allPoliciesInfo = this.fs.getAllErasureCodingPolicies();
        for (ErasureCodingPolicyInfo info : allPoliciesInfo) {
            if (this.ecPolicy.equals((Object)info.getPolicy())) continue;
            ecPolicyName = info.getPolicy().getName();
            break;
        }
        Assert.assertNotNull(ecPolicyName);
        this.fs.createFile(filePath).build().close();
        Assert.assertEquals((Object)this.ecPolicy, (Object)this.fs.getErasureCodingPolicy(filePath));
        this.fs.delete(filePath, true);
        this.fs.createFile(filePath).ecPolicyName(ecPolicyName).build().close();
        Assert.assertEquals((Object)ecPolicyName, (Object)this.fs.getErasureCodingPolicy(filePath).getName());
        this.fs.delete(filePath, true);
        try {
            this.fs.createFile(filePath).ecPolicyName(ecPolicyName).replicate().build().close();
            Assert.fail((String)"shouldReplicate and ecPolicyName are exclusive parameters. Set both is not allowed.");
        }
        catch (Exception e) {
            GenericTestUtils.assertExceptionContains((String)"SHOULD_REPLICATE flag and ecPolicyName are exclusive parameters.", (Throwable)e);
        }
        try {
            DFSClient dfsClient = this.fs.getClient();
            dfsClient.create(filePath.toString(), null, EnumSet.of(CreateFlag.CREATE, CreateFlag.OVERWRITE, CreateFlag.SHOULD_REPLICATE), false, (short)1, 1024L, null, 1024, null, null, ecPolicyName);
            Assert.fail((String)"SHOULD_REPLICATE flag and ecPolicyName are exclusive parameters. Set both is not allowed.");
        }
        catch (Exception e) {
            GenericTestUtils.assertExceptionContains((String)"SHOULD_REPLICATE flag and ecPolicyName are exclusive parameters. Set both is not allowed!", (Throwable)e);
        }
        this.fs.createFile(filePath).replicate().build().close();
        Assert.assertNull((Object)this.fs.getErasureCodingPolicy(filePath));
        this.fs.delete(dirPath, true);
    }

    @Test
    public void testGetAllErasureCodingCodecs() throws Exception {
        Map allECCodecs = this.fs.getAllErasureCodingCodecs();
        Assert.assertTrue((String)"At least 3 system codecs should be enabled", (allECCodecs.size() >= 3 ? 1 : 0) != 0);
        System.out.println("Erasure Coding Codecs: Codec [Coder List]");
        for (String codec : allECCodecs.keySet()) {
            String coders = (String)allECCodecs.get(codec);
            if (codec == null || coders == null) continue;
            System.out.println("\t" + codec.toUpperCase() + "[" + coders.toUpperCase() + "]");
        }
    }

    @Test
    public void testAddErasureCodingPolicies() throws Exception {
        int[] cellSizes;
        ECSchema toAddSchema = new ECSchema("testcodec", 3, 2);
        ErasureCodingPolicy newPolicy = new ErasureCodingPolicy(toAddSchema, 131072);
        ErasureCodingPolicy[] policyArray = new ErasureCodingPolicy[]{newPolicy};
        AddErasureCodingPolicyResponse[] responses = this.fs.addErasureCodingPolicies(policyArray);
        Assert.assertEquals((long)1L, (long)responses.length);
        Assert.assertFalse((boolean)responses[0].isSucceed());
        toAddSchema = new ECSchema("rs", 3, 2);
        newPolicy = new ErasureCodingPolicy(toAddSchema, 0x8000000);
        policyArray = new ErasureCodingPolicy[]{newPolicy};
        responses = this.fs.addErasureCodingPolicies(policyArray);
        Assert.assertEquals((long)1L, (long)responses.length);
        Assert.assertFalse((boolean)responses[0].isSucceed());
        toAddSchema = new ECSchema("rs", 3, 2);
        for (int cellSize : cellSizes = new int[]{0, -1, 1023}) {
            try {
                new ErasureCodingPolicy(toAddSchema, cellSize);
                Assert.fail((String)"Invalid cell size should be detected.");
            }
            catch (Exception e) {
                GenericTestUtils.assertExceptionContains((String)"cellSize must be", (Throwable)e);
            }
        }
        ErasureCodingPolicy policy0 = (ErasureCodingPolicy)SystemErasureCodingPolicies.getPolicies().get(0);
        policyArray = new ErasureCodingPolicy[]{policy0};
        responses = this.fs.addErasureCodingPolicies(policyArray);
        Assert.assertEquals((long)1L, (long)responses.length);
        Assert.assertTrue((boolean)responses[0].isSucceed());
        newPolicy = new ErasureCodingPolicy(toAddSchema, 0x400000);
        policyArray = new ErasureCodingPolicy[]{newPolicy};
        responses = this.fs.addErasureCodingPolicies(policyArray);
        Assert.assertEquals((long)1L, (long)responses.length);
        Assert.assertTrue((boolean)responses[0].isSucceed());
        Assert.assertEquals((long)(SystemErasureCodingPolicies.getPolicies().size() + 1), (long)ErasureCodingPolicyManager.getInstance().getPolicies().length);
        UserGroupInformation fakeUGI = UserGroupInformation.createUserForTesting((String)"ProbablyNotARealUserName", (String[])new String[]{"ShangriLa"});
        final ErasureCodingPolicy ecPolicy = newPolicy;
        fakeUGI.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<Object>(){

            @Override
            public Object run() throws Exception {
                DistributedFileSystem fs = TestErasureCodingPolicies.this.cluster.getFileSystem();
                try {
                    fs.addErasureCodingPolicies(new ErasureCodingPolicy[]{ecPolicy});
                    Assert.fail();
                }
                catch (AccessControlException ace) {
                    GenericTestUtils.assertExceptionContains((String)"Access denied for user ProbablyNotARealUserName. Superuser privilege is required", (Throwable)ace);
                }
                return null;
            }
        });
    }

    @Test
    public void testAddECPoliciesExceeded() throws Exception {
        ECSchema toAddSchema = new ECSchema("rs", 3, 2);
        int allowNumPolicies = 64;
        for (int i = 0; i < allowNumPolicies; ++i) {
            ErasureCodingPolicy erasureCodingPolicy = new ErasureCodingPolicy(toAddSchema, 1024 + 1024 * i);
            ErasureCodingPolicy[] policyArray = new ErasureCodingPolicy[]{erasureCodingPolicy};
            AddErasureCodingPolicyResponse[] responses = this.fs.addErasureCodingPolicies(policyArray);
            Assert.assertEquals((long)1L, (long)responses.length);
            Assert.assertTrue((boolean)responses[0].isSucceed());
            Assert.assertEquals((long)responses[0].getPolicy().getId(), (long)(64 + i));
        }
        ErasureCodingPolicy erasureCodingPolicy = new ErasureCodingPolicy(toAddSchema, 1024 + 1024 * allowNumPolicies);
        ErasureCodingPolicy[] policyArray = new ErasureCodingPolicy[]{erasureCodingPolicy};
        AddErasureCodingPolicyResponse[] responses = this.fs.addErasureCodingPolicies(policyArray);
        Assert.assertEquals((long)1L, (long)responses.length);
        Assert.assertFalse((boolean)responses[0].isSucceed());
    }

    @Test
    public void testReplicationPolicy() throws Exception {
        ErasureCodingPolicy replicaPolicy = SystemErasureCodingPolicies.getReplicationPolicy();
        Path rootDir = new Path("/striped");
        Path replicaDir = new Path(rootDir, "replica");
        Path subReplicaDir = new Path(replicaDir, "replica");
        Path replicaFile = new Path(replicaDir, "file");
        Path subReplicaFile = new Path(subReplicaDir, "file");
        this.fs.mkdirs(rootDir);
        this.fs.setErasureCodingPolicy(rootDir, this.ecPolicy.getName());
        this.fs.mkdirs(replicaDir);
        this.fs.createFile(replicaFile).build().close();
        HdfsFileStatus fileStatus = (HdfsFileStatus)this.fs.getFileStatus(replicaFile);
        Assert.assertEquals((String)"File should inherit EC policy.", (Object)this.ecPolicy, (Object)fileStatus.getErasureCodingPolicy());
        Assert.assertEquals((String)"File should be a EC file.", (Object)true, (Object)fileStatus.isErasureCoded());
        Assert.assertEquals((String)"File should have the same EC policy as its ancestor.", (Object)this.ecPolicy, (Object)this.fs.getErasureCodingPolicy(replicaFile));
        this.fs.delete(replicaFile, false);
        this.fs.setErasureCodingPolicy(replicaDir, replicaPolicy.getName());
        ErasureCodingPolicy temp = this.fs.getErasureCodingPolicy(replicaDir);
        Assert.assertEquals((String)"Directory should hide replication EC policy.", null, (Object)temp);
        this.fs.createFile(replicaFile).build().close();
        Assert.assertEquals((String)"Replication file should have default replication factor.", (long)this.fs.getDefaultReplication(), (long)this.fs.getFileStatus(replicaFile).getReplication());
        this.fs.setReplication(replicaFile, (short)2);
        Assert.assertEquals((String)"File should have replication factor as expected.", (long)2L, (long)this.fs.getFileStatus(replicaFile).getReplication());
        fileStatus = (HdfsFileStatus)this.fs.getFileStatus(replicaFile);
        Assert.assertEquals((String)"File should not have EC policy.", null, (Object)fileStatus.getErasureCodingPolicy());
        Assert.assertEquals((String)"File should not be a EC file.", (Object)false, (Object)fileStatus.isErasureCoded());
        ErasureCodingPolicy ecPolicyOnFile = this.fs.getErasureCodingPolicy(replicaFile);
        Assert.assertEquals((String)"File should not have EC policy.", null, (Object)ecPolicyOnFile);
        this.fs.delete(replicaFile, false);
        this.fs.mkdirs(subReplicaDir);
        Assert.assertEquals((String)"Directory should inherit hiding replication EC policy.", null, (Object)this.fs.getErasureCodingPolicy(subReplicaDir));
        this.fs.createFile(subReplicaFile).build().close();
        Assert.assertEquals((String)"File should have default replication factor.", (long)this.fs.getDefaultReplication(), (long)this.fs.getFileStatus(subReplicaFile).getReplication());
        fileStatus = (HdfsFileStatus)this.fs.getFileStatus(subReplicaFile);
        Assert.assertEquals((String)"File should not have EC policy.", null, (Object)fileStatus.getErasureCodingPolicy());
        Assert.assertEquals((String)"File should not be a EC file.", (Object)false, (Object)fileStatus.isErasureCoded());
        Assert.assertEquals((String)"File should not have EC policy.", null, (Object)this.fs.getErasureCodingPolicy(subReplicaFile));
        this.fs.delete(subReplicaFile, false);
        this.fs.unsetErasureCodingPolicy(replicaDir);
        this.fs.createFile(subReplicaFile).build().close();
        fileStatus = (HdfsFileStatus)this.fs.getFileStatus(subReplicaFile);
        Assert.assertEquals((String)"File should inherit EC policy.", (Object)this.ecPolicy, (Object)fileStatus.getErasureCodingPolicy());
        Assert.assertEquals((String)"File should be a EC file.", (Object)true, (Object)fileStatus.isErasureCoded());
        Assert.assertEquals((String)"File should have the same EC policy as its ancestor", (Object)this.ecPolicy, (Object)this.fs.getErasureCodingPolicy(subReplicaFile));
        this.fs.delete(subReplicaFile, false);
    }

    @Test
    public void testDifferentErasureCodingPolicyCellSize() throws Exception {
        ErasureCodingPolicy newPolicy1 = new ErasureCodingPolicy(ErasureCodeConstants.RS_3_2_SCHEMA, 8192);
        ErasureCodingPolicy[] policyArray = new ErasureCodingPolicy[]{newPolicy1};
        AddErasureCodingPolicyResponse[] responses = this.fs.addErasureCodingPolicies(policyArray);
        Assert.assertEquals((long)1L, (long)responses.length);
        Assert.assertTrue((boolean)responses[0].isSucceed());
        newPolicy1 = responses[0].getPolicy();
        ErasureCodingPolicy newPolicy2 = new ErasureCodingPolicy(ErasureCodeConstants.RS_3_2_SCHEMA, 4096);
        policyArray = new ErasureCodingPolicy[]{newPolicy2};
        responses = this.fs.addErasureCodingPolicies(policyArray);
        Assert.assertEquals((long)1L, (long)responses.length);
        Assert.assertTrue((boolean)responses[0].isSucceed());
        newPolicy2 = responses[0].getPolicy();
        this.fs.enableErasureCodingPolicy(newPolicy1.getName());
        this.fs.enableErasureCodingPolicy(newPolicy2.getName());
        Path stripedDir1 = new Path("/striped1");
        Path stripedDir2 = new Path("/striped2");
        Path file1 = new Path(stripedDir1, "file");
        Path file2 = new Path(stripedDir2, "file");
        this.fs.mkdirs(stripedDir1);
        this.fs.setErasureCodingPolicy(stripedDir1, newPolicy1.getName());
        this.fs.mkdirs(stripedDir2);
        this.fs.setErasureCodingPolicy(stripedDir2, newPolicy2.getName());
        int fileLength = 16384 * newPolicy1.getNumDataUnits();
        byte[] bytes = StripedFileTestUtil.generateBytes(fileLength);
        DFSTestUtil.writeFile((FileSystem)this.fs, file1, bytes);
        DFSTestUtil.writeFile((FileSystem)this.fs, file2, bytes);
        this.fs.delete(stripedDir1, true);
        this.fs.delete(stripedDir2, true);
    }
}

