/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.IOException;
import java.util.concurrent.ThreadLocalRandom;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FsShell;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.AppendTestUtil;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.datanode.FsDatasetTestUtils;
import org.apache.hadoop.hdfs.server.namenode.FSDirTruncateOp;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogLoader;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.NameNodeAdapter;
import org.apache.hadoop.hdfs.tools.DFSAdmin;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.apache.log4j.Level;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.core.Is;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;

public class TestFileTruncate {
    static final Log LOG;
    static final int BLOCK_SIZE = 4;
    static final short REPLICATION = 3;
    static final int DATANODE_NUM = 3;
    static final int SUCCESS_ATTEMPTS = 300;
    static final int RECOVERY_ATTEMPTS = 600;
    static final long SLEEP = 100L;
    static final long LOW_SOFTLIMIT = 100L;
    static final long LOW_HARDLIMIT = 200L;
    static final int SHORT_HEARTBEAT = 1;
    static Configuration conf;
    static MiniDFSCluster cluster;
    static DistributedFileSystem fs;
    private Path parent;

    @Before
    public void setUp() throws IOException {
        conf = new HdfsConfiguration();
        conf.setLong("dfs.namenode.fs-limits.min-block-size", 4L);
        conf.setInt("dfs.bytes-per-checksum", 4);
        conf.setInt("dfs.heartbeat.interval", 1);
        conf.setLong("dfs.namenode.reconstruction.pending.timeout-sec", 1L);
        cluster = new MiniDFSCluster.Builder(conf).format(true).numDataNodes(3).waitSafeMode(true).build();
        fs = cluster.getFileSystem();
        this.parent = new Path("/test");
    }

    @After
    public void tearDown() throws IOException {
        if (fs != null) {
            fs.close();
            fs = null;
        }
        if (cluster != null) {
            cluster.shutdown();
            cluster = null;
        }
    }

    @Test
    public void testBasicTruncate() throws IOException {
        int startingFileSize = 12;
        fs.mkdirs(this.parent);
        fs.setQuota(this.parent, 100L, 1000L);
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        for (int fileLength = startingFileSize; fileLength > 0; fileLength -= 3) {
            for (int toTruncate = 0; toTruncate <= fileLength; ++toTruncate) {
                Path p = new Path(this.parent, "testBasicTruncate" + fileLength);
                TestFileTruncate.writeContents(contents, fileLength, p);
                int newLength = fileLength - toTruncate;
                boolean isReady = fs.truncate(p, (long)newLength);
                LOG.info((Object)("fileLength=" + fileLength + ", newLength=" + newLength + ", toTruncate=" + toTruncate + ", isReady=" + isReady));
                Assert.assertEquals((String)"File must be closed for zero truncate or truncating at the block boundary", (Object)isReady, (Object)(toTruncate == 0 || newLength % 4 == 0 ? 1 : 0));
                if (!isReady) {
                    TestFileTruncate.checkBlockRecovery(p);
                }
                ContentSummary cs = fs.getContentSummary(this.parent);
                Assert.assertEquals((String)"Bad disk space usage", (long)cs.getSpaceConsumed(), (long)(newLength * 3));
                TestFileTruncate.checkFullFile(p, newLength, contents);
            }
        }
        fs.delete(this.parent, true);
    }

    @Test
    public void testMultipleTruncate() throws IOException {
        Path dir = new Path("/testMultipleTruncate");
        fs.mkdirs(dir);
        Path p = new Path(dir, "file");
        byte[] data = new byte[400];
        ThreadLocalRandom.current().nextBytes(data);
        TestFileTruncate.writeContents(data, data.length, p);
        int n = data.length;
        while (n > 0) {
            int newLength = ThreadLocalRandom.current().nextInt(n);
            boolean isReady = fs.truncate(p, (long)newLength);
            LOG.info((Object)("newLength=" + newLength + ", isReady=" + isReady));
            Assert.assertEquals((String)"File must be closed for truncating at the block boundary", (Object)isReady, (Object)(newLength % 4 == 0 ? 1 : 0));
            Assert.assertEquals((String)"Truncate is not idempotent", (Object)isReady, (Object)fs.truncate(p, (long)newLength));
            if (!isReady) {
                TestFileTruncate.checkBlockRecovery(p);
            }
            TestFileTruncate.checkFullFile(p, newLength, data);
            n = newLength;
        }
        fs.delete(dir, true);
    }

    @Test
    public void testSnapshotTruncateThenDeleteSnapshot() throws IOException {
        Path dir = new Path("/testSnapshotTruncateThenDeleteSnapshot");
        fs.mkdirs(dir);
        fs.allowSnapshot(dir);
        Path p = new Path(dir, "file");
        byte[] data = new byte[4];
        ThreadLocalRandom.current().nextBytes(data);
        TestFileTruncate.writeContents(data, data.length, p);
        String snapshot = "s0";
        fs.createSnapshot(dir, "s0");
        Block lastBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock().getBlock().getLocalBlock();
        int newLength = data.length - 1;
        assert (newLength % 4 != 0) : " newLength must not be multiple of BLOCK_SIZE";
        boolean isReady = fs.truncate(p, (long)newLength);
        LOG.info((Object)("newLength=" + newLength + ", isReady=" + isReady));
        Assert.assertEquals((String)"File must be closed for truncating at the block boundary", (Object)isReady, (Object)(newLength % 4 == 0 ? 1 : 0));
        fs.deleteSnapshot(dir, "s0");
        if (!isReady) {
            TestFileTruncate.checkBlockRecovery(p);
        }
        TestFileTruncate.checkFullFile(p, newLength, data);
        TestFileTruncate.assertBlockNotPresent(lastBlock);
        fs.delete(dir, true);
    }

    @Test
    public void testTruncateWithOtherOperations() throws IOException {
        Path dir = new Path("/testTruncateOtherOperations");
        fs.mkdirs(dir);
        Path p = new Path(dir, "file");
        byte[] data = new byte[8];
        ThreadLocalRandom.current().nextBytes(data);
        TestFileTruncate.writeContents(data, data.length, p);
        int newLength = data.length - 1;
        boolean isReady = fs.truncate(p, (long)newLength);
        Assert.assertFalse((boolean)isReady);
        fs.setReplication(p, (short)2);
        fs.setPermission(p, FsPermission.createImmutable((short)292));
        Path q = new Path(dir, "newFile");
        fs.rename(p, q);
        TestFileTruncate.checkBlockRecovery(q);
        TestFileTruncate.checkFullFile(q, newLength, data);
        cluster.restartNameNode(new String[0]);
        TestFileTruncate.checkFullFile(q, newLength, data);
        fs.delete(dir, true);
    }

    @Test
    public void testSnapshotWithAppendTruncate() throws IOException {
        this.testSnapshotWithAppendTruncate(0, 1, 2);
        this.testSnapshotWithAppendTruncate(0, 2, 1);
        this.testSnapshotWithAppendTruncate(1, 0, 2);
        this.testSnapshotWithAppendTruncate(1, 2, 0);
        this.testSnapshotWithAppendTruncate(2, 0, 1);
        this.testSnapshotWithAppendTruncate(2, 1, 0);
    }

    void testSnapshotWithAppendTruncate(int ... deleteOrder) throws IOException {
        FSDirectory fsDir = cluster.getNamesystem().getFSDirectory();
        fs.mkdirs(this.parent);
        fs.setQuota(this.parent, 100L, 1000L);
        fs.allowSnapshot(this.parent);
        String truncateFile = "testSnapshotWithAppendTruncate";
        Path src = new Path(this.parent, truncateFile);
        int[] length = new int[4];
        length[0] = 10;
        DFSTestUtil.createFile((FileSystem)fs, src, 64, length[0], 4L, (short)3, 0L);
        Block firstBlk = TestFileTruncate.getLocatedBlocks(src).get(0).getBlock().getLocalBlock();
        Path[] snapshotFiles = new Path[4];
        ContentSummary contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)30L));
        String[] ss = new String[]{"ss0", "ss1", "ss2", "ss3"};
        Path snapshotDir = fs.createSnapshot(this.parent, ss[0]);
        snapshotFiles[0] = new Path(snapshotDir, truncateFile);
        length[1] = length[2] = length[0] + 4 + 1;
        DFSTestUtil.appendFile((FileSystem)fs, src, 5);
        Block lastBlk = TestFileTruncate.getLocatedBlocks(src).getLastLocatedBlock().getBlock().getLocalBlock();
        contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)45L));
        snapshotDir = fs.createSnapshot(this.parent, ss[1]);
        snapshotFiles[1] = new Path(snapshotDir, truncateFile);
        snapshotDir = fs.createSnapshot(this.parent, ss[2]);
        snapshotFiles[2] = new Path(snapshotDir, truncateFile);
        DFSTestUtil.appendFile((FileSystem)fs, src, 5);
        Block appendedBlk = TestFileTruncate.getLocatedBlocks(src).getLastLocatedBlock().getBlock().getLocalBlock();
        contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)60L));
        int newLength = length[0] + 2;
        boolean isReady = fs.truncate(src, (long)newLength);
        Assert.assertTrue((String)"Recovery is not expected.", (boolean)isReady);
        TestFileTruncate.assertFileLength(snapshotFiles[2], length[2]);
        TestFileTruncate.assertFileLength(snapshotFiles[1], length[1]);
        TestFileTruncate.assertFileLength(snapshotFiles[0], length[0]);
        TestFileTruncate.assertBlockNotPresent(appendedBlk);
        contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)48L));
        newLength = length[0] - 2;
        isReady = fs.truncate(src, (long)newLength);
        Assert.assertTrue((String)"Recovery is not expected.", (boolean)isReady);
        TestFileTruncate.assertFileLength(snapshotFiles[2], length[2]);
        TestFileTruncate.assertFileLength(snapshotFiles[1], length[1]);
        TestFileTruncate.assertFileLength(snapshotFiles[0], length[0]);
        contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)48L));
        isReady = fs.truncate(src, (long)(newLength -= 2));
        Assert.assertFalse((String)"Recovery is expected.", (boolean)isReady);
        TestFileTruncate.checkBlockRecovery(src);
        TestFileTruncate.assertFileLength(snapshotFiles[2], length[2]);
        TestFileTruncate.assertFileLength(snapshotFiles[1], length[1]);
        TestFileTruncate.assertFileLength(snapshotFiles[0], length[0]);
        Block replacedBlk = TestFileTruncate.getLocatedBlocks(src).getLastLocatedBlock().getBlock().getLocalBlock();
        contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)54L));
        snapshotDir = fs.createSnapshot(this.parent, ss[3]);
        snapshotFiles[3] = new Path(snapshotDir, truncateFile);
        length[3] = newLength;
        int numINodes = fsDir.getInodeMapSize();
        isReady = fs.delete(src, false);
        Assert.assertTrue((String)"Delete failed.", (boolean)isReady);
        TestFileTruncate.assertFileLength(snapshotFiles[3], length[3]);
        TestFileTruncate.assertFileLength(snapshotFiles[2], length[2]);
        TestFileTruncate.assertFileLength(snapshotFiles[1], length[1]);
        TestFileTruncate.assertFileLength(snapshotFiles[0], length[0]);
        Assert.assertEquals((String)"Number of INodes should not change", (long)numINodes, (long)fsDir.getInodeMapSize());
        fs.deleteSnapshot(this.parent, ss[3]);
        TestFileTruncate.assertBlockExists(firstBlk);
        TestFileTruncate.assertBlockExists(lastBlk);
        TestFileTruncate.assertBlockNotPresent(replacedBlk);
        contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)48L));
        fs.deleteSnapshot(this.parent, ss[deleteOrder[0]]);
        TestFileTruncate.assertFileLength(snapshotFiles[deleteOrder[1]], length[deleteOrder[1]]);
        TestFileTruncate.assertFileLength(snapshotFiles[deleteOrder[2]], length[deleteOrder[2]]);
        TestFileTruncate.assertBlockExists(firstBlk);
        TestFileTruncate.assertBlockExists(lastBlk);
        Assert.assertEquals((String)"Number of INodes should not change", (long)numINodes, (long)fsDir.getInodeMapSize());
        contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)48L));
        fs.deleteSnapshot(this.parent, ss[deleteOrder[1]]);
        TestFileTruncate.assertFileLength(snapshotFiles[deleteOrder[2]], length[deleteOrder[2]]);
        TestFileTruncate.assertBlockExists(firstBlk);
        contentSummary = fs.getContentSummary(this.parent);
        if (fs.exists(snapshotFiles[0])) {
            TestFileTruncate.assertBlockNotPresent(lastBlk);
            Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)36L));
        } else {
            Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)48L));
        }
        Assert.assertEquals((String)"Number of INodes should not change", (long)numINodes, (long)fsDir.getInodeMapSize());
        fs.deleteSnapshot(this.parent, ss[deleteOrder[2]]);
        TestFileTruncate.assertBlockNotPresent(firstBlk);
        TestFileTruncate.assertBlockNotPresent(lastBlk);
        contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)0L));
        Assert.assertNotEquals((String)"Number of INodes should change", (long)numINodes, (long)fsDir.getInodeMapSize());
    }

    @Test
    public void testSnapshotWithTruncates() throws IOException {
        this.testSnapshotWithTruncates(0, 1, 2);
        this.testSnapshotWithTruncates(0, 2, 1);
        this.testSnapshotWithTruncates(1, 0, 2);
        this.testSnapshotWithTruncates(1, 2, 0);
        this.testSnapshotWithTruncates(2, 0, 1);
        this.testSnapshotWithTruncates(2, 1, 0);
    }

    void testSnapshotWithTruncates(int ... deleteOrder) throws IOException {
        fs.mkdirs(this.parent);
        fs.setQuota(this.parent, 100L, 1000L);
        fs.allowSnapshot(this.parent);
        String truncateFile = "testSnapshotWithTruncates";
        Path src = new Path(this.parent, truncateFile);
        int[] length = new int[3];
        length[0] = 12;
        DFSTestUtil.createFile((FileSystem)fs, src, 64, length[0], 4L, (short)3, 0L);
        Block firstBlk = TestFileTruncate.getLocatedBlocks(src).get(0).getBlock().getLocalBlock();
        Block lastBlk = TestFileTruncate.getLocatedBlocks(src).getLastLocatedBlock().getBlock().getLocalBlock();
        Path[] snapshotFiles = new Path[3];
        ContentSummary contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)36L));
        String[] ss = new String[]{"ss0", "ss1", "ss2"};
        Path snapshotDir = fs.createSnapshot(this.parent, ss[0]);
        snapshotFiles[0] = new Path(snapshotDir, truncateFile);
        length[1] = 8;
        boolean isReady = fs.truncate(src, 8L);
        Assert.assertTrue((String)"Recovery is not expected.", (boolean)isReady);
        contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)36L));
        snapshotDir = fs.createSnapshot(this.parent, ss[1]);
        snapshotFiles[1] = new Path(snapshotDir, truncateFile);
        length[2] = 6;
        isReady = fs.truncate(src, 6L);
        Assert.assertFalse((String)"Recovery is expected.", (boolean)isReady);
        TestFileTruncate.checkBlockRecovery(src);
        snapshotDir = fs.createSnapshot(this.parent, ss[2]);
        snapshotFiles[2] = new Path(snapshotDir, truncateFile);
        TestFileTruncate.assertFileLength(snapshotFiles[0], length[0]);
        TestFileTruncate.assertBlockExists(lastBlk);
        contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)42L));
        fs.deleteSnapshot(this.parent, ss[deleteOrder[0]]);
        TestFileTruncate.assertFileLength(snapshotFiles[deleteOrder[1]], length[deleteOrder[1]]);
        TestFileTruncate.assertFileLength(snapshotFiles[deleteOrder[2]], length[deleteOrder[2]]);
        TestFileTruncate.assertFileLength(src, length[2]);
        TestFileTruncate.assertBlockExists(firstBlk);
        contentSummary = fs.getContentSummary(this.parent);
        if (fs.exists(snapshotFiles[0])) {
            Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)42L));
            TestFileTruncate.assertBlockExists(lastBlk);
        } else {
            Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)30L));
            TestFileTruncate.assertBlockNotPresent(lastBlk);
        }
        fs.deleteSnapshot(this.parent, ss[deleteOrder[1]]);
        TestFileTruncate.assertFileLength(snapshotFiles[deleteOrder[2]], length[deleteOrder[2]]);
        TestFileTruncate.assertFileLength(src, length[2]);
        TestFileTruncate.assertBlockExists(firstBlk);
        contentSummary = fs.getContentSummary(this.parent);
        if (fs.exists(snapshotFiles[0])) {
            Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)42L));
            TestFileTruncate.assertBlockExists(lastBlk);
        } else if (fs.exists(snapshotFiles[1])) {
            Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)30L));
            TestFileTruncate.assertBlockNotPresent(lastBlk);
        } else {
            Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)18L));
            TestFileTruncate.assertBlockNotPresent(lastBlk);
        }
        fs.deleteSnapshot(this.parent, ss[deleteOrder[2]]);
        TestFileTruncate.assertFileLength(src, length[2]);
        TestFileTruncate.assertBlockExists(firstBlk);
        contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)18L));
        Assert.assertThat((Object)contentSummary.getLength(), (Matcher)Is.is((Object)6L));
        fs.delete(src, false);
        TestFileTruncate.assertBlockNotPresent(firstBlk);
        contentSummary = fs.getContentSummary(this.parent);
        Assert.assertThat((Object)contentSummary.getSpaceConsumed(), (Matcher)Is.is((Object)0L));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testTruncateFailure() throws IOException {
        int startingFileSize = 10;
        int toTruncate = 1;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path dir = new Path("/dir");
        Path p = new Path(dir, "testTruncateFailure");
        out.write(contents, 0, startingFileSize);
        try (FSDataOutputStream out = fs.create(p, false, 4, (short)3, 4L);){
            fs.truncate(p, 0L);
            Assert.fail((String)"Truncate must fail on open file.");
        }
        out = fs.append(p);
        try {
            fs.truncate(p, 0L);
            Assert.fail((String)"Truncate must fail for append.");
        }
        catch (IOException expected) {
            GenericTestUtils.assertExceptionContains((String)"Failed to TRUNCATE_FILE", (Throwable)expected);
        }
        finally {
            out.close();
        }
        try {
            fs.truncate(p, -1L);
            Assert.fail((String)"Truncate must fail for a negative new length.");
        }
        catch (HadoopIllegalArgumentException expected) {
            GenericTestUtils.assertExceptionContains((String)"Cannot truncate to a negative file size", (Throwable)expected);
        }
        try {
            fs.truncate(p, (long)(startingFileSize + 1));
            Assert.fail((String)"Truncate must fail for a larger new length.");
        }
        catch (Exception expected) {
            GenericTestUtils.assertExceptionContains((String)"Cannot truncate to a larger file size", (Throwable)expected);
        }
        try {
            fs.truncate(dir, 0L);
            Assert.fail((String)"Truncate must fail for a directory.");
        }
        catch (Exception expected) {
            GenericTestUtils.assertExceptionContains((String)"Path is not a file", (Throwable)expected);
        }
        try {
            fs.truncate(new Path(dir, "non-existing"), 0L);
            Assert.fail((String)"Truncate must fail for a non-existing file.");
        }
        catch (Exception expected) {
            GenericTestUtils.assertExceptionContains((String)"File does not exist", (Throwable)expected);
        }
        fs.setPermission(p, FsPermission.createImmutable((short)436));
        UserGroupInformation fooUgi = UserGroupInformation.createUserForTesting((String)"foo", (String[])new String[]{"foo"});
        try {
            FileSystem foofs = DFSTestUtil.getFileSystemAs(fooUgi, conf);
            foofs.truncate(p, 0L);
            Assert.fail((String)"Truncate must fail for no WRITE permission.");
        }
        catch (Exception expected) {
            GenericTestUtils.assertExceptionContains((String)"Permission denied", (Throwable)expected);
        }
        cluster.shutdownDataNodes();
        NameNodeAdapter.getLeaseManager(cluster.getNamesystem()).setLeasePeriod(100L, 200L);
        int newLength = startingFileSize - toTruncate;
        boolean isReady = fs.truncate(p, (long)newLength);
        Assert.assertThat((String)"truncate should have triggered block recovery.", (Object)isReady, (Matcher)Is.is((Object)false));
        try {
            fs.truncate(p, 0L);
            Assert.fail((String)"Truncate must fail since a trancate is already in pregress.");
        }
        catch (IOException expected) {
            GenericTestUtils.assertExceptionContains((String)"Failed to TRUNCATE_FILE", (Throwable)expected);
        }
        boolean recoveryTriggered = false;
        for (int i = 0; i < 600; ++i) {
            String leaseHolder = NameNodeAdapter.getLeaseHolderForPath(cluster.getNameNode(), p.toUri().getPath());
            if (leaseHolder.startsWith("HDFS_NameNode")) {
                recoveryTriggered = true;
                break;
            }
            try {
                Thread.sleep(100L);
                continue;
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        Assert.assertThat((String)"lease recovery should have occurred in ~60000 ms.", (Object)recoveryTriggered, (Matcher)Is.is((Object)true));
        cluster.startDataNodes(conf, 3, true, HdfsServerConstants.StartupOption.REGULAR, null);
        cluster.waitActive();
        TestFileTruncate.checkBlockRecovery(p);
        NameNodeAdapter.getLeaseManager(cluster.getNamesystem()).setLeasePeriod(60000L, conf.getLong("dfs.namenode.lease-hard-limit-sec", 1200L) * 1000L);
        TestFileTruncate.checkFullFile(p, newLength, contents);
        fs.delete(p, false);
    }

    @Test(timeout=60000L)
    public void testTruncateWithDataNodesRestart() throws Exception {
        int startingFileSize = 12;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path p = new Path(this.parent, "testTruncateWithDataNodesRestart");
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        LocatedBlock oldBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        int dn = 0;
        int toTruncateLength = 1;
        int newLength = startingFileSize - toTruncateLength;
        cluster.getDataNodes().get(dn).shutdown();
        this.truncateAndRestartDN(p, dn, newLength);
        TestFileTruncate.checkBlockRecovery(p);
        LocatedBlock newBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        Assert.assertEquals((long)newBlock.getBlock().getBlockId(), (long)oldBlock.getBlock().getBlockId());
        Assert.assertEquals((long)newBlock.getBlock().getGenerationStamp(), (long)(oldBlock.getBlock().getGenerationStamp() + 1L));
        Thread.sleep(2000L);
        cluster.triggerBlockReports();
        DFSTestUtil.waitReplication((FileSystem)fs, p, (short)3);
        FsDatasetTestUtils utils = cluster.getFsDatasetTestUtils(dn);
        Assert.assertEquals((long)utils.getStoredDataLength(newBlock.getBlock()), (long)newBlock.getBlockSize());
        Assert.assertEquals((long)utils.getStoredGenerationStamp(newBlock.getBlock()), (long)newBlock.getBlock().getGenerationStamp());
        FileStatus fileStatus = fs.getFileStatus(p);
        Assert.assertThat((Object)fileStatus.getLen(), (Matcher)Is.is((Object)newLength));
        TestFileTruncate.checkFullFile(p, newLength, contents);
        fs.delete(this.parent, true);
    }

    @Test(timeout=60000L)
    public void testCopyOnTruncateWithDataNodesRestart() throws Exception {
        int startingFileSize = 12;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path p = new Path(this.parent, "testCopyOnTruncateWithDataNodesRestart");
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        LocatedBlock oldBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        fs.allowSnapshot(this.parent);
        fs.createSnapshot(this.parent, "ss0");
        int dn = 1;
        int toTruncateLength = 1;
        int newLength = startingFileSize - toTruncateLength;
        cluster.getDataNodes().get(dn).shutdown();
        this.truncateAndRestartDN(p, dn, newLength);
        TestFileTruncate.checkBlockRecovery(p);
        LocatedBlock newBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        Assert.assertNotEquals((long)newBlock.getBlock().getBlockId(), (long)oldBlock.getBlock().getBlockId());
        Assert.assertEquals((long)newBlock.getBlock().getGenerationStamp(), (long)(oldBlock.getBlock().getGenerationStamp() + 1L));
        DFSTestUtil.waitReplication((FileSystem)fs, p, (short)3);
        FsDatasetTestUtils utils = cluster.getFsDatasetTestUtils(dn);
        Assert.assertEquals((long)utils.getStoredDataLength(newBlock.getBlock()), (long)newBlock.getBlockSize());
        Assert.assertEquals((long)utils.getStoredDataLength(oldBlock.getBlock()), (long)oldBlock.getBlockSize());
        Assert.assertEquals((long)utils.getStoredGenerationStamp(oldBlock.getBlock()), (long)oldBlock.getBlock().getGenerationStamp());
        FileStatus fileStatus = fs.getFileStatus(p);
        Assert.assertThat((Object)fileStatus.getLen(), (Matcher)Is.is((Object)newLength));
        TestFileTruncate.checkFullFile(p, newLength, contents);
        fs.deleteSnapshot(this.parent, "ss0");
        fs.delete(this.parent, true);
    }

    @Test(timeout=60000L)
    public void testTruncateWithDataNodesRestartImmediately() throws Exception {
        int startingFileSize = 12;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path p = new Path(this.parent, "testTruncateWithDataNodesRestartImmediately");
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        LocatedBlock oldBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        int dn0 = 0;
        int dn1 = 1;
        int toTruncateLength = 1;
        int newLength = startingFileSize - toTruncateLength;
        boolean isReady = fs.truncate(p, (long)newLength);
        Assert.assertFalse((boolean)isReady);
        cluster.restartDataNode(dn0, false, true);
        cluster.restartDataNode(dn1, false, true);
        cluster.waitActive();
        TestFileTruncate.checkBlockRecovery(p);
        LocatedBlock newBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        Assert.assertEquals((long)newBlock.getBlock().getBlockId(), (long)oldBlock.getBlock().getBlockId());
        Assert.assertEquals((long)newBlock.getBlock().getGenerationStamp(), (long)(oldBlock.getBlock().getGenerationStamp() + 1L));
        Thread.sleep(2000L);
        cluster.triggerBlockReports();
        DFSTestUtil.waitReplication((FileSystem)fs, p, (short)3);
        FsDatasetTestUtils utils = cluster.getFsDatasetTestUtils(dn0);
        Assert.assertEquals((long)utils.getStoredDataLength(newBlock.getBlock()), (long)newBlock.getBlockSize());
        Assert.assertEquals((long)utils.getStoredGenerationStamp(newBlock.getBlock()), (long)newBlock.getBlock().getGenerationStamp());
        utils = cluster.getFsDatasetTestUtils(dn1);
        Assert.assertEquals((long)utils.getStoredDataLength(newBlock.getBlock()), (long)newBlock.getBlockSize());
        Assert.assertEquals((long)utils.getStoredGenerationStamp(newBlock.getBlock()), (long)newBlock.getBlock().getGenerationStamp());
        FileStatus fileStatus = fs.getFileStatus(p);
        Assert.assertThat((Object)fileStatus.getLen(), (Matcher)Is.is((Object)newLength));
        TestFileTruncate.checkFullFile(p, newLength, contents);
        fs.delete(this.parent, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=60000L)
    public void testTruncateWithDataNodesShutdownImmediately() throws Exception {
        int startingFileSize = 12;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path p = new Path(this.parent, "testTruncateWithDataNodesShutdownImmediately");
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        int toTruncateLength = 1;
        int newLength = startingFileSize - toTruncateLength;
        boolean isReady = fs.truncate(p, (long)newLength);
        Assert.assertFalse((boolean)isReady);
        cluster.shutdownDataNodes();
        cluster.setDataNodesDead();
        try {
            for (int i = 0; i < 300 && cluster.isDataNodeUp(); ++i) {
                Thread.sleep(100L);
            }
            Assert.assertFalse((String)"All DataNodes should be down.", (boolean)cluster.isDataNodeUp());
            LocatedBlocks blocks = TestFileTruncate.getLocatedBlocks(p);
            Assert.assertTrue((boolean)blocks.isUnderConstruction());
        }
        finally {
            cluster.startDataNodes(conf, 3, true, HdfsServerConstants.StartupOption.REGULAR, null);
            cluster.waitActive();
        }
        TestFileTruncate.checkBlockRecovery(p);
        fs.delete(this.parent, true);
    }

    @Test
    public void testTruncateEditLogLoad() throws IOException {
        fs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_ENTER);
        fs.saveNamespace();
        fs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_LEAVE);
        int startingFileSize = 10;
        int toTruncate = 1;
        String s = "/testTruncateEditLogLoad";
        Path p = new Path("/testTruncateEditLogLoad");
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        int newLength = startingFileSize - toTruncate;
        boolean isReady = fs.truncate(p, (long)newLength);
        Assert.assertThat((String)"truncate should have triggered block recovery.", (Object)isReady, (Matcher)Is.is((Object)false));
        cluster.restartNameNode(new String[0]);
        String holder = UserGroupInformation.getCurrentUser().getUserName();
        cluster.getNamesystem().recoverLease("/testTruncateEditLogLoad", holder, "");
        TestFileTruncate.checkBlockRecovery(p);
        TestFileTruncate.checkFullFile(p, newLength, contents);
        fs.delete(p, false);
    }

    @Test
    public void testUpgradeAndRestart() throws IOException {
        fs.mkdirs(this.parent);
        fs.setQuota(this.parent, 100L, 1000L);
        fs.allowSnapshot(this.parent);
        String truncateFile = "testUpgrade";
        Path p = new Path(this.parent, truncateFile);
        int startingFileSize = 8;
        int toTruncate = 1;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        Path snapshotDir = fs.createSnapshot(this.parent, "ss0");
        Path snapshotFile = new Path(snapshotDir, truncateFile);
        int newLengthBeforeUpgrade = startingFileSize - toTruncate;
        boolean isReady = fs.truncate(p, (long)newLengthBeforeUpgrade);
        Assert.assertThat((String)"truncate should have triggered block recovery.", (Object)isReady, (Matcher)Is.is((Object)false));
        TestFileTruncate.checkBlockRecovery(p);
        TestFileTruncate.checkFullFile(p, newLengthBeforeUpgrade, contents);
        TestFileTruncate.assertFileLength(snapshotFile, startingFileSize);
        long totalBlockBefore = cluster.getNamesystem().getBlocksTotal();
        TestFileTruncate.restartCluster(HdfsServerConstants.StartupOption.UPGRADE);
        Assert.assertThat((String)"SafeMode should be OFF", (Object)cluster.getNamesystem().isInSafeMode(), (Matcher)Is.is((Object)false));
        Assert.assertThat((String)"NameNode should be performing upgrade.", (Object)cluster.getNamesystem().isUpgradeFinalized(), (Matcher)Is.is((Object)false));
        FileStatus fileStatus = fs.getFileStatus(p);
        Assert.assertThat((Object)fileStatus.getLen(), (Matcher)Is.is((Object)newLengthBeforeUpgrade));
        int newLengthAfterUpgrade = newLengthBeforeUpgrade - toTruncate;
        Block oldBlk = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock().getBlock().getLocalBlock();
        isReady = fs.truncate(p, (long)newLengthAfterUpgrade);
        Assert.assertThat((String)"truncate should have triggered block recovery.", (Object)isReady, (Matcher)Is.is((Object)false));
        fileStatus = fs.getFileStatus(p);
        Assert.assertThat((Object)fileStatus.getLen(), (Matcher)Is.is((Object)newLengthAfterUpgrade));
        Assert.assertThat((String)"Should copy on truncate during upgrade", (Object)TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock().getBlock().getLocalBlock().getBlockId(), (Matcher)Is.is((Matcher)CoreMatchers.not((Matcher)CoreMatchers.equalTo((Object)oldBlk.getBlockId()))));
        TestFileTruncate.checkBlockRecovery(p);
        TestFileTruncate.checkFullFile(p, newLengthAfterUpgrade, contents);
        Assert.assertThat((String)"Total block count should be unchanged from copy-on-truncate", (Object)cluster.getNamesystem().getBlocksTotal(), (Matcher)Is.is((Object)totalBlockBefore));
        TestFileTruncate.restartCluster(HdfsServerConstants.StartupOption.ROLLBACK);
        Assert.assertThat((String)("File does not exist " + p), (Object)fs.exists(p), (Matcher)Is.is((Object)true));
        fileStatus = fs.getFileStatus(p);
        Assert.assertThat((Object)fileStatus.getLen(), (Matcher)Is.is((Object)newLengthBeforeUpgrade));
        TestFileTruncate.checkFullFile(p, newLengthBeforeUpgrade, contents);
        Assert.assertThat((String)"Total block count should be unchanged from rolling back", (Object)cluster.getNamesystem().getBlocksTotal(), (Matcher)Is.is((Object)totalBlockBefore));
        TestFileTruncate.restartCluster(HdfsServerConstants.StartupOption.REGULAR);
        Assert.assertThat((String)"Total block count should be unchanged from start-up", (Object)cluster.getNamesystem().getBlocksTotal(), (Matcher)Is.is((Object)totalBlockBefore));
        TestFileTruncate.checkFullFile(p, newLengthBeforeUpgrade, contents);
        TestFileTruncate.assertFileLength(snapshotFile, startingFileSize);
        fs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_ENTER);
        fs.saveNamespace();
        cluster.restartNameNode(true);
        Assert.assertThat((String)"Total block count should be unchanged from start-up", (Object)cluster.getNamesystem().getBlocksTotal(), (Matcher)Is.is((Object)totalBlockBefore));
        TestFileTruncate.checkFullFile(p, newLengthBeforeUpgrade, contents);
        TestFileTruncate.assertFileLength(snapshotFile, startingFileSize);
        fs.deleteSnapshot(this.parent, "ss0");
        fs.delete(this.parent, true);
        Assert.assertThat((String)("File " + p + " shouldn't exist"), (Object)fs.exists(p), (Matcher)Is.is((Object)false));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testTruncateRecovery() throws IOException {
        long blockRecoveryId;
        Block truncateBlock;
        BlockInfo oldBlock;
        FSNamesystem fsn = cluster.getNamesystem();
        String client = "client";
        String clientMachine = "clientMachine";
        String src = "/test/testTruncateRecovery";
        Path srcPath = new Path(src);
        byte[] contents = AppendTestUtil.initBuffer(4);
        TestFileTruncate.writeContents(contents, 4, srcPath);
        INodesInPath iip = fsn.getFSDirectory().getINodesInPath(src, FSDirectory.DirOp.WRITE);
        INodeFile file = iip.getLastINode().asFile();
        long initialGenStamp = file.getLastBlock().getGenerationStamp();
        fsn.writeLock();
        try {
            oldBlock = file.getLastBlock();
            truncateBlock = FSDirTruncateOp.prepareFileForTruncate((FSNamesystem)fsn, (INodesInPath)iip, (String)client, (String)clientMachine, (long)1L, null);
            Assert.assertThat((Object)truncateBlock.getBlockId(), (Matcher)Is.is((Matcher)CoreMatchers.equalTo((Object)oldBlock.getBlockId())));
            Assert.assertThat((Object)truncateBlock.getNumBytes(), (Matcher)Is.is((Object)oldBlock.getNumBytes()));
            Assert.assertThat((Object)truncateBlock.getGenerationStamp(), (Matcher)Is.is((Object)fsn.getBlockManager().getBlockIdManager().getGenerationStamp()));
            Assert.assertThat((Object)file.getLastBlock().getBlockUCState(), (Matcher)Is.is((Object)HdfsServerConstants.BlockUCState.UNDER_RECOVERY));
            blockRecoveryId = file.getLastBlock().getUnderConstructionFeature().getBlockRecoveryId();
            Assert.assertThat((Object)blockRecoveryId, (Matcher)Is.is((Object)(initialGenStamp + 1L)));
            fsn.getEditLog().logTruncate(src, client, clientMachine, 3L, Time.now(), truncateBlock);
        }
        finally {
            fsn.writeUnlock();
        }
        TestFileTruncate.writeContents(contents, 4, srcPath);
        fs.allowSnapshot(this.parent);
        fs.createSnapshot(this.parent, "ss0");
        iip = fsn.getFSDirectory().getINodesInPath(src, FSDirectory.DirOp.WRITE);
        file = iip.getLastINode().asFile();
        file.recordModification(iip.getLatestSnapshotId(), true);
        Assert.assertThat((Object)file.isBlockInLatestSnapshot((BlockInfo)((BlockInfoContiguous)file.getLastBlock())), (Matcher)Is.is((Object)true));
        initialGenStamp = file.getLastBlock().getGenerationStamp();
        fsn.writeLock();
        try {
            oldBlock = file.getLastBlock();
            truncateBlock = FSDirTruncateOp.prepareFileForTruncate((FSNamesystem)fsn, (INodesInPath)iip, (String)client, (String)clientMachine, (long)1L, null);
            Assert.assertThat((Object)truncateBlock.getBlockId(), (Matcher)Is.is((Matcher)CoreMatchers.not((Matcher)CoreMatchers.equalTo((Object)oldBlock.getBlockId()))));
            Assert.assertThat((Object)(truncateBlock.getNumBytes() < oldBlock.getNumBytes() ? 1 : 0), (Matcher)Is.is((Object)true));
            Assert.assertThat((Object)truncateBlock.getGenerationStamp(), (Matcher)Is.is((Object)fsn.getBlockManager().getBlockIdManager().getGenerationStamp()));
            Assert.assertThat((Object)file.getLastBlock().getBlockUCState(), (Matcher)Is.is((Object)HdfsServerConstants.BlockUCState.UNDER_RECOVERY));
            blockRecoveryId = file.getLastBlock().getUnderConstructionFeature().getBlockRecoveryId();
            Assert.assertThat((Object)blockRecoveryId, (Matcher)Is.is((Object)(initialGenStamp + 1L)));
            fsn.getEditLog().logTruncate(src, client, clientMachine, 3L, Time.now(), truncateBlock);
        }
        finally {
            fsn.writeUnlock();
        }
        TestFileTruncate.checkBlockRecovery(srcPath);
        fs.deleteSnapshot(this.parent, "ss0");
        fs.delete(this.parent, true);
    }

    @Test
    public void testTruncateShellCommand() throws Exception {
        Path src = new Path("/test/testTruncateShellCommand");
        int oldLength = 9;
        int newLength = 5;
        String[] argv = new String[]{"-truncate", String.valueOf(5), src.toString()};
        this.runTruncateShellCommand(src, 9, argv);
        TestFileTruncate.checkBlockRecovery(src);
        Assert.assertThat((Object)fs.getFileStatus(src).getLen(), (Matcher)Is.is((Object)5L));
        fs.delete(this.parent, true);
    }

    @Test
    public void testTruncateShellCommandOnBlockBoundary() throws Exception {
        Path src = new Path("/test/testTruncateShellCommandOnBoundary");
        int oldLength = 8;
        int newLength = 4;
        String[] argv = new String[]{"-truncate", String.valueOf(4), src.toString()};
        this.runTruncateShellCommand(src, 8, argv);
        Assert.assertThat((Object)fs.getFileStatus(src).getLen(), (Matcher)Is.is((Object)4L));
        fs.delete(this.parent, true);
    }

    @Test
    public void testTruncateShellCommandWithWaitOption() throws Exception {
        Path src = new Path("/test/testTruncateShellCommandWithWaitOption");
        int oldLength = 9;
        int newLength = 5;
        String[] argv = new String[]{"-truncate", "-w", String.valueOf(5), src.toString()};
        this.runTruncateShellCommand(src, 9, argv);
        Assert.assertThat((Object)fs.getFileStatus(src).getLen(), (Matcher)Is.is((Object)5L));
        fs.delete(this.parent, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void runTruncateShellCommand(Path src, int oldLength, String[] shellOpts) throws Exception {
        TestFileTruncate.writeContents(AppendTestUtil.initBuffer(oldLength), oldLength, src);
        Assert.assertThat((Object)fs.getFileStatus(src).getLen(), (Matcher)Is.is((Object)oldLength));
        try (FsShell shell = null;){
            shell = new FsShell(conf);
            Assert.assertThat((Object)ToolRunner.run((Tool)shell, (String[])shellOpts), (Matcher)Is.is((Object)0));
        }
    }

    @Test
    public void testTruncate4Symlink() throws IOException {
        int fileLength = 12;
        fs.mkdirs(this.parent);
        byte[] contents = AppendTestUtil.initBuffer(12);
        Path file = new Path(this.parent, "testTruncate4Symlink");
        TestFileTruncate.writeContents(contents, 12, file);
        Path link = new Path(this.parent, "link");
        fs.createSymlink(file, link, false);
        int newLength = 4;
        boolean isReady = fs.truncate(link, 4L);
        Assert.assertTrue((String)"Recovery is not expected.", (boolean)isReady);
        FileStatus fileStatus = fs.getFileStatus(file);
        Assert.assertThat((Object)fileStatus.getLen(), (Matcher)Is.is((Object)4L));
        ContentSummary cs = fs.getContentSummary(this.parent);
        Assert.assertEquals((String)"Bad disk space usage", (long)cs.getSpaceConsumed(), (long)12L);
        TestFileTruncate.checkFullFile(file, 4, contents);
        fs.delete(this.parent, true);
    }

    @Test
    public void testTruncateWithRollingUpgrade() throws Exception {
        DFSAdmin dfsadmin = new DFSAdmin(cluster.getConfiguration(0));
        DistributedFileSystem dfs = cluster.getFileSystem();
        dfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_ENTER);
        int status = dfsadmin.run(new String[]{"-rollingUpgrade", "prepare"});
        Assert.assertEquals((String)"could not prepare for rolling upgrade", (long)0L, (long)status);
        dfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_LEAVE);
        Path dir = new Path("/testTruncateWithRollingUpgrade");
        fs.mkdirs(dir);
        Path p = new Path(dir, "file");
        byte[] data = new byte[3];
        ThreadLocalRandom.current().nextBytes(data);
        TestFileTruncate.writeContents(data, data.length, p);
        Assert.assertEquals((String)"block num should 1", (long)1L, (long)cluster.getNamesystem().getFSDirectory().getBlockManager().getTotalBlocks());
        boolean isReady = fs.truncate(p, 2L);
        Assert.assertFalse((String)"should be copy-on-truncate", (boolean)isReady);
        Assert.assertEquals((String)"block num should 2", (long)2L, (long)cluster.getNamesystem().getFSDirectory().getBlockManager().getTotalBlocks());
        fs.delete(p, true);
        Assert.assertEquals((String)"block num should 0", (long)0L, (long)cluster.getNamesystem().getFSDirectory().getBlockManager().getTotalBlocks());
        status = dfsadmin.run(new String[]{"-rollingUpgrade", "finalize"});
        Assert.assertEquals((String)"could not finalize rolling upgrade", (long)0L, (long)status);
    }

    static void writeContents(byte[] contents, int fileLength, Path p) throws IOException {
        FSDataOutputStream out = fs.create(p, true, 4, (short)3, 4L);
        out.write(contents, 0, fileLength);
        out.close();
    }

    static void checkBlockRecovery(Path p) throws IOException {
        TestFileTruncate.checkBlockRecovery(p, fs);
    }

    public static void checkBlockRecovery(Path p, DistributedFileSystem dfs) throws IOException {
        TestFileTruncate.checkBlockRecovery(p, dfs, 300, 100L);
    }

    public static void checkBlockRecovery(Path p, DistributedFileSystem dfs, int attempts, long sleepMs) throws IOException {
        boolean success = false;
        for (int i = 0; i < attempts; ++i) {
            boolean noLastBlock;
            LocatedBlocks blocks = TestFileTruncate.getLocatedBlocks(p, dfs);
            boolean bl = noLastBlock = blocks.getLastLocatedBlock() == null;
            if (!blocks.isUnderConstruction() && (noLastBlock || blocks.isLastBlockComplete())) {
                success = true;
                break;
            }
            try {
                Thread.sleep(sleepMs);
                continue;
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        Assert.assertThat((String)("inode should complete in ~" + sleepMs * (long)attempts + " ms."), (Object)success, (Matcher)Is.is((Object)true));
    }

    static LocatedBlocks getLocatedBlocks(Path src) throws IOException {
        return TestFileTruncate.getLocatedBlocks(src, fs);
    }

    static LocatedBlocks getLocatedBlocks(Path src, DistributedFileSystem dfs) throws IOException {
        return dfs.getClient().getLocatedBlocks(src.toString(), 0L, Long.MAX_VALUE);
    }

    static void assertBlockExists(Block blk) {
        Assert.assertNotNull((String)("BlocksMap does not contain block: " + blk), (Object)cluster.getNamesystem().getStoredBlock(blk));
    }

    static void assertBlockNotPresent(Block blk) {
        Assert.assertNull((String)("BlocksMap should not contain block: " + blk), (Object)cluster.getNamesystem().getStoredBlock(blk));
    }

    static void assertFileLength(Path file, long length) throws IOException {
        byte[] data = DFSTestUtil.readFileBuffer((FileSystem)fs, file);
        Assert.assertEquals((String)"Wrong data size in snapshot.", (long)length, (long)data.length);
    }

    static void checkFullFile(Path p, int newLength, byte[] contents) throws IOException {
        AppendTestUtil.checkFullFile((FileSystem)fs, p, newLength, contents, p.toString());
    }

    static void restartCluster(HdfsServerConstants.StartupOption o) throws IOException {
        cluster.shutdown();
        if (HdfsServerConstants.StartupOption.ROLLBACK == o) {
            NameNode.doRollback((Configuration)conf, (boolean)false);
        }
        cluster = new MiniDFSCluster.Builder(conf).numDataNodes(3).format(false).startupOption(o == HdfsServerConstants.StartupOption.ROLLBACK ? HdfsServerConstants.StartupOption.REGULAR : o).dnStartupOption(o != HdfsServerConstants.StartupOption.ROLLBACK ? HdfsServerConstants.StartupOption.REGULAR : o).build();
        fs = cluster.getFileSystem();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void truncateAndRestartDN(Path p, int dn, int newLength) throws IOException {
        try {
            boolean isReady = fs.truncate(p, (long)newLength);
            Assert.assertFalse((boolean)isReady);
        }
        finally {
            cluster.restartDataNode(dn, false, true);
            cluster.waitActive();
        }
    }

    @Test
    public void testConcatOnInodeRefernce() throws IOException {
        String dir = "/testConcat";
        Path trgDir = new Path(dir);
        fs.mkdirs(new Path(dir), FsPermission.getDirDefault());
        Path trg = new Path(dir, "file");
        DFSTestUtil.createFile((FileSystem)fs, trg, 512L, (short)2, 0L);
        String dir2 = "/dir2";
        Path srcDir = new Path(dir2);
        fs.mkdirs(srcDir);
        fs.allowSnapshot(srcDir);
        Path src = new Path(srcDir, "file1");
        DFSTestUtil.createFile((FileSystem)fs, src, 512L, (short)2, 0L);
        fs.createSnapshot(srcDir, "s1");
        fs.rename(src, trgDir);
        fs.deleteSnapshot(srcDir, "s1");
        Path[] srcs = new Path[]{new Path(dir, "file1")};
        Assert.assertEquals((long)2L, (long)fs.getContentSummary(new Path(dir)).getFileCount());
        fs.concat(trg, srcs);
        Assert.assertEquals((long)1L, (long)fs.getContentSummary(new Path(dir)).getFileCount());
    }

    static {
        GenericTestUtils.setLogLevel((Logger)NameNode.stateChangeLog, (Level)Level.ALL);
        GenericTestUtils.setLogLevel((Log)FSEditLogLoader.LOG, (Level)Level.ALL);
        LOG = LogFactory.getLog(TestFileTruncate.class);
    }
}

