/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.qjournal.server;

import com.google.common.annotations.VisibleForTesting;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.qjournal.server.Journal;
import org.apache.hadoop.hdfs.server.namenode.EditLogFileOutputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogLoader;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogOp;
import org.apache.hadoop.util.AutoCloseableLock;

@InterfaceAudience.Private
@InterfaceStability.Evolving
class JournaledEditsCache {
    private static final int INVALID_LAYOUT_VERSION = 0;
    private static final long INVALID_TXN_ID = -1L;
    private final int capacity;
    private final AutoCloseableLock readLock;
    private final AutoCloseableLock writeLock;
    private final NavigableMap<Long, byte[]> dataMap = new TreeMap<Long, byte[]>();
    private int layoutVersion = 0;
    private ByteBuffer layoutHeader;
    private long lowestTxnId;
    private long highestTxnId;
    private long initialTxnId;
    private int totalSize;

    JournaledEditsCache(Configuration conf) {
        this.capacity = conf.getInt("dfs.journalnode.edit-cache-size.bytes", 0x100000);
        if ((double)this.capacity > 0.9 * (double)Runtime.getRuntime().maxMemory()) {
            Journal.LOG.warn((Object)String.format("Cache capacity is set at %d bytes but maximum JVM memory is only %d bytes. It is recommended that you decrease the cache size or increase the heap size.", this.capacity, Runtime.getRuntime().maxMemory()));
        }
        Journal.LOG.info((Object)("Enabling the journaled edits cache with a capacity of bytes: " + this.capacity));
        ReentrantReadWriteLock lock = new ReentrantReadWriteLock(true);
        this.readLock = new AutoCloseableLock(lock.readLock());
        this.writeLock = new AutoCloseableLock(lock.writeLock());
        this.initialize(-1L);
    }

    int retrieveEdits(long requestedStartTxn, int maxTxns, List<ByteBuffer> outputBuffers) throws IOException {
        int txnCount = 0;
        try (AutoCloseableLock l = this.readLock.acquire();){
            if (this.lowestTxnId == -1L || requestedStartTxn < this.lowestTxnId) {
                throw this.getCacheMissException(requestedStartTxn);
            }
            if (requestedStartTxn > this.highestTxnId) {
                int n = 0;
                return n;
            }
            outputBuffers.add(this.layoutHeader);
            Iterator incrBuffIter = this.dataMap.tailMap(this.dataMap.floorKey(requestedStartTxn), true).entrySet().iterator();
            long prevTxn = requestedStartTxn;
            byte[] prevBuf = null;
            while (txnCount < maxTxns && (incrBuffIter.hasNext() || prevBuf != null)) {
                byte[] currBuf;
                long currTxn;
                if (incrBuffIter.hasNext()) {
                    Map.Entry ent = incrBuffIter.next();
                    currTxn = (Long)ent.getKey();
                    currBuf = (byte[])ent.getValue();
                } else {
                    currTxn = this.highestTxnId + 1L;
                    currBuf = null;
                }
                if (prevBuf != null) {
                    outputBuffers.add(ByteBuffer.wrap(prevBuf));
                    txnCount = (int)((long)txnCount + (currTxn - Math.max(requestedStartTxn, prevTxn)));
                }
                prevTxn = currTxn;
                prevBuf = currBuf;
            }
        }
        ByteBuffer firstBuf = outputBuffers.get(1);
        firstBuf.position(this.findTransactionPosition(firstBuf.array(), requestedStartTxn));
        if (txnCount > maxTxns) {
            ByteBuffer lastBuf = outputBuffers.get(outputBuffers.size() - 1);
            int limit = this.findTransactionPosition(lastBuf.array(), requestedStartTxn + (long)maxTxns);
            lastBuf.limit(limit);
            txnCount = maxTxns;
        }
        return txnCount;
    }

    void storeEdits(byte[] inputData, long newStartTxn, long newEndTxn, int newLayoutVersion) {
        if (newStartTxn < 0L || newEndTxn < newStartTxn) {
            Journal.LOG.error((Object)String.format("Attempted to cache data of length %d with newStartTxn %d and newEndTxn %d", inputData.length, newStartTxn, newEndTxn));
            return;
        }
        try (AutoCloseableLock l = this.writeLock.acquire();){
            if (newLayoutVersion != this.layoutVersion) {
                try {
                    this.updateLayoutVersion(newLayoutVersion, newStartTxn);
                }
                catch (IOException ioe) {
                    Journal.LOG.error((Object)String.format("Unable to save new edits [%d, %d] due to exception when updating to new layout version %d", newStartTxn, newEndTxn, newLayoutVersion), (Throwable)ioe);
                    if (l != null) {
                        if (var8_6 != null) {
                            try {
                                l.close();
                            }
                            catch (Throwable throwable) {
                                var8_6.addSuppressed(throwable);
                            }
                        } else {
                            l.close();
                        }
                    }
                    return;
                }
            } else if (this.lowestTxnId == -1L) {
                Journal.LOG.info((Object)("Initializing edits cache starting from txn ID " + newStartTxn));
                this.initialize(newStartTxn);
            } else if (this.highestTxnId + 1L != newStartTxn) {
                Journal.LOG.error((Object)String.format("Edits cache is out of sync; looked for next txn id at %d but got start txn id for cache put request at %d. Reinitializing at new request.", this.highestTxnId + 1L, newStartTxn));
                this.initialize(newStartTxn);
            }
            while (this.totalSize + inputData.length > this.capacity && !this.dataMap.isEmpty()) {
                Map.Entry<Long, byte[]> lowest = this.dataMap.firstEntry();
                this.dataMap.remove(lowest.getKey());
                this.totalSize -= lowest.getValue().length;
            }
            if (inputData.length > this.capacity) {
                this.initialize(-1L);
                Journal.LOG.warn((Object)String.format("A single batch of edits was too large to fit into the cache: startTxn = %d, endTxn = %d, input length = %d. The capacity of the cache (%s) must be increased for it to work properly (current capacity %d).Cache is now empty.", newStartTxn, newEndTxn, inputData.length, "dfs.journalnode.edit-cache-size.bytes", this.capacity));
                return;
            }
            this.lowestTxnId = this.dataMap.isEmpty() ? newStartTxn : (Long)this.dataMap.firstKey();
            this.dataMap.put(newStartTxn, inputData);
            this.highestTxnId = newEndTxn;
            this.totalSize += inputData.length;
        }
    }

    private int findTransactionPosition(byte[] buf, long txnId) throws IOException {
        ByteArrayInputStream bais = new ByteArrayInputStream(buf);
        FSEditLogLoader.PositionTrackingInputStream tracker = new FSEditLogLoader.PositionTrackingInputStream(bais);
        FSEditLogOp.Reader reader = FSEditLogOp.Reader.create(new DataInputStream(tracker), tracker, this.layoutVersion);
        long previousPos = 0L;
        while (reader.scanOp() < txnId) {
            previousPos = tracker.getPos();
        }
        return (int)previousPos;
    }

    private void updateLayoutVersion(int newLayoutVersion, long newStartTxn) throws IOException {
        StringBuilder logMsg = new StringBuilder().append("Updating edits cache to use layout version ").append(newLayoutVersion).append(" starting from txn ID ").append(newStartTxn);
        if (this.layoutVersion != 0) {
            logMsg.append("; previous version was ").append(this.layoutVersion).append("; old entries will be cleared.");
        }
        Journal.LOG.info((Object)logMsg.toString());
        this.initialize(newStartTxn);
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        EditLogFileOutputStream.writeHeader(newLayoutVersion, new DataOutputStream(baos));
        this.layoutVersion = newLayoutVersion;
        this.layoutHeader = ByteBuffer.wrap(baos.toByteArray());
    }

    private void initialize(long newInitialTxnId) {
        this.dataMap.clear();
        this.totalSize = 0;
        this.lowestTxnId = this.initialTxnId = newInitialTxnId;
        this.highestTxnId = -1L;
    }

    @VisibleForTesting
    byte[] getRawDataForTests(long txnId) {
        try (AutoCloseableLock l = this.readLock.acquire();){
            byte[] byArray = this.dataMap.floorEntry(txnId).getValue();
            return byArray;
        }
    }

    private CacheMissException getCacheMissException(long requestedTxnId) {
        if (this.lowestTxnId == -1L) {
            return new CacheMissException(0L, "Cache is empty; either it was never written to or the last write overflowed the cache capacity.", new Object[0]);
        }
        if (requestedTxnId < this.initialTxnId) {
            return new CacheMissException(this.initialTxnId - requestedTxnId, "Cache started at txn ID %d but requested txns starting at %d.", this.initialTxnId, requestedTxnId);
        }
        return new CacheMissException(this.lowestTxnId - requestedTxnId, "Oldest txn ID available in the cache is %d, but requested txns starting at %d. The cache size (%s) may need to be increased to hold more transactions (currently %d bytes containing %d transactions)", this.lowestTxnId, requestedTxnId, "dfs.journalnode.edit-cache-size.bytes", this.capacity, this.highestTxnId - this.lowestTxnId + 1L);
    }

    static class CacheMissException
    extends IOException {
        private static final long serialVersionUID = 0L;
        private final long cacheMissAmount;

        CacheMissException(long cacheMissAmount, String msgFormat, Object ... msgArgs) {
            super(String.format(msgFormat, msgArgs));
            this.cacheMissAmount = cacheMissAmount;
        }

        long getCacheMissAmount() {
            return this.cacheMissAmount;
        }
    }
}

