/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.mapred.uploader;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.NotLinkException;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.GZIPOutputStream;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.SystemErasureCodingPolicies;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.mapred.uploader.UploaderException;
import org.apache.hadoop.util.GenericOptionsParser;
import org.apache.hadoop.util.Shell;
import org.apache.hadoop.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FrameworkUploader
implements Runnable {
    private static final Pattern VAR_SUBBER = Pattern.compile(Shell.getEnvironmentVariableRegex());
    private static final Logger LOG = LoggerFactory.getLogger(FrameworkUploader.class);
    private Configuration conf = new Configuration();
    @VisibleForTesting
    String input = null;
    @VisibleForTesting
    String whitelist = null;
    @VisibleForTesting
    String blacklist = null;
    @VisibleForTesting
    String target = null;
    @VisibleForTesting
    Path targetPath = null;
    @VisibleForTesting
    short initialReplication = (short)3;
    @VisibleForTesting
    short finalReplication = (short)10;
    @VisibleForTesting
    short acceptableReplication = (short)9;
    @VisibleForTesting
    int timeout = 10;
    private boolean ignoreSymlink = false;
    @VisibleForTesting
    Set<String> filteredInputFiles = new HashSet<String>();
    @VisibleForTesting
    List<Pattern> whitelistedFiles = new LinkedList<Pattern>();
    @VisibleForTesting
    List<Pattern> blacklistedFiles = new LinkedList<Pattern>();
    private OutputStream targetStream = null;
    private String alias = null;

    @VisibleForTesting
    void setConf(Configuration configuration) {
        this.conf = configuration;
    }

    private void printHelp(Options options) {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp("mapred frameworkuploader", options);
    }

    @Override
    public void run() {
        try {
            this.collectPackages();
            this.buildPackage();
            LOG.info("Uploaded " + this.target);
            System.out.println("Suggested mapreduce.application.framework.path " + this.target);
            LOG.info("Suggested mapreduce.application.classpath $PWD/" + this.alias + "/*");
            System.out.println("Suggested classpath $PWD/" + this.alias + "/*");
        }
        catch (IOException | InterruptedException | UploaderException e) {
            LOG.error("Error in execution " + e.getMessage());
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    @VisibleForTesting
    void collectPackages() throws UploaderException {
        String[] list;
        this.parseLists();
        for (String item : list = StringUtils.split((String)this.input, (char)File.pathSeparatorChar)) {
            LOG.info("Original source " + item);
            String expanded = this.expandEnvironmentVariables(item, System.getenv());
            LOG.info("Expanded source " + expanded);
            if (expanded.endsWith("*")) {
                File path = new File(expanded.substring(0, expanded.length() - 1));
                if (path.isDirectory()) {
                    File[] files = path.listFiles();
                    if (files != null) {
                        for (File jar : files) {
                            if (!jar.isDirectory()) {
                                this.addJar(jar);
                                continue;
                            }
                            LOG.info("Ignored " + jar + " because it is a directory");
                        }
                        continue;
                    }
                    LOG.warn("Could not list directory " + path);
                    continue;
                }
                LOG.warn("Ignored " + expanded + ". It is not a directory");
                continue;
            }
            if (expanded.endsWith(".jar")) {
                File jarFile = new File(expanded);
                this.addJar(jarFile);
                continue;
            }
            if (expanded.isEmpty()) continue;
            LOG.warn("Ignored " + expanded + " only jars are supported");
        }
    }

    @VisibleForTesting
    void beginUpload() throws IOException, UploaderException {
        if (this.targetStream == null) {
            this.validateTargetPath();
            int lastIndex = this.target.indexOf(35);
            this.targetPath = new Path(this.target.substring(0, lastIndex == -1 ? this.target.length() : lastIndex));
            this.alias = lastIndex != -1 ? this.target.substring(lastIndex + 1) : this.targetPath.getName();
            LOG.info("Target " + this.targetPath);
            FileSystem fileSystem = this.targetPath.getFileSystem(this.conf);
            this.targetStream = null;
            if (fileSystem instanceof DistributedFileSystem) {
                LOG.info("Set replication to " + this.initialReplication + " for path: " + this.targetPath);
                LOG.info("Disabling Erasure Coding for path: " + this.targetPath);
                DistributedFileSystem dfs = (DistributedFileSystem)fileSystem;
                DistributedFileSystem.HdfsDataOutputStreamBuilder builder = ((DistributedFileSystem.HdfsDataOutputStreamBuilder)dfs.createFile(this.targetPath).overwrite(true)).ecPolicyName(SystemErasureCodingPolicies.getReplicationPolicy().getName());
                if (this.initialReplication > 0) {
                    builder.replication(this.initialReplication);
                }
                this.targetStream = builder.build();
            } else {
                LOG.warn("Cannot set replication to " + this.initialReplication + " for path: " + this.targetPath + " on a non-distributed fileystem " + fileSystem.getClass().getName());
            }
            if (this.targetStream == null) {
                this.targetStream = fileSystem.create(this.targetPath, true);
            }
            if (this.targetPath.getName().endsWith("gz") || this.targetPath.getName().endsWith("tgz")) {
                LOG.info("Creating GZip");
                this.targetStream = new GZIPOutputStream(this.targetStream);
            }
        }
    }

    private long getSmallestReplicatedBlockCount() throws IOException {
        BlockLocation[] locations;
        FileSystem fileSystem = this.targetPath.getFileSystem(this.conf);
        FileStatus status = fileSystem.getFileStatus(this.targetPath);
        long length = status.getLen();
        HashMap<Long, Integer> blockCount = new HashMap<Long, Integer>();
        for (long offset = 0L; offset < length; offset += status.getBlockSize()) {
            blockCount.put(offset, 0);
        }
        for (BlockLocation location : locations = fileSystem.getFileBlockLocations(this.targetPath, 0L, length)) {
            int replicas = location.getHosts().length;
            blockCount.compute(location.getOffset(), (key, value) -> value == null ? 0 : value + replicas);
        }
        for (long offset = 0L; offset < length; offset += status.getBlockSize()) {
            LOG.info(String.format("Replication counts offset:%d blocks:%d", offset, blockCount.get(offset)));
        }
        return ((Integer)Collections.min(blockCount.values())).intValue();
    }

    private void endUpload() throws IOException, InterruptedException {
        FileSystem fileSystem = this.targetPath.getFileSystem(this.conf);
        if (fileSystem instanceof DistributedFileSystem) {
            long startTime;
            fileSystem.setReplication(this.targetPath, this.finalReplication);
            LOG.info("Set replication to " + this.finalReplication + " for path: " + this.targetPath);
            long endTime = startTime = System.currentTimeMillis();
            long currentReplication = 0L;
            while (endTime - startTime < (long)(this.timeout * 1000) && currentReplication < (long)this.acceptableReplication) {
                Thread.sleep(1000L);
                endTime = System.currentTimeMillis();
                currentReplication = this.getSmallestReplicatedBlockCount();
            }
            if (endTime - startTime >= (long)(this.timeout * 1000)) {
                LOG.error(String.format("Timed out after %d seconds while waiting for acceptable replication of %d (current replication is %d)", this.timeout, this.acceptableReplication, currentReplication));
            }
        } else {
            LOG.info("Cannot set replication to " + this.finalReplication + " for path: " + this.targetPath + " on a non-distributed fileystem " + fileSystem.getClass().getName());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    void buildPackage() throws IOException, UploaderException, InterruptedException {
        this.beginUpload();
        LOG.info("Compressing tarball");
        try (TarArchiveOutputStream out = new TarArchiveOutputStream(this.targetStream);){
            for (String fullPath : this.filteredInputFiles) {
                LOG.info("Adding " + fullPath);
                File file = new File(fullPath);
                FileInputStream inputStream = new FileInputStream(file);
                Throwable throwable = null;
                try {
                    ArchiveEntry entry = out.createArchiveEntry(file, file.getName());
                    out.putArchiveEntry(entry);
                    IOUtils.copyBytes((InputStream)inputStream, (OutputStream)out, (int)0x100000);
                    out.closeArchiveEntry();
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (inputStream == null) continue;
                    if (throwable != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    inputStream.close();
                }
            }
            this.endUpload();
        }
        finally {
            if (this.targetStream != null) {
                this.targetStream.close();
            }
        }
    }

    private void parseLists() throws UploaderException {
        String[] stringArray;
        String[] whiteListItems;
        Map<String, String> env = System.getenv();
        for (Map.Entry<String, String> entry : env.entrySet()) {
            LOG.info("Environment " + entry.getKey() + " " + entry.getValue());
        }
        for (String pattern : whiteListItems = StringUtils.split((String)this.whitelist)) {
            String expandedPattern = this.expandEnvironmentVariables(pattern, env);
            Pattern compiledPattern = Pattern.compile("^" + expandedPattern + "$");
            LOG.info("Whitelisted " + compiledPattern.toString());
            this.whitelistedFiles.add(compiledPattern);
        }
        for (String pattern : stringArray = StringUtils.split((String)this.blacklist)) {
            String expandedPattern = this.expandEnvironmentVariables(pattern, env);
            Pattern compiledPattern = Pattern.compile("^" + expandedPattern + "$");
            LOG.info("Blacklisted " + compiledPattern.toString());
            this.blacklistedFiles.add(compiledPattern);
        }
    }

    @VisibleForTesting
    String expandEnvironmentVariables(String innerInput, Map<String, String> env) throws UploaderException {
        boolean found;
        do {
            found = false;
            Matcher matcher = VAR_SUBBER.matcher(innerInput);
            StringBuffer stringBuffer = new StringBuffer();
            while (matcher.find()) {
                found = true;
                String var = matcher.group(1);
                String replace = env.get(var);
                if (replace == null) {
                    throw new UploaderException("Environment variable does not exist " + var);
                }
                matcher.appendReplacement(stringBuffer, Matcher.quoteReplacement(replace));
            }
            matcher.appendTail(stringBuffer);
            innerInput = stringBuffer.toString();
        } while (found);
        return innerInput;
    }

    private void addJar(File jar) throws UploaderException {
        boolean found = false;
        if (!jar.getName().endsWith(".jar")) {
            LOG.info("Ignored non-jar " + jar.getAbsolutePath());
        }
        for (Pattern pattern : this.whitelistedFiles) {
            Matcher matcher = pattern.matcher(jar.getAbsolutePath());
            if (!matcher.matches()) continue;
            LOG.info("Whitelisted " + jar.getAbsolutePath());
            found = true;
            break;
        }
        boolean excluded = false;
        for (Pattern pattern : this.blacklistedFiles) {
            Matcher matcher = pattern.matcher(jar.getAbsolutePath());
            if (!matcher.matches()) continue;
            LOG.info("Blacklisted " + jar.getAbsolutePath());
            excluded = true;
            break;
        }
        if (this.ignoreSymlink && !excluded) {
            excluded = this.checkSymlink(jar);
        }
        if (found && !excluded) {
            LOG.info("Whitelisted " + jar.getAbsolutePath());
            if (!this.filteredInputFiles.add(jar.getAbsolutePath())) {
                throw new UploaderException("Duplicate jar" + jar.getAbsolutePath());
            }
        }
        if (!found) {
            LOG.info("Ignored " + jar.getAbsolutePath() + " because it is missing from the whitelist");
        } else if (excluded) {
            LOG.info("Ignored " + jar.getAbsolutePath() + " because it is on the the blacklist");
        }
    }

    @VisibleForTesting
    boolean checkSymlink(File jar) {
        if (Files.isSymbolicLink(jar.toPath())) {
            try {
                java.nio.file.Path normalizedLinkPath;
                java.nio.file.Path link = Files.readSymbolicLink(jar.toPath());
                java.nio.file.Path jarPath = Paths.get(jar.getAbsolutePath(), new String[0]);
                String linkString = link.toString();
                java.nio.file.Path jarParent = jarPath.getParent();
                java.nio.file.Path linkPath = jarParent == null ? null : jarParent.resolve(linkString);
                java.nio.file.Path linkPathParent = linkPath == null ? null : linkPath.getParent();
                java.nio.file.Path path = normalizedLinkPath = linkPathParent == null ? null : linkPathParent.normalize();
                if (normalizedLinkPath != null && jarParent.equals(normalizedLinkPath)) {
                    LOG.info(String.format("Ignoring same directory link %s to %s", jarPath.toString(), link.toString()));
                    return true;
                }
            }
            catch (NotLinkException ex) {
                LOG.debug("Not a link", (Object)jar);
            }
            catch (IOException ex) {
                LOG.warn("Cannot read symbolic link on", (Object)jar);
            }
        }
        return false;
    }

    private void validateTargetPath() throws UploaderException {
        if (!this.target.startsWith("hdfs:/") && !this.target.startsWith("file:/")) {
            throw new UploaderException("Target path is not hdfs or local " + this.target);
        }
    }

    @VisibleForTesting
    boolean parseArguments(String[] args) throws IOException {
        boolean isFullPath;
        Options opts = new Options();
        opts.addOption(OptionBuilder.create((String)"h"));
        opts.addOption(OptionBuilder.create((String)"help"));
        OptionBuilder.withDescription((String)"Input class path. Defaults to the default classpath.");
        OptionBuilder.hasArg();
        opts.addOption(OptionBuilder.create((String)"input"));
        OptionBuilder.withDescription((String)"Regex specifying the full path of jars to include in the framework tarball. Default is a hardcoded set of jars considered necessary to include");
        OptionBuilder.hasArg();
        opts.addOption(OptionBuilder.create((String)"whitelist"));
        OptionBuilder.withDescription((String)"Regex specifying the full path of jars to exclude in the framework tarball. Default is a hardcoded set of jars considered unnecessary to include");
        OptionBuilder.hasArg();
        opts.addOption(OptionBuilder.create((String)"blacklist"));
        OptionBuilder.withDescription((String)"Target file system to upload to. Example: hdfs://foo.com:8020");
        OptionBuilder.hasArg();
        opts.addOption(OptionBuilder.create((String)"fs"));
        OptionBuilder.withDescription((String)"Target file to upload to with a reference name. Example: /usr/mr-framework.tar.gz#mr-framework");
        OptionBuilder.hasArg();
        opts.addOption(OptionBuilder.create((String)"target"));
        OptionBuilder.withDescription((String)"Desired initial replication count. Default 3.");
        OptionBuilder.hasArg();
        opts.addOption(OptionBuilder.create((String)"initialReplication"));
        OptionBuilder.withDescription((String)"Desired final replication count. Default 10.");
        OptionBuilder.hasArg();
        opts.addOption(OptionBuilder.create((String)"finalReplication"));
        OptionBuilder.withDescription((String)"Desired acceptable replication count. Default 9.");
        OptionBuilder.hasArg();
        opts.addOption(OptionBuilder.create((String)"acceptableReplication"));
        OptionBuilder.withDescription((String)"Desired timeout for the acceptable replication in seconds. Default 10");
        OptionBuilder.hasArg();
        opts.addOption(OptionBuilder.create((String)"timeout"));
        OptionBuilder.withDescription((String)"Ignore symlinks into the same directory");
        opts.addOption(OptionBuilder.create((String)"nosymlink"));
        GenericOptionsParser parser = new GenericOptionsParser(opts, args);
        if (parser.getCommandLine().hasOption("help") || parser.getCommandLine().hasOption("h")) {
            this.printHelp(opts);
            return false;
        }
        this.input = parser.getCommandLine().getOptionValue("input", System.getProperty("java.class.path"));
        this.whitelist = parser.getCommandLine().getOptionValue("whitelist", "$HADOOP_HOME/share/hadoop/common/.*\\.jar,$HADOOP_HOME/share/hadoop/common/lib/.*\\.jar,$HADOOP_HOME/share/hadoop/hdfs/.*\\.jar,$HADOOP_HOME/share/hadoop/hdfs/lib/.*\\.jar,$HADOOP_HOME/share/hadoop/mapreduce/.*\\.jar,$HADOOP_HOME/share/hadoop/mapreduce/lib/.*\\.jar,$HADOOP_HOME/share/hadoop/yarn/.*\\.jar,$HADOOP_HOME/share/hadoop/yarn/lib/.*\\.jar,");
        this.blacklist = parser.getCommandLine().getOptionValue("blacklist", ".*hadoop-yarn-server-applicationhistoryservice.*\\.jar,.*hadoop-yarn-server-nodemanager.*\\.jar,.*hadoop-yarn-server-resourcemanager.*\\.jar,.*hadoop-yarn-server-router.*\\.jar,.*hadoop-yarn-server-sharedcachemanager.*\\.jar,.*hadoop-yarn-server-timeline-pluginstorage.*\\.jar,.*hadoop-yarn-server-timelineservice.*\\.jar,.*hadoop-yarn-server-timelineservice-hbase.*\\.jar,");
        this.initialReplication = Short.parseShort(parser.getCommandLine().getOptionValue("initialReplication", "3"));
        this.finalReplication = Short.parseShort(parser.getCommandLine().getOptionValue("finalReplication", "10"));
        this.acceptableReplication = Short.parseShort(parser.getCommandLine().getOptionValue("acceptableReplication", "9"));
        this.timeout = Integer.parseInt(parser.getCommandLine().getOptionValue("timeout", "10"));
        if (parser.getCommandLine().hasOption("nosymlink")) {
            this.ignoreSymlink = true;
        }
        String fs = parser.getCommandLine().getOptionValue("fs", null);
        String path = parser.getCommandLine().getOptionValue("target", "/usr/lib/mr-framework.tar.gz#mr-framework");
        boolean bl = isFullPath = path.startsWith("hdfs://") || path.startsWith("file://");
        if (fs == null) {
            fs = this.conf.get("fs.defaultFS");
            if (fs == null && !isFullPath) {
                LOG.error("No filesystem specified in either fs or target.");
                this.printHelp(opts);
                return false;
            }
            LOG.info(String.format("Target file system not specified. Using default %s", fs));
        }
        if (path.isEmpty()) {
            LOG.error("Target directory not specified");
            this.printHelp(opts);
            return false;
        }
        StringBuilder absolutePath = new StringBuilder();
        if (!isFullPath) {
            absolutePath.append(fs);
            absolutePath.append(path.startsWith("/") ? "" : "/");
        }
        absolutePath.append(path);
        this.target = absolutePath.toString();
        if (parser.getRemainingArgs().length > 0) {
            LOG.warn("Unexpected parameters");
            this.printHelp(opts);
            return false;
        }
        return true;
    }

    public static void main(String[] args) throws IOException {
        FrameworkUploader uploader = new FrameworkUploader();
        if (uploader.parseArguments(args)) {
            uploader.run();
        }
    }
}

