/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.swift.snative;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.swift.exceptions.SwiftConfigurationException;
import org.apache.hadoop.fs.swift.exceptions.SwiftOperationFailedException;
import org.apache.hadoop.fs.swift.exceptions.SwiftUnsupportedFeatureException;
import org.apache.hadoop.fs.swift.snative.StrictBufferedFSInputStream;
import org.apache.hadoop.fs.swift.snative.SwiftNativeFileSystemStore;
import org.apache.hadoop.fs.swift.snative.SwiftNativeInputStream;
import org.apache.hadoop.fs.swift.snative.SwiftNativeOutputStream;
import org.apache.hadoop.fs.swift.util.DurationStats;
import org.apache.hadoop.fs.swift.util.SwiftObjectPath;
import org.apache.hadoop.fs.swift.util.SwiftUtils;
import org.apache.hadoop.util.Progressable;

public class SwiftNativeFileSystem
extends FileSystem {
    public static final String SWIFT = "swift";
    private static final Log LOG = LogFactory.getLog(SwiftNativeFileSystem.class);
    private Path workingDir;
    private URI uri;
    private SwiftNativeFileSystemStore store;

    public SwiftNativeFileSystem() {
    }

    public SwiftNativeFileSystem(SwiftNativeFileSystemStore store) {
        this.store = store;
    }

    public SwiftNativeFileSystemStore getStore() {
        return this.store;
    }

    public String getScheme() {
        return SWIFT;
    }

    public void initialize(URI fsuri, Configuration conf) throws IOException {
        super.initialize(fsuri, conf);
        this.setConf(conf);
        if (this.store == null) {
            this.store = new SwiftNativeFileSystemStore();
        }
        this.uri = fsuri;
        String username = System.getProperty("user.name");
        this.workingDir = new Path("/user", username).makeQualified(this.uri, new Path(username));
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Initializing SwiftNativeFileSystem against URI " + this.uri + " and working dir " + this.workingDir));
        }
        this.store.initialize(this.uri, conf);
        LOG.debug((Object)"SwiftFileSystem initialized");
    }

    public URI getUri() {
        return this.uri;
    }

    public String toString() {
        return "Swift FileSystem " + this.store;
    }

    public Path getWorkingDirectory() {
        return this.workingDir;
    }

    public void setWorkingDirectory(Path dir) {
        this.workingDir = this.makeAbsolute(dir);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("SwiftFileSystem.setWorkingDirectory to " + dir));
        }
    }

    public FileStatus getFileStatus(Path path) throws IOException {
        Path absolutePath = this.makeAbsolute(path);
        return this.store.getObjectMetadata(absolutePath);
    }

    public long getDefaultBlockSize() {
        return this.store.getBlocksize();
    }

    public long getDefaultBlockSize(Path f) {
        return this.store.getBlocksize();
    }

    public long getBlockSize(Path path) throws IOException {
        return this.store.getBlocksize();
    }

    public boolean isFile(Path f) throws IOException {
        try {
            FileStatus fileStatus = this.getFileStatus(f);
            return !SwiftUtils.isDirectory(fileStatus);
        }
        catch (FileNotFoundException e) {
            return false;
        }
    }

    public boolean isDirectory(Path f) throws IOException {
        try {
            FileStatus fileStatus = this.getFileStatus(f);
            return SwiftUtils.isDirectory(fileStatus);
        }
        catch (FileNotFoundException e) {
            return false;
        }
    }

    public BlockLocation[] getFileBlockLocations(FileStatus file, long start, long len) throws IOException {
        if (file == null) {
            return null;
        }
        if (start < 0L || len < 0L) {
            throw new IllegalArgumentException("Negative start or len parameter to getFileBlockLocations");
        }
        if (file.getLen() <= start) {
            return new BlockLocation[0];
        }
        FileStatus[] listOfFileBlocks = this.store.listSubPaths(file.getPath(), false, true);
        List<Object> locations = new ArrayList();
        if (listOfFileBlocks.length > 1) {
            for (FileStatus fileStatus : listOfFileBlocks) {
                if (SwiftObjectPath.fromPath(this.uri, fileStatus.getPath()).equals(SwiftObjectPath.fromPath(this.uri, file.getPath()))) continue;
                locations.addAll(this.store.getObjectLocation(fileStatus.getPath()));
            }
        } else {
            locations = this.store.getObjectLocation(file.getPath());
        }
        if (locations.isEmpty()) {
            LOG.debug((Object)("No locations returned for " + file.getPath()));
            String[] name = new String[]{"/default-rack/swift"};
            String[] host = new String[]{"localhost"};
            String[] topology = new String[]{"/swift/unknown"};
            return new BlockLocation[]{new BlockLocation(name, host, topology, 0L, file.getLen())};
        }
        String[] names = new String[locations.size()];
        String[] hosts = new String[locations.size()];
        int i = 0;
        for (URI uRI : locations) {
            hosts[i] = uRI.getHost();
            names[i] = uRI.getAuthority();
            ++i;
        }
        return new BlockLocation[]{new BlockLocation(names, hosts, 0L, file.getLen())};
    }

    public boolean mkdirs(Path path, FsPermission permission) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("SwiftFileSystem.mkdirs: " + path));
        }
        Path directory = this.makeAbsolute(path);
        ArrayList<Path> paths = new ArrayList<Path>();
        while (this.shouldCreate(directory)) {
            paths.add(0, directory);
            directory = directory.getParent();
        }
        for (Path p : paths) {
            if (!this.isNotRoot(p)) continue;
            this.forceMkdir(p);
        }
        return true;
    }

    private boolean isNotRoot(Path absolutePath) {
        return !this.isRoot(absolutePath);
    }

    private boolean isRoot(Path absolutePath) {
        return absolutePath.getParent() == null;
    }

    private boolean mkdir(Path path) throws IOException {
        Path directory = this.makeAbsolute(path);
        boolean shouldCreate = this.shouldCreate(directory);
        if (shouldCreate) {
            this.forceMkdir(directory);
        }
        return shouldCreate;
    }

    private boolean shouldCreate(Path directory) throws IOException {
        boolean shouldCreate;
        if (this.isRoot(directory)) {
            return false;
        }
        try {
            FileStatus fileStatus = this.getFileStatus(directory);
            if (!SwiftUtils.isDirectory(fileStatus)) {
                throw new ParentNotDirectoryException(String.format("%s: can't mkdir since it exists and is not a directory: %s", directory, fileStatus));
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("skipping mkdir(" + directory + ") as it exists already"));
            }
            shouldCreate = false;
        }
        catch (FileNotFoundException e) {
            shouldCreate = true;
        }
        return shouldCreate;
    }

    private void forceMkdir(Path absolutePath) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Making dir '" + absolutePath + "' in Swift"));
        }
        this.store.createDirectory(absolutePath);
    }

    public FileStatus[] listStatus(Path path) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("SwiftFileSystem.listStatus for: " + path));
        }
        return this.store.listSubPaths(this.makeAbsolute(path), false, true);
    }

    public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
        LOG.debug((Object)"SwiftFileSystem.append");
        throw new SwiftUnsupportedFeatureException("Not supported: append()");
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public FSDataOutputStream create(Path file, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        LOG.debug((Object)"SwiftFileSystem.create");
        FileStatus fileStatus = null;
        Path absolutePath = this.makeAbsolute(file);
        try {
            fileStatus = this.getFileStatus(absolutePath);
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
        if (fileStatus != null) {
            if (fileStatus.isDirectory() && LOG.isDebugEnabled()) {
                LOG.debug((Object)"Overwriting either an empty file or a directory");
            }
            if (!overwrite) throw new FileAlreadyExistsException("Path exists: " + file);
            this.store.delete(absolutePath, true);
        } else {
            Path parent = file.getParent();
            if (parent != null && !this.mkdirs(parent)) {
                throw new SwiftOperationFailedException("Mkdirs failed to create " + parent);
            }
        }
        SwiftNativeOutputStream out = this.createSwiftOutputStream(file);
        return new FSDataOutputStream((OutputStream)out, this.statistics);
    }

    protected SwiftNativeOutputStream createSwiftOutputStream(Path path) throws IOException {
        long partSizeKB = this.getStore().getPartsizeKB();
        return new SwiftNativeOutputStream(this.getConf(), this.getStore(), path.toUri().toString(), partSizeKB);
    }

    public FSDataInputStream open(Path path, int bufferSize) throws IOException {
        int bufferSizeKB = this.getStore().getBufferSizeKB();
        long readBlockSize = (long)bufferSizeKB * 1024L;
        return this.open(path, bufferSize, readBlockSize);
    }

    public FSDataInputStream open(Path path, int bufferSize, long readBlockSize) throws IOException {
        if (readBlockSize <= 0L) {
            throw new SwiftConfigurationException("Bad remote buffer size");
        }
        Path absolutePath = this.makeAbsolute(path);
        return new FSDataInputStream((InputStream)((Object)new StrictBufferedFSInputStream(new SwiftNativeInputStream(this.store, this.statistics, absolutePath, readBlockSize), bufferSize)));
    }

    public boolean rename(Path src, Path dst) throws IOException {
        try {
            this.store.rename(this.makeAbsolute(src), this.makeAbsolute(dst));
            return true;
        }
        catch (SwiftOperationFailedException e) {
            return false;
        }
        catch (FileAlreadyExistsException e) {
            return false;
        }
        catch (FileNotFoundException e) {
            return false;
        }
    }

    public boolean delete(Path path, boolean recursive) throws IOException {
        try {
            return this.store.delete(path, recursive);
        }
        catch (FileNotFoundException e) {
            return false;
        }
    }

    public boolean delete(Path f) throws IOException {
        return this.delete(f, true);
    }

    protected Path makeAbsolute(Path path) {
        if (path.isAbsolute()) {
            return path;
        }
        return new Path(this.workingDir, path);
    }

    public List<DurationStats> getOperationStatistics() {
        return this.store.getOperationStatistics();
    }

    @InterfaceAudience.Private
    public FileStatus[] listRawFileStatus(Path path, boolean newest) throws IOException {
        return this.store.listSubPaths(this.makeAbsolute(path), true, newest);
    }

    @InterfaceAudience.Private
    public static int getPartitionsWritten(FSDataOutputStream outputStream) {
        SwiftNativeOutputStream snos = SwiftNativeFileSystem.getSwiftNativeOutputStream(outputStream);
        return snos.getPartitionsWritten();
    }

    private static SwiftNativeOutputStream getSwiftNativeOutputStream(FSDataOutputStream outputStream) {
        OutputStream wrappedStream = outputStream.getWrappedStream();
        return (SwiftNativeOutputStream)wrappedStream;
    }

    @InterfaceAudience.Private
    public static long getPartitionSize(FSDataOutputStream outputStream) {
        SwiftNativeOutputStream snos = SwiftNativeFileSystem.getSwiftNativeOutputStream(outputStream);
        return snos.getFilePartSize();
    }

    @InterfaceAudience.Private
    public static long getBytesWritten(FSDataOutputStream outputStream) {
        SwiftNativeOutputStream snos = SwiftNativeFileSystem.getSwiftNativeOutputStream(outputStream);
        return snos.getBytesWritten();
    }

    @InterfaceAudience.Private
    public static long getBytesUploaded(FSDataOutputStream outputStream) {
        SwiftNativeOutputStream snos = SwiftNativeFileSystem.getSwiftNativeOutputStream(outputStream);
        return snos.getBytesUploaded();
    }
}

