/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.nodemanager.containermanager.localizer;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.yarn.api.records.ApplicationId;
import org.apache.hadoop.yarn.api.records.LocalResource;
import org.apache.hadoop.yarn.api.records.LocalResourceType;
import org.apache.hadoop.yarn.api.records.LocalResourceVisibility;
import org.apache.hadoop.yarn.api.records.URL;
import org.apache.hadoop.yarn.api.records.impl.pb.LocalResourcePBImpl;
import org.apache.hadoop.yarn.event.Dispatcher;
import org.apache.hadoop.yarn.proto.YarnProtos;
import org.apache.hadoop.yarn.proto.YarnServerNodemanagerRecoveryProtos;
import org.apache.hadoop.yarn.server.nodemanager.DeletionService;
import org.apache.hadoop.yarn.server.nodemanager.LocalDirsHandlerService;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.deletion.task.FileDeletionTask;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.localizer.LocalCacheDirectoryManager;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.localizer.LocalResourceRequest;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.localizer.LocalResourcesTracker;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.localizer.LocalizedResource;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.localizer.ResourceState;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.localizer.event.ResourceEvent;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.localizer.event.ResourceEventType;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.localizer.event.ResourceRecoveredEvent;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.localizer.event.ResourceReleaseEvent;
import org.apache.hadoop.yarn.server.nodemanager.recovery.NMStateStoreService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class LocalResourcesTrackerImpl
implements LocalResourcesTracker {
    static final Logger LOG = LoggerFactory.getLogger(LocalResourcesTrackerImpl.class);
    private static final String RANDOM_DIR_REGEX = "-?\\d+";
    private static final Pattern RANDOM_DIR_PATTERN = Pattern.compile("-?\\d+");
    private final String user;
    private final ApplicationId appId;
    private final Dispatcher dispatcher;
    @VisibleForTesting
    final ConcurrentMap<LocalResourceRequest, LocalizedResource> localrsrc;
    private Configuration conf;
    private LocalDirsHandlerService dirsHandler;
    private final boolean useLocalCacheDirectoryManager;
    private ConcurrentHashMap<Path, LocalCacheDirectoryManager> directoryManagers;
    private ConcurrentHashMap<LocalResourceRequest, Path> inProgressLocalResourcesMap;
    private AtomicLong uniqueNumberGenerator = new AtomicLong(9L);
    private NMStateStoreService stateStore;

    public LocalResourcesTrackerImpl(String user, ApplicationId appId, Dispatcher dispatcher, boolean useLocalCacheDirectoryManager, Configuration conf, NMStateStoreService stateStore, LocalDirsHandlerService dirHandler) {
        this(user, appId, dispatcher, new ConcurrentHashMap<LocalResourceRequest, LocalizedResource>(), useLocalCacheDirectoryManager, conf, stateStore, dirHandler);
    }

    LocalResourcesTrackerImpl(String user, ApplicationId appId, Dispatcher dispatcher, ConcurrentMap<LocalResourceRequest, LocalizedResource> localrsrc, boolean useLocalCacheDirectoryManager, Configuration conf, NMStateStoreService stateStore, LocalDirsHandlerService dirHandler) {
        this.appId = appId;
        this.user = user;
        this.dispatcher = dispatcher;
        this.localrsrc = localrsrc;
        this.useLocalCacheDirectoryManager = useLocalCacheDirectoryManager;
        if (this.useLocalCacheDirectoryManager) {
            this.directoryManagers = new ConcurrentHashMap();
            this.inProgressLocalResourcesMap = new ConcurrentHashMap();
        }
        this.conf = conf;
        this.stateStore = stateStore;
        this.dirsHandler = dirHandler;
    }

    public synchronized void handle(ResourceEvent event) {
        LocalResourceRequest req = event.getLocalResourceRequest();
        LocalizedResource rsrc = (LocalizedResource)this.localrsrc.get(req);
        switch ((ResourceEventType)event.getType()) {
            case LOCALIZED: {
                if (!this.useLocalCacheDirectoryManager) break;
                this.inProgressLocalResourcesMap.remove(req);
                break;
            }
            case REQUEST: {
                if (rsrc != null && !this.isResourcePresent(rsrc)) {
                    LOG.info("Resource " + rsrc.getLocalPath() + " is missing, localizing it again");
                    this.removeResource(req);
                    rsrc = null;
                }
                if (null != rsrc) break;
                rsrc = new LocalizedResource(req, this.dispatcher);
                this.localrsrc.put(req, rsrc);
                break;
            }
            case RELEASE: {
                if (null != rsrc) break;
                ResourceReleaseEvent relEvent = (ResourceReleaseEvent)event;
                LOG.info("Container " + relEvent.getContainer() + " sent RELEASE event on a resource request " + req + " not present in cache.");
                return;
            }
            case LOCALIZATION_FAILED: {
                this.removeResource(req);
                break;
            }
            case RECOVERED: {
                if (rsrc != null) {
                    LOG.warn("Ignoring attempt to recover existing resource " + rsrc);
                    return;
                }
                rsrc = this.recoverResource(req, (ResourceRecoveredEvent)event);
                this.localrsrc.put(req, rsrc);
            }
        }
        if (rsrc == null) {
            LOG.warn("Received " + event.getType() + " event for request " + req + " but localized resource is missing");
            return;
        }
        rsrc.handle(event);
        if (event.getType() == ResourceEventType.RELEASE && rsrc.getState() == ResourceState.DOWNLOADING && rsrc.getRefCount() <= 0 && rsrc.getRequest().getVisibility() != LocalResourceVisibility.PUBLIC) {
            this.removeResource(req);
        }
        if (event.getType() == ResourceEventType.LOCALIZED) {
            if (rsrc.getLocalPath() != null) {
                try {
                    this.stateStore.finishResourceLocalization(this.user, this.appId, this.buildLocalizedResourceProto(rsrc));
                }
                catch (IOException ioe) {
                    LOG.error("Error storing resource state for " + rsrc, (Throwable)ioe);
                }
            } else {
                LOG.warn("Resource " + rsrc + " localized without a location");
            }
        }
    }

    private LocalizedResource recoverResource(LocalResourceRequest req, ResourceRecoveredEvent event) {
        long nextRsrcId;
        long currentRsrcId;
        Path localDir = event.getLocalPath().getParent();
        long rsrcId = Long.parseLong(localDir.getName());
        while (!this.uniqueNumberGenerator.compareAndSet(currentRsrcId = this.uniqueNumberGenerator.get(), nextRsrcId = Math.max(currentRsrcId, rsrcId))) {
        }
        this.incrementFileCountForLocalCacheDirectory(localDir.getParent());
        return new LocalizedResource(req, this.dispatcher);
    }

    private YarnServerNodemanagerRecoveryProtos.LocalizedResourceProto buildLocalizedResourceProto(LocalizedResource rsrc) {
        return YarnServerNodemanagerRecoveryProtos.LocalizedResourceProto.newBuilder().setResource(this.buildLocalResourceProto(rsrc.getRequest())).setLocalPath(rsrc.getLocalPath().toString()).setSize(rsrc.getSize()).build();
    }

    private YarnProtos.LocalResourceProto buildLocalResourceProto(LocalResource lr) {
        if (!(lr instanceof LocalResourcePBImpl)) {
            lr = LocalResource.newInstance((URL)lr.getResource(), (LocalResourceType)lr.getType(), (LocalResourceVisibility)lr.getVisibility(), (long)lr.getSize(), (long)lr.getTimestamp(), (String)lr.getPattern());
        }
        LocalResourcePBImpl lrpb = (LocalResourcePBImpl)lr;
        return lrpb.getProto();
    }

    public void incrementFileCountForLocalCacheDirectory(Path cacheDir) {
        Path cacheRoot;
        if (this.useLocalCacheDirectoryManager && (cacheRoot = LocalCacheDirectoryManager.getCacheDirectoryRoot(cacheDir)) != null) {
            LocalCacheDirectoryManager otherDir;
            LocalCacheDirectoryManager dir = this.directoryManagers.get(cacheRoot);
            if (dir == null && (otherDir = this.directoryManagers.putIfAbsent(cacheRoot, dir = new LocalCacheDirectoryManager(this.conf))) != null) {
                dir = otherDir;
            }
            if (cacheDir.equals((Object)cacheRoot)) {
                dir.incrementFileCountForPath("");
            } else {
                String dirStr = cacheDir.toUri().getRawPath();
                String rootStr = cacheRoot.toUri().getRawPath();
                dir.incrementFileCountForPath(dirStr.substring(rootStr.length() + 1));
            }
        }
    }

    private void decrementFileCountForLocalCacheDirectory(LocalResourceRequest req, LocalizedResource rsrc) {
        if (this.useLocalCacheDirectoryManager) {
            Path rsrcPath = null;
            if (this.inProgressLocalResourcesMap.containsKey(req)) {
                rsrcPath = this.inProgressLocalResourcesMap.remove(req);
            } else if (rsrc != null && rsrc.getLocalPath() != null) {
                rsrcPath = rsrc.getLocalPath().getParent().getParent();
            }
            if (rsrcPath != null) {
                Path parentPath = new Path(rsrcPath.toUri().getRawPath());
                while (!this.directoryManagers.containsKey(parentPath)) {
                    if ((parentPath = parentPath.getParent()) != null) continue;
                    return;
                }
                if (parentPath != null) {
                    String parentDir = parentPath.toUri().getRawPath().toString();
                    LocalCacheDirectoryManager dir = this.directoryManagers.get(parentPath);
                    String rsrcDir = rsrcPath.toUri().getRawPath();
                    if (rsrcDir.equals(parentDir)) {
                        dir.decrementFileCountForPath("");
                    } else {
                        dir.decrementFileCountForPath(rsrcDir.substring(parentDir.length() + 1));
                    }
                }
            }
        }
    }

    public boolean isResourcePresent(LocalizedResource rsrc) {
        boolean ret = true;
        if (rsrc.getState() == ResourceState.LOCALIZED) {
            File file = new File(rsrc.getLocalPath().toUri().getRawPath().toString());
            if (!file.exists()) {
                ret = false;
            } else if (this.dirsHandler != null) {
                ret = this.checkLocalResource(rsrc);
            }
        }
        return ret;
    }

    @VisibleForTesting
    boolean checkLocalResource(LocalizedResource rsrc) {
        List<String> localDirs = this.dirsHandler.getLocalDirsForRead();
        for (String dir : localDirs) {
            if (!this.isParent(rsrc.getLocalPath().toUri().getPath(), dir)) continue;
            return true;
        }
        return false;
    }

    private boolean isParent(String path, String parentdir) {
        if (path.charAt(path.length() - 1) != File.separatorChar) {
            path = path + File.separator;
        }
        return path.startsWith(parentdir);
    }

    @Override
    public boolean remove(LocalizedResource rem, DeletionService delService) {
        LocalizedResource rsrc = (LocalizedResource)this.localrsrc.get(rem.getRequest());
        if (null == rsrc) {
            LOG.error("Attempt to remove absent resource: " + rem.getRequest() + " from " + this.getUser());
            return true;
        }
        if (rsrc.getRefCount() > 0 || ResourceState.DOWNLOADING.equals((Object)rsrc.getState()) || rsrc != rem) {
            LOG.error("Attempt to remove resource: " + rsrc + " with non-zero refcount");
            return false;
        }
        if (ResourceState.LOCALIZED.equals((Object)rsrc.getState())) {
            FileDeletionTask deletionTask = new FileDeletionTask(delService, this.getUser(), this.getPathToDelete(rsrc.getLocalPath()), null);
            delService.delete(deletionTask);
        }
        this.removeResource(rem.getRequest());
        LOG.info("Removed " + rsrc.getLocalPath() + " from localized cache");
        return true;
    }

    private void removeResource(LocalResourceRequest req) {
        Path localPath;
        LocalizedResource rsrc = (LocalizedResource)this.localrsrc.remove(req);
        this.decrementFileCountForLocalCacheDirectory(req, rsrc);
        if (rsrc != null && (localPath = rsrc.getLocalPath()) != null) {
            try {
                this.stateStore.removeLocalizedResource(this.user, this.appId, localPath);
            }
            catch (IOException e) {
                LOG.error("Unable to remove resource " + rsrc + " from state store", (Throwable)e);
            }
        }
    }

    private Path getPathToDelete(Path localPath) {
        Path delPath = localPath.getParent();
        String name = delPath.getName();
        Matcher matcher = RANDOM_DIR_PATTERN.matcher(name);
        if (matcher.matches()) {
            return delPath;
        }
        LOG.warn("Random directory component did not match. Deleting localized path only");
        return localPath;
    }

    @Override
    public String getUser() {
        return this.user;
    }

    @Override
    public Iterator<LocalizedResource> iterator() {
        return this.localrsrc.values().iterator();
    }

    @Override
    public Path getPathForLocalization(LocalResourceRequest req, Path localDirPath, DeletionService delService) {
        Path uniquePath;
        Path rPath = localDirPath;
        if (this.useLocalCacheDirectoryManager && localDirPath != null) {
            if (!this.directoryManagers.containsKey(localDirPath)) {
                this.directoryManagers.putIfAbsent(localDirPath, new LocalCacheDirectoryManager(this.conf));
            }
            LocalCacheDirectoryManager dir = this.directoryManagers.get(localDirPath);
            rPath = localDirPath;
            String hierarchicalPath = dir.getRelativePathForLocalization();
            if (!hierarchicalPath.isEmpty()) {
                rPath = new Path(localDirPath, hierarchicalPath);
            }
            this.inProgressLocalResourcesMap.put(req, rPath);
        }
        while (true) {
            File file;
            if (!(file = new File((uniquePath = new Path(rPath, Long.toString(this.uniqueNumberGenerator.incrementAndGet()))).toUri().getRawPath())).exists()) break;
            LOG.warn("Directory " + uniquePath + " already exists, " + "try next one.");
            if (delService == null) continue;
            FileDeletionTask deletionTask = new FileDeletionTask(delService, this.getUser(), uniquePath, null);
            delService.delete(deletionTask);
        }
        rPath = uniquePath;
        Path localPath = new Path(rPath, req.getPath().getName());
        LocalizedResource rsrc = (LocalizedResource)this.localrsrc.get(req);
        rsrc.setLocalPath(localPath);
        LocalResource lr = LocalResource.newInstance((URL)req.getResource(), (LocalResourceType)req.getType(), (LocalResourceVisibility)req.getVisibility(), (long)req.getSize(), (long)req.getTimestamp());
        try {
            this.stateStore.startResourceLocalization(this.user, this.appId, ((LocalResourcePBImpl)lr).getProto(), localPath);
        }
        catch (IOException e) {
            LOG.error("Unable to record localization start for " + rsrc, (Throwable)e);
        }
        return rPath;
    }

    @Override
    public LocalizedResource getLocalizedResource(LocalResourceRequest request) {
        return (LocalizedResource)this.localrsrc.get(request);
    }

    @VisibleForTesting
    LocalCacheDirectoryManager getDirectoryManager(Path localDirPath) {
        LocalCacheDirectoryManager mgr = null;
        if (this.useLocalCacheDirectoryManager) {
            mgr = this.directoryManagers.get(localDirPath);
        }
        return mgr;
    }

    @VisibleForTesting
    LocalDirsHandlerService getDirsHandler() {
        return this.dirsHandler;
    }
}

