/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.nodemanager.containermanager.scheduler;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.service.AbstractService;
import org.apache.hadoop.yarn.api.records.ContainerId;
import org.apache.hadoop.yarn.api.records.ExecutionType;
import org.apache.hadoop.yarn.api.records.ResourceUtilization;
import org.apache.hadoop.yarn.event.AsyncDispatcher;
import org.apache.hadoop.yarn.event.Event;
import org.apache.hadoop.yarn.event.EventHandler;
import org.apache.hadoop.yarn.server.api.records.ContainerQueuingLimit;
import org.apache.hadoop.yarn.server.api.records.OpportunisticContainersStatus;
import org.apache.hadoop.yarn.server.nodemanager.Context;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.container.Container;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.monitor.ContainersMonitor;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.scheduler.AllocationBasedResourceUtilizationTracker;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.scheduler.ContainerSchedulerEvent;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.scheduler.ContainerSchedulerEventType;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.scheduler.ResourceUtilizationTracker;
import org.apache.hadoop.yarn.server.nodemanager.metrics.NodeManagerMetrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerScheduler
extends AbstractService
implements EventHandler<ContainerSchedulerEvent> {
    private static final Logger LOG = LoggerFactory.getLogger(ContainerScheduler.class);
    private final Context context;
    private final int maxOppQueueLength;
    private final LinkedHashMap<ContainerId, Container> queuedGuaranteedContainers = new LinkedHashMap();
    private final LinkedHashMap<ContainerId, Container> queuedOpportunisticContainers = new LinkedHashMap();
    private final Map<ContainerId, Container> oppContainersToKill = new HashMap<ContainerId, Container>();
    private final LinkedHashMap<ContainerId, Container> runningContainers = new LinkedHashMap();
    private final ContainerQueuingLimit queuingLimit = ContainerQueuingLimit.newInstance();
    private final OpportunisticContainersStatus opportunisticContainersStatus;
    private ResourceUtilizationTracker utilizationTracker;
    private final AsyncDispatcher dispatcher;
    private final NodeManagerMetrics metrics;

    public ContainerScheduler(Context context, AsyncDispatcher dispatcher, NodeManagerMetrics metrics) {
        this(context, dispatcher, metrics, context.getConf().getInt("yarn.nodemanager.opportunistic-containers-max-queue-length", 0));
    }

    @VisibleForTesting
    public ContainerScheduler(Context context, AsyncDispatcher dispatcher, NodeManagerMetrics metrics, int qLength) {
        super(ContainerScheduler.class.getName());
        this.context = context;
        this.dispatcher = dispatcher;
        this.metrics = metrics;
        this.maxOppQueueLength = qLength <= 0 ? 0 : qLength;
        this.utilizationTracker = new AllocationBasedResourceUtilizationTracker(this);
        this.opportunisticContainersStatus = OpportunisticContainersStatus.newInstance();
    }

    public void handle(ContainerSchedulerEvent event) {
        switch ((ContainerSchedulerEventType)event.getType()) {
            case SCHEDULE_CONTAINER: {
                this.scheduleContainer(event.getContainer());
                break;
            }
            case CONTAINER_COMPLETED: {
                this.onContainerCompleted(event.getContainer());
                break;
            }
            case SHED_QUEUED_CONTAINERS: {
                this.shedQueuedOpportunisticContainers();
                break;
            }
            default: {
                LOG.error("Unknown event arrived at ContainerScheduler: " + event.toString());
            }
        }
    }

    public int getNumQueuedContainers() {
        return this.queuedGuaranteedContainers.size() + this.queuedOpportunisticContainers.size();
    }

    @VisibleForTesting
    public int getNumQueuedGuaranteedContainers() {
        return this.queuedGuaranteedContainers.size();
    }

    @VisibleForTesting
    public int getNumQueuedOpportunisticContainers() {
        return this.queuedOpportunisticContainers.size();
    }

    public OpportunisticContainersStatus getOpportunisticContainersStatus() {
        this.opportunisticContainersStatus.setQueuedOpportContainers(this.getNumQueuedOpportunisticContainers());
        this.opportunisticContainersStatus.setWaitQueueLength(this.getNumQueuedContainers());
        this.opportunisticContainersStatus.setOpportMemoryUsed(this.metrics.getAllocatedOpportunisticGB());
        this.opportunisticContainersStatus.setOpportCoresUsed(this.metrics.getAllocatedOpportunisticVCores());
        this.opportunisticContainersStatus.setRunningOpportContainers(this.metrics.getRunningOpportunisticContainers());
        return this.opportunisticContainersStatus;
    }

    private void onContainerCompleted(Container container) {
        Container completedContainer;
        this.oppContainersToKill.remove(container.getContainerId());
        Container queued = (Container)this.queuedOpportunisticContainers.remove(container.getContainerId());
        if (queued == null) {
            this.queuedGuaranteedContainers.remove(container.getContainerId());
        }
        if ((completedContainer = (Container)this.runningContainers.remove(container.getContainerId())) != null) {
            this.utilizationTracker.subtractContainerResource(container);
            if (container.getContainerTokenIdentifier().getExecutionType() == ExecutionType.OPPORTUNISTIC) {
                this.metrics.completeOpportunisticContainer(container.getResource());
            }
            this.startPendingContainers();
        }
    }

    private void startPendingContainers() {
        boolean resourcesAvailable = this.startContainersFromQueue(this.queuedGuaranteedContainers.values());
        if (resourcesAvailable) {
            this.startContainersFromQueue(this.queuedOpportunisticContainers.values());
        }
    }

    private boolean startContainersFromQueue(Collection<Container> queuedContainers) {
        Iterator<Container> cIter = queuedContainers.iterator();
        boolean resourcesAvailable = true;
        while (cIter.hasNext() && resourcesAvailable) {
            Container container = cIter.next();
            if (this.utilizationTracker.hasResourcesAvailable(container)) {
                this.startAllocatedContainer(container);
                cIter.remove();
                continue;
            }
            resourcesAvailable = false;
        }
        return resourcesAvailable;
    }

    @VisibleForTesting
    protected void scheduleContainer(Container container) {
        if (this.maxOppQueueLength <= 0) {
            this.startAllocatedContainer(container);
            return;
        }
        if (this.queuedGuaranteedContainers.isEmpty() && this.queuedOpportunisticContainers.isEmpty() && this.utilizationTracker.hasResourcesAvailable(container)) {
            this.startAllocatedContainer(container);
        } else {
            LOG.info("No available resources for container {} to start its execution immediately.", (Object)container.getContainerId());
            boolean isQueued = true;
            if (container.getContainerTokenIdentifier().getExecutionType() == ExecutionType.GUARANTEED) {
                this.queuedGuaranteedContainers.put(container.getContainerId(), container);
                this.killOpportunisticContainers(container);
            } else if (this.queuedOpportunisticContainers.size() <= this.maxOppQueueLength) {
                LOG.info("Opportunistic container {} will be queued at the NM.", (Object)container.getContainerId());
                this.queuedOpportunisticContainers.put(container.getContainerId(), container);
            } else {
                isQueued = false;
                LOG.info("Opportunistic container [{}] will not be queued at the NMsince max queue length [{}] has been reached", (Object)container.getContainerId(), (Object)this.maxOppQueueLength);
                container.sendKillEvent(-108, "Opportunistic container queue is full.");
            }
            if (isQueued) {
                try {
                    this.context.getNMStateStore().storeContainerQueued(container.getContainerId());
                }
                catch (IOException e) {
                    LOG.warn("Could not store container [" + container.getContainerId() + "] state. The Container has been queued.", (Throwable)e);
                }
            }
        }
    }

    private void killOpportunisticContainers(Container container) {
        List<Container> extraOpportContainersToKill = this.pickOpportunisticContainersToKill(container.getContainerId());
        for (Container contToKill : extraOpportContainersToKill) {
            contToKill.sendKillEvent(-108, "Container Killed to make room for Guaranteed Container.");
            this.oppContainersToKill.put(contToKill.getContainerId(), contToKill);
            LOG.info("Opportunistic container {} will be killed in order to start the execution of guaranteed container {}.", (Object)contToKill.getContainerId(), (Object)container.getContainerId());
        }
    }

    private void startAllocatedContainer(Container container) {
        LOG.info("Starting container [" + container.getContainerId() + "]");
        this.runningContainers.put(container.getContainerId(), container);
        this.utilizationTracker.addContainerResources(container);
        if (container.getContainerTokenIdentifier().getExecutionType() == ExecutionType.OPPORTUNISTIC) {
            this.metrics.startOpportunisticContainer(container.getResource());
        }
        container.sendLaunchEvent();
    }

    private List<Container> pickOpportunisticContainersToKill(ContainerId containerToStartId) {
        ArrayList<Container> extraOpportContainersToKill = new ArrayList<Container>();
        ResourceUtilization resourcesToFreeUp = this.resourcesToFreeUp(containerToStartId);
        Iterator<Container> lifoIterator = new LinkedList<Container>(this.runningContainers.values()).descendingIterator();
        while (lifoIterator.hasNext() && !this.hasSufficientResources(resourcesToFreeUp)) {
            Container runningCont = lifoIterator.next();
            if (runningCont.getContainerTokenIdentifier().getExecutionType() != ExecutionType.OPPORTUNISTIC || this.oppContainersToKill.containsKey(runningCont.getContainerId())) continue;
            extraOpportContainersToKill.add(runningCont);
            ContainersMonitor.decreaseResourceUtilization(this.getContainersMonitor(), resourcesToFreeUp, runningCont.getResource());
        }
        if (!this.hasSufficientResources(resourcesToFreeUp)) {
            LOG.warn("There are no sufficient resources to start guaranteed [{}]at the moment. Opportunistic containers are in the process ofbeing killed to make room.", (Object)containerToStartId);
        }
        return extraOpportContainersToKill;
    }

    private boolean hasSufficientResources(ResourceUtilization resourcesToFreeUp) {
        return resourcesToFreeUp.getPhysicalMemory() <= 0 && resourcesToFreeUp.getVirtualMemory() <= 0 && resourcesToFreeUp.getCPU() <= 0.0f;
    }

    private ResourceUtilization resourcesToFreeUp(ContainerId containerToStartId) {
        ResourceUtilization resourceAllocationToFreeUp = ResourceUtilization.newInstance((ResourceUtilization)this.utilizationTracker.getCurrentUtilization());
        for (Container container : this.queuedGuaranteedContainers.values()) {
            ContainersMonitor.increaseResourceUtilization(this.getContainersMonitor(), resourceAllocationToFreeUp, container.getResource());
            if (!container.getContainerId().equals((Object)containerToStartId)) continue;
            break;
        }
        for (Container container : this.oppContainersToKill.values()) {
            ContainersMonitor.decreaseResourceUtilization(this.getContainersMonitor(), resourceAllocationToFreeUp, container.getResource());
        }
        this.getContainersMonitor().subtractNodeResourcesFromResourceUtilization(resourceAllocationToFreeUp);
        return resourceAllocationToFreeUp;
    }

    public void updateQueuingLimit(ContainerQueuingLimit limit) {
        this.queuingLimit.setMaxQueueLength(limit.getMaxQueueLength());
        if (this.queuingLimit.getMaxQueueLength() > -1 && this.queuingLimit.getMaxQueueLength() < this.queuedOpportunisticContainers.size()) {
            this.dispatcher.getEventHandler().handle((Event)new ContainerSchedulerEvent(null, ContainerSchedulerEventType.SHED_QUEUED_CONTAINERS));
        }
    }

    private void shedQueuedOpportunisticContainers() {
        int numAllowed = this.queuingLimit.getMaxQueueLength();
        Iterator<Container> containerIter = this.queuedOpportunisticContainers.values().iterator();
        while (containerIter.hasNext()) {
            Container container = containerIter.next();
            if (numAllowed <= 0) {
                container.sendKillEvent(-108, "Container De-queued to meet NM queuing limits.");
                containerIter.remove();
                LOG.info("Opportunistic container {} will be killed to meet NM queuing limits.", (Object)container.getContainerId());
            }
            --numAllowed;
        }
    }

    public ContainersMonitor getContainersMonitor() {
        return this.context.getContainerManager().getContainersMonitor();
    }
}

