/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.procedure2;

import edu.umd.cs.findbugs.annotations.Nullable;
import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Deque;
import java.util.HashSet;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.exceptions.IllegalArgumentIOException;
import org.apache.hadoop.hbase.log.HBaseMarkers;
import org.apache.hadoop.hbase.procedure2.CompletedProcedureCleaner;
import org.apache.hadoop.hbase.procedure2.CompletedProcedureRetainer;
import org.apache.hadoop.hbase.procedure2.FailedProcedure;
import org.apache.hadoop.hbase.procedure2.InlineChore;
import org.apache.hadoop.hbase.procedure2.Procedure;
import org.apache.hadoop.hbase.procedure2.ProcedureInMemoryChore;
import org.apache.hadoop.hbase.procedure2.ProcedureScheduler;
import org.apache.hadoop.hbase.procedure2.ProcedureSuspendedException;
import org.apache.hadoop.hbase.procedure2.ProcedureYieldException;
import org.apache.hadoop.hbase.procedure2.RemoteProcedureException;
import org.apache.hadoop.hbase.procedure2.RootProcedureState;
import org.apache.hadoop.hbase.procedure2.SimpleProcedureScheduler;
import org.apache.hadoop.hbase.procedure2.StoppableThread;
import org.apache.hadoop.hbase.procedure2.TimeoutExecutorThread;
import org.apache.hadoop.hbase.procedure2.store.ProcedureStore;
import org.apache.hadoop.hbase.procedure2.trace.ProcedureSpanBuilder;
import org.apache.hadoop.hbase.procedure2.util.StringUtils;
import org.apache.hadoop.hbase.security.User;
import org.apache.hadoop.hbase.shaded.protobuf.generated.ProcedureProtos;
import org.apache.hadoop.hbase.trace.TraceUtil;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.IdLock;
import org.apache.hadoop.hbase.util.NonceKey;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hbase.thirdparty.com.google.common.base.Preconditions;
import org.apache.hbase.thirdparty.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class ProcedureExecutor<TEnvironment> {
    private static final Logger LOG = LoggerFactory.getLogger(ProcedureExecutor.class);
    public static final String CHECK_OWNER_SET_CONF_KEY = "hbase.procedure.check.owner.set";
    private static final boolean DEFAULT_CHECK_OWNER_SET = false;
    public static final String WORKER_KEEP_ALIVE_TIME_CONF_KEY = "hbase.procedure.worker.keep.alive.time.msec";
    private static final long DEFAULT_WORKER_KEEP_ALIVE_TIME = TimeUnit.MINUTES.toMillis(1L);
    public static final String EVICT_TTL_CONF_KEY = "hbase.procedure.cleaner.evict.ttl";
    static final int DEFAULT_EVICT_TTL = 900000;
    public static final String EVICT_ACKED_TTL_CONF_KEY = "hbase.procedure.cleaner.acked.evict.ttl";
    static final int DEFAULT_ACKED_EVICT_TTL = 300000;
    volatile Testing testing = null;
    private final ConcurrentHashMap<Long, CompletedProcedureRetainer<TEnvironment>> completed = new ConcurrentHashMap();
    private final ConcurrentHashMap<Long, RootProcedureState<TEnvironment>> rollbackStack = new ConcurrentHashMap();
    private final ConcurrentHashMap<Long, Procedure<TEnvironment>> procedures = new ConcurrentHashMap();
    private final ConcurrentHashMap<NonceKey, Long> nonceKeysToProcIdsMap = new ConcurrentHashMap();
    private final CopyOnWriteArrayList<ProcedureExecutorListener> listeners = new CopyOnWriteArrayList();
    private Configuration conf;
    private ThreadGroup threadGroup;
    private CopyOnWriteArrayList<WorkerThread> workerThreads;
    private TimeoutExecutorThread<TEnvironment> timeoutExecutor;
    private TimeoutExecutorThread<TEnvironment> workerMonitorExecutor;
    private int corePoolSize;
    private int maxPoolSize;
    private volatile long keepAliveTime;
    private final ProcedureScheduler scheduler;
    private final Executor forceUpdateExecutor = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setDaemon(true).setNameFormat("Force-Update-PEWorker-%d").build());
    private final AtomicLong lastProcId = new AtomicLong(-1L);
    private final AtomicLong workerId = new AtomicLong(0L);
    private final AtomicInteger activeExecutorCount = new AtomicInteger(0);
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final TEnvironment environment;
    private final ProcedureStore store;
    private final boolean checkOwnerSet;
    private final IdLock procExecutionLock = new IdLock();

    public ProcedureExecutor(Configuration conf, TEnvironment environment, ProcedureStore store) {
        this(conf, environment, store, new SimpleProcedureScheduler());
    }

    private boolean isRootFinished(Procedure<?> proc) {
        Procedure<TEnvironment> rootProc = this.procedures.get(proc.getRootProcId());
        return rootProc == null || rootProc.isFinished();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void forceUpdateProcedure(long procId) throws IOException {
        IdLock.Entry lockEntry = this.procExecutionLock.getLockEntry(procId);
        try {
            Procedure<TEnvironment> proc = this.procedures.get(procId);
            if (proc != null) {
                if (proc.isFinished() && proc.hasParent() && this.isRootFinished(proc)) {
                    LOG.debug("Procedure {} has already been finished and parent is succeeded, skip force updating", proc);
                    return;
                }
            } else {
                CompletedProcedureRetainer<TEnvironment> retainer = this.completed.get(procId);
                if (retainer == null || retainer.getProcedure() instanceof FailedProcedure) {
                    LOG.debug("No pending procedure with id = {}, skip force updating.", (Object)procId);
                    return;
                }
                long evictTtl = this.conf.getInt(EVICT_TTL_CONF_KEY, 900000);
                long evictAckTtl = this.conf.getInt(EVICT_ACKED_TTL_CONF_KEY, 300000);
                if (retainer.isExpired(EnvironmentEdgeManager.currentTime(), evictTtl, evictAckTtl)) {
                    LOG.debug("Procedure {} has already been finished and expired, skip force updating", (Object)procId);
                    return;
                }
                proc = retainer.getProcedure();
            }
            LOG.debug("Force update procedure {}", proc);
            this.store.update(proc);
        }
        finally {
            this.procExecutionLock.releaseLockEntry(lockEntry);
        }
    }

    public ProcedureExecutor(Configuration conf, TEnvironment environment, ProcedureStore store, ProcedureScheduler scheduler) {
        this.environment = environment;
        this.scheduler = scheduler;
        this.store = store;
        this.conf = conf;
        this.checkOwnerSet = conf.getBoolean(CHECK_OWNER_SET_CONF_KEY, false);
        this.refreshConfiguration(conf);
        store.registerListener(new ProcedureStore.ProcedureStoreListener(){

            @Override
            public void forceUpdate(long[] procIds) {
                Arrays.stream(procIds).forEach(procId -> ProcedureExecutor.this.forceUpdateExecutor.execute(() -> {
                    try {
                        ProcedureExecutor.this.forceUpdateProcedure(procId);
                    }
                    catch (IOException e) {
                        LOG.warn("Failed to force update procedure with pid={}", (Object)procId);
                    }
                }));
            }
        });
    }

    private void load(final boolean abortOnCorruption) throws IOException {
        Preconditions.checkArgument((boolean)this.completed.isEmpty(), (String)"completed not empty: %s", this.completed);
        Preconditions.checkArgument((boolean)this.rollbackStack.isEmpty(), (String)"rollback state not empty: %s", this.rollbackStack);
        Preconditions.checkArgument((boolean)this.procedures.isEmpty(), (String)"procedure map not empty: %s", this.procedures);
        Preconditions.checkArgument((this.scheduler.size() == 0 ? 1 : 0) != 0, (String)"scheduler queue not empty: %s", (Object)this.scheduler);
        this.store.load(new ProcedureStore.ProcedureLoader(){

            @Override
            public void setMaxProcId(long maxProcId) {
                assert (ProcedureExecutor.this.lastProcId.get() < 0L) : "expected only one call to setMaxProcId()";
                ProcedureExecutor.this.lastProcId.set(maxProcId);
            }

            @Override
            public void load(ProcedureStore.ProcedureIterator procIter) throws IOException {
                ProcedureExecutor.this.loadProcedures(procIter);
            }

            @Override
            public void handleCorrupted(ProcedureStore.ProcedureIterator procIter) throws IOException {
                int corruptedCount = 0;
                while (procIter.hasNext()) {
                    Procedure proc = procIter.next();
                    LOG.error("Corrupt " + proc);
                    ++corruptedCount;
                }
                if (abortOnCorruption && corruptedCount > 0) {
                    throw new IOException("found " + corruptedCount + " corrupted procedure(s) on replay");
                }
            }
        });
    }

    private void restoreLock(Procedure<TEnvironment> proc, Set<Long> restored) {
        proc.restoreLock(this.getEnvironment());
        restored.add(proc.getProcId());
    }

    private void restoreLocks(Deque<Procedure<TEnvironment>> stack, Set<Long> restored) {
        while (!stack.isEmpty()) {
            this.restoreLock(stack.pop(), restored);
        }
    }

    private void restoreLocks() {
        HashSet restored = new HashSet();
        ArrayDeque stack = new ArrayDeque();
        this.procedures.values().forEach(proc -> {
            while (true) {
                if (restored.contains(proc.getProcId())) {
                    this.restoreLocks(stack, restored);
                    return;
                }
                if (!proc.hasParent()) {
                    this.restoreLock((Procedure<TEnvironment>)proc, restored);
                    this.restoreLocks(stack, restored);
                    return;
                }
                stack.push(proc);
                proc = this.procedures.get(proc.getParentProcId());
            }
        });
    }

    private void initializeStacks(ProcedureStore.ProcedureIterator procIter, List<Procedure<TEnvironment>> runnableList, List<Procedure<TEnvironment>> failedList, List<Procedure<TEnvironment>> waitingList, List<Procedure<TEnvironment>> waitingTimeoutList) throws IOException {
        procIter.reset();
        while (procIter.hasNext()) {
            Procedure<TEnvironment> parent;
            if (procIter.isNextFinished()) {
                procIter.skipNext();
                continue;
            }
            Procedure proc = procIter.next();
            assert (!proc.isFinished() || proc.hasParent()) : "unexpected completed proc=" + proc;
            LOG.debug("Loading {}", (Object)proc);
            Long rootProcId2 = this.getRootProcedureId(proc);
            assert (rootProcId2 != null);
            if (proc.hasParent() && (parent = this.procedures.get(proc.getParentProcId())) != null && !proc.isFinished()) {
                parent.incChildrenLatch();
            }
            RootProcedureState<TEnvironment> procStack2 = this.rollbackStack.get(rootProcId2);
            procStack2.loadStack(proc);
            proc.setRootProcId(rootProcId2);
            switch (proc.getState()) {
                case RUNNABLE: {
                    runnableList.add(proc);
                    break;
                }
                case WAITING: {
                    waitingList.add(proc);
                    break;
                }
                case WAITING_TIMEOUT: {
                    waitingTimeoutList.add(proc);
                    break;
                }
                case FAILED: {
                    failedList.add(proc);
                    break;
                }
                case ROLLEDBACK: 
                case INITIALIZING: {
                    String msg = "Unexpected " + proc.getState() + " state for " + proc;
                    LOG.error(msg);
                    throw new UnsupportedOperationException(msg);
                }
            }
        }
        this.rollbackStack.forEach((rootProcId, procStack) -> {
            if (procStack.getSubproceduresStack() != null) {
                procStack.setRollbackSupported(true);
            } else {
                procStack.setRollbackSupported(this.procedures.get(rootProcId).isRollbackSupported());
            }
        });
    }

    private void processWaitingProcedures(List<Procedure<TEnvironment>> waitingList, List<Procedure<TEnvironment>> runnableList) {
        waitingList.forEach(proc -> {
            if (!proc.hasChildren()) {
                proc.setState(ProcedureProtos.ProcedureState.RUNNABLE);
                runnableList.add((Procedure<TEnvironment>)proc);
            } else {
                proc.afterReplay(this.getEnvironment());
            }
        });
    }

    private void processWaitingTimeoutProcedures(List<Procedure<TEnvironment>> waitingTimeoutList) {
        waitingTimeoutList.forEach(proc -> {
            proc.afterReplay(this.getEnvironment());
            this.timeoutExecutor.add((Procedure<TEnvironment>)proc);
        });
    }

    private void pushProceduresAfterLoad(List<Procedure<TEnvironment>> runnableList, List<Procedure<TEnvironment>> failedList) {
        failedList.forEach(this.scheduler::addBack);
        runnableList.forEach(p -> {
            p.afterReplay(this.getEnvironment());
            if (!p.hasParent()) {
                this.sendProcedureLoadedNotification(p.getProcId());
            }
            this.scheduler.addBack((Procedure)p);
        });
    }

    private void loadProcedures(ProcedureStore.ProcedureIterator procIter) throws IOException {
        int runnableCount = 0;
        int failedCount = 0;
        int waitingCount = 0;
        int waitingTimeoutCount = 0;
        while (procIter.hasNext()) {
            boolean finished = procIter.isNextFinished();
            Procedure proc = procIter.next();
            NonceKey nonceKey = proc.getNonceKey();
            long procId = proc.getProcId();
            if (finished) {
                this.completed.put(proc.getProcId(), new CompletedProcedureRetainer(proc));
                LOG.debug("Completed {}", (Object)proc);
            } else {
                if (!proc.hasParent()) {
                    assert (!proc.isFinished()) : "unexpected finished procedure";
                    this.rollbackStack.put(proc.getProcId(), new RootProcedureState());
                }
                proc.beforeReplay(this.getEnvironment());
                this.procedures.put(proc.getProcId(), proc);
                switch (proc.getState()) {
                    case RUNNABLE: {
                        ++runnableCount;
                        break;
                    }
                    case FAILED: {
                        ++failedCount;
                        break;
                    }
                    case WAITING: {
                        ++waitingCount;
                        break;
                    }
                    case WAITING_TIMEOUT: {
                        ++waitingTimeoutCount;
                        break;
                    }
                }
            }
            if (nonceKey == null) continue;
            this.nonceKeysToProcIdsMap.put(nonceKey, procId);
        }
        ArrayList<Procedure<TEnvironment>> runnableList = new ArrayList<Procedure<TEnvironment>>(runnableCount);
        ArrayList<Procedure<TEnvironment>> failedList = new ArrayList<Procedure<TEnvironment>>(failedCount);
        ArrayList<Procedure<TEnvironment>> waitingList = new ArrayList<Procedure<TEnvironment>>(waitingCount);
        ArrayList<Procedure<TEnvironment>> waitingTimeoutList = new ArrayList<Procedure<TEnvironment>>(waitingTimeoutCount);
        this.initializeStacks(procIter, runnableList, failedList, waitingList, waitingTimeoutList);
        this.processWaitingProcedures(waitingList, runnableList);
        this.restoreLocks();
        this.processWaitingTimeoutProcedures(waitingTimeoutList);
        this.pushProceduresAfterLoad(runnableList, failedList);
        this.scheduler.signalAll();
    }

    public void init(int numThreads, boolean abortOnCorruption) throws IOException {
        this.corePoolSize = numThreads;
        this.maxPoolSize = 10 * numThreads;
        LOG.info("Starting {} core workers (bigger of cpus/4 or 16) with max (burst) worker count={}", (Object)this.corePoolSize, (Object)this.maxPoolSize);
        this.threadGroup = new ThreadGroup("PEWorkerGroup");
        this.timeoutExecutor = new TimeoutExecutorThread(this, this.threadGroup, "ProcExecTimeout");
        this.workerMonitorExecutor = new TimeoutExecutorThread(this, this.threadGroup, "WorkerMonitor");
        this.workerId.set(0L);
        this.workerThreads = new CopyOnWriteArrayList();
        for (int i = 0; i < this.corePoolSize; ++i) {
            this.workerThreads.add(new WorkerThread(this.threadGroup));
        }
        long st = System.nanoTime();
        this.store.recoverLease();
        long et = System.nanoTime();
        LOG.info("Recovered {} lease in {}", (Object)this.store.getClass().getSimpleName(), (Object)StringUtils.humanTimeDiff(TimeUnit.NANOSECONDS.toMillis(et - st)));
        this.scheduler.start();
        st = System.nanoTime();
        this.load(abortOnCorruption);
        et = System.nanoTime();
        LOG.info("Loaded {} in {}", (Object)this.store.getClass().getSimpleName(), (Object)StringUtils.humanTimeDiff(TimeUnit.NANOSECONDS.toMillis(et - st)));
    }

    public void startWorkers() throws IOException {
        if (!this.running.compareAndSet(false, true)) {
            LOG.warn("Already running");
            return;
        }
        LOG.trace("Start workers {}", (Object)this.workerThreads.size());
        this.timeoutExecutor.start();
        this.workerMonitorExecutor.start();
        for (WorkerThread worker : this.workerThreads) {
            worker.start();
        }
        this.workerMonitorExecutor.add(new WorkerMonitor());
        this.addChore(new CompletedProcedureCleaner<TEnvironment>(this.conf, this.store, this.procExecutionLock, this.completed, this.nonceKeysToProcIdsMap));
    }

    public void stop() {
        if (!this.running.getAndSet(false)) {
            return;
        }
        LOG.info("Stopping");
        this.scheduler.stop();
        this.timeoutExecutor.sendStopSignal();
        this.workerMonitorExecutor.sendStopSignal();
    }

    public void join() {
        assert (!this.isRunning()) : "expected not running";
        this.timeoutExecutor.awaitTermination();
        this.workerMonitorExecutor.awaitTermination();
        for (WorkerThread worker : this.workerThreads) {
            worker.awaitTermination();
        }
        if (this.threadGroup.activeCount() > 0) {
            LOG.error("There are still active thread in group {}, see STDOUT", (Object)this.threadGroup);
            this.threadGroup.list();
        }
        this.completed.clear();
        this.rollbackStack.clear();
        this.procedures.clear();
        this.nonceKeysToProcIdsMap.clear();
        this.scheduler.clear();
        this.lastProcId.set(-1L);
    }

    public void refreshConfiguration(Configuration conf) {
        this.conf = conf;
        this.setKeepAliveTime(conf.getLong(WORKER_KEEP_ALIVE_TIME_CONF_KEY, DEFAULT_WORKER_KEEP_ALIVE_TIME), TimeUnit.MILLISECONDS);
    }

    public boolean isRunning() {
        return this.running.get();
    }

    public int getWorkerThreadCount() {
        return this.workerThreads.size();
    }

    public int getCorePoolSize() {
        return this.corePoolSize;
    }

    public int getActiveExecutorCount() {
        return this.activeExecutorCount.get();
    }

    public TEnvironment getEnvironment() {
        return this.environment;
    }

    public ProcedureStore getStore() {
        return this.store;
    }

    ProcedureScheduler getScheduler() {
        return this.scheduler;
    }

    public void setKeepAliveTime(long keepAliveTime, TimeUnit timeUnit) {
        this.keepAliveTime = timeUnit.toMillis(keepAliveTime);
        this.scheduler.signalAll();
    }

    public long getKeepAliveTime(TimeUnit timeUnit) {
        return timeUnit.convert(this.keepAliveTime, TimeUnit.MILLISECONDS);
    }

    public void addChore(@Nullable ProcedureInMemoryChore<TEnvironment> chore) {
        if (chore == null) {
            return;
        }
        chore.setState(ProcedureProtos.ProcedureState.WAITING_TIMEOUT);
        this.timeoutExecutor.add(chore);
    }

    public boolean removeChore(@Nullable ProcedureInMemoryChore<TEnvironment> chore) {
        if (chore == null) {
            return true;
        }
        chore.setState(ProcedureProtos.ProcedureState.SUCCESS);
        return this.timeoutExecutor.remove(chore);
    }

    public NonceKey createNonceKey(long nonceGroup, long nonce) {
        return nonce == 0L ? null : new NonceKey(nonceGroup, nonce);
    }

    public long registerNonce(NonceKey nonceKey) {
        long newProcId;
        if (nonceKey == null) {
            return -1L;
        }
        Long oldProcId = this.nonceKeysToProcIdsMap.get(nonceKey);
        if (oldProcId == null && (oldProcId = this.nonceKeysToProcIdsMap.putIfAbsent(nonceKey, newProcId = this.nextProcId())) == null) {
            return -1L;
        }
        boolean traceEnabled = LOG.isTraceEnabled();
        while (this.isRunning() && !this.procedures.containsKey(oldProcId) && !this.completed.containsKey(oldProcId) && this.nonceKeysToProcIdsMap.containsKey(nonceKey)) {
            if (traceEnabled) {
                LOG.trace("Waiting for pid=" + oldProcId + " to be submitted");
            }
            Threads.sleep((long)100L);
        }
        return oldProcId;
    }

    public void unregisterNonceIfProcedureWasNotSubmitted(NonceKey nonceKey) {
        if (nonceKey == null) {
            return;
        }
        Long procId = this.nonceKeysToProcIdsMap.get(nonceKey);
        if (procId == null) {
            return;
        }
        if (!this.procedures.containsKey(procId) && !this.completed.containsKey(procId)) {
            this.nonceKeysToProcIdsMap.remove(nonceKey);
        }
    }

    public void setFailureResultForNonce(NonceKey nonceKey, String procName, User procOwner, IOException exception) {
        if (nonceKey == null) {
            return;
        }
        Long procId = this.nonceKeysToProcIdsMap.get(nonceKey);
        if (procId == null || this.completed.containsKey(procId)) {
            return;
        }
        this.completed.computeIfAbsent(procId, key -> {
            FailedProcedure proc = new FailedProcedure(procId, procName, procOwner, nonceKey, exception);
            return new CompletedProcedureRetainer(proc);
        });
    }

    public long submitProcedure(Procedure<TEnvironment> proc) {
        return this.submitProcedure(proc, null);
    }

    public List<Boolean> bypassProcedure(List<Long> pids, long lockWait, boolean force, boolean recursive) throws IOException {
        ArrayList<Boolean> result = new ArrayList<Boolean>(pids.size());
        for (long pid : pids) {
            result.add(this.bypassProcedure(pid, lockWait, force, recursive));
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean bypassProcedure(long pid, long lockWait, boolean override, boolean recursive) throws IOException {
        Preconditions.checkArgument((lockWait > 0L ? 1 : 0) != 0, (Object)"lockWait should be positive");
        Procedure procedure = this.getProcedure(pid);
        if (procedure == null) {
            LOG.debug("Procedure pid={} does not exist, skipping bypass", (Object)pid);
            return false;
        }
        LOG.debug("Begin bypass {} with lockWait={}, override={}, recursive={}", new Object[]{procedure, lockWait, override, recursive});
        IdLock.Entry lockEntry = this.procExecutionLock.tryLockEntry(procedure.getProcId(), lockWait);
        if (lockEntry == null && !override) {
            LOG.debug("Waited {} ms, but {} is still running, skipping bypass with force={}", new Object[]{lockWait, procedure, override});
            return false;
        }
        if (lockEntry == null) {
            LOG.debug("Waited {} ms, but {} is still running, begin bypass with force={}", new Object[]{lockWait, procedure, override});
        }
        try {
            if (procedure.isFinished()) {
                LOG.debug("{} is already finished, skipping bypass", procedure);
                boolean bl = false;
                return bl;
            }
            if (procedure.hasChildren()) {
                if (recursive) {
                    LOG.info("Recursive bypass on children of pid={}", (Object)procedure.getProcId());
                    this.procedures.forEachValue(1L, v -> v.getParentProcId() == procedure.getProcId() ? v : null, v -> {
                        try {
                            this.bypassProcedure(v.getProcId(), lockWait, override, recursive);
                        }
                        catch (IOException e) {
                            LOG.warn("Recursive bypass of pid={}", (Object)v.getProcId(), (Object)e);
                        }
                    });
                } else {
                    LOG.debug("{} has children, skipping bypass", procedure);
                    boolean bl = false;
                    return bl;
                }
            }
            if (procedure.hasParent() && procedure.getState() != ProcedureProtos.ProcedureState.RUNNABLE && procedure.getState() != ProcedureProtos.ProcedureState.WAITING && procedure.getState() != ProcedureProtos.ProcedureState.WAITING_TIMEOUT) {
                LOG.debug("Bypassing procedures in RUNNABLE, WAITING and WAITING_TIMEOUT states (with no parent), {}", procedure);
                boolean bl = false;
                return bl;
            }
            Procedure current = procedure;
            while (current != null) {
                LOG.debug("Bypassing {}", current);
                current.bypass(this.getEnvironment());
                this.store.update(current);
                long parentID = current.getParentProcId();
                current = this.getProcedure(parentID);
            }
            if (procedure.getState() == ProcedureProtos.ProcedureState.WAITING) {
                procedure.setState(ProcedureProtos.ProcedureState.RUNNABLE);
                this.store.update(procedure);
            }
            if (procedure.getState() == ProcedureProtos.ProcedureState.WAITING_TIMEOUT) {
                LOG.debug("transform procedure {} from WAITING_TIMEOUT to RUNNABLE", procedure);
                if (this.timeoutExecutor.remove(procedure)) {
                    LOG.debug("removed procedure {} from timeoutExecutor", procedure);
                    this.timeoutExecutor.executeTimedoutProcedure(procedure);
                }
            } else if (lockEntry != null) {
                this.scheduler.addFront(procedure);
                LOG.debug("Bypassing {} and its ancestors successfully, adding to queue", procedure);
            } else {
                LOG.debug("Bypassing {} and its ancestors successfully, but since it is already running, skipping add to queue", procedure);
            }
            boolean bl = true;
            return bl;
        }
        finally {
            if (lockEntry != null) {
                this.procExecutionLock.releaseLockEntry(lockEntry);
            }
        }
    }

    @SuppressWarnings(value={"NP_NULL_ON_SOME_PATH"}, justification="FindBugs is blind to the check-for-null")
    public long submitProcedure(Procedure<TEnvironment> proc, NonceKey nonceKey) {
        Long currentProcId;
        Preconditions.checkArgument((this.lastProcId.get() >= 0L ? 1 : 0) != 0);
        this.prepareProcedure(proc);
        if (nonceKey != null) {
            currentProcId = this.nonceKeysToProcIdsMap.get(nonceKey);
            Preconditions.checkArgument((currentProcId != null ? 1 : 0) != 0, (Object)("Expected nonceKey=" + nonceKey + " to be reserved, use registerNonce(); proc=" + proc));
        } else {
            currentProcId = this.nextProcId();
        }
        proc.setNonceKey(nonceKey);
        proc.setProcId(currentProcId);
        this.store.insert(proc, null);
        LOG.debug("Stored {}", proc);
        return this.pushProcedure(proc);
    }

    public void submitProcedures(Procedure<TEnvironment>[] procs) {
        int i;
        Preconditions.checkArgument((this.lastProcId.get() >= 0L ? 1 : 0) != 0);
        if (procs == null || procs.length <= 0) {
            return;
        }
        for (i = 0; i < procs.length; ++i) {
            this.prepareProcedure(procs[i]).setProcId(this.nextProcId());
        }
        this.store.insert(procs);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Stored " + Arrays.toString(procs));
        }
        for (i = 0; i < procs.length; ++i) {
            this.pushProcedure(procs[i]);
        }
    }

    private Procedure<TEnvironment> prepareProcedure(Procedure<TEnvironment> proc) {
        Preconditions.checkArgument((proc.getState() == ProcedureProtos.ProcedureState.INITIALIZING ? 1 : 0) != 0);
        Preconditions.checkArgument((!proc.hasParent() ? 1 : 0) != 0, (String)"unexpected parent", proc);
        if (this.checkOwnerSet) {
            Preconditions.checkArgument((boolean)proc.hasOwner(), (Object)"missing owner");
        }
        return proc;
    }

    private long pushProcedure(Procedure<TEnvironment> proc) {
        long currentProcId = proc.getProcId();
        proc.updateMetricsOnSubmit(this.getEnvironment());
        RootProcedureState stack = new RootProcedureState();
        stack.setRollbackSupported(proc.isRollbackSupported());
        this.rollbackStack.put(currentProcId, stack);
        assert (!this.procedures.containsKey(currentProcId));
        this.procedures.put(currentProcId, proc);
        this.sendProcedureAddedNotification(currentProcId);
        this.scheduler.addBack(proc);
        return proc.getProcId();
    }

    public boolean abort(long procId) {
        return this.abort(procId, true);
    }

    public boolean abort(long procId, boolean mayInterruptIfRunning) {
        Procedure<TEnvironment> proc = this.procedures.get(procId);
        if (proc != null) {
            if (!mayInterruptIfRunning && proc.wasExecuted()) {
                return false;
            }
            return proc.abort(this.getEnvironment());
        }
        return false;
    }

    public Procedure<TEnvironment> getProcedure(long procId) {
        return this.procedures.get(procId);
    }

    public <T extends Procedure<TEnvironment>> T getProcedure(Class<T> clazz, long procId) {
        Procedure<TEnvironment> proc = this.getProcedure(procId);
        if (clazz.isInstance(proc)) {
            return (T)((Procedure)clazz.cast(proc));
        }
        return null;
    }

    public Procedure<TEnvironment> getResult(long procId) {
        CompletedProcedureRetainer<TEnvironment> retainer = this.completed.get(procId);
        if (retainer == null) {
            return null;
        }
        return retainer.getProcedure();
    }

    public boolean isFinished(long procId) {
        return !this.procedures.containsKey(procId);
    }

    public boolean isStarted(long procId) {
        Procedure<TEnvironment> proc = this.procedures.get(procId);
        if (proc == null) {
            return this.completed.get(procId) != null;
        }
        return proc.wasExecuted();
    }

    public void removeResult(long procId) {
        CompletedProcedureRetainer<TEnvironment> retainer = this.completed.get(procId);
        if (retainer == null) {
            assert (!this.procedures.containsKey(procId)) : "pid=" + procId + " is still running";
            LOG.debug("pid={} already removed by the cleaner.", (Object)procId);
            return;
        }
        retainer.setClientAckTime(EnvironmentEdgeManager.currentTime());
    }

    public Procedure<TEnvironment> getResultOrProcedure(long procId) {
        CompletedProcedureRetainer<TEnvironment> retainer = this.completed.get(procId);
        if (retainer == null) {
            return this.procedures.get(procId);
        }
        return retainer.getProcedure();
    }

    public boolean isProcedureOwner(long procId, User user) {
        if (user == null) {
            return false;
        }
        Procedure<TEnvironment> runningProc = this.procedures.get(procId);
        if (runningProc != null) {
            return runningProc.getOwner().equals(user.getShortName());
        }
        CompletedProcedureRetainer<TEnvironment> retainer = this.completed.get(procId);
        if (retainer != null) {
            return retainer.getProcedure().getOwner().equals(user.getShortName());
        }
        return false;
    }

    public Collection<Procedure<TEnvironment>> getActiveProceduresNoCopy() {
        return this.procedures.values();
    }

    public List<Procedure<TEnvironment>> getProcedures() {
        ArrayList<Procedure<TEnvironment>> procedureList = new ArrayList<Procedure<TEnvironment>>(this.procedures.size() + this.completed.size());
        procedureList.addAll(this.procedures.values());
        this.completed.values().stream().map(CompletedProcedureRetainer::getProcedure).forEach(procedureList::add);
        return procedureList;
    }

    public void registerListener(ProcedureExecutorListener listener) {
        this.listeners.add(listener);
    }

    public boolean unregisterListener(ProcedureExecutorListener listener) {
        return this.listeners.remove(listener);
    }

    private void sendProcedureLoadedNotification(long procId) {
        if (!this.listeners.isEmpty()) {
            for (ProcedureExecutorListener listener : this.listeners) {
                try {
                    listener.procedureLoaded(procId);
                }
                catch (Throwable e) {
                    LOG.error("Listener " + listener + " had an error: " + e.getMessage(), e);
                }
            }
        }
    }

    private void sendProcedureAddedNotification(long procId) {
        if (!this.listeners.isEmpty()) {
            for (ProcedureExecutorListener listener : this.listeners) {
                try {
                    listener.procedureAdded(procId);
                }
                catch (Throwable e) {
                    LOG.error("Listener " + listener + " had an error: " + e.getMessage(), e);
                }
            }
        }
    }

    private void sendProcedureFinishedNotification(long procId) {
        if (!this.listeners.isEmpty()) {
            for (ProcedureExecutorListener listener : this.listeners) {
                try {
                    listener.procedureFinished(procId);
                }
                catch (Throwable e) {
                    LOG.error("Listener " + listener + " had an error: " + e.getMessage(), e);
                }
            }
        }
    }

    private long nextProcId() {
        long procId = this.lastProcId.incrementAndGet();
        if (procId < 0L) {
            while (!this.lastProcId.compareAndSet(procId, 0L) && (procId = this.lastProcId.get()) < 0L) {
            }
            while (this.procedures.containsKey(procId)) {
                procId = this.lastProcId.incrementAndGet();
            }
        }
        assert (procId >= 0L) : "Invalid procId " + procId;
        return procId;
    }

    protected long getLastProcId() {
        return this.lastProcId.get();
    }

    public Set<Long> getActiveProcIds() {
        return this.procedures.keySet();
    }

    Long getRootProcedureId(Procedure<TEnvironment> proc) {
        return Procedure.getRootProcedureId(this.procedures, proc);
    }

    private void executeProcedure(Procedure<TEnvironment> proc) {
        if (proc.isFinished()) {
            LOG.debug("{} is already finished, skipping execution", proc);
            return;
        }
        Long rootProcId = this.getRootProcedureId(proc);
        if (rootProcId == null) {
            LOG.warn("Rollback because parent is done/rolledback proc=" + proc);
            this.executeRollback(proc);
            return;
        }
        RootProcedureState<TEnvironment> procStack = this.rollbackStack.get(rootProcId);
        if (procStack == null) {
            LOG.warn("RootProcedureState is null for " + proc.getProcId());
            return;
        }
        block15: do {
            if (!procStack.acquire(proc)) {
                if (procStack.setRollback()) {
                    switch (this.executeRollback(rootProcId, procStack)) {
                        case LOCK_ACQUIRED: {
                            break block15;
                        }
                        case LOCK_YIELD_WAIT: {
                            procStack.unsetRollback();
                            this.scheduler.yield(proc);
                            break block15;
                        }
                        case LOCK_EVENT_WAIT: {
                            LOG.info("LOCK_EVENT_WAIT rollback..." + proc);
                            procStack.unsetRollback();
                            break block15;
                        }
                        default: {
                            throw new UnsupportedOperationException();
                        }
                    }
                }
                if (proc.wasExecuted()) break;
                switch (this.executeRollback(proc)) {
                    case LOCK_ACQUIRED: {
                        break block15;
                    }
                    case LOCK_YIELD_WAIT: {
                        this.scheduler.yield(proc);
                        break block15;
                    }
                    case LOCK_EVENT_WAIT: {
                        LOG.info("LOCK_EVENT_WAIT can't rollback child running?..." + proc);
                        break block15;
                    }
                    default: {
                        throw new UnsupportedOperationException();
                    }
                }
            }
            assert (proc.getState() == ProcedureProtos.ProcedureState.RUNNABLE) : proc;
            Procedure.LockState lockState = this.acquireLock(proc);
            switch (lockState) {
                case LOCK_ACQUIRED: {
                    this.execProcedure(procStack, proc);
                    break;
                }
                case LOCK_YIELD_WAIT: {
                    LOG.info((Object)((Object)lockState) + " " + proc);
                    this.scheduler.yield(proc);
                    break;
                }
                case LOCK_EVENT_WAIT: {
                    LOG.debug((Object)((Object)lockState) + " " + proc);
                    break;
                }
                default: {
                    throw new UnsupportedOperationException();
                }
            }
            procStack.release(proc);
            if (!proc.isSuccess()) continue;
            proc.updateMetricsOnFinish(this.getEnvironment(), proc.elapsedTime(), true);
            LOG.info("Finished " + proc + " in " + StringUtils.humanTimeDiff(proc.elapsedTime()));
            if (proc.getProcId() == rootProcId.longValue()) {
                this.procedureFinished(proc);
                break;
            }
            this.execCompletionCleanup(proc);
            break;
        } while (procStack.isFailed());
    }

    private Procedure.LockState acquireLock(Procedure<TEnvironment> proc) {
        TEnvironment env = this.getEnvironment();
        if (proc.hasLock()) {
            return Procedure.LockState.LOCK_ACQUIRED;
        }
        return proc.doAcquireLock(env, this.store);
    }

    private void releaseLock(Procedure<TEnvironment> proc, boolean force) {
        TEnvironment env = this.getEnvironment();
        if (force || !proc.holdLock(env) || proc.isFinished()) {
            proc.doReleaseLock(env, this.store);
        }
    }

    private IdLock.Entry getLockEntryForRollback(long procId) {
        if (!this.procExecutionLock.isHeldByCurrentThread(procId)) {
            try {
                return this.procExecutionLock.getLockEntry(procId);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void executeUnexpectedRollback(Procedure<TEnvironment> rootProc, RootProcedureState<TEnvironment> procStack) {
        if (procStack.getSubprocs() != null) {
            Procedure subproc;
            PriorityQueue<Procedure> pq = new PriorityQueue<Procedure>(procStack.getSubprocs().size(), Comparator.comparingLong(Procedure::getProcId).reversed());
            pq.addAll(procStack.getSubprocs());
            while ((subproc = pq.poll()) != null) {
                if (!this.procedures.containsKey(subproc.getProcId())) continue;
                IdLock.Entry lockEntry = this.getLockEntryForRollback(subproc.getProcId());
                try {
                    this.cleanupAfterRollbackOneStep(subproc);
                    this.execCompletionCleanup(subproc);
                }
                finally {
                    if (lockEntry != null) {
                        this.procExecutionLock.releaseLockEntry(lockEntry);
                    }
                }
            }
        }
        IdLock.Entry lockEntry = this.getLockEntryForRollback(rootProc.getProcId());
        try {
            this.cleanupAfterRollbackOneStep(rootProc);
        }
        finally {
            if (lockEntry != null) {
                this.procExecutionLock.releaseLockEntry(lockEntry);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Procedure.LockState executeNormalRollback(Procedure<TEnvironment> rootProc, RootProcedureState<TEnvironment> procStack) {
        List<Procedure<TEnvironment>> subprocStack = procStack.getSubproceduresStack();
        assert (subprocStack != null) : "Called rollback with no steps executed rootProc=" + rootProc;
        int stackTail = subprocStack.size();
        while (stackTail-- > 0) {
            Procedure<TEnvironment> proc = subprocStack.get(stackTail);
            IdLock.Entry lockEntry = this.getLockEntryForRollback(proc.getProcId());
            try {
                if (proc.isSuccess()) {
                    subprocStack.remove(stackTail);
                    this.cleanupAfterRollbackOneStep(proc);
                    continue;
                }
                Procedure.LockState lockState = this.acquireLock(proc);
                if (lockState != Procedure.LockState.LOCK_ACQUIRED) {
                    Procedure.LockState lockState2 = lockState;
                    return lockState2;
                }
                lockState = this.executeRollback(proc);
                this.releaseLock(proc, false);
                boolean abortRollback = lockState != Procedure.LockState.LOCK_ACQUIRED;
                if (abortRollback |= !this.isRunning() || !this.store.isRunning()) {
                    Procedure.LockState lockState3 = lockState;
                    return lockState3;
                }
                subprocStack.remove(stackTail);
                if (!proc.isFinished() && proc.isYieldAfterExecutionStep(this.getEnvironment())) {
                    Procedure.LockState lockState4 = Procedure.LockState.LOCK_YIELD_WAIT;
                    return lockState4;
                }
                if (proc == rootProc) continue;
                this.execCompletionCleanup(proc);
            }
            finally {
                if (lockEntry == null) continue;
                this.procExecutionLock.releaseLockEntry(lockEntry);
            }
        }
        return Procedure.LockState.LOCK_ACQUIRED;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Procedure.LockState executeRollback(long rootProcId, RootProcedureState<TEnvironment> procStack) {
        Procedure<TEnvironment> rootProc = this.procedures.get(rootProcId);
        RemoteProcedureException exception = rootProc.getException();
        if (exception == null) {
            exception = procStack.getException();
            rootProc.setFailure(exception);
            this.store.update(rootProc);
        }
        if (procStack.isRollbackSupported()) {
            Procedure.LockState lockState = this.executeNormalRollback(rootProc, procStack);
            if (lockState != Procedure.LockState.LOCK_ACQUIRED) {
                return lockState;
            }
        } else {
            LOG.error(HBaseMarkers.FATAL, "Root Procedure {} does not support rollback but the execution failed and try to rollback, code bug?", rootProc, (Object)exception);
            this.executeUnexpectedRollback(rootProc, procStack);
        }
        IdLock.Entry lockEntry = this.getLockEntryForRollback(rootProc.getProcId());
        try {
            LOG.info("Rolled back {} exec-time={}", rootProc, (Object)StringUtils.humanTimeDiff(rootProc.elapsedTime()));
            this.procedureFinished(rootProc);
        }
        finally {
            if (lockEntry != null) {
                this.procExecutionLock.releaseLockEntry(lockEntry);
            }
        }
        return Procedure.LockState.LOCK_ACQUIRED;
    }

    private void cleanupAfterRollbackOneStep(Procedure<TEnvironment> proc) {
        if (this.testing != null && this.testing.shouldKillBeforeStoreUpdateInRollback()) {
            this.kill("TESTING: Kill BEFORE store update in rollback: " + proc);
        }
        if (proc.removeStackIndex()) {
            if (!proc.isSuccess()) {
                proc.setState(ProcedureProtos.ProcedureState.ROLLEDBACK);
            }
            proc.updateMetricsOnFinish(this.getEnvironment(), proc.elapsedTime(), false);
            if (proc.hasParent()) {
                this.store.delete(proc.getProcId());
                this.procedures.remove(proc.getProcId());
            } else {
                long[] childProcIds = this.rollbackStack.get(proc.getProcId()).getSubprocedureIds();
                if (childProcIds != null) {
                    this.store.delete(proc, childProcIds);
                } else {
                    this.store.update(proc);
                }
            }
        } else {
            this.store.update(proc);
        }
    }

    private Procedure.LockState executeRollback(Procedure<TEnvironment> proc) {
        try {
            proc.doRollback(this.getEnvironment());
        }
        catch (IOException e) {
            LOG.debug("Roll back attempt failed for {}", proc, (Object)e);
            return Procedure.LockState.LOCK_YIELD_WAIT;
        }
        catch (InterruptedException e) {
            this.handleInterruptedException(proc, e);
            return Procedure.LockState.LOCK_YIELD_WAIT;
        }
        catch (Throwable e) {
            LOG.error(HBaseMarkers.FATAL, "CODE-BUG: Uncaught runtime exception for " + proc, e);
        }
        this.cleanupAfterRollbackOneStep(proc);
        return Procedure.LockState.LOCK_ACQUIRED;
    }

    private void yieldProcedure(Procedure<TEnvironment> proc) {
        this.releaseLock(proc, false);
        this.scheduler.yield(proc);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void execProcedure(RootProcedureState<TEnvironment> procStack, Procedure<TEnvironment> procedure) {
        Preconditions.checkArgument((procedure.getState() == ProcedureProtos.ProcedureState.RUNNABLE ? 1 : 0) != 0, (Object)("NOT RUNNABLE! " + procedure.toString()));
        boolean suspended = false;
        boolean reExecute = false;
        Procedure<TEnvironment>[] subprocs = null;
        do {
            reExecute = false;
            procedure.resetPersistence();
            try {
                subprocs = procedure.doExecute(this.getEnvironment());
                if (subprocs != null && subprocs.length == 0) {
                    subprocs = null;
                }
            }
            catch (ProcedureSuspendedException e2) {
                LOG.trace("Suspend {}", procedure);
                suspended = true;
            }
            catch (ProcedureYieldException e3) {
                LOG.trace("Yield {}", procedure, (Object)e3);
                this.yieldProcedure(procedure);
                return;
            }
            catch (InterruptedException e4) {
                LOG.trace("Yield interrupt {}", procedure, (Object)e4);
                this.handleInterruptedException(procedure, e4);
                this.yieldProcedure(procedure);
                return;
            }
            catch (Throwable e5) {
                String msg = "CODE-BUG: Uncaught runtime exception: " + procedure;
                LOG.error(msg, e5);
                procedure.setFailure(new RemoteProcedureException(msg, e5));
            }
            if (!procedure.isFailed()) {
                if (subprocs != null) {
                    if (subprocs.length == 1 && subprocs[0] == procedure) {
                        subprocs = null;
                        reExecute = true;
                        LOG.trace("Short-circuit to next step on pid={}", (Object)procedure.getProcId());
                    } else {
                        subprocs = this.initializeChildren(procStack, procedure, subprocs);
                        LOG.info("Initialized subprocedures=" + (subprocs == null ? null : Stream.of(subprocs).map(e -> "{" + e.toString() + "}").collect(Collectors.toList()).toString()));
                    }
                } else if (procedure.getState() == ProcedureProtos.ProcedureState.WAITING_TIMEOUT) {
                    LOG.trace("Added to timeoutExecutor {}", procedure);
                    this.timeoutExecutor.add(procedure);
                } else if (!suspended) {
                    procedure.setState(ProcedureProtos.ProcedureState.SUCCESS);
                }
            }
            if (this.testing != null && this.testing.shouldKillBeforeStoreUpdate(suspended, procedure.hasParent())) {
                this.kill("TESTING: Kill BEFORE store update: " + procedure);
            }
            if (procedure.needPersistence()) {
                boolean needUpdateStoreOutsideLock = false;
                RootProcedureState<TEnvironment> rootProcedureState = procStack;
                synchronized (rootProcedureState) {
                    if (procStack.addRollbackStep(procedure)) {
                        this.updateStoreOnExec(procStack, procedure, subprocs);
                    } else {
                        needUpdateStoreOutsideLock = true;
                    }
                }
                if (needUpdateStoreOutsideLock) {
                    this.updateStoreOnExec(procStack, procedure, subprocs);
                }
            }
            if (!this.store.isRunning()) {
                return;
            }
            if (procedure.isRunnable() && !suspended && procedure.isYieldAfterExecutionStep(this.getEnvironment())) {
                this.yieldProcedure(procedure);
                return;
            }
            assert (reExecute && subprocs == null || !reExecute);
        } while (reExecute);
        if (this.testing != null && this.testing.shouldKillAfterStoreUpdate(suspended)) {
            this.kill("TESTING: Kill AFTER store update: " + procedure);
        }
        if (subprocs != null && !procedure.isFailed()) {
            this.submitChildrenProcedures(subprocs);
        }
        this.releaseLock(procedure, false);
        if (!suspended && procedure.isFinished() && procedure.hasParent()) {
            this.countDownChildren(procStack, procedure);
        }
    }

    private void kill(String msg) {
        LOG.debug(msg);
        this.stop();
        throw new RuntimeException(msg);
    }

    private Procedure<TEnvironment>[] initializeChildren(RootProcedureState<TEnvironment> procStack, Procedure<TEnvironment> procedure, Procedure<TEnvironment>[] subprocs) {
        assert (subprocs != null) : "expected subprocedures";
        long rootProcId = this.getRootProcedureId(procedure);
        for (int i = 0; i < subprocs.length; ++i) {
            Procedure<TEnvironment> subproc = subprocs[i];
            if (subproc == null) {
                String msg = "subproc[" + i + "] is null, aborting the procedure";
                procedure.setFailure(new RemoteProcedureException(msg, (Throwable)new IllegalArgumentIOException(msg)));
                return null;
            }
            assert (subproc.getState() == ProcedureProtos.ProcedureState.INITIALIZING) : subproc;
            subproc.setParentProcId(procedure.getProcId());
            subproc.setRootProcId(rootProcId);
            subproc.setProcId(this.nextProcId());
            procStack.addSubProcedure(subproc);
        }
        if (!procedure.isFailed()) {
            procedure.setChildrenLatch(subprocs.length);
            switch (procedure.getState()) {
                case RUNNABLE: {
                    procedure.setState(ProcedureProtos.ProcedureState.WAITING);
                    break;
                }
                case WAITING_TIMEOUT: {
                    this.timeoutExecutor.add(procedure);
                    break;
                }
            }
        }
        return subprocs;
    }

    private void submitChildrenProcedures(Procedure<TEnvironment>[] subprocs) {
        for (int i = 0; i < subprocs.length; ++i) {
            Procedure<TEnvironment> subproc = subprocs[i];
            subproc.updateMetricsOnSubmit(this.getEnvironment());
            assert (!this.procedures.containsKey(subproc.getProcId()));
            this.procedures.put(subproc.getProcId(), subproc);
            this.scheduler.addFront(subproc);
        }
    }

    private void countDownChildren(RootProcedureState<TEnvironment> procStack, Procedure<TEnvironment> procedure) {
        Procedure<TEnvironment> parent = this.procedures.get(procedure.getParentProcId());
        if (parent == null) {
            assert (procStack.isRollingback());
            return;
        }
        if (parent.tryRunnable()) {
            this.store.update(parent);
            this.scheduler.addFront(parent);
            LOG.info("Finished subprocedure pid={}, resume processing ppid={}", (Object)procedure.getProcId(), (Object)parent.getProcId());
            return;
        }
    }

    private void updateStoreOnExec(RootProcedureState<TEnvironment> procStack, Procedure<TEnvironment> procedure, Procedure<TEnvironment>[] subprocs) {
        if (subprocs != null && !procedure.isFailed()) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Stored " + procedure + ", children " + Arrays.toString(subprocs));
            }
            this.store.insert(procedure, subprocs);
        } else {
            LOG.trace("Store update {}", procedure);
            if (procedure.isFinished() && !procedure.hasParent()) {
                long[] childProcIds = procStack.getSubprocedureIds();
                if (childProcIds != null) {
                    this.store.delete(procedure, childProcIds);
                    for (int i = 0; i < childProcIds.length; ++i) {
                        this.procedures.remove(childProcIds[i]);
                    }
                } else {
                    this.store.update(procedure);
                }
            } else {
                this.store.update(procedure);
            }
        }
    }

    private void handleInterruptedException(Procedure<TEnvironment> proc, InterruptedException e) {
        LOG.trace("Interrupt during {}. suspend and retry it later.", proc, (Object)e);
    }

    private void execCompletionCleanup(Procedure<TEnvironment> proc) {
        TEnvironment env = this.getEnvironment();
        if (proc.hasLock()) {
            LOG.warn("Usually this should not happen, we will release the lock before if the procedure is finished, even if the holdLock is true, arrive here means we have some holes where we do not release the lock. And the releaseLock below may fail since the procedure may have already been deleted from the procedure store.");
            this.releaseLock(proc, true);
        }
        try {
            proc.completionCleanup(env);
        }
        catch (Throwable e) {
            LOG.error("CODE-BUG: uncatched runtime exception for procedure: " + proc, e);
        }
    }

    private void procedureFinished(Procedure<TEnvironment> proc) {
        this.execCompletionCleanup(proc);
        CompletedProcedureRetainer<TEnvironment> retainer = new CompletedProcedureRetainer<TEnvironment>(proc);
        if (!proc.shouldWaitClientAck(this.getEnvironment())) {
            retainer.setClientAckTime(0L);
        }
        this.completed.put(proc.getProcId(), retainer);
        this.rollbackStack.remove(proc.getProcId());
        this.procedures.remove(proc.getProcId());
        try {
            this.scheduler.completionCleanup(proc);
        }
        catch (Throwable e) {
            LOG.error("CODE-BUG: uncatched runtime exception for completion cleanup: {}", proc, (Object)e);
        }
        this.sendProcedureFinishedNotification(proc.getProcId());
    }

    RootProcedureState<TEnvironment> getProcStack(long rootProcId) {
        return this.rollbackStack.get(rootProcId);
    }

    ProcedureScheduler getProcedureScheduler() {
        return this.scheduler;
    }

    int getCompletedSize() {
        return this.completed.size();
    }

    public IdLock getProcExecutionLock() {
        return this.procExecutionLock;
    }

    private final class WorkerMonitor
    extends InlineChore {
        public static final String WORKER_MONITOR_INTERVAL_CONF_KEY = "hbase.procedure.worker.monitor.interval.msec";
        private static final int DEFAULT_WORKER_MONITOR_INTERVAL = 5000;
        public static final String WORKER_STUCK_THRESHOLD_CONF_KEY = "hbase.procedure.worker.stuck.threshold.msec";
        private static final int DEFAULT_WORKER_STUCK_THRESHOLD = 10000;
        public static final String WORKER_ADD_STUCK_PERCENTAGE_CONF_KEY = "hbase.procedure.worker.add.stuck.percentage";
        private static final float DEFAULT_WORKER_ADD_STUCK_PERCENTAGE = 0.5f;
        private float addWorkerStuckPercentage = 0.5f;
        private int timeoutInterval = 5000;
        private int stuckThreshold = 10000;

        public WorkerMonitor() {
            this.refreshConfig();
        }

        @Override
        public void run() {
            int stuckCount = this.checkForStuckWorkers();
            this.checkThreadCount(stuckCount);
            this.refreshConfig();
        }

        private int checkForStuckWorkers() {
            int stuckCount = 0;
            for (WorkerThread worker : ProcedureExecutor.this.workerThreads) {
                if (worker.getCurrentRunTime() < (long)this.stuckThreshold) continue;
                ++stuckCount;
                LOG.warn("Worker stuck {}, run time {}", (Object)worker, (Object)StringUtils.humanTimeDiff(worker.getCurrentRunTime()));
            }
            return stuckCount;
        }

        private void checkThreadCount(int stuckCount) {
            if (stuckCount < 1 || !ProcedureExecutor.this.scheduler.hasRunnables()) {
                return;
            }
            float stuckPerc = (float)stuckCount / (float)ProcedureExecutor.this.workerThreads.size();
            if (stuckPerc >= this.addWorkerStuckPercentage && ProcedureExecutor.this.workerThreads.size() < ProcedureExecutor.this.maxPoolSize) {
                KeepAliveWorkerThread worker = new KeepAliveWorkerThread(ProcedureExecutor.this.threadGroup);
                ProcedureExecutor.this.workerThreads.add(worker);
                worker.start();
                LOG.debug("Added new worker thread {}", (Object)worker);
            }
        }

        private void refreshConfig() {
            this.addWorkerStuckPercentage = ProcedureExecutor.this.conf.getFloat(WORKER_ADD_STUCK_PERCENTAGE_CONF_KEY, 0.5f);
            this.timeoutInterval = ProcedureExecutor.this.conf.getInt(WORKER_MONITOR_INTERVAL_CONF_KEY, 5000);
            this.stuckThreshold = ProcedureExecutor.this.conf.getInt(WORKER_STUCK_THRESHOLD_CONF_KEY, 10000);
        }

        @Override
        public int getTimeoutInterval() {
            return this.timeoutInterval;
        }
    }

    private final class KeepAliveWorkerThread
    extends WorkerThread {
        public KeepAliveWorkerThread(ThreadGroup group) {
            super(group, "KeepAlivePEWorker-");
        }

        @Override
        protected boolean keepAlive(long lastUpdate) {
            return EnvironmentEdgeManager.currentTime() - lastUpdate < ProcedureExecutor.this.keepAliveTime;
        }
    }

    private class WorkerThread
    extends StoppableThread {
        private final AtomicLong executionStartTime;
        private volatile Procedure<TEnvironment> activeProcedure;

        public WorkerThread(ThreadGroup group) {
            this(group, "PEWorker-");
        }

        protected WorkerThread(ThreadGroup group, String prefix) {
            super(group, prefix + ProcedureExecutor.this.workerId.incrementAndGet());
            this.executionStartTime = new AtomicLong(Long.MAX_VALUE);
            this.setDaemon(true);
        }

        @Override
        public void sendStopSignal() {
            ProcedureExecutor.this.scheduler.signalAll();
        }

        private long runProcedure() throws IOException {
            Procedure proc = this.activeProcedure;
            int activeCount = ProcedureExecutor.this.activeExecutorCount.incrementAndGet();
            int runningCount = ProcedureExecutor.this.store.setRunningProcedureCount(activeCount);
            LOG.trace("Execute pid={} runningCount={}, activeCount={}", new Object[]{proc.getProcId(), runningCount, activeCount});
            this.executionStartTime.set(EnvironmentEdgeManager.currentTime());
            IdLock.Entry lockEntry = ProcedureExecutor.this.procExecutionLock.getLockEntry(proc.getProcId());
            try {
                ProcedureExecutor.this.executeProcedure(proc);
            }
            catch (AssertionError e) {
                try {
                    LOG.info("ASSERT pid=" + proc.getProcId(), (Throwable)((Object)e));
                    throw e;
                }
                catch (Throwable throwable) {
                    ProcedureExecutor.this.procExecutionLock.releaseLockEntry(lockEntry);
                    activeCount = ProcedureExecutor.this.activeExecutorCount.decrementAndGet();
                    runningCount = ProcedureExecutor.this.store.setRunningProcedureCount(activeCount);
                    LOG.trace("Halt pid={} runningCount={}, activeCount={}", new Object[]{proc.getProcId(), runningCount, activeCount});
                    this.activeProcedure = null;
                    this.executionStartTime.set(Long.MAX_VALUE);
                    throw throwable;
                }
            }
            ProcedureExecutor.this.procExecutionLock.releaseLockEntry(lockEntry);
            activeCount = ProcedureExecutor.this.activeExecutorCount.decrementAndGet();
            runningCount = ProcedureExecutor.this.store.setRunningProcedureCount(activeCount);
            LOG.trace("Halt pid={} runningCount={}, activeCount={}", new Object[]{proc.getProcId(), runningCount, activeCount});
            this.activeProcedure = null;
            this.executionStartTime.set(Long.MAX_VALUE);
            return EnvironmentEdgeManager.currentTime();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            long lastUpdate = EnvironmentEdgeManager.currentTime();
            try {
                while (ProcedureExecutor.this.isRunning() && this.keepAlive(lastUpdate)) {
                    Procedure proc = ProcedureExecutor.this.scheduler.poll(ProcedureExecutor.this.keepAliveTime, TimeUnit.MILLISECONDS);
                    if (proc == null) continue;
                    this.activeProcedure = proc;
                    lastUpdate = (Long)TraceUtil.trace(this::runProcedure, (Supplier)new ProcedureSpanBuilder(proc));
                }
            }
            catch (Throwable t) {
                LOG.warn("Worker terminating UNNATURALLY {}", this.activeProcedure, (Object)t);
            }
            finally {
                LOG.trace("Worker terminated.");
            }
            ProcedureExecutor.this.workerThreads.remove(this);
        }

        @Override
        public String toString() {
            Procedure p = this.activeProcedure;
            return this.getName() + "(pid=" + (p == null ? Long.valueOf(-1L) : p.getProcId() + ")");
        }

        public long getCurrentRunTime() {
            return EnvironmentEdgeManager.currentTime() - this.executionStartTime.get();
        }

        protected boolean keepAlive(long lastUpdate) {
            return true;
        }
    }

    public static interface ProcedureExecutorListener {
        public void procedureLoaded(long var1);

        public void procedureAdded(long var1);

        public void procedureFinished(long var1);
    }

    public static class Testing {
        protected volatile boolean killIfHasParent = true;
        protected volatile boolean killIfSuspended = false;
        protected volatile boolean killBeforeStoreUpdate = false;
        protected volatile boolean toggleKillBeforeStoreUpdate = false;
        protected volatile boolean killAfterStoreUpdate = false;
        protected volatile boolean toggleKillAfterStoreUpdate = false;
        protected volatile boolean killBeforeStoreUpdateInRollback = false;
        protected volatile boolean toggleKillBeforeStoreUpdateInRollback = false;

        protected boolean shouldKillBeforeStoreUpdate() {
            boolean kill = this.killBeforeStoreUpdate;
            if (this.toggleKillBeforeStoreUpdate) {
                this.killBeforeStoreUpdate = !kill;
                LOG.warn("Toggle KILL before store update to: " + this.killBeforeStoreUpdate);
            }
            return kill;
        }

        protected boolean shouldKillBeforeStoreUpdate(boolean isSuspended, boolean hasParent) {
            if (isSuspended && !this.killIfSuspended) {
                return false;
            }
            if (hasParent && !this.killIfHasParent) {
                return false;
            }
            return this.shouldKillBeforeStoreUpdate();
        }

        protected boolean shouldKillAfterStoreUpdate() {
            boolean kill = this.killAfterStoreUpdate;
            if (this.toggleKillAfterStoreUpdate) {
                this.killAfterStoreUpdate = !kill;
                LOG.warn("Toggle KILL after store update to: " + this.killAfterStoreUpdate);
            }
            return kill;
        }

        protected boolean shouldKillAfterStoreUpdate(boolean isSuspended) {
            return isSuspended && !this.killIfSuspended ? false : this.shouldKillAfterStoreUpdate();
        }

        protected boolean shouldKillBeforeStoreUpdateInRollback() {
            boolean kill = this.killBeforeStoreUpdateInRollback;
            if (this.toggleKillBeforeStoreUpdateInRollback) {
                this.killBeforeStoreUpdateInRollback = !kill;
                LOG.warn("Toggle KILL before store update in rollback to: " + this.killBeforeStoreUpdateInRollback);
            }
            return kill;
        }
    }
}

