/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile.bucket;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedByInterruptException;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.io.hfile.bucket.PersistentIOEngine;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.util.Shell;
import org.apache.hadoop.util.StringUtils;

@InterfaceAudience.Private
public class FileIOEngine
implements PersistentIOEngine {
    private static final Log LOG = LogFactory.getLog(FileIOEngine.class);
    public static final String FILE_DELIMITER = ",";
    private static final DuFileCommand DU = new DuFileCommand(new String[]{"du", ""});
    private final String[] filePaths;
    private final FileChannel[] fileChannels;
    private final RandomAccessFile[] rafs;
    private final ReentrantLock[] channelLocks;
    private final long sizePerFile;
    private final long capacity;
    private FileReadAccessor readAccessor = new FileReadAccessor();
    private FileWriteAccessor writeAccessor = new FileWriteAccessor();

    public FileIOEngine(long capacity, String ... filePaths) throws IOException {
        this.sizePerFile = capacity / (long)filePaths.length;
        this.capacity = this.sizePerFile * (long)filePaths.length;
        this.filePaths = filePaths;
        this.fileChannels = new FileChannel[filePaths.length];
        this.rafs = new RandomAccessFile[filePaths.length];
        this.channelLocks = new ReentrantLock[filePaths.length];
        for (int i = 0; i < filePaths.length; ++i) {
            String filePath = filePaths[i];
            try {
                File file;
                this.rafs[i] = new RandomAccessFile(filePath, "rw");
                long totalSpace = new File(filePath).getTotalSpace();
                if (totalSpace < this.sizePerFile) {
                    String msg = "Only " + StringUtils.byteDesc((long)totalSpace) + " total space under " + filePath + ", not enough for requested " + StringUtils.byteDesc((long)this.sizePerFile);
                    LOG.warn((Object)msg);
                }
                if ((file = new File(filePath)).length() != this.sizePerFile) {
                    this.rafs[i].setLength(this.sizePerFile);
                }
                this.fileChannels[i] = this.rafs[i].getChannel();
                this.channelLocks[i] = new ReentrantLock();
                LOG.info((Object)("Allocating cache " + StringUtils.byteDesc((long)this.sizePerFile) + ", on the path: " + filePath));
                continue;
            }
            catch (IOException fex) {
                LOG.error((Object)("Failed allocating cache on " + filePath), (Throwable)fex);
                this.shutdown();
                throw fex;
            }
        }
    }

    @Override
    public boolean verifyFileIntegrity(byte[] persistentChecksum, String algorithm) {
        byte[] calculateChecksum = this.calculateChecksum(algorithm);
        if (!Bytes.equals((byte[])persistentChecksum, (byte[])calculateChecksum)) {
            LOG.error((Object)("Mismatch of checksum! The persistent checksum is " + Bytes.toString((byte[])persistentChecksum) + ", but the calculate checksum is " + Bytes.toString((byte[])calculateChecksum)));
            return false;
        }
        return true;
    }

    public String toString() {
        return "ioengine=" + this.getClass().getSimpleName() + ", paths=" + Arrays.asList(this.filePaths) + ", capacity=" + String.format("%,d", this.capacity);
    }

    @Override
    public boolean isPersistent() {
        return true;
    }

    @Override
    public int read(ByteBuffer dstBuffer, long offset) throws IOException {
        if (dstBuffer.remaining() != 0) {
            return this.accessFile(this.readAccessor, dstBuffer, offset);
        }
        return 0;
    }

    void closeFileChannels() {
        for (FileChannel fileChannel : this.fileChannels) {
            try {
                fileChannel.close();
            }
            catch (IOException e) {
                LOG.warn((Object)"Failed to close FileChannel", (Throwable)e);
            }
        }
    }

    @Override
    public void write(ByteBuffer srcBuffer, long offset) throws IOException {
        if (!srcBuffer.hasRemaining()) {
            return;
        }
        this.accessFile(this.writeAccessor, srcBuffer, offset);
    }

    @Override
    public void sync() throws IOException {
        for (int i = 0; i < this.fileChannels.length; ++i) {
            try {
                if (this.fileChannels[i] == null) continue;
                this.fileChannels[i].force(true);
                continue;
            }
            catch (IOException ie) {
                LOG.warn((Object)("Failed syncing data to " + this.filePaths[i]));
                throw ie;
            }
        }
    }

    @Override
    public void shutdown() {
        for (int i = 0; i < this.filePaths.length; ++i) {
            try {
                if (this.fileChannels[i] != null) {
                    this.fileChannels[i].close();
                }
                if (this.rafs[i] == null) continue;
                this.rafs[i].close();
                continue;
            }
            catch (IOException ex) {
                LOG.error((Object)("Failed closing " + this.filePaths[i] + " when shudown the IOEngine"), (Throwable)ex);
            }
        }
    }

    private int accessFile(FileAccessor accessor, ByteBuffer buffer, long globalOffset) throws IOException {
        int bufLimit;
        block5: {
            int startFileNum = this.getFileNum(globalOffset);
            int remainingAccessDataLen = buffer.remaining();
            int endFileNum = this.getFileNum(globalOffset + (long)remainingAccessDataLen - 1L);
            int accessFileNum = startFileNum;
            long accessOffset = this.getAbsoluteOffsetInFile(accessFileNum, globalOffset);
            bufLimit = buffer.limit();
            while (true) {
                FileChannel fileChannel = this.fileChannels[accessFileNum];
                int accessLen = 0;
                if (endFileNum > accessFileNum) {
                    buffer.limit((int)((long)(buffer.limit() - remainingAccessDataLen) + this.sizePerFile - accessOffset));
                }
                try {
                    accessLen = accessor.access(fileChannel, buffer, accessOffset);
                }
                catch (ClosedByInterruptException e) {
                    throw e;
                }
                catch (ClosedChannelException e) {
                    this.refreshFileConnection(accessFileNum, e);
                    continue;
                }
                buffer.limit(bufLimit);
                if (accessLen >= remainingAccessDataLen) break block5;
                remainingAccessDataLen -= accessLen;
                accessOffset = 0L;
                if (++accessFileNum >= this.fileChannels.length) break;
            }
            throw new IOException("Required data len " + StringUtils.byteDesc((long)buffer.remaining()) + " exceed the engine's capacity " + StringUtils.byteDesc((long)this.capacity) + " where offset=" + globalOffset);
        }
        return bufLimit;
    }

    private long getAbsoluteOffsetInFile(int fileNum, long globalOffset) {
        return globalOffset - (long)fileNum * this.sizePerFile;
    }

    private int getFileNum(long offset) {
        if (offset < 0L) {
            throw new IllegalArgumentException("Unexpected offset " + offset);
        }
        int fileNum = (int)(offset / this.sizePerFile);
        if (fileNum >= this.fileChannels.length) {
            throw new RuntimeException("Not expected offset " + offset + " where capacity=" + this.capacity);
        }
        return fileNum;
    }

    FileChannel[] getFileChannels() {
        return this.fileChannels;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void refreshFileConnection(int accessFileNum, IOException ioe) throws IOException {
        ReentrantLock channelLock = this.channelLocks[accessFileNum];
        channelLock.lock();
        try {
            FileChannel fileChannel = this.fileChannels[accessFileNum];
            if (fileChannel != null) {
                if (fileChannel.isOpen()) {
                    return;
                }
                fileChannel.close();
            }
            LOG.warn((Object)("Caught ClosedChannelException accessing BucketCache, reopening file: " + this.filePaths[accessFileNum]), (Throwable)ioe);
            this.rafs[accessFileNum] = new RandomAccessFile(this.filePaths[accessFileNum], "rw");
            this.fileChannels[accessFileNum] = this.rafs[accessFileNum].getChannel();
        }
        finally {
            channelLock.unlock();
        }
    }

    @Override
    public byte[] calculateChecksum(String algorithm) {
        if (this.filePaths == null) {
            return null;
        }
        try {
            StringBuilder sb = new StringBuilder();
            for (String filePath : this.filePaths) {
                File file = new File(filePath);
                sb.append(filePath);
                sb.append(FileIOEngine.getFileSize(filePath));
                sb.append(file.lastModified());
            }
            MessageDigest messageDigest = MessageDigest.getInstance(algorithm);
            messageDigest.update(Bytes.toBytes((String)sb.toString()));
            return messageDigest.digest();
        }
        catch (IOException ioex) {
            LOG.error((Object)"Calculating checksum failed.", (Throwable)ioex);
            return null;
        }
        catch (NoSuchAlgorithmException e) {
            LOG.error((Object)("No such algorithm : " + algorithm + "!"));
            return null;
        }
    }

    private static long getFileSize(String filePath) throws IOException {
        DU.setExecCommand(filePath);
        DU.execute();
        return Long.parseLong(DU.getOutput().split("\t")[0]);
    }

    private static class FileWriteAccessor
    implements FileAccessor {
        private FileWriteAccessor() {
        }

        @Override
        public int access(FileChannel fileChannel, ByteBuffer byteBuffer, long accessOffset) throws IOException {
            return fileChannel.write(byteBuffer, accessOffset);
        }
    }

    private static class FileReadAccessor
    implements FileAccessor {
        private FileReadAccessor() {
        }

        @Override
        public int access(FileChannel fileChannel, ByteBuffer byteBuffer, long accessOffset) throws IOException {
            return fileChannel.read(byteBuffer, accessOffset);
        }
    }

    private static interface FileAccessor {
        public int access(FileChannel var1, ByteBuffer var2, long var3) throws IOException;
    }

    private static class DuFileCommand
    extends Shell.ShellCommandExecutor {
        private String[] execCommand;

        DuFileCommand(String[] execString) {
            super(execString);
            this.execCommand = execString;
        }

        void setExecCommand(String filePath) {
            this.execCommand[1] = filePath;
        }

        public String[] getExecString() {
            return this.execCommand;
        }
    }
}

