/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.client;

import java.io.Closeable;
import java.io.IOException;
import java.lang.invoke.LambdaMetafactory;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.NotServingRegionException;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.TableNotFoundException;
import org.apache.hadoop.hbase.Waiter;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.AsyncConnectionImpl;
import org.apache.hadoop.hbase.client.AsyncNonMetaRegionLocator;
import org.apache.hadoop.hbase.client.AsyncRegistry;
import org.apache.hadoop.hbase.client.AsyncRegistryFactory;
import org.apache.hadoop.hbase.client.RegionLocateType;
import org.apache.hadoop.hbase.security.User;
import org.apache.hadoop.hbase.testclassification.ClientTests;
import org.apache.hadoop.hbase.testclassification.MediumTests;
import org.apache.hadoop.hbase.util.Bytes;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;

@Category(value={MediumTests.class, ClientTests.class})
public class TestAsyncNonMetaRegionLocator {
    private static final HBaseTestingUtility TEST_UTIL = new HBaseTestingUtility();
    private static TableName TABLE_NAME = TableName.valueOf((String)"async");
    private static byte[] FAMILY = Bytes.toBytes((String)"cf");
    private static AsyncConnectionImpl CONN;
    private static AsyncNonMetaRegionLocator LOCATOR;
    private static byte[][] SPLIT_KEYS;

    @BeforeClass
    public static void setUp() throws Exception {
        TEST_UTIL.startMiniCluster(3);
        TEST_UTIL.getAdmin().setBalancerRunning(false, true);
        AsyncRegistry registry = AsyncRegistryFactory.getRegistry((Configuration)TEST_UTIL.getConfiguration());
        CONN = new AsyncConnectionImpl(TEST_UTIL.getConfiguration(), registry, (String)registry.getClusterId().get(), User.getCurrent());
        LOCATOR = new AsyncNonMetaRegionLocator(CONN);
        SPLIT_KEYS = new byte[8][];
        for (int i = 111; i < 999; i += 111) {
            TestAsyncNonMetaRegionLocator.SPLIT_KEYS[i / 111 - 1] = Bytes.toBytes((String)String.format("%03d", i));
        }
    }

    @AfterClass
    public static void tearDown() throws Exception {
        IOUtils.closeQuietly((Closeable)CONN);
        TEST_UTIL.shutdownMiniCluster();
    }

    @After
    public void tearDownAfterTest() throws IOException {
        Admin admin = TEST_UTIL.getAdmin();
        if (admin.tableExists(TABLE_NAME)) {
            if (admin.isTableEnabled(TABLE_NAME)) {
                TEST_UTIL.getAdmin().disableTable(TABLE_NAME);
            }
            TEST_UTIL.getAdmin().deleteTable(TABLE_NAME);
        }
        LOCATOR.clearCache(TABLE_NAME);
    }

    private void createSingleRegionTable() throws IOException, InterruptedException {
        TEST_UTIL.createTable(TABLE_NAME, FAMILY);
        TEST_UTIL.waitTableAvailable(TABLE_NAME);
    }

    @Test
    public void testNoTable() throws InterruptedException {
        for (RegionLocateType locateType : RegionLocateType.values()) {
            try {
                LOCATOR.getRegionLocation(TABLE_NAME, HConstants.EMPTY_START_ROW, locateType, false).get();
            }
            catch (ExecutionException e) {
                Assert.assertThat((Object)e.getCause(), (Matcher)CoreMatchers.instanceOf(TableNotFoundException.class));
            }
        }
    }

    @Test
    public void testDisableTable() throws IOException, InterruptedException {
        this.createSingleRegionTable();
        TEST_UTIL.getAdmin().disableTable(TABLE_NAME);
        for (RegionLocateType locateType : RegionLocateType.values()) {
            try {
                LOCATOR.getRegionLocation(TABLE_NAME, HConstants.EMPTY_START_ROW, locateType, false).get();
            }
            catch (ExecutionException e) {
                Assert.assertThat((Object)e.getCause(), (Matcher)CoreMatchers.instanceOf(TableNotFoundException.class));
            }
        }
    }

    private void assertLocEquals(byte[] startKey, byte[] endKey, ServerName serverName, HRegionLocation loc) {
        HRegionInfo info = loc.getRegionInfo();
        Assert.assertEquals((Object)TABLE_NAME, (Object)info.getTable());
        Assert.assertArrayEquals((byte[])startKey, (byte[])info.getStartKey());
        Assert.assertArrayEquals((byte[])endKey, (byte[])info.getEndKey());
        Assert.assertEquals((Object)serverName, (Object)loc.getServerName());
    }

    @Test
    public void testSingleRegionTable() throws IOException, InterruptedException, ExecutionException {
        this.createSingleRegionTable();
        ServerName serverName = TEST_UTIL.getRSForFirstRegionInTable(TABLE_NAME).getServerName();
        for (RegionLocateType locateType : RegionLocateType.values()) {
            this.assertLocEquals(HConstants.EMPTY_START_ROW, HConstants.EMPTY_END_ROW, serverName, (HRegionLocation)LOCATOR.getRegionLocation(TABLE_NAME, HConstants.EMPTY_START_ROW, locateType, false).get());
        }
        byte[] randKey = new byte[ThreadLocalRandom.current().nextInt(128)];
        ThreadLocalRandom.current().nextBytes(randKey);
        for (RegionLocateType locateType : RegionLocateType.values()) {
            this.assertLocEquals(HConstants.EMPTY_START_ROW, HConstants.EMPTY_END_ROW, serverName, (HRegionLocation)LOCATOR.getRegionLocation(TABLE_NAME, randKey, locateType, false).get());
        }
    }

    private void createMultiRegionTable() throws IOException, InterruptedException {
        TEST_UTIL.createTable(TABLE_NAME, FAMILY, SPLIT_KEYS);
        TEST_UTIL.waitTableAvailable(TABLE_NAME);
    }

    private static byte[][] getStartKeys() {
        byte[][] startKeys = new byte[SPLIT_KEYS.length + 1][];
        startKeys[0] = HConstants.EMPTY_START_ROW;
        System.arraycopy(SPLIT_KEYS, 0, startKeys, 1, SPLIT_KEYS.length);
        return startKeys;
    }

    private static byte[][] getEndKeys() {
        byte[][] endKeys = (byte[][])Arrays.copyOf(SPLIT_KEYS, SPLIT_KEYS.length + 1);
        endKeys[endKeys.length - 1] = HConstants.EMPTY_START_ROW;
        return endKeys;
    }

    private ServerName[] getLocations(byte[][] startKeys) {
        ServerName[] serverNames = new ServerName[startKeys.length];
        TEST_UTIL.getHBaseCluster().getRegionServerThreads().stream().map(t -> t.getRegionServer()).forEach(rs -> rs.getRegions(TABLE_NAME).forEach(r -> {
            serverNames[Arrays.binarySearch(startKeys, r.getRegionInfo().getStartKey(), (Comparator)LambdaMetafactory.metafactory(null, null, null, (Ljava/lang/Object;Ljava/lang/Object;)I, compareTo(byte[] byte[] ), ([B[B)I)())] = rs.getServerName();
        }));
        return serverNames;
    }

    @Test
    public void testMultiRegionTable() throws IOException, InterruptedException {
        this.createMultiRegionTable();
        byte[][] startKeys = TestAsyncNonMetaRegionLocator.getStartKeys();
        ServerName[] serverNames = this.getLocations(startKeys);
        IntStream.range(0, 2).forEach(n -> IntStream.range(0, startKeys.length).forEach(i -> {
            try {
                this.assertLocEquals(startKeys[i], i == startKeys.length - 1 ? HConstants.EMPTY_END_ROW : startKeys[i + 1], serverNames[i], (HRegionLocation)LOCATOR.getRegionLocation(TABLE_NAME, startKeys[i], RegionLocateType.CURRENT, false).get());
            }
            catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException(e);
            }
        }));
        LOCATOR.clearCache(TABLE_NAME);
        IntStream.range(0, 2).forEach(n -> IntStream.range(0, startKeys.length).forEach(i -> {
            try {
                this.assertLocEquals(startKeys[i], i == startKeys.length - 1 ? HConstants.EMPTY_END_ROW : startKeys[i + 1], serverNames[i], (HRegionLocation)LOCATOR.getRegionLocation(TABLE_NAME, startKeys[i], RegionLocateType.AFTER, false).get());
            }
            catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException(e);
            }
        }));
        LOCATOR.clearCache(TABLE_NAME);
        byte[][] endKeys = TestAsyncNonMetaRegionLocator.getEndKeys();
        IntStream.range(0, 2).forEach(n -> IntStream.range(0, endKeys.length).map(i -> endKeys.length - 1 - i).forEach(i -> {
            try {
                this.assertLocEquals(i == 0 ? HConstants.EMPTY_START_ROW : endKeys[i - 1], endKeys[i], serverNames[i], (HRegionLocation)LOCATOR.getRegionLocation(TABLE_NAME, endKeys[i], RegionLocateType.BEFORE, false).get());
            }
            catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException(e);
            }
        }));
    }

    @Test
    public void testRegionMove() throws IOException, InterruptedException, ExecutionException {
        this.createSingleRegionTable();
        ServerName serverName = TEST_UTIL.getRSForFirstRegionInTable(TABLE_NAME).getServerName();
        HRegionLocation loc = (HRegionLocation)LOCATOR.getRegionLocation(TABLE_NAME, HConstants.EMPTY_START_ROW, RegionLocateType.CURRENT, false).get();
        this.assertLocEquals(HConstants.EMPTY_START_ROW, HConstants.EMPTY_END_ROW, serverName, loc);
        ServerName newServerName = TEST_UTIL.getHBaseCluster().getRegionServerThreads().stream().map(t -> t.getRegionServer().getServerName()).filter(sn -> !sn.equals((Object)serverName)).findAny().get();
        TEST_UTIL.getAdmin().move(Bytes.toBytes((String)loc.getRegionInfo().getEncodedName()), Bytes.toBytes((String)newServerName.getServerName()));
        while (!TEST_UTIL.getRSForFirstRegionInTable(TABLE_NAME).getServerName().equals((Object)newServerName)) {
            Thread.sleep(100L);
        }
        Assert.assertSame((Object)loc, LOCATOR.getRegionLocation(TABLE_NAME, HConstants.EMPTY_START_ROW, RegionLocateType.CURRENT, false).get());
        LOCATOR.updateCachedLocation(loc, null);
        Assert.assertSame((Object)loc, LOCATOR.getRegionLocation(TABLE_NAME, HConstants.EMPTY_START_ROW, RegionLocateType.CURRENT, false).get());
        LOCATOR.updateCachedLocation(loc, (Throwable)new NotServingRegionException());
        this.assertLocEquals(HConstants.EMPTY_START_ROW, HConstants.EMPTY_END_ROW, newServerName, (HRegionLocation)LOCATOR.getRegionLocation(TABLE_NAME, HConstants.EMPTY_START_ROW, RegionLocateType.CURRENT, false).get());
    }

    @Test
    public void testLocateAfter() throws IOException, InterruptedException, ExecutionException {
        byte[] row = Bytes.toBytes((String)"1");
        byte[] splitKey = Arrays.copyOf(row, 2);
        TEST_UTIL.createTable(TABLE_NAME, FAMILY, (byte[][])new byte[][]{splitKey});
        TEST_UTIL.waitTableAvailable(TABLE_NAME);
        HRegionLocation currentLoc = (HRegionLocation)LOCATOR.getRegionLocation(TABLE_NAME, row, RegionLocateType.CURRENT, false).get();
        ServerName currentServerName = TEST_UTIL.getRSForFirstRegionInTable(TABLE_NAME).getServerName();
        this.assertLocEquals(HConstants.EMPTY_START_ROW, splitKey, currentServerName, currentLoc);
        HRegionLocation afterLoc = (HRegionLocation)LOCATOR.getRegionLocation(TABLE_NAME, row, RegionLocateType.AFTER, false).get();
        ServerName afterServerName = TEST_UTIL.getHBaseCluster().getRegionServerThreads().stream().map(t -> t.getRegionServer()).filter(rs -> rs.getRegions(TABLE_NAME).stream().anyMatch(r -> Bytes.equals((byte[])splitKey, (byte[])r.getRegionInfo().getStartKey()))).findAny().get().getServerName();
        this.assertLocEquals(splitKey, HConstants.EMPTY_END_ROW, afterServerName, afterLoc);
        Assert.assertSame((Object)afterLoc, LOCATOR.getRegionLocation(TABLE_NAME, row, RegionLocateType.AFTER, false).get());
    }

    @Test
    public void testConcurrentLocate() throws IOException, InterruptedException, ExecutionException {
        this.createMultiRegionTable();
        byte[][] startKeys = TestAsyncNonMetaRegionLocator.getStartKeys();
        byte[][] endKeys = TestAsyncNonMetaRegionLocator.getEndKeys();
        ServerName[] serverNames = this.getLocations(startKeys);
        for (int i = 0; i < 100; ++i) {
            LOCATOR.clearCache(TABLE_NAME);
            List futures = IntStream.range(0, 1000).mapToObj(n -> String.format("%03d", n)).map(s -> Bytes.toBytes((String)s)).map(r -> LOCATOR.getRegionLocation(TABLE_NAME, r, RegionLocateType.CURRENT, false)).collect(Collectors.toList());
            for (int j = 0; j < 1000; ++j) {
                int index = Math.min(8, j / 111);
                this.assertLocEquals(startKeys[index], endKeys[index], serverNames[index], (HRegionLocation)((CompletableFuture)futures.get(j)).get());
            }
        }
    }

    @Test
    public void testReload() throws Exception {
        this.createSingleRegionTable();
        final ServerName serverName = TEST_UTIL.getRSForFirstRegionInTable(TABLE_NAME).getServerName();
        for (RegionLocateType locateType : RegionLocateType.values()) {
            this.assertLocEquals(HConstants.EMPTY_START_ROW, HConstants.EMPTY_END_ROW, serverName, (HRegionLocation)LOCATOR.getRegionLocation(TABLE_NAME, HConstants.EMPTY_START_ROW, locateType, false).get());
        }
        ServerName newServerName = TEST_UTIL.getHBaseCluster().getRegionServerThreads().stream().map(t -> t.getRegionServer().getServerName()).filter(sn -> !sn.equals((Object)serverName)).findAny().get();
        Admin admin = TEST_UTIL.getAdmin();
        final HRegionInfo region = (HRegionInfo)admin.getTableRegions(TABLE_NAME).stream().findAny().get();
        admin.move(region.getEncodedNameAsBytes(), Bytes.toBytes((String)newServerName.getServerName()));
        TEST_UTIL.waitFor(30000L, new Waiter.ExplainingPredicate<Exception>(){

            public boolean evaluate() throws Exception {
                ServerName newServerName = TEST_UTIL.getRSForFirstRegionInTable(TABLE_NAME).getServerName();
                return newServerName != null && !newServerName.equals((Object)serverName);
            }

            public String explainFailure() throws Exception {
                return region.getRegionNameAsString() + " is still on " + serverName;
            }
        });
        for (RegionLocateType locateType : RegionLocateType.values()) {
            this.assertLocEquals(HConstants.EMPTY_START_ROW, HConstants.EMPTY_END_ROW, serverName, (HRegionLocation)LOCATOR.getRegionLocation(TABLE_NAME, HConstants.EMPTY_START_ROW, locateType, false).get());
        }
        this.assertLocEquals(HConstants.EMPTY_START_ROW, HConstants.EMPTY_END_ROW, newServerName, (HRegionLocation)LOCATOR.getRegionLocation(TABLE_NAME, HConstants.EMPTY_START_ROW, RegionLocateType.CURRENT, true).get());
        for (RegionLocateType locateType : RegionLocateType.values()) {
            this.assertLocEquals(HConstants.EMPTY_START_ROW, HConstants.EMPTY_END_ROW, newServerName, (HRegionLocation)LOCATOR.getRegionLocation(TABLE_NAME, HConstants.EMPTY_START_ROW, locateType, false).get());
        }
    }
}

