/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.assignment;

import java.io.IOException;
import java.net.SocketTimeoutException;
import java.util.NavigableMap;
import java.util.Random;
import java.util.Set;
import java.util.SortedSet;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.CategoryBasedTimeout;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.NotServingRegionException;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.RegionInfoBuilder;
import org.apache.hadoop.hbase.client.RetriesExhaustedException;
import org.apache.hadoop.hbase.exceptions.UnexpectedStateException;
import org.apache.hadoop.hbase.ipc.ServerNotRunningYetException;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.RegionState;
import org.apache.hadoop.hbase.master.assignment.AssignProcedure;
import org.apache.hadoop.hbase.master.assignment.AssignmentManager;
import org.apache.hadoop.hbase.master.assignment.MockMasterServices;
import org.apache.hadoop.hbase.master.assignment.UnassignProcedure;
import org.apache.hadoop.hbase.master.procedure.ProcedureSyncWait;
import org.apache.hadoop.hbase.master.procedure.RSProcedureDispatcher;
import org.apache.hadoop.hbase.procedure2.Procedure;
import org.apache.hadoop.hbase.procedure2.ProcedureMetrics;
import org.apache.hadoop.hbase.procedure2.ProcedureTestingUtility;
import org.apache.hadoop.hbase.procedure2.RemoteProcedureDispatcher;
import org.apache.hadoop.hbase.procedure2.util.StringUtils;
import org.apache.hadoop.hbase.regionserver.RegionServerAbortedException;
import org.apache.hadoop.hbase.regionserver.RegionServerStoppedException;
import org.apache.hadoop.hbase.shaded.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.shaded.protobuf.generated.AdminProtos;
import org.apache.hadoop.hbase.shaded.protobuf.generated.HBaseProtos;
import org.apache.hadoop.hbase.shaded.protobuf.generated.RegionServerStatusProtos;
import org.apache.hadoop.hbase.testclassification.MasterTests;
import org.apache.hadoop.hbase.testclassification.MediumTests;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.ipc.RemoteException;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.ExpectedException;
import org.junit.rules.TestName;
import org.junit.rules.TestRule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Category(value={MasterTests.class, MediumTests.class})
public class TestAssignmentManager {
    private static final Logger LOG = LoggerFactory.getLogger(TestAssignmentManager.class);
    @Rule
    public TestName name = new TestName();
    @Rule
    public final TestRule timeout = CategoryBasedTimeout.builder().withTimeout(this.getClass()).withLookingForStuckThread(true).build();
    @Rule
    public final ExpectedException exception = ExpectedException.none();
    private static final int PROC_NTHREADS = 64;
    private static final int NREGIONS = 1000;
    private static final int NSERVERS = Math.max(1, 10);
    private HBaseTestingUtility UTIL;
    private MockRSProcedureDispatcher rsDispatcher;
    private MockMasterServices master;
    private AssignmentManager am;
    private NavigableMap<ServerName, SortedSet<byte[]>> regionsToRegionServers = new ConcurrentSkipListMap<ServerName, SortedSet<byte[]>>();
    private ScheduledExecutorService executor;
    private ProcedureMetrics assignProcMetrics;
    private ProcedureMetrics unassignProcMetrics;
    private long assignSubmittedCount = 0L;
    private long assignFailedCount = 0L;
    private long unassignSubmittedCount = 0L;
    private long unassignFailedCount = 0L;

    private void setupConfiguration(Configuration conf) throws Exception {
        FSUtils.setRootDir((Configuration)conf, (Path)this.UTIL.getDataTestDir());
        conf.setBoolean("hbase.procedure.store.wal.use.hsync", false);
        conf.setInt("hbase.procedure.store.wal.sync.wait.msec", 10);
        conf.setInt("hbase.master.procedure.threads", 64);
        conf.setInt("hbase.regionserver.rpc.startup.waittime", 1000);
        conf.setInt("hbase.assignment.maximum.attempts", 100);
    }

    @Before
    public void setUp() throws Exception {
        this.UTIL = new HBaseTestingUtility();
        this.executor = Executors.newSingleThreadScheduledExecutor();
        this.setupConfiguration(this.UTIL.getConfiguration());
        this.master = new MockMasterServices(this.UTIL.getConfiguration(), this.regionsToRegionServers);
        this.rsDispatcher = new MockRSProcedureDispatcher(this.master);
        this.master.start(NSERVERS, this.rsDispatcher);
        this.am = this.master.getAssignmentManager();
        this.assignProcMetrics = this.am.getAssignmentManagerMetrics().getAssignProcMetrics();
        this.unassignProcMetrics = this.am.getAssignmentManagerMetrics().getUnassignProcMetrics();
        this.setUpMeta();
    }

    private void setUpMeta() throws Exception {
        this.rsDispatcher.setMockRsExecutor(new GoodRsExecutor());
        this.am.assign(RegionInfoBuilder.FIRST_META_REGIONINFO);
        this.am.wakeMetaLoadedEvent();
        this.am.setFailoverCleanupDone(true);
    }

    @After
    public void tearDown() throws Exception {
        this.master.stop("tearDown");
        this.executor.shutdownNow();
    }

    @Test(expected=NullPointerException.class)
    public void testWaitServerReportEventWithNullServer() throws UnexpectedStateException {
        if (this.am.waitServerReportEvent(null, null)) {
            throw new UnexpectedStateException();
        }
    }

    @Test
    public void testAssignWithGoodExec() throws Exception {
        this.collectAssignmentManagerMetrics();
        this.testAssign(new GoodRsExecutor());
        Assert.assertEquals((long)(this.assignSubmittedCount + 1000L), (long)this.assignProcMetrics.getSubmittedCounter().getCount());
        Assert.assertEquals((long)this.assignFailedCount, (long)this.assignProcMetrics.getFailedCounter().getCount());
    }

    @Test
    public void testAssignAndCrashBeforeResponse() throws Exception {
        TableName tableName = TableName.valueOf((String)"testAssignAndCrashBeforeResponse");
        RegionInfo hri = this.createRegionInfo(tableName, 1L);
        this.rsDispatcher.setMockRsExecutor(new HangThenRSCrashExecutor());
        AssignProcedure proc = this.am.createAssignProcedure(hri);
        this.waitOnFuture(this.submitProcedure((Procedure)proc));
    }

    @Test
    public void testUnassignAndCrashBeforeResponse() throws Exception {
        TableName tableName = TableName.valueOf((String)"testAssignAndCrashBeforeResponse");
        RegionInfo hri = this.createRegionInfo(tableName, 1L);
        this.rsDispatcher.setMockRsExecutor(new HangOnCloseThenRSCrashExecutor());
        for (int i = 0; i < 6; ++i) {
            AssignProcedure assign = this.am.createAssignProcedure(hri);
            this.waitOnFuture(this.submitProcedure((Procedure)assign));
            UnassignProcedure unassign = this.am.createUnassignProcedure(hri, this.am.getRegionStates().getRegionServerOfRegion(hri), false);
            this.waitOnFuture(this.submitProcedure((Procedure)unassign));
        }
    }

    @Test
    public void testAssignWithRandExec() throws Exception {
        TableName tableName = TableName.valueOf((String)"testAssignWithRandExec");
        RegionInfo hri = this.createRegionInfo(tableName, 1L);
        this.rsDispatcher.setMockRsExecutor(new RandRsExecutor());
        for (int i = 0; i < 10; ++i) {
            LOG.info("" + i);
            AssignProcedure proc = this.am.createAssignProcedure(hri);
            this.waitOnFuture(this.submitProcedure((Procedure)proc));
        }
    }

    @Ignore
    @Test
    public void testSocketTimeout() throws Exception {
        TableName tableName = TableName.valueOf((String)this.name.getMethodName());
        RegionInfo hri = this.createRegionInfo(tableName, 1L);
        this.collectAssignmentManagerMetrics();
        this.rsDispatcher.setMockRsExecutor(new SocketTimeoutRsExecutor(20, 3));
        this.waitOnFuture(this.submitProcedure((Procedure)this.am.createAssignProcedure(hri)));
        this.rsDispatcher.setMockRsExecutor(new SocketTimeoutRsExecutor(20, 1));
        this.waitOnFuture(this.submitProcedure((Procedure)this.am.createUnassignProcedure(hri, null, false)));
        Assert.assertEquals((long)(this.assignSubmittedCount + 1L), (long)this.assignProcMetrics.getSubmittedCounter().getCount());
        Assert.assertEquals((long)this.assignFailedCount, (long)this.assignProcMetrics.getFailedCounter().getCount());
        Assert.assertEquals((long)(this.unassignSubmittedCount + 1L), (long)this.unassignProcMetrics.getSubmittedCounter().getCount());
        Assert.assertEquals((long)(this.unassignFailedCount + 1L), (long)this.unassignProcMetrics.getFailedCounter().getCount());
    }

    @Test
    public void testServerNotYetRunning() throws Exception {
        this.testRetriesExhaustedFailure(TableName.valueOf((String)this.name.getMethodName()), new ServerNotYetRunningRsExecutor());
    }

    private void testRetriesExhaustedFailure(TableName tableName, MockRSExecutor executor) throws Exception {
        RegionInfo hri = this.createRegionInfo(tableName, 1L);
        this.collectAssignmentManagerMetrics();
        this.rsDispatcher.setMockRsExecutor(executor);
        try {
            this.waitOnFuture(this.submitProcedure((Procedure)this.am.createAssignProcedure(hri)));
            Assert.fail((String)"unexpected assign completion");
        }
        catch (RetriesExhaustedException e) {
            LOG.info("expected exception from assign operation: " + e.getMessage(), (Throwable)e);
        }
        this.rsDispatcher.setMockRsExecutor(new GoodRsExecutor());
        this.waitOnFuture(this.submitProcedure((Procedure)this.am.createAssignProcedure(hri)));
    }

    @Test
    public void testIOExceptionOnAssignment() throws Exception {
        this.collectAssignmentManagerMetrics();
        this.testFailedOpen(TableName.valueOf((String)"testExceptionOnAssignment"), new FaultyRsExecutor(new IOException("test fault")));
        Assert.assertEquals((long)(this.assignSubmittedCount + 1L), (long)this.assignProcMetrics.getSubmittedCounter().getCount());
        Assert.assertEquals((long)(this.assignFailedCount + 1L), (long)this.assignProcMetrics.getFailedCounter().getCount());
    }

    @Test
    public void testDoNotRetryExceptionOnAssignment() throws Exception {
        this.collectAssignmentManagerMetrics();
        this.testFailedOpen(TableName.valueOf((String)"testDoNotRetryExceptionOnAssignment"), new FaultyRsExecutor((IOException)((Object)new DoNotRetryIOException("test do not retry fault"))));
        Assert.assertEquals((long)(this.assignSubmittedCount + 1L), (long)this.assignProcMetrics.getSubmittedCounter().getCount());
        Assert.assertEquals((long)(this.assignFailedCount + 1L), (long)this.assignProcMetrics.getFailedCounter().getCount());
    }

    private void testFailedOpen(TableName tableName, MockRSExecutor executor) throws Exception {
        RegionInfo hri = this.createRegionInfo(tableName, 1L);
        this.rsDispatcher.setMockRsExecutor(executor);
        try {
            this.waitOnFuture(this.submitProcedure((Procedure)this.am.createAssignProcedure(hri)));
            Assert.fail((String)"unexpected assign completion");
        }
        catch (RetriesExhaustedException e) {
            LOG.info("REGION STATE " + this.am.getRegionStates().getRegionStateNode(hri));
            LOG.info("expected exception from assign operation: " + e.getMessage(), (Throwable)e);
            Assert.assertEquals((Object)true, (Object)this.am.getRegionStates().getRegionState(hri).isFailedOpen());
        }
    }

    private void testAssign(MockRSExecutor executor) throws Exception {
        this.testAssign(executor, 1000);
    }

    private void testAssign(MockRSExecutor executor, int nregions) throws Exception {
        this.rsDispatcher.setMockRsExecutor(executor);
        AssignProcedure[] assignments = new AssignProcedure[nregions];
        long st = System.currentTimeMillis();
        this.bulkSubmit(assignments);
        for (int i = 0; i < assignments.length; ++i) {
            ProcedureTestingUtility.waitProcedure(this.master.getMasterProcedureExecutor(), (Procedure)assignments[i]);
            Assert.assertTrue((String)assignments[i].toString(), (boolean)assignments[i].isSuccess());
        }
        long et = System.currentTimeMillis();
        float sec = (float)(et - st) / 1000.0f;
        LOG.info(String.format("[T] Assigning %dprocs in %s (%.2fproc/sec)", assignments.length, StringUtils.humanTimeDiff((long)(et - st)), Float.valueOf((float)assignments.length / sec)));
    }

    @Test
    public void testAssignAnAssignedRegion() throws Exception {
        TableName tableName = TableName.valueOf((String)"testAssignAnAssignedRegion");
        RegionInfo hri = this.createRegionInfo(tableName, 1L);
        this.collectAssignmentManagerMetrics();
        this.rsDispatcher.setMockRsExecutor(new GoodRsExecutor());
        Future<byte[]> futureA = this.submitProcedure((Procedure)this.am.createAssignProcedure(hri));
        this.waitOnFuture(futureA);
        this.am.getRegionStates().isRegionInState(hri, new RegionState.State[]{RegionState.State.OPEN});
        Future<byte[]> futureB = this.submitProcedure((Procedure)this.am.createAssignProcedure(hri));
        this.waitOnFuture(futureB);
        this.am.getRegionStates().isRegionInState(hri, new RegionState.State[]{RegionState.State.OPEN});
        Assert.assertEquals((long)(this.assignSubmittedCount + 2L), (long)this.assignProcMetrics.getSubmittedCounter().getCount());
        Assert.assertEquals((long)this.assignFailedCount, (long)this.assignProcMetrics.getFailedCounter().getCount());
    }

    @Test
    public void testUnassignAnUnassignedRegion() throws Exception {
        TableName tableName = TableName.valueOf((String)"testUnassignAnUnassignedRegion");
        RegionInfo hri = this.createRegionInfo(tableName, 1L);
        this.collectAssignmentManagerMetrics();
        this.rsDispatcher.setMockRsExecutor(new GoodRsExecutor());
        this.waitOnFuture(this.submitProcedure((Procedure)this.am.createAssignProcedure(hri)));
        Future<byte[]> futureA = this.submitProcedure((Procedure)this.am.createUnassignProcedure(hri, null, false));
        this.waitOnFuture(futureA);
        this.am.getRegionStates().isRegionInState(hri, new RegionState.State[]{RegionState.State.CLOSED});
        Future<byte[]> futureB = this.submitProcedure((Procedure)this.am.createUnassignProcedure(hri, ServerName.valueOf((String)"example.org,1234,1"), false));
        this.waitOnFuture(futureB);
        this.am.getRegionStates().isRegionInState(hri, new RegionState.State[]{RegionState.State.CLOSED});
        Assert.assertEquals((long)(this.assignSubmittedCount + 1L), (long)this.assignProcMetrics.getSubmittedCounter().getCount());
        Assert.assertEquals((long)this.assignFailedCount, (long)this.assignProcMetrics.getFailedCounter().getCount());
        Assert.assertEquals((long)(this.unassignSubmittedCount + 2L), (long)this.unassignProcMetrics.getSubmittedCounter().getCount());
        Assert.assertEquals((long)this.unassignFailedCount, (long)this.unassignProcMetrics.getFailedCounter().getCount());
    }

    @Test
    public void testAssignMetaAndCrashBeforeResponse() throws Exception {
        this.tearDown();
        this.UTIL = new HBaseTestingUtility();
        this.executor = Executors.newSingleThreadScheduledExecutor();
        this.setupConfiguration(this.UTIL.getConfiguration());
        this.master = new MockMasterServices(this.UTIL.getConfiguration(), this.regionsToRegionServers);
        this.rsDispatcher = new MockRSProcedureDispatcher(this.master);
        this.master.start(NSERVERS, this.rsDispatcher);
        this.am = this.master.getAssignmentManager();
        this.master.setServerCrashProcessingEnabled(false);
        this.rsDispatcher.setMockRsExecutor(new HangThenRSRestartExecutor());
        this.am.assign(RegionInfoBuilder.FIRST_META_REGIONINFO);
        Assert.assertEquals((Object)true, (Object)this.am.isMetaInitialized());
        this.master.setServerCrashProcessingEnabled(true);
        this.am.wakeMetaLoadedEvent();
        this.am.setFailoverCleanupDone(true);
    }

    private Future<byte[]> submitProcedure(Procedure proc) {
        return ProcedureSyncWait.submitProcedure(this.master.getMasterProcedureExecutor(), (Procedure)proc);
    }

    private byte[] waitOnFuture(Future<byte[]> future) throws Exception {
        try {
            return future.get(5L, TimeUnit.SECONDS);
        }
        catch (ExecutionException e) {
            LOG.info("ExecutionException", (Throwable)e);
            throw (Exception)e.getCause();
        }
    }

    private void bulkSubmit(final AssignProcedure[] procs) throws Exception {
        int i;
        final Thread[] threads = new Thread[64];
        for (i = 0; i < threads.length; ++i) {
            final int threadId = i;
            threads[i] = new Thread(){

                @Override
                public void run() {
                    TableName tableName = TableName.valueOf((String)("table-" + threadId));
                    int n = procs.length / threads.length;
                    int start = threadId * n;
                    int stop = start + n;
                    for (int j = start; j < stop; ++j) {
                        procs[j] = TestAssignmentManager.this.createAndSubmitAssign(tableName, j);
                    }
                }
            };
            threads[i].start();
        }
        for (i = 0; i < threads.length; ++i) {
            threads[i].join();
        }
        for (i = procs.length - 1; i >= 0 && procs[i] == null; --i) {
            procs[i] = this.createAndSubmitAssign(TableName.valueOf((String)"table-sync"), i);
        }
    }

    private AssignProcedure createAndSubmitAssign(TableName tableName, int regionId) {
        RegionInfo hri = this.createRegionInfo(tableName, regionId);
        AssignProcedure proc = this.am.createAssignProcedure(hri);
        this.master.getMasterProcedureExecutor().submitProcedure((Procedure)proc);
        return proc;
    }

    private UnassignProcedure createAndSubmitUnassign(TableName tableName, int regionId) {
        RegionInfo hri = this.createRegionInfo(tableName, regionId);
        UnassignProcedure proc = this.am.createUnassignProcedure(hri, null, false);
        this.master.getMasterProcedureExecutor().submitProcedure((Procedure)proc);
        return proc;
    }

    private RegionInfo createRegionInfo(TableName tableName, long regionId) {
        return RegionInfoBuilder.newBuilder((TableName)tableName).setStartKey(Bytes.toBytes((long)regionId)).setEndKey(Bytes.toBytes((long)(regionId + 1L))).setSplit(false).setRegionId(0L).build();
    }

    private void sendTransitionReport(ServerName serverName, HBaseProtos.RegionInfo regionInfo, RegionServerStatusProtos.RegionStateTransition.TransitionCode state) throws IOException {
        RegionServerStatusProtos.ReportRegionStateTransitionRequest.Builder req = RegionServerStatusProtos.ReportRegionStateTransitionRequest.newBuilder();
        req.setServer(ProtobufUtil.toServerName((ServerName)serverName));
        req.addTransition(RegionServerStatusProtos.RegionStateTransition.newBuilder().addRegionInfo(regionInfo).setTransitionCode(state).setOpenSeqNum(1L).build());
        this.am.reportRegionStateTransition(req.build());
    }

    private void doCrash(ServerName serverName) {
        this.am.submitServerCrash(serverName, false);
    }

    private void doRestart(ServerName serverName) {
        try {
            this.master.restartRegionServer(serverName);
        }
        catch (IOException e) {
            LOG.warn("Can not restart RS with new startcode");
        }
    }

    private void collectAssignmentManagerMetrics() {
        this.assignSubmittedCount = this.assignProcMetrics.getSubmittedCounter().getCount();
        this.assignFailedCount = this.assignProcMetrics.getFailedCounter().getCount();
        this.unassignSubmittedCount = this.unassignProcMetrics.getSubmittedCounter().getCount();
        this.unassignFailedCount = this.unassignProcMetrics.getFailedCounter().getCount();
    }

    private class MockRSProcedureDispatcher
    extends RSProcedureDispatcher {
        private MockRSExecutor mockRsExec;

        public MockRSProcedureDispatcher(MasterServices master) {
            super(master);
        }

        public void setMockRsExecutor(MockRSExecutor mockRsExec) {
            this.mockRsExec = mockRsExec;
        }

        protected void remoteDispatch(ServerName serverName, Set<RemoteProcedureDispatcher.RemoteProcedure> remoteProcedures) {
            this.submitTask((Callable)((Object)new MockRemoteCall(serverName, remoteProcedures)));
        }

        private class MockRemoteCall
        extends RSProcedureDispatcher.ExecuteProceduresRemoteCall {
            public MockRemoteCall(ServerName serverName, Set<RemoteProcedureDispatcher.RemoteProcedure> operations) {
                super((RSProcedureDispatcher)MockRSProcedureDispatcher.this, serverName, operations);
            }

            protected AdminProtos.ExecuteProceduresResponse sendRequest(ServerName serverName, AdminProtos.ExecuteProceduresRequest request) throws IOException {
                return MockRSProcedureDispatcher.this.mockRsExec.sendRequest(serverName, request);
            }
        }
    }

    private static interface MockRSExecutor {
        public AdminProtos.ExecuteProceduresResponse sendRequest(ServerName var1, AdminProtos.ExecuteProceduresRequest var2) throws IOException;
    }

    private class RandRsExecutor
    extends NoopRsExecutor {
        private final Random rand;

        private RandRsExecutor() {
            this.rand = new Random();
        }

        @Override
        public AdminProtos.ExecuteProceduresResponse sendRequest(ServerName server, AdminProtos.ExecuteProceduresRequest req) throws IOException {
            switch (this.rand.nextInt(5)) {
                case 0: {
                    throw new ServerNotRunningYetException("wait on server startup");
                }
                case 1: {
                    throw new SocketTimeoutException("simulate socket timeout");
                }
                case 2: {
                    throw new RemoteException("java.io.IOException", "unexpected exception");
                }
            }
            return super.sendRequest(server, req);
        }

        @Override
        protected AdminProtos.OpenRegionResponse.RegionOpeningState execOpenRegion(final ServerName server, AdminProtos.OpenRegionRequest.RegionOpenInfo openReq) throws IOException {
            switch (this.rand.nextInt(6)) {
                case 0: {
                    LOG.info("Return OPENED response");
                    TestAssignmentManager.this.sendTransitionReport(server, openReq.getRegion(), RegionServerStatusProtos.RegionStateTransition.TransitionCode.OPENED);
                    return AdminProtos.OpenRegionResponse.RegionOpeningState.OPENED;
                }
                case 1: {
                    LOG.info("Return transition report that OPENED/ALREADY_OPENED response");
                    TestAssignmentManager.this.sendTransitionReport(server, openReq.getRegion(), RegionServerStatusProtos.RegionStateTransition.TransitionCode.OPENED);
                    return AdminProtos.OpenRegionResponse.RegionOpeningState.ALREADY_OPENED;
                }
                case 2: {
                    LOG.info("Return transition report that FAILED_OPEN/FAILED_OPENING response");
                    TestAssignmentManager.this.sendTransitionReport(server, openReq.getRegion(), RegionServerStatusProtos.RegionStateTransition.TransitionCode.FAILED_OPEN);
                    return AdminProtos.OpenRegionResponse.RegionOpeningState.FAILED_OPENING;
                }
            }
            LOG.info("Return null as response; means proc stuck so we send in a crash report after a few seconds...");
            TestAssignmentManager.this.executor.schedule(new Runnable(){

                @Override
                public void run() {
                    LOG.info("Delayed CRASHING of " + server);
                    TestAssignmentManager.this.doCrash(server);
                }
            }, 5L, TimeUnit.SECONDS);
            return null;
        }

        @Override
        protected AdminProtos.CloseRegionResponse execCloseRegion(ServerName server, byte[] regionName) throws IOException {
            AdminProtos.CloseRegionResponse.Builder resp = AdminProtos.CloseRegionResponse.newBuilder();
            boolean closed = this.rand.nextBoolean();
            if (closed) {
                RegionInfo hri = TestAssignmentManager.this.am.getRegionInfo(regionName);
                TestAssignmentManager.this.sendTransitionReport(server, ProtobufUtil.toRegionInfo((RegionInfo)hri), RegionServerStatusProtos.RegionStateTransition.TransitionCode.CLOSED);
            }
            resp.setClosed(closed);
            return resp.build();
        }
    }

    private class HangOnCloseThenRSCrashExecutor
    extends GoodRsExecutor {
        public static final int TYPES_OF_FAILURE = 6;
        private int invocations;

        private HangOnCloseThenRSCrashExecutor() {
        }

        @Override
        protected AdminProtos.CloseRegionResponse execCloseRegion(final ServerName server, byte[] regionName) throws IOException {
            switch (this.invocations++) {
                case 0: {
                    throw new NotServingRegionException("Fake");
                }
                case 1: {
                    throw new RegionServerAbortedException("Fake!");
                }
                case 2: {
                    throw new RegionServerStoppedException("Fake!");
                }
                case 3: {
                    throw new ServerNotRunningYetException("Fake!");
                }
                case 4: {
                    LOG.info("Return null response from serverName=" + server + "; means STUCK...TODO timeout");
                    TestAssignmentManager.this.executor.schedule(new Runnable(){

                        @Override
                        public void run() {
                            LOG.info("Sending in CRASH of " + server);
                            TestAssignmentManager.this.doCrash(server);
                        }
                    }, 1L, TimeUnit.SECONDS);
                    return null;
                }
            }
            return super.execCloseRegion(server, regionName);
        }
    }

    private class HangThenRSRestartExecutor
    extends GoodRsExecutor {
        private int invocations;

        private HangThenRSRestartExecutor() {
        }

        @Override
        protected AdminProtos.OpenRegionResponse.RegionOpeningState execOpenRegion(final ServerName server, AdminProtos.OpenRegionRequest.RegionOpenInfo openReq) throws IOException {
            if (this.invocations++ > 0) {
                return super.execOpenRegion(server, openReq);
            }
            LOG.info("Return null response from serverName=" + server + "; means STUCK...TODO timeout");
            TestAssignmentManager.this.executor.schedule(new Runnable(){

                @Override
                public void run() {
                    LOG.info("Restarting RS of " + server);
                    TestAssignmentManager.this.doRestart(server);
                }
            }, 1L, TimeUnit.SECONDS);
            return null;
        }
    }

    private class HangThenRSCrashExecutor
    extends GoodRsExecutor {
        private int invocations;

        private HangThenRSCrashExecutor() {
        }

        @Override
        protected AdminProtos.OpenRegionResponse.RegionOpeningState execOpenRegion(final ServerName server, AdminProtos.OpenRegionRequest.RegionOpenInfo openReq) throws IOException {
            if (this.invocations++ > 0) {
                return super.execOpenRegion(server, openReq);
            }
            LOG.info("Return null response from serverName=" + server + "; means STUCK...TODO timeout");
            TestAssignmentManager.this.executor.schedule(new Runnable(){

                @Override
                public void run() {
                    LOG.info("Sending in CRASH of " + server);
                    TestAssignmentManager.this.doCrash(server);
                }
            }, 1L, TimeUnit.SECONDS);
            return null;
        }
    }

    private class SocketTimeoutRsExecutor
    extends GoodRsExecutor {
        private final int maxSocketTimeoutRetries;
        private final int maxServerRetries;
        private ServerName lastServer;
        private int sockTimeoutRetries;
        private int serverRetries;

        public SocketTimeoutRsExecutor(int maxSocketTimeoutRetries, int maxServerRetries) {
            this.maxServerRetries = maxServerRetries;
            this.maxSocketTimeoutRetries = maxSocketTimeoutRetries;
        }

        @Override
        public AdminProtos.ExecuteProceduresResponse sendRequest(ServerName server, AdminProtos.ExecuteProceduresRequest req) throws IOException {
            if (this.sockTimeoutRetries++ < this.maxSocketTimeoutRetries) {
                if (this.sockTimeoutRetries == 1) {
                    Assert.assertNotEquals((Object)this.lastServer, (Object)server);
                }
                this.lastServer = server;
                LOG.debug("Socket timeout for server=" + server + " retries=" + this.sockTimeoutRetries);
                throw new SocketTimeoutException("simulate socket timeout");
            }
            if (this.serverRetries++ < this.maxServerRetries) {
                LOG.info("Mark server=" + server + " as dead. serverRetries=" + this.serverRetries);
                TestAssignmentManager.this.master.getServerManager().moveFromOnlineToDeadServers(server);
                this.sockTimeoutRetries = 0;
                throw new SocketTimeoutException("simulate socket timeout");
            }
            return super.sendRequest(server, req);
        }
    }

    private static class FaultyRsExecutor
    implements MockRSExecutor {
        private final IOException exception;

        public FaultyRsExecutor(IOException exception) {
            this.exception = exception;
        }

        @Override
        public AdminProtos.ExecuteProceduresResponse sendRequest(ServerName server, AdminProtos.ExecuteProceduresRequest req) throws IOException {
            throw this.exception;
        }
    }

    private static class ServerNotYetRunningRsExecutor
    implements MockRSExecutor {
        private ServerNotYetRunningRsExecutor() {
        }

        @Override
        public AdminProtos.ExecuteProceduresResponse sendRequest(ServerName server, AdminProtos.ExecuteProceduresRequest req) throws IOException {
            throw new ServerNotRunningYetException("wait on server startup");
        }
    }

    private class GoodRsExecutor
    extends NoopRsExecutor {
        private GoodRsExecutor() {
        }

        @Override
        protected AdminProtos.OpenRegionResponse.RegionOpeningState execOpenRegion(ServerName server, AdminProtos.OpenRegionRequest.RegionOpenInfo openReq) throws IOException {
            RegionInfo hri;
            TestAssignmentManager.this.sendTransitionReport(server, openReq.getRegion(), RegionServerStatusProtos.RegionStateTransition.TransitionCode.OPENED);
            ConcurrentSkipListSet<byte[]> regions = (ConcurrentSkipListSet<byte[]>)TestAssignmentManager.this.regionsToRegionServers.get(server);
            if (regions == null) {
                regions = new ConcurrentSkipListSet<byte[]>(Bytes.BYTES_COMPARATOR);
                TestAssignmentManager.this.regionsToRegionServers.put(server, regions);
            }
            if (regions.contains((hri = ProtobufUtil.toRegionInfo((HBaseProtos.RegionInfo)openReq.getRegion())).getRegionName())) {
                throw new UnsupportedOperationException(hri.getRegionNameAsString());
            }
            regions.add(hri.getRegionName());
            return AdminProtos.OpenRegionResponse.RegionOpeningState.OPENED;
        }

        @Override
        protected AdminProtos.CloseRegionResponse execCloseRegion(ServerName server, byte[] regionName) throws IOException {
            RegionInfo hri = TestAssignmentManager.this.am.getRegionInfo(regionName);
            TestAssignmentManager.this.sendTransitionReport(server, ProtobufUtil.toRegionInfo((RegionInfo)hri), RegionServerStatusProtos.RegionStateTransition.TransitionCode.CLOSED);
            return AdminProtos.CloseRegionResponse.newBuilder().setClosed(true).build();
        }
    }

    private class NoopRsExecutor
    implements MockRSExecutor {
        private NoopRsExecutor() {
        }

        @Override
        public AdminProtos.ExecuteProceduresResponse sendRequest(ServerName server, AdminProtos.ExecuteProceduresRequest request) throws IOException {
            AdminProtos.OpenRegionResponse.Builder resp;
            AdminProtos.ExecuteProceduresResponse.Builder builder = AdminProtos.ExecuteProceduresResponse.newBuilder();
            if (request.getOpenRegionCount() > 0) {
                for (AdminProtos.OpenRegionRequest req : request.getOpenRegionList()) {
                    resp = AdminProtos.OpenRegionResponse.newBuilder();
                    for (AdminProtos.OpenRegionRequest.RegionOpenInfo openReq : req.getOpenInfoList()) {
                        AdminProtos.OpenRegionResponse.RegionOpeningState state = this.execOpenRegion(server, openReq);
                        if (state == null) continue;
                        resp.addOpeningState(state);
                    }
                    builder.addOpenRegion(resp.build());
                }
            }
            if (request.getCloseRegionCount() > 0) {
                for (AdminProtos.OpenRegionRequest req : request.getCloseRegionList()) {
                    resp = this.execCloseRegion(server, req.getRegion().getValue().toByteArray());
                    if (resp == null) continue;
                    builder.addCloseRegion((AdminProtos.CloseRegionResponse)resp);
                }
            }
            return AdminProtos.ExecuteProceduresResponse.newBuilder().build();
        }

        protected AdminProtos.OpenRegionResponse.RegionOpeningState execOpenRegion(ServerName server, AdminProtos.OpenRegionRequest.RegionOpenInfo regionInfo) throws IOException {
            return null;
        }

        protected AdminProtos.CloseRegionResponse execCloseRegion(ServerName server, byte[] regionName) throws IOException {
            return null;
        }
    }
}

