/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.cleaner;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.RecursiveTask;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.ScheduledChore;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.conf.ConfigurationObserver;
import org.apache.hadoop.hbase.master.cleaner.FileCleanerDelegate;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hbase.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hbase.thirdparty.com.google.common.base.Predicate;
import org.apache.hbase.thirdparty.com.google.common.collect.ImmutableSet;
import org.apache.hbase.thirdparty.com.google.common.collect.Iterables;
import org.apache.hbase.thirdparty.com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class CleanerChore<T extends FileCleanerDelegate>
extends ScheduledChore
implements ConfigurationObserver {
    private static final Logger LOG = LoggerFactory.getLogger(CleanerChore.class);
    private static final int AVAIL_PROCESSORS = Runtime.getRuntime().availableProcessors();
    public static final String CHORE_POOL_SIZE = "hbase.cleaner.scan.dir.concurrent.size";
    private static final String DEFAULT_CHORE_POOL_SIZE = "0.5";
    private static volatile ForkJoinPool chorePool;
    private static volatile int chorePoolSize;
    protected final FileSystem fs;
    private final Path oldFileDir;
    private final Configuration conf;
    protected final Map<String, Object> params;
    private final AtomicBoolean enabled = new AtomicBoolean(true);
    private final AtomicBoolean reconfig = new AtomicBoolean(false);
    protected List<T> cleanersChain;

    public CleanerChore(String name, int sleepPeriod, Stoppable s, Configuration conf, FileSystem fs, Path oldFileDir, String confKey) {
        this(name, sleepPeriod, s, conf, fs, oldFileDir, confKey, null);
    }

    public CleanerChore(String name, int sleepPeriod, Stoppable s, Configuration conf, FileSystem fs, Path oldFileDir, String confKey, Map<String, Object> params) {
        super(name, s, sleepPeriod);
        this.fs = fs;
        this.oldFileDir = oldFileDir;
        this.conf = conf;
        this.params = params;
        this.initCleanerChain(confKey);
        if (chorePool == null) {
            String poolSize = conf.get(CHORE_POOL_SIZE, DEFAULT_CHORE_POOL_SIZE);
            chorePoolSize = this.calculatePoolSize(poolSize);
            chorePoolSize = chorePoolSize == 0 ? this.calculatePoolSize(DEFAULT_CHORE_POOL_SIZE) : chorePoolSize;
            chorePool = new ForkJoinPool(chorePoolSize);
            LOG.info("Cleaner pool size is {}", (Object)chorePoolSize);
        }
    }

    int calculatePoolSize(String poolSize) {
        if (poolSize.matches("[1-9][0-9]*")) {
            int size = Math.min(Integer.valueOf(poolSize), AVAIL_PROCESSORS);
            if (size == AVAIL_PROCESSORS) {
                LOG.warn("Use full core processors to scan dir, size={}", (Object)size);
            }
            return size;
        }
        if (poolSize.matches("0.[0-9]+|1.0")) {
            int computedThreads = (int)((double)AVAIL_PROCESSORS * Double.valueOf(poolSize));
            if (computedThreads < 1) {
                LOG.debug("Computed {} threads for CleanerChore, using 1 instead", (Object)computedThreads);
                return 1;
            }
            return computedThreads;
        }
        LOG.error("Unrecognized value: " + poolSize + " for " + CHORE_POOL_SIZE + ", use default config: " + DEFAULT_CHORE_POOL_SIZE + " instead.");
        return this.calculatePoolSize(DEFAULT_CHORE_POOL_SIZE);
    }

    protected abstract boolean validate(Path var1);

    private void initCleanerChain(String confKey) {
        this.cleanersChain = new LinkedList<T>();
        String[] logCleaners = this.conf.getStrings(confKey);
        if (logCleaners != null) {
            for (String className : logCleaners) {
                T logCleaner = this.newFileCleaner(className, this.conf);
                if (logCleaner == null) continue;
                LOG.debug("Initialize cleaner={}", (Object)className);
                this.cleanersChain.add(logCleaner);
            }
        }
    }

    @Override
    public void onConfigurationChange(Configuration conf) {
        int updatedSize = this.calculatePoolSize(conf.get(CHORE_POOL_SIZE, DEFAULT_CHORE_POOL_SIZE));
        if (updatedSize == chorePoolSize) {
            LOG.trace("Size from configuration is same as previous={}, no need to update.", (Object)updatedSize);
            return;
        }
        chorePoolSize = updatedSize;
        if (chorePool.getPoolSize() == 0) {
            this.updateChorePoolSize(updatedSize);
            return;
        }
        this.reconfig.set(true);
    }

    private void updateChorePoolSize(int updatedSize) {
        chorePool.shutdownNow();
        LOG.info("Update chore's pool size from {} to {}", (Object)chorePool.getParallelism(), (Object)updatedSize);
        chorePool = new ForkJoinPool(updatedSize);
    }

    private T newFileCleaner(String className, Configuration conf) {
        try {
            Class<FileCleanerDelegate> c = Class.forName(className).asSubclass(FileCleanerDelegate.class);
            FileCleanerDelegate cleaner = c.newInstance();
            cleaner.setConf(conf);
            cleaner.init(this.params);
            return (T)cleaner;
        }
        catch (Exception e) {
            LOG.warn("Can NOT create CleanerDelegate={}", (Object)className, (Object)e);
            return null;
        }
    }

    protected void chore() {
        if (this.getEnabled()) {
            if (this.runCleaner().booleanValue()) {
                LOG.debug("Cleaned old files/dirs under {} successfully", (Object)this.oldFileDir);
            } else {
                LOG.warn("Failed to fully clean old files/dirs under {}", (Object)this.oldFileDir);
            }
            if (this.reconfig.compareAndSet(true, false)) {
                this.updateChorePoolSize(chorePoolSize);
            }
        } else {
            LOG.debug("Cleaner chore disabled! Not cleaning.");
        }
    }

    private void preRunCleaner() {
        this.cleanersChain.forEach(FileCleanerDelegate::preClean);
    }

    public Boolean runCleaner() {
        this.preRunCleaner();
        CleanerTask task = new CleanerTask(this.oldFileDir, true);
        chorePool.submit(task);
        return (Boolean)task.join();
    }

    private void sortByConsumedSpace(List<FileStatus> dirs) {
        if (dirs == null || dirs.size() < 2) {
            return;
        }
        dirs.sort(new Comparator<FileStatus>(){
            HashMap<FileStatus, Long> directorySpaces = new HashMap();

            @Override
            public int compare(FileStatus f1, FileStatus f2) {
                long f1ConsumedSpace = this.getSpace(f1);
                long f2ConsumedSpace = this.getSpace(f2);
                return Long.compare(f2ConsumedSpace, f1ConsumedSpace);
            }

            private long getSpace(FileStatus f) {
                Long cached = this.directorySpaces.get(f);
                if (cached != null) {
                    return cached;
                }
                try {
                    long space = f.isDirectory() ? CleanerChore.this.fs.getContentSummary(f.getPath()).getSpaceConsumed() : f.getLen();
                    this.directorySpaces.put(f, space);
                    return space;
                }
                catch (IOException e) {
                    LOG.trace("Failed to get space consumed by path={}", (Object)f, (Object)e);
                    return -1L;
                }
            }
        });
    }

    private boolean checkAndDeleteFiles(List<FileStatus> files) {
        if (files == null) {
            return true;
        }
        ArrayList validFiles = Lists.newArrayListWithCapacity((int)files.size());
        ArrayList invalidFiles = Lists.newArrayList();
        for (FileStatus fileStatus : files) {
            if (this.validate(fileStatus.getPath())) {
                validFiles.add(fileStatus);
                continue;
            }
            LOG.warn("Found a wrongly formatted file: " + fileStatus.getPath() + " - will delete it.");
            invalidFiles.add(fileStatus);
        }
        Iterable<Object> deletableValidFiles = validFiles;
        for (FileCleanerDelegate cleaner : this.cleanersChain) {
            if (cleaner.isStopped() || this.getStopper().isStopped()) {
                LOG.warn("A file cleaner" + this.getName() + " is stopped, won't delete any more files in:" + this.oldFileDir);
                return false;
            }
            Iterable<FileStatus> filteredFiles = cleaner.getDeletableFiles((Iterable<FileStatus>)deletableValidFiles);
            if (LOG.isTraceEnabled()) {
                ImmutableSet filteredFileSet = ImmutableSet.copyOf(filteredFiles);
                for (FileStatus fileStatus : deletableValidFiles) {
                    if (filteredFileSet.contains((Object)fileStatus)) continue;
                    LOG.trace(fileStatus.getPath() + " is not deletable according to:" + cleaner);
                }
            }
            deletableValidFiles = filteredFiles;
        }
        Iterable iterable = Iterables.concat((Iterable)invalidFiles, (Iterable)deletableValidFiles);
        return this.deleteFiles(iterable) == files.size();
    }

    protected int deleteFiles(Iterable<FileStatus> filesToDelete) {
        int deletedFileCount = 0;
        for (FileStatus file : filesToDelete) {
            Path filePath = file.getPath();
            LOG.trace("Removing {} from archive", (Object)filePath);
            try {
                boolean success = this.fs.delete(filePath, false);
                if (success) {
                    ++deletedFileCount;
                    continue;
                }
                LOG.warn("Attempted to delete:" + filePath + ", but couldn't. Run cleaner chain and attempt to delete on next pass.");
            }
            catch (IOException e) {
                e = e instanceof RemoteException ? ((RemoteException)((Object)e)).unwrapRemoteException() : e;
                LOG.warn("Error while deleting: " + filePath, (Throwable)e);
            }
        }
        return deletedFileCount;
    }

    public void cleanup() {
        for (FileCleanerDelegate lc : this.cleanersChain) {
            try {
                lc.stop("Exiting");
            }
            catch (Throwable t) {
                LOG.warn("Stopping", t);
            }
        }
    }

    @VisibleForTesting
    int getChorePoolSize() {
        return chorePoolSize;
    }

    public boolean setEnabled(boolean enabled) {
        return this.enabled.getAndSet(enabled);
    }

    public boolean getEnabled() {
        return this.enabled.get();
    }

    private class CleanerTask
    extends RecursiveTask<Boolean> {
        private final Path dir;
        private final boolean root;

        CleanerTask(FileStatus dir, boolean root) {
            this(dir.getPath(), root);
        }

        CleanerTask(Path dir, boolean root) {
            this.dir = dir;
            this.root = root;
        }

        @Override
        protected Boolean compute() {
            boolean nullSubDirs;
            List<FileStatus> files;
            List<FileStatus> subDirs;
            if (LOG.isDebugEnabled()) {
                LOG.debug("CleanerTask " + Thread.currentThread().getId() + " starts cleaning dirs and files under " + this.dir + " and itself.");
            }
            try {
                subDirs = this.getFilteredStatus((Predicate<FileStatus>)((Predicate)status -> status.isDirectory()));
                files = this.getFilteredStatus((Predicate<FileStatus>)((Predicate)status -> status.isFile()));
            }
            catch (IOException ioe) {
                LOG.warn(this.dir + " doesn't exist, just skip it. ", (Throwable)ioe);
                return true;
            }
            boolean bl = nullSubDirs = subDirs == null;
            if (nullSubDirs) {
                LOG.trace("There is no subdir under {}", (Object)this.dir);
            }
            if (files == null) {
                LOG.trace("There is no file under {}", (Object)this.dir);
            }
            int capacity = nullSubDirs ? 0 : subDirs.size();
            ArrayList tasks = Lists.newArrayListWithCapacity((int)capacity);
            if (!nullSubDirs) {
                CleanerChore.this.sortByConsumedSpace(subDirs);
                for (FileStatus subdir : subDirs) {
                    CleanerTask task = new CleanerTask(subdir, false);
                    tasks.add(task);
                    task.fork();
                }
            }
            boolean result = true;
            result &= this.deleteAction(() -> CleanerChore.this.checkAndDeleteFiles(files), "files");
            if ((result &= this.deleteAction(() -> this.getCleanResult(tasks), "subdirs")) && !this.root) {
                result &= this.deleteAction(() -> CleanerChore.this.fs.delete(this.dir, false), "dir");
            }
            return result;
        }

        private List<FileStatus> getFilteredStatus(Predicate<FileStatus> function) throws IOException {
            return FSUtils.listStatusWithStatusFilter(CleanerChore.this.fs, this.dir, status -> function.test((Object)status));
        }

        private boolean deleteAction(Action<Boolean> deletion, String type) {
            boolean deleted;
            String errorMsg = null;
            try {
                LOG.trace("Start deleting {} under {}", (Object)type, (Object)this.dir);
                deleted = deletion.act();
            }
            catch (IOException ioe) {
                errorMsg = ioe.getMessage();
                LOG.warn("Could not delete {} under {}; {}", new Object[]{type, this.dir, errorMsg});
                deleted = false;
            }
            LOG.trace("Finish deleting {} under {}, deleted=", new Object[]{type, this.dir, deleted});
            return deleted;
        }

        private boolean getCleanResult(List<CleanerTask> tasks) throws IOException {
            boolean cleaned = true;
            try {
                for (CleanerTask task : tasks) {
                    cleaned &= ((Boolean)task.get()).booleanValue();
                }
            }
            catch (InterruptedException | ExecutionException e) {
                throw new IOException(e);
            }
            return cleaned;
        }
    }

    private static interface Action<T> {
        public T act() throws IOException;
    }
}

