/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Predicate;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.ClusterMetrics;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.ConnectionFactory;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.filter.Filter;
import org.apache.hadoop.hbase.filter.FirstKeyOnlyFilter;
import org.apache.hadoop.hbase.util.AbstractHBaseTool;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hbase.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hbase.thirdparty.org.apache.commons.cli.CommandLine;
import org.apache.hbase.thirdparty.org.apache.commons.collections4.CollectionUtils;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Public
public class RegionMover
extends AbstractHBaseTool
implements Closeable {
    public static final String MOVE_RETRIES_MAX_KEY = "hbase.move.retries.max";
    public static final String MOVE_WAIT_MAX_KEY = "hbase.move.wait.max";
    public static final String SERVERSTART_WAIT_MAX_KEY = "hbase.serverstart.wait.max";
    public static final int DEFAULT_MOVE_RETRIES_MAX = 5;
    public static final int DEFAULT_MOVE_WAIT_MAX = 60;
    public static final int DEFAULT_SERVERSTART_WAIT_MAX = 180;
    private static final Logger LOG = LoggerFactory.getLogger(RegionMover.class);
    private RegionMoverBuilder rmbuilder;
    private boolean ack = true;
    private int maxthreads = 1;
    private int timeout;
    private String loadUnload;
    private String hostname;
    private String filename;
    private String excludeFile;
    private int port;
    private Connection conn;
    private Admin admin;

    private RegionMover(RegionMoverBuilder builder) throws IOException {
        this.hostname = builder.hostname;
        this.filename = builder.filename;
        this.excludeFile = builder.excludeFile;
        this.maxthreads = builder.maxthreads;
        this.ack = builder.ack;
        this.port = builder.port;
        this.timeout = builder.timeout;
        this.setConf(builder.conf);
        this.conn = ConnectionFactory.createConnection((Configuration)this.conf);
        this.admin = this.conn.getAdmin();
    }

    private RegionMover() {
    }

    @Override
    public void close() {
        IOUtils.closeQuietly((Closeable)this.admin);
        IOUtils.closeQuietly((Closeable)this.conn);
    }

    public boolean load() throws ExecutionException, InterruptedException, TimeoutException {
        ExecutorService loadPool = Executors.newFixedThreadPool(1);
        Future<Boolean> loadTask = loadPool.submit(() -> {
            try {
                List<RegionInfo> regionsToMove = this.readRegionsFromFile(this.filename);
                if (regionsToMove.isEmpty()) {
                    LOG.info("No regions to load.Exiting");
                    return true;
                }
                this.loadRegions(regionsToMove);
            }
            catch (Exception e) {
                LOG.error("Error while loading regions to " + this.hostname, (Throwable)e);
                return false;
            }
            return true;
        });
        return this.waitTaskToFinish(loadPool, loadTask, "loading");
    }

    private void loadRegions(List<RegionInfo> regionsToMove) throws Exception {
        ServerName server = this.getTargetServer();
        List<RegionInfo> movedRegions = Collections.synchronizedList(new ArrayList());
        LOG.info("Moving " + regionsToMove.size() + " regions to " + server + " using " + this.maxthreads + " threads.Ack mode:" + this.ack);
        ExecutorService moveRegionsPool = Executors.newFixedThreadPool(this.maxthreads);
        ArrayList<Future<Boolean>> taskList = new ArrayList<Future<Boolean>>();
        int counter = 0;
        while (counter < regionsToMove.size()) {
            Future<Boolean> task;
            RegionInfo region = regionsToMove.get(counter);
            ServerName currentServer = this.getServerNameForRegion(region);
            if (currentServer == null) {
                LOG.warn("Could not get server for Region:" + region.getRegionNameAsString() + " moving on");
                ++counter;
                continue;
            }
            if (server.equals((Object)currentServer)) {
                LOG.info("Region " + region.getRegionNameAsString() + " is already on target server=" + server);
                ++counter;
                continue;
            }
            if (this.ack) {
                task = moveRegionsPool.submit(new MoveWithAck(region, currentServer, server, movedRegions));
                taskList.add(task);
            } else {
                task = moveRegionsPool.submit(new MoveWithoutAck(region, currentServer, server, movedRegions));
                taskList.add(task);
            }
            ++counter;
        }
        moveRegionsPool.shutdown();
        long timeoutInSeconds = (long)regionsToMove.size() * this.admin.getConfiguration().getLong(MOVE_WAIT_MAX_KEY, 60L);
        this.waitMoveTasksToFinish(moveRegionsPool, taskList, timeoutInSeconds);
    }

    public boolean unload() throws InterruptedException, ExecutionException, TimeoutException {
        this.deleteFile(this.filename);
        ExecutorService unloadPool = Executors.newFixedThreadPool(1);
        Future<Boolean> unloadTask = unloadPool.submit(() -> {
            List<RegionInfo> movedRegions = Collections.synchronizedList(new ArrayList());
            try {
                ArrayList<ServerName> regionServers = new ArrayList<ServerName>();
                regionServers.addAll(this.admin.getRegionServers());
                ServerName server = this.stripServer(regionServers, this.hostname, this.port);
                if (server == null) {
                    LOG.info("Could not find server '{}:{}' in the set of region servers. giving up.", (Object)this.hostname, (Object)this.port);
                    LOG.debug("List of region servers: {}", regionServers);
                    Boolean bl = false;
                    return bl;
                }
                this.stripExcludes(regionServers);
                HashSet decommissionedRS = new HashSet(this.admin.listDecommissionedRegionServers());
                if (CollectionUtils.isNotEmpty(decommissionedRS)) {
                    regionServers.removeIf(decommissionedRS::contains);
                    LOG.debug("Excluded RegionServers from unloading regions to because they are marked as decommissioned. Servers: {}", decommissionedRS);
                }
                this.stripMaster(regionServers);
                if (regionServers.isEmpty()) {
                    LOG.warn("No Regions were moved - no servers available");
                    Boolean bl = false;
                    return bl;
                }
                this.unloadRegions(server, regionServers, movedRegions);
            }
            catch (Exception e) {
                LOG.error("Error while unloading regions ", (Throwable)e);
                Boolean bl = false;
                return bl;
            }
            finally {
                if (movedRegions != null) {
                    this.writeFile(this.filename, movedRegions);
                }
            }
            return true;
        });
        return this.waitTaskToFinish(unloadPool, unloadTask, "unloading");
    }

    private void unloadRegions(ServerName server, List<ServerName> regionServers, List<RegionInfo> movedRegions) throws Exception {
        while (true) {
            List regionsToMove = this.admin.getRegions(server);
            regionsToMove.removeAll(movedRegions);
            if (regionsToMove.isEmpty()) break;
            LOG.info("Moving " + regionsToMove.size() + " regions from " + this.hostname + " to " + regionServers.size() + " servers using " + this.maxthreads + " threads .Ack Mode:" + this.ack);
            ExecutorService moveRegionsPool = Executors.newFixedThreadPool(this.maxthreads);
            ArrayList<Future<Boolean>> taskList = new ArrayList<Future<Boolean>>();
            int serverIndex = 0;
            for (int counter = 0; counter < regionsToMove.size(); ++counter) {
                Future<Boolean> task;
                if (this.ack) {
                    task = moveRegionsPool.submit(new MoveWithAck((RegionInfo)regionsToMove.get(counter), server, regionServers.get(serverIndex), movedRegions));
                    taskList.add(task);
                } else {
                    task = moveRegionsPool.submit(new MoveWithoutAck((RegionInfo)regionsToMove.get(counter), server, regionServers.get(serverIndex), movedRegions));
                    taskList.add(task);
                }
                serverIndex = (serverIndex + 1) % regionServers.size();
            }
            moveRegionsPool.shutdown();
            long timeoutInSeconds = (long)regionsToMove.size() * this.admin.getConfiguration().getLong(MOVE_WAIT_MAX_KEY, 60L);
            this.waitMoveTasksToFinish(moveRegionsPool, taskList, timeoutInSeconds);
        }
        LOG.info("No Regions to move....Quitting now");
    }

    private boolean waitTaskToFinish(ExecutorService pool, Future<Boolean> task, String operation) throws TimeoutException, InterruptedException, ExecutionException {
        pool.shutdown();
        try {
            if (!pool.awaitTermination(this.timeout, TimeUnit.SECONDS)) {
                LOG.warn("Timed out before finishing the " + operation + " operation. Timeout: " + this.timeout + "sec");
                pool.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            pool.shutdownNow();
            Thread.currentThread().interrupt();
        }
        try {
            return task.get(5L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            LOG.warn("Interrupted while " + operation + " Regions on " + this.hostname, (Throwable)e);
            throw e;
        }
        catch (ExecutionException e) {
            LOG.error("Error while " + operation + " regions on RegionServer " + this.hostname, (Throwable)e);
            throw e;
        }
    }

    private void waitMoveTasksToFinish(ExecutorService moveRegionsPool, List<Future<Boolean>> taskList, long timeoutInSeconds) throws Exception {
        try {
            if (!moveRegionsPool.awaitTermination(timeoutInSeconds, TimeUnit.SECONDS)) {
                moveRegionsPool.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            moveRegionsPool.shutdownNow();
            Thread.currentThread().interrupt();
        }
        for (Future<Boolean> future : taskList) {
            try {
                if (future.get(5L, TimeUnit.SECONDS).booleanValue()) continue;
                LOG.error("Was Not able to move region....Exiting Now");
                throw new Exception("Could not move region Exception");
            }
            catch (InterruptedException e) {
                LOG.error("Interrupted while waiting for Thread to Complete " + e.getMessage(), (Throwable)e);
                throw e;
            }
            catch (ExecutionException e) {
                LOG.error("Got Exception From Thread While moving region " + e.getMessage(), (Throwable)e);
                throw e;
            }
            catch (CancellationException e) {
                LOG.error("Thread for moving region cancelled. Timeout for cancellation:" + timeoutInSeconds + "secs", (Throwable)e);
                throw e;
            }
        }
    }

    private ServerName getTargetServer() throws Exception {
        ServerName server = null;
        int maxWaitInSeconds = this.admin.getConfiguration().getInt(SERVERSTART_WAIT_MAX_KEY, 180);
        long maxWait = EnvironmentEdgeManager.currentTime() + (long)(maxWaitInSeconds * 1000);
        while (EnvironmentEdgeManager.currentTime() < maxWait) {
            try {
                ArrayList<ServerName> regionServers = new ArrayList<ServerName>();
                regionServers.addAll(this.admin.getRegionServers());
                server = this.stripServer(regionServers, this.hostname, this.port);
                if (server != null) break;
                LOG.warn("Server " + this.hostname + ":" + this.port + " is not up yet, waiting");
            }
            catch (IOException e) {
                LOG.warn("Could not get list of region servers", (Throwable)e);
            }
            Thread.sleep(500L);
        }
        if (server == null) {
            LOG.error("Server " + this.hostname + ":" + this.port + " is not up. Giving up.");
            throw new Exception("Server " + this.hostname + ":" + this.port + " to load regions not online");
        }
        return server;
    }

    private List<RegionInfo> readRegionsFromFile(String filename) throws IOException {
        ArrayList<RegionInfo> regions = new ArrayList<RegionInfo>();
        File f = new File(filename);
        if (!f.exists()) {
            return regions;
        }
        try (DataInputStream dis = new DataInputStream(new BufferedInputStream(new FileInputStream(f)));){
            int numRegions = dis.readInt();
            for (int index = 0; index < numRegions; ++index) {
                regions.add(RegionInfo.parseFromOrNull((byte[])Bytes.readByteArray((DataInput)dis)));
            }
        }
        catch (IOException e) {
            LOG.error("Error while reading regions from file:" + filename, (Throwable)e);
            throw e;
        }
        return regions;
    }

    private void writeFile(String filename, List<RegionInfo> movedRegions) throws IOException {
        try (DataOutputStream dos = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(filename)));){
            dos.writeInt(movedRegions.size());
            for (RegionInfo region : movedRegions) {
                Bytes.writeByteArray((DataOutput)dos, (byte[])RegionInfo.toByteArray((RegionInfo)region));
            }
        }
        catch (IOException e) {
            LOG.error("ERROR: Was Not able to write regions moved to output file but moved " + movedRegions.size() + " regions", (Throwable)e);
            throw e;
        }
    }

    private void deleteFile(String filename) {
        File f = new File(filename);
        if (f.exists()) {
            f.delete();
        }
    }

    private List<String> readExcludes(String excludeFile) throws IOException {
        ArrayList<String> excludeServers = new ArrayList<String>();
        if (excludeFile == null) {
            return excludeServers;
        }
        try {
            Files.readAllLines(Paths.get(excludeFile, new String[0])).stream().map(String::trim).filter(((Predicate<String>)String::isEmpty).negate()).map(String::toLowerCase).forEach(excludeServers::add);
        }
        catch (IOException e) {
            LOG.warn("Exception while reading excludes file, continuing anyways", (Throwable)e);
        }
        return excludeServers;
    }

    private void stripExcludes(List<ServerName> regionServers) throws IOException {
        if (this.excludeFile != null) {
            List<String> excludes = this.readExcludes(this.excludeFile);
            Iterator<ServerName> i = regionServers.iterator();
            while (i.hasNext()) {
                String rs = i.next().getServerName();
                String rsPort = rs.split(",")[0].toLowerCase() + ":" + rs.split(",")[1];
                if (!excludes.contains(rsPort)) continue;
                i.remove();
            }
            LOG.info("Valid Region server targets are:" + regionServers.toString());
            LOG.info("Excluded Servers are" + excludes.toString());
        }
    }

    private void stripMaster(List<ServerName> regionServers) throws IOException {
        ServerName master = this.admin.getClusterMetrics(EnumSet.of(ClusterMetrics.Option.MASTER)).getMasterName();
        this.stripServer(regionServers, master.getHostname(), master.getPort());
    }

    private ServerName stripServer(List<ServerName> regionServers, String hostname, int port) {
        Iterator<ServerName> iter = regionServers.iterator();
        while (iter.hasNext()) {
            ServerName server = iter.next();
            if (!server.getAddress().getHostname().equalsIgnoreCase(hostname) || server.getAddress().getPort() != port) continue;
            iter.remove();
            return server;
        }
        return null;
    }

    private void isSuccessfulScan(RegionInfo region) throws IOException {
        Scan scan = new Scan().withStartRow(region.getStartKey()).setRaw(true).setOneRowLimit().setMaxResultSize(1L).setCaching(1).setFilter((Filter)new FirstKeyOnlyFilter()).setCacheBlocks(false);
        try (Table table = this.conn.getTable(region.getTable());
             ResultScanner scanner = table.getScanner(scan);){
            scanner.next();
        }
        catch (IOException e) {
            LOG.error("Could not scan region:" + region.getEncodedName(), (Throwable)e);
            throw e;
        }
    }

    private boolean isSameServer(RegionInfo region, ServerName serverName) throws IOException {
        ServerName serverForRegion = this.getServerNameForRegion(region);
        return serverForRegion != null && serverForRegion.equals((Object)serverName);
    }

    private ServerName getServerNameForRegion(RegionInfo region) throws IOException {
        if (!this.admin.isTableEnabled(region.getTable())) {
            return null;
        }
        HRegionLocation loc = this.conn.getRegionLocator(region.getTable()).getRegionLocation(region.getStartKey(), region.getReplicaId(), true);
        if (loc != null) {
            return loc.getServerName();
        }
        return null;
    }

    protected void addOptions() {
        this.addRequiredOptWithArg("r", "regionserverhost", "region server <hostname>|<hostname:port>");
        this.addRequiredOptWithArg("o", "operation", "Expected: load/unload");
        this.addOptWithArg("m", "maxthreads", "Define the maximum number of threads to use to unload and reload the regions");
        this.addOptWithArg("x", "excludefile", "File with <hostname:port> per line to exclude as unload targets; default excludes only target host; useful for rack decommisioning.");
        this.addOptWithArg("f", "filename", "File to save regions list into unloading, or read from loading; default /tmp/<usernamehostname:port>");
        this.addOptNoArg("n", "noack", "Turn on No-Ack mode(default: false) which won't check if region is online on target RegionServer, hence best effort. This is more performant in unloading and loading but might lead to region being unavailable for some time till master reassigns it in case the move failed");
        this.addOptWithArg("t", "timeout", "timeout in seconds after which the tool will exit irrespective of whether it finished or not;default Integer.MAX_VALUE");
    }

    protected void processOptions(CommandLine cmd) {
        String hostname = cmd.getOptionValue("r");
        this.rmbuilder = new RegionMoverBuilder(hostname);
        if (cmd.hasOption('m')) {
            this.rmbuilder.maxthreads(Integer.parseInt(cmd.getOptionValue('m')));
        }
        if (cmd.hasOption('n')) {
            this.rmbuilder.ack(false);
        }
        if (cmd.hasOption('f')) {
            this.rmbuilder.filename(cmd.getOptionValue('f'));
        }
        if (cmd.hasOption('x')) {
            this.rmbuilder.excludeFile(cmd.getOptionValue('x'));
        }
        if (cmd.hasOption('t')) {
            this.rmbuilder.timeout(Integer.parseInt(cmd.getOptionValue('t')));
        }
        this.loadUnload = cmd.getOptionValue("o").toLowerCase(Locale.ROOT);
    }

    protected int doWork() throws Exception {
        boolean success;
        try (RegionMover rm = this.rmbuilder.build();){
            if (this.loadUnload.equalsIgnoreCase("load")) {
                success = rm.load();
            } else if (this.loadUnload.equalsIgnoreCase("unload")) {
                success = rm.unload();
            } else {
                this.printUsage();
                success = false;
            }
        }
        return success ? 0 : 1;
    }

    public static void main(String[] args) {
        try (RegionMover mover = new RegionMover();){
            mover.doStaticMain(args);
        }
    }

    private class MoveWithoutAck
    implements Callable<Boolean> {
        private RegionInfo region;
        private ServerName targetServer;
        private List<RegionInfo> movedRegions;
        private ServerName sourceServer;

        public MoveWithoutAck(RegionInfo regionInfo, ServerName sourceServer, ServerName targetServer, List<RegionInfo> movedRegions) {
            this.region = regionInfo;
            this.targetServer = targetServer;
            this.movedRegions = movedRegions;
            this.sourceServer = sourceServer;
        }

        @Override
        public Boolean call() {
            try {
                LOG.info("Moving region:" + this.region.getEncodedName() + " from " + this.sourceServer + " to " + this.targetServer);
                RegionMover.this.admin.move(this.region.getEncodedNameAsBytes(), this.targetServer);
                LOG.info("Moved " + this.region.getEncodedName() + " from " + this.sourceServer + " to " + this.targetServer);
            }
            catch (Exception e) {
                LOG.error("Error Moving Region:" + this.region.getEncodedName(), (Throwable)e);
            }
            finally {
                this.movedRegions.add(this.region);
            }
            return true;
        }
    }

    private class MoveWithAck
    implements Callable<Boolean> {
        private RegionInfo region;
        private ServerName targetServer;
        private List<RegionInfo> movedRegions;
        private ServerName sourceServer;

        public MoveWithAck(RegionInfo regionInfo, ServerName sourceServer, ServerName targetServer, List<RegionInfo> movedRegions) {
            this.region = regionInfo;
            this.targetServer = targetServer;
            this.movedRegions = movedRegions;
            this.sourceServer = sourceServer;
        }

        @Override
        public Boolean call() throws IOException, InterruptedException {
            boolean moved = false;
            int count = 0;
            int retries = RegionMover.this.admin.getConfiguration().getInt(RegionMover.MOVE_RETRIES_MAX_KEY, 5);
            int maxWaitInSeconds = RegionMover.this.admin.getConfiguration().getInt(RegionMover.MOVE_WAIT_MAX_KEY, 60);
            long startTime = EnvironmentEdgeManager.currentTime();
            boolean sameServer = true;
            RegionMover.this.isSuccessfulScan(this.region);
            LOG.info("Moving region:" + this.region.getEncodedName() + " from " + this.sourceServer + " to " + this.targetServer);
            while (count < retries && sameServer) {
                if (count > 0) {
                    LOG.info("Retry " + Integer.toString(count) + " of maximum " + Integer.toString(retries));
                }
                ++count;
                RegionMover.this.admin.move(this.region.getEncodedNameAsBytes(), this.targetServer);
                long maxWait = startTime + (long)(maxWaitInSeconds * 1000);
                while (EnvironmentEdgeManager.currentTime() < maxWait && (sameServer = RegionMover.this.isSameServer(this.region, this.sourceServer))) {
                    Thread.sleep(100L);
                }
            }
            if (sameServer) {
                LOG.error("Region: " + this.region.getRegionNameAsString() + " stuck on " + this.sourceServer + ",newServer=" + this.targetServer);
            } else {
                RegionMover.this.isSuccessfulScan(this.region);
                LOG.info("Moved Region " + this.region.getRegionNameAsString() + " cost:" + String.format("%.3f", Float.valueOf((float)(EnvironmentEdgeManager.currentTime() - startTime) / 1000.0f)));
                moved = true;
                this.movedRegions.add(this.region);
            }
            return moved;
        }
    }

    public static class RegionMoverBuilder {
        private boolean ack = true;
        private int maxthreads = 1;
        private int timeout = Integer.MAX_VALUE;
        private String hostname;
        private String filename;
        private String excludeFile = null;
        private String defaultDir = System.getProperty("java.io.tmpdir");
        @VisibleForTesting
        final int port;
        private final Configuration conf;

        public RegionMoverBuilder(String hostname) {
            this(hostname, RegionMoverBuilder.createConf());
        }

        private static Configuration createConf() {
            Configuration conf = HBaseConfiguration.create();
            conf.setInt("hbase.client.prefetch.limit", 1);
            conf.setInt("hbase.client.pause", 500);
            conf.setInt("hbase.client.retries.number", 100);
            return conf;
        }

        public RegionMoverBuilder(String hostname, Configuration conf) {
            String[] splitHostname = hostname.toLowerCase().split(":");
            this.hostname = splitHostname[0];
            this.port = splitHostname.length == 2 ? Integer.parseInt(splitHostname[1]) : conf.getInt("hbase.regionserver.port", 16020);
            this.filename = this.defaultDir + File.separator + System.getProperty("user.name") + this.hostname + ":" + Integer.toString(this.port);
            this.conf = conf;
        }

        public RegionMoverBuilder filename(String filename) {
            this.filename = filename;
            return this;
        }

        public RegionMoverBuilder maxthreads(int threads) {
            this.maxthreads = threads;
            return this;
        }

        public RegionMoverBuilder excludeFile(String excludefile) {
            this.excludeFile = excludefile;
            return this;
        }

        public RegionMoverBuilder ack(boolean ack) {
            this.ack = ack;
            return this;
        }

        public RegionMoverBuilder timeout(int timeout) {
            this.timeout = timeout;
            return this;
        }

        public RegionMover build() throws IOException {
            return new RegionMover(this);
        }
    }
}

