/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import edu.umd.cs.findbugs.annotations.CheckForNull;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.InetSocketAddress;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.Vector;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hbase.ClusterId;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HDFSBlocksDistribution;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.RegionInfoBuilder;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.fs.HFileSystem;
import org.apache.hadoop.hbase.io.HFileLink;
import org.apache.hadoop.hbase.master.HMaster;
import org.apache.hadoop.hbase.regionserver.StoreFileInfo;
import org.apache.hadoop.hbase.shaded.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.shaded.protobuf.generated.FSProtos;
import org.apache.hadoop.hbase.util.AbstractFileStatusFilter;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.CommonFSUtils;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.FSRegionScanner;
import org.apache.hadoop.hbase.util.FileStatusFilter;
import org.apache.hadoop.hbase.util.FileSystemVersionException;
import org.apache.hadoop.hbase.util.HbckErrorReporter;
import org.apache.hadoop.hbase.util.ServerRegionReplicaUtil;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSHedgedReadMetrics;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.StringUtils;
import org.apache.hbase.thirdparty.com.google.common.base.Throwables;
import org.apache.hbase.thirdparty.com.google.common.collect.Iterators;
import org.apache.hbase.thirdparty.com.google.common.collect.Sets;
import org.apache.hbase.thirdparty.com.google.common.primitives.Ints;
import org.apache.hbase.thirdparty.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.hbase.thirdparty.com.google.protobuf.Message;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public final class FSUtils {
    private static final Logger LOG = LoggerFactory.getLogger(FSUtils.class);
    private static final String THREAD_POOLSIZE = "hbase.client.localityCheck.threadPoolSize";
    private static final int DEFAULT_THREAD_POOLSIZE = 2;
    public static final boolean WINDOWS = System.getProperty("os.name").startsWith("Windows");

    private FSUtils() {
    }

    public static boolean isDistributedFileSystem(FileSystem fs) throws IOException {
        FileSystem fileSystem = fs;
        if (fs instanceof HFileSystem) {
            fileSystem = ((HFileSystem)fs).getBackingFs();
        }
        return fileSystem instanceof DistributedFileSystem;
    }

    public static boolean isMatchingTail(Path pathToSearch, Path pathTail) {
        String toSearchName;
        String tailName;
        Path tailPath = pathTail;
        Path toSearch = pathToSearch;
        boolean result = false;
        if (pathToSearch.depth() != pathTail.depth()) {
            return false;
        }
        do {
            if ((tailName = tailPath.getName()) == null || tailName.isEmpty()) {
                result = true;
                break;
            }
            toSearchName = toSearch.getName();
            if (toSearchName == null || toSearchName.isEmpty()) break;
            tailPath = tailPath.getParent();
            toSearch = toSearch.getParent();
        } while (tailName.equals(toSearchName));
        return result;
    }

    public static boolean deleteRegionDir(Configuration conf, RegionInfo hri) throws IOException {
        Path rootDir = CommonFSUtils.getRootDir((Configuration)conf);
        FileSystem fs = rootDir.getFileSystem(conf);
        return CommonFSUtils.deleteDirectory((FileSystem)fs, (Path)new Path(CommonFSUtils.getTableDir((Path)rootDir, (TableName)hri.getTable()), hri.getEncodedName()));
    }

    public static FSDataOutputStream create(Configuration conf, FileSystem fs, Path path, FsPermission perm, InetSocketAddress[] favoredNodes) throws IOException {
        FileSystem backingFs;
        if (fs instanceof HFileSystem && (backingFs = ((HFileSystem)fs).getBackingFs()) instanceof DistributedFileSystem) {
            short replication = Short.parseShort(conf.get("DFS_REPLICATION", String.valueOf(0)));
            try {
                return (FSDataOutputStream)DistributedFileSystem.class.getDeclaredMethod("create", Path.class, FsPermission.class, Boolean.TYPE, Integer.TYPE, Short.TYPE, Long.TYPE, Progressable.class, InetSocketAddress[].class).invoke((Object)backingFs, path, perm, true, CommonFSUtils.getDefaultBufferSize((FileSystem)backingFs), replication > 0 ? replication : CommonFSUtils.getDefaultReplication((FileSystem)backingFs, (Path)path), CommonFSUtils.getDefaultBlockSize((FileSystem)backingFs, (Path)path), null, favoredNodes);
            }
            catch (InvocationTargetException ite) {
                throw new IOException(ite.getCause());
            }
            catch (NoSuchMethodException e) {
                LOG.debug("DFS Client does not support most favored nodes create; using default create");
                LOG.trace("Ignoring; use default create", (Throwable)e);
            }
            catch (IllegalAccessException | IllegalArgumentException | SecurityException e) {
                LOG.debug("Ignoring (most likely Reflection related exception) " + e);
            }
        }
        return CommonFSUtils.create((FileSystem)fs, (Path)path, (FsPermission)perm, (boolean)true);
    }

    public static void checkFileSystemAvailable(FileSystem fs) throws IOException {
        if (!(fs instanceof DistributedFileSystem)) {
            return;
        }
        IOException exception = null;
        DistributedFileSystem dfs = (DistributedFileSystem)fs;
        try {
            if (dfs.exists(new Path("/"))) {
                return;
            }
        }
        catch (IOException e) {
            exception = e instanceof RemoteException ? ((RemoteException)((Object)e)).unwrapRemoteException() : e;
        }
        try {
            fs.close();
        }
        catch (Exception e) {
            LOG.error("file system close failed: ", (Throwable)e);
        }
        throw new IOException("File system is not available", exception);
    }

    private static boolean isInSafeMode(DistributedFileSystem dfs) throws IOException {
        boolean inSafeMode = false;
        try {
            Method m = DistributedFileSystem.class.getMethod("setSafeMode", HdfsConstants.SafeModeAction.class, Boolean.TYPE);
            inSafeMode = (Boolean)m.invoke((Object)dfs, HdfsConstants.SafeModeAction.SAFEMODE_GET, true);
        }
        catch (Exception e) {
            if (e instanceof IOException) {
                throw (IOException)e;
            }
            inSafeMode = dfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_GET);
        }
        return inSafeMode;
    }

    public static void checkDfsSafeMode(Configuration conf) throws IOException {
        boolean isInSafeMode = false;
        FileSystem fs = FileSystem.get((Configuration)conf);
        if (fs instanceof DistributedFileSystem) {
            DistributedFileSystem dfs = (DistributedFileSystem)fs;
            isInSafeMode = FSUtils.isInSafeMode(dfs);
        }
        if (isInSafeMode) {
            throw new IOException("File system is in safemode, it can't be written now");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String getVersion(FileSystem fs, Path rootdir) throws IOException, DeserializationException {
        String version;
        block21: {
            Path versionFile = new Path(rootdir, "hbase.version");
            FileStatus[] status = null;
            try {
                status = fs.listStatus(versionFile);
            }
            catch (FileNotFoundException fnfe) {
                return null;
            }
            if (ArrayUtils.getLength((Object)status) == 0) {
                return null;
            }
            version = null;
            byte[] content = new byte[(int)status[0].getLen()];
            try (FSDataInputStream s = fs.open(versionFile);){
                IOUtils.readFully((InputStream)s, (byte[])content, (int)0, (int)content.length);
                if (ProtobufUtil.isPBMagicPrefix((byte[])content)) {
                    version = FSUtils.parseVersionFrom(content);
                    break block21;
                }
                try (DataInputStream dis = new DataInputStream(new ByteArrayInputStream(content));){
                    version = dis.readUTF();
                }
            }
        }
        return version;
    }

    static String parseVersionFrom(byte[] bytes) throws DeserializationException {
        ProtobufUtil.expectPBMagicPrefix((byte[])bytes);
        int pblen = ProtobufUtil.lengthOfPBMagic();
        FSProtos.HBaseVersionFileContent.Builder builder = FSProtos.HBaseVersionFileContent.newBuilder();
        try {
            ProtobufUtil.mergeFrom((Message.Builder)builder, (byte[])bytes, (int)pblen, (int)(bytes.length - pblen));
            return builder.getVersion();
        }
        catch (IOException e) {
            throw new DeserializationException((Throwable)e);
        }
    }

    static byte[] toVersionByteArray(String version) {
        FSProtos.HBaseVersionFileContent.Builder builder = FSProtos.HBaseVersionFileContent.newBuilder();
        return ProtobufUtil.prependPBMagic((byte[])builder.setVersion(version).build().toByteArray());
    }

    public static void checkVersion(FileSystem fs, Path rootdir, boolean message) throws IOException, DeserializationException {
        FSUtils.checkVersion(fs, rootdir, message, 0, 3);
    }

    public static void checkVersion(FileSystem fs, Path rootdir, boolean message, int wait, int retries) throws IOException, DeserializationException {
        String msg;
        String version = FSUtils.getVersion(fs, rootdir);
        if (version == null) {
            if (!FSUtils.metaRegionExists(fs, rootdir)) {
                FSUtils.setVersion(fs, rootdir, wait, retries);
                return;
            }
            msg = "hbase.version file is missing. Is your hbase.rootdir valid? You can restore hbase.version file by running 'HBCK2 filesystem -fix'. See https://github.com/apache/hbase-operator-tools/tree/master/hbase-hbck2";
        } else {
            if (version.compareTo("8") == 0) {
                return;
            }
            msg = "HBase file layout needs to be upgraded. Current filesystem version is " + version + " but software requires version " + "8" + ". Consult http://hbase.apache.org/book.html for further information about upgrading HBase.";
        }
        if (message) {
            System.out.println("WARNING! " + msg);
        }
        throw new FileSystemVersionException(msg);
    }

    public static void setVersion(FileSystem fs, Path rootdir) throws IOException {
        FSUtils.setVersion(fs, rootdir, "8", 0, 3);
    }

    public static void setVersion(FileSystem fs, Path rootdir, int wait, int retries) throws IOException {
        FSUtils.setVersion(fs, rootdir, "8", wait, retries);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void setVersion(FileSystem fs, Path rootdir, String version, int wait, int retries) throws IOException {
        Path versionFile = new Path(rootdir, "hbase.version");
        Path tempVersionFile = new Path(rootdir, ".tmp/hbase.version");
        while (true) {
            try {
                FSDataOutputStream s = fs.create(tempVersionFile);
                try {
                    s.write(FSUtils.toVersionByteArray(version));
                    s.close();
                    s = null;
                    if (!fs.rename(tempVersionFile, versionFile)) {
                        throw new IOException("Unable to move temp version file to " + versionFile);
                    }
                }
                finally {
                    try {
                        if (s != null) {
                            s.close();
                        }
                    }
                    catch (IOException iOException) {}
                }
                LOG.info("Created version file at " + rootdir.toString() + " with version=" + version);
                return;
            }
            catch (IOException e) {
                if (retries > 0) {
                    LOG.debug("Unable to create version file at " + rootdir.toString() + ", retrying", (Throwable)e);
                    fs.delete(versionFile, false);
                    try {
                        if (wait > 0) {
                            Thread.sleep(wait);
                        }
                    }
                    catch (InterruptedException ie) {
                        throw (InterruptedIOException)new InterruptedIOException().initCause(ie);
                    }
                    --retries;
                    continue;
                }
                throw e;
            }
            break;
        }
    }

    public static boolean checkClusterIdExists(FileSystem fs, Path rootdir, long wait) throws IOException {
        while (true) {
            try {
                Path filePath = new Path(rootdir, "hbase.id");
                return fs.exists(filePath);
            }
            catch (IOException ioe) {
                if (wait > 0L) {
                    LOG.warn("Unable to check cluster ID file in {}, retrying in {}ms", new Object[]{rootdir, wait, ioe});
                    try {
                        Thread.sleep(wait);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        throw (InterruptedIOException)new InterruptedIOException().initCause(e);
                    }
                    continue;
                }
                throw ioe;
            }
            break;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static ClusterId getClusterId(FileSystem fs, Path rootdir) throws IOException {
        FileStatus status;
        Path idPath = new Path(rootdir, "hbase.id");
        ClusterId clusterId = null;
        FileStatus fileStatus = status = fs.exists(idPath) ? fs.getFileStatus(idPath) : null;
        if (status != null) {
            int len = Ints.checkedCast((long)status.getLen());
            byte[] content = new byte[len];
            try (FSDataInputStream in = fs.open(idPath);){
                in.readFully(content);
            }
            try {
                clusterId = ClusterId.parseFrom((byte[])content);
            }
            catch (DeserializationException e) {
                throw new IOException("content=" + Bytes.toString((byte[])content), e);
            }
            if (!ProtobufUtil.isPBMagicPrefix((byte[])content)) {
                String cid = null;
                in = fs.open(idPath);
                try {
                    cid = in.readUTF();
                    clusterId = new ClusterId(cid);
                }
                catch (EOFException eof) {
                    LOG.warn("Cluster ID file {} is empty", (Object)idPath);
                }
                finally {
                    in.close();
                }
                FSUtils.rewriteAsPb(fs, rootdir, idPath, clusterId);
            }
            return clusterId;
        }
        LOG.warn("Cluster ID file does not exist at {}", (Object)idPath);
        return clusterId;
    }

    private static void rewriteAsPb(FileSystem fs, Path rootdir, Path p, ClusterId cid) throws IOException {
        Path movedAsideName = new Path(p + "." + System.currentTimeMillis());
        if (!fs.rename(p, movedAsideName)) {
            throw new IOException("Failed rename of " + p);
        }
        FSUtils.setClusterId(fs, rootdir, cid, 100);
        if (!fs.delete(movedAsideName, false)) {
            throw new IOException("Failed delete of " + movedAsideName);
        }
        LOG.debug("Rewrote the hbase.id file as pb");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void setClusterId(FileSystem fs, Path rootdir, ClusterId clusterId, int wait) throws IOException {
        while (true) {
            try {
                Path idFile = new Path(rootdir, "hbase.id");
                Path tempIdFile = new Path(rootdir, ".tmp/hbase.id");
                FSDataOutputStream s = fs.create(tempIdFile);
                try {
                    s.write(clusterId.toByteArray());
                    s.close();
                    s = null;
                    if (!fs.rename(tempIdFile, idFile)) {
                        throw new IOException("Unable to move temp version file to " + idFile);
                    }
                }
                finally {
                    try {
                        if (s != null) {
                            s.close();
                        }
                    }
                    catch (IOException iOException) {}
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Created cluster ID file at " + idFile.toString() + " with ID: " + clusterId);
                }
                return;
            }
            catch (IOException ioe) {
                if (wait > 0) {
                    LOG.warn("Unable to create cluster ID file in " + rootdir.toString() + ", retrying in " + wait + "msec: " + StringUtils.stringifyException((Throwable)ioe));
                    try {
                        Thread.sleep(wait);
                    }
                    catch (InterruptedException e) {
                        throw (InterruptedIOException)new InterruptedIOException().initCause(e);
                    }
                    continue;
                }
                throw ioe;
            }
            break;
        }
    }

    public static void waitOnSafeMode(Configuration conf, long wait) throws IOException {
        FileSystem fs = FileSystem.get((Configuration)conf);
        if (!(fs instanceof DistributedFileSystem)) {
            return;
        }
        DistributedFileSystem dfs = (DistributedFileSystem)fs;
        while (FSUtils.isInSafeMode(dfs)) {
            LOG.info("Waiting for dfs to exit safe mode...");
            try {
                Thread.sleep(wait);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw (InterruptedIOException)new InterruptedIOException().initCause(e);
            }
        }
    }

    public static boolean metaRegionExists(FileSystem fs, Path rootDir) throws IOException {
        Path metaRegionDir = FSUtils.getRegionDirFromRootDir(rootDir, RegionInfoBuilder.FIRST_META_REGIONINFO);
        return fs.exists(metaRegionDir);
    }

    public static HDFSBlocksDistribution computeHDFSBlocksDistribution(FileSystem fs, FileStatus status, long start, long length) throws IOException {
        HDFSBlocksDistribution blocksDistribution = new HDFSBlocksDistribution();
        BlockLocation[] blockLocations = fs.getFileBlockLocations(status, start, length);
        FSUtils.addToHDFSBlocksDistribution(blocksDistribution, blockLocations);
        return blocksDistribution;
    }

    public static void addToHDFSBlocksDistribution(HDFSBlocksDistribution blocksDistribution, BlockLocation[] blockLocations) throws IOException {
        for (BlockLocation bl : blockLocations) {
            String[] hosts = bl.getHosts();
            long len = bl.getLength();
            StorageType[] storageTypes = bl.getStorageTypes();
            blocksDistribution.addHostsAndBlockWeight(hosts, len, storageTypes);
        }
    }

    public static int getTotalTableFragmentation(HMaster master) throws IOException {
        Map<String, Integer> map = FSUtils.getTableFragmentation(master);
        return map.isEmpty() ? -1 : map.get("-TOTAL-");
    }

    public static Map<String, Integer> getTableFragmentation(HMaster master) throws IOException {
        Path path = CommonFSUtils.getRootDir((Configuration)master.getConfiguration());
        FileSystem fs = path.getFileSystem(master.getConfiguration());
        return FSUtils.getTableFragmentation(fs, path);
    }

    public static Map<String, Integer> getTableFragmentation(FileSystem fs, Path hbaseRootDir) throws IOException {
        HashMap<String, Integer> frags = new HashMap<String, Integer>();
        int cfCountTotal = 0;
        int cfFragTotal = 0;
        RegionDirFilter regionFilter = new RegionDirFilter(fs);
        FamilyDirFilter familyFilter = new FamilyDirFilter(fs);
        List<Path> tableDirs = FSUtils.getTableDirs(fs, hbaseRootDir);
        for (Path d : tableDirs) {
            FileStatus[] regionDirs;
            int cfCount = 0;
            int cfFrag = 0;
            for (FileStatus regionDir : regionDirs = fs.listStatus(d, (PathFilter)regionFilter)) {
                FileStatus[] familyDirs;
                Path dd = regionDir.getPath();
                for (FileStatus familyDir : familyDirs = fs.listStatus(dd, (PathFilter)familyFilter)) {
                    ++cfCount;
                    ++cfCountTotal;
                    Path family = familyDir.getPath();
                    FileStatus[] familyStatus = fs.listStatus(family);
                    if (familyStatus.length <= 1) continue;
                    ++cfFrag;
                    ++cfFragTotal;
                }
            }
            frags.put(CommonFSUtils.getTableName((Path)d).getNameAsString(), cfCount == 0 ? 0 : Math.round((float)cfFrag / (float)cfCount * 100.0f));
        }
        frags.put("-TOTAL-", cfCountTotal == 0 ? 0 : Math.round((float)cfFragTotal / (float)cfCountTotal * 100.0f));
        return frags;
    }

    public static List<Path> getTableDirs(FileSystem fs, Path rootdir) throws IOException {
        ArrayList<Path> tableDirs = new ArrayList<Path>();
        for (FileStatus status : fs.globStatus(new Path(rootdir, new Path("data", "*")))) {
            tableDirs.addAll(FSUtils.getLocalTableDirs(fs, status.getPath()));
        }
        return tableDirs;
    }

    public static List<Path> getLocalTableDirs(FileSystem fs, Path rootdir) throws IOException {
        FileStatus[] dirs = fs.listStatus(rootdir, (PathFilter)new UserTableDirFilter(fs));
        ArrayList<Path> tabledirs = new ArrayList<Path>(dirs.length);
        for (FileStatus dir : dirs) {
            tabledirs.add(dir.getPath());
        }
        return tabledirs;
    }

    public static List<Path> getRegionDirs(FileSystem fs, Path tableDir) throws IOException {
        List<FileStatus> rds = FSUtils.listStatusWithStatusFilter(fs, tableDir, new RegionDirFilter(fs));
        if (rds == null) {
            return Collections.emptyList();
        }
        ArrayList<Path> regionDirs = new ArrayList<Path>(rds.size());
        for (FileStatus rdfs : rds) {
            Path rdPath = rdfs.getPath();
            regionDirs.add(rdPath);
        }
        return regionDirs;
    }

    public static Path getRegionDirFromRootDir(Path rootDir, RegionInfo region) {
        return FSUtils.getRegionDirFromTableDir(CommonFSUtils.getTableDir((Path)rootDir, (TableName)region.getTable()), region);
    }

    public static Path getRegionDirFromTableDir(Path tableDir, RegionInfo region) {
        return FSUtils.getRegionDirFromTableDir(tableDir, ServerRegionReplicaUtil.getRegionInfoForFs(region).getEncodedName());
    }

    public static Path getRegionDirFromTableDir(Path tableDir, String encodedRegionName) {
        return new Path(tableDir, encodedRegionName);
    }

    public static List<Path> getFamilyDirs(FileSystem fs, Path regionDir) throws IOException {
        FileStatus[] fds = fs.listStatus(regionDir, (PathFilter)new FamilyDirFilter(fs));
        ArrayList<Path> familyDirs = new ArrayList<Path>(fds.length);
        for (FileStatus fdfs : fds) {
            Path fdPath = fdfs.getPath();
            familyDirs.add(fdPath);
        }
        return familyDirs;
    }

    public static List<Path> getReferenceFilePaths(FileSystem fs, Path familyDir) throws IOException {
        List<FileStatus> fds = FSUtils.listStatusWithStatusFilter(fs, familyDir, new ReferenceFileFilter(fs));
        if (fds == null) {
            return Collections.emptyList();
        }
        ArrayList<Path> referenceFiles = new ArrayList<Path>(fds.size());
        for (FileStatus fdfs : fds) {
            Path fdPath = fdfs.getPath();
            referenceFiles.add(fdPath);
        }
        return referenceFiles;
    }

    public static Map<String, Path> getTableStoreFilePathMap(Map<String, Path> map, FileSystem fs, Path hbaseRootDir, TableName tableName) throws IOException, InterruptedException {
        return FSUtils.getTableStoreFilePathMap(map, fs, hbaseRootDir, tableName, null, null, (ProgressReporter)null);
    }

    @Deprecated
    public static Map<String, Path> getTableStoreFilePathMap(Map<String, Path> resultMap, FileSystem fs, Path hbaseRootDir, TableName tableName, PathFilter sfFilter, ExecutorService executor, final HbckErrorReporter progressReporter) throws IOException, InterruptedException {
        return FSUtils.getTableStoreFilePathMap(resultMap, fs, hbaseRootDir, tableName, sfFilter, executor, new ProgressReporter(){

            @Override
            public void progress(FileStatus status) {
                progressReporter.progress();
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Map<String, Path> getTableStoreFilePathMap(Map<String, Path> resultMap, final FileSystem fs, Path hbaseRootDir, TableName tableName, final PathFilter sfFilter, ExecutorService executor, final ProgressReporter progressReporter) throws IOException, InterruptedException {
        ConcurrentHashMap<String, Path> finalResultMap;
        block14: {
            finalResultMap = resultMap == null ? new ConcurrentHashMap<String, Path>(128, 0.75f, 32) : resultMap;
            Path tableDir = CommonFSUtils.getTableDir((Path)hbaseRootDir, (TableName)tableName);
            final FamilyDirFilter familyFilter = new FamilyDirFilter(fs);
            final Vector<IOException> exceptions = new Vector<IOException>();
            try {
                List<FileStatus> regionDirs = FSUtils.listStatusWithStatusFilter(fs, tableDir, new RegionDirFilter(fs));
                if (regionDirs == null) {
                    ConcurrentHashMap<String, Path> concurrentHashMap = finalResultMap;
                    return concurrentHashMap;
                }
                ArrayList futures = new ArrayList(regionDirs.size());
                for (FileStatus fileStatus : regionDirs) {
                    Future<?> future;
                    if (null != progressReporter) {
                        progressReporter.progress(fileStatus);
                    }
                    final Path dd = fileStatus.getPath();
                    if (!exceptions.isEmpty()) break;
                    Runnable getRegionStoreFileMapCall = new Runnable(){

                        @Override
                        public void run() {
                            try {
                                HashMap<String, Path> regionStoreFileMap = new HashMap<String, Path>();
                                List<FileStatus> familyDirs = FSUtils.listStatusWithStatusFilter(fs, dd, familyFilter);
                                if (familyDirs == null) {
                                    if (!fs.exists(dd)) {
                                        LOG.warn("Skipping region because it no longer exists: " + dd);
                                    } else {
                                        LOG.warn("Skipping region because it has no family dirs: " + dd);
                                    }
                                    return;
                                }
                                for (FileStatus familyDir : familyDirs) {
                                    FileStatus[] familyStatus;
                                    Path family;
                                    if (null != progressReporter) {
                                        progressReporter.progress(familyDir);
                                    }
                                    if ((family = familyDir.getPath()).getName().equals("recovered.edits")) continue;
                                    for (FileStatus sfStatus : familyStatus = fs.listStatus(family)) {
                                        if (null != progressReporter) {
                                            progressReporter.progress(sfStatus);
                                        }
                                        Path sf = sfStatus.getPath();
                                        if (sfFilter != null && !sfFilter.accept(sf)) continue;
                                        regionStoreFileMap.put(sf.getName(), sf);
                                    }
                                }
                                finalResultMap.putAll(regionStoreFileMap);
                            }
                            catch (Exception e) {
                                LOG.error("Could not get region store file map for region: " + dd, (Throwable)e);
                                exceptions.add(e);
                            }
                        }
                    };
                    if (executor != null) {
                        future = executor.submit(getRegionStoreFileMapCall);
                        futures.add(future);
                        continue;
                    }
                    future = new FutureTask<Object>(getRegionStoreFileMapCall, null);
                    ((FutureTask)future).run();
                    futures.add(future);
                }
                for (Future future : futures) {
                    if (!exceptions.isEmpty()) {
                        break;
                    }
                    try {
                        future.get();
                    }
                    catch (ExecutionException e) {
                        LOG.error("Unexpected exec exception!  Should've been caught already.  (Bug?)", (Throwable)e);
                    }
                }
            }
            catch (IOException e) {
                LOG.error("Cannot execute getTableStoreFilePathMap for " + tableName, (Throwable)e);
                exceptions.add(e);
            }
            finally {
                if (exceptions.isEmpty()) break block14;
                Throwables.propagateIfPossible((Throwable)((Throwable)exceptions.firstElement()), IOException.class);
                throw new IOException((Throwable)exceptions.firstElement());
            }
        }
        return finalResultMap;
    }

    public static int getRegionReferenceFileCount(FileSystem fs, Path p) {
        int result = 0;
        try {
            for (Path familyDir : FSUtils.getFamilyDirs(fs, p)) {
                result += FSUtils.getReferenceFilePaths(fs, familyDir).size();
            }
        }
        catch (IOException e) {
            LOG.warn("Error counting reference files", (Throwable)e);
        }
        return result;
    }

    public static Map<String, Path> getTableStoreFilePathMap(FileSystem fs, Path hbaseRootDir) throws IOException, InterruptedException {
        return FSUtils.getTableStoreFilePathMap(fs, hbaseRootDir, null, null, (ProgressReporter)null);
    }

    @Deprecated
    public static Map<String, Path> getTableStoreFilePathMap(FileSystem fs, Path hbaseRootDir, PathFilter sfFilter, ExecutorService executor, final HbckErrorReporter progressReporter) throws IOException, InterruptedException {
        return FSUtils.getTableStoreFilePathMap(fs, hbaseRootDir, sfFilter, executor, new ProgressReporter(){

            @Override
            public void progress(FileStatus status) {
                progressReporter.progress();
            }
        });
    }

    public static Map<String, Path> getTableStoreFilePathMap(FileSystem fs, Path hbaseRootDir, PathFilter sfFilter, ExecutorService executor, ProgressReporter progressReporter) throws IOException, InterruptedException {
        ConcurrentHashMap<String, Path> map = new ConcurrentHashMap<String, Path>(1024, 0.75f, 32);
        for (Path tableDir : FSUtils.getTableDirs(fs, hbaseRootDir)) {
            FSUtils.getTableStoreFilePathMap(map, fs, hbaseRootDir, CommonFSUtils.getTableName((Path)tableDir), sfFilter, executor, progressReporter);
        }
        return map;
    }

    public static List<FileStatus> filterFileStatuses(FileStatus[] input, FileStatusFilter filter) {
        if (input == null) {
            return null;
        }
        return FSUtils.filterFileStatuses((Iterator<FileStatus>)Iterators.forArray((Object[])input), filter);
    }

    public static List<FileStatus> filterFileStatuses(Iterator<FileStatus> input, FileStatusFilter filter) {
        if (input == null) {
            return null;
        }
        ArrayList<FileStatus> results = new ArrayList<FileStatus>();
        while (input.hasNext()) {
            FileStatus f = input.next();
            if (!filter.accept(f)) continue;
            results.add(f);
        }
        return results;
    }

    public static List<FileStatus> listStatusWithStatusFilter(FileSystem fs, Path dir, FileStatusFilter filter) throws IOException {
        FileStatus[] status = null;
        try {
            status = fs.listStatus(dir);
        }
        catch (FileNotFoundException fnfe) {
            LOG.trace("{} does not exist", (Object)dir);
            return null;
        }
        if (ArrayUtils.getLength((Object)status) == 0) {
            return null;
        }
        if (filter == null) {
            return Arrays.asList(status);
        }
        List<FileStatus> status2 = FSUtils.filterFileStatuses(status, filter);
        if (status2 == null || status2.isEmpty()) {
            return null;
        }
        return status2;
    }

    public static Map<String, Map<String, Float>> getRegionDegreeLocalityMappingFromFS(Configuration conf) throws IOException {
        return FSUtils.getRegionDegreeLocalityMappingFromFS(conf, null, conf.getInt(THREAD_POOLSIZE, 2));
    }

    public static Map<String, Map<String, Float>> getRegionDegreeLocalityMappingFromFS(Configuration conf, String desiredTable, int threadPoolSize) throws IOException {
        ConcurrentHashMap<String, Map<String, Float>> regionDegreeLocalityMapping = new ConcurrentHashMap<String, Map<String, Float>>();
        FSUtils.getRegionLocalityMappingFromFS(conf, desiredTable, threadPoolSize, regionDegreeLocalityMapping);
        return regionDegreeLocalityMapping;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void getRegionLocalityMappingFromFS(Configuration conf, String desiredTable, int threadPoolSize, Map<String, Map<String, Float>> regionDegreeLocalityMapping) throws IOException {
        FileSystem fs = FileSystem.get((Configuration)conf);
        Path rootPath = CommonFSUtils.getRootDir((Configuration)conf);
        long startTime = EnvironmentEdgeManager.currentTime();
        Path queryPath = null == desiredTable ? new Path(new Path(rootPath, "data").toString() + "/*/*/*/") : new Path(CommonFSUtils.getTableDir((Path)rootPath, (TableName)TableName.valueOf((String)desiredTable)).toString() + "/*/");
        PathFilter pathFilter = new PathFilter(){

            public boolean accept(Path path) {
                if (null == path) {
                    return false;
                }
                Path parent = path.getParent();
                if (null == parent) {
                    return false;
                }
                String regionName = path.getName();
                if (null == regionName) {
                    return false;
                }
                return regionName.toLowerCase(Locale.ROOT).matches("[0-9a-f]+");
            }
        };
        Object[] statusList = fs.globStatus(queryPath, pathFilter);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Query Path: {} ; # list of files: {}", (Object)queryPath, (Object)Arrays.toString(statusList));
        }
        if (null == statusList) {
            return;
        }
        threadPoolSize = Math.min(threadPoolSize, statusList.length);
        ExecutorService tpe = Executors.newFixedThreadPool(threadPoolSize, new ThreadFactoryBuilder().setNameFormat("FSRegionQuery-pool-%d").setDaemon(true).setUncaughtExceptionHandler(Threads.LOGGING_EXCEPTION_HANDLER).build());
        try {
            for (Object regionStatus : statusList) {
                Path regionPath;
                if (null == regionStatus || !regionStatus.isDirectory() || null == (regionPath = regionStatus.getPath())) continue;
                tpe.execute(new FSRegionScanner(fs, regionPath, null, regionDegreeLocalityMapping));
            }
        }
        finally {
            tpe.shutdown();
            long threadWakeFrequency = conf.getInt("hbase.server.thread.wakefrequency", 10000);
            try {
                while (!tpe.awaitTermination(threadWakeFrequency, TimeUnit.MILLISECONDS)) {
                    LOG.info("Locality checking is underway: { Scanned Regions : " + ((ThreadPoolExecutor)tpe).getCompletedTaskCount() + "/" + ((ThreadPoolExecutor)tpe).getTaskCount() + " }");
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw (InterruptedIOException)new InterruptedIOException().initCause(e);
            }
        }
        long overhead = EnvironmentEdgeManager.currentTime() - startTime;
        LOG.info("Scan DFS for locality info takes {}ms", (Object)overhead);
    }

    public static void setupShortCircuitRead(Configuration conf) {
        boolean shortCircuitSkipChecksum = conf.getBoolean("dfs.client.read.shortcircuit.skip.checksum", false);
        boolean useHBaseChecksum = conf.getBoolean("hbase.regionserver.checksum.verify", true);
        if (shortCircuitSkipChecksum) {
            LOG.warn("Configuration \"dfs.client.read.shortcircuit.skip.checksum\" should not be set to true." + (useHBaseChecksum ? " HBase checksum doesn't require it, see https://issues.apache.org/jira/browse/HBASE-6868." : ""));
            assert (!shortCircuitSkipChecksum);
        }
        FSUtils.checkShortCircuitReadBufferSize(conf);
    }

    public static void checkShortCircuitReadBufferSize(Configuration conf) {
        int defaultSize = 131072;
        int notSet = -1;
        String dfsKey = "dfs.client.read.shortcircuit.buffer.size";
        int size = conf.getInt("dfs.client.read.shortcircuit.buffer.size", -1);
        if (size != -1) {
            return;
        }
        int hbaseSize = conf.getInt("hbase.dfs.client.read.shortcircuit.buffer.size", 131072);
        conf.setIfUnset("dfs.client.read.shortcircuit.buffer.size", Integer.toString(hbaseSize));
    }

    public static DFSHedgedReadMetrics getDFSHedgedReadMetrics(Configuration c) throws IOException {
        Method m;
        if (!CommonFSUtils.isHDFS((Configuration)c)) {
            return null;
        }
        String name = "getHedgedReadMetrics";
        DFSClient dfsclient = ((DistributedFileSystem)FileSystem.get((Configuration)c)).getClient();
        try {
            m = dfsclient.getClass().getDeclaredMethod("getHedgedReadMetrics", new Class[0]);
        }
        catch (NoSuchMethodException e) {
            LOG.warn("Failed find method getHedgedReadMetrics in dfsclient; no hedged read metrics: " + e.getMessage());
            return null;
        }
        catch (SecurityException e) {
            LOG.warn("Failed find method getHedgedReadMetrics in dfsclient; no hedged read metrics: " + e.getMessage());
            return null;
        }
        m.setAccessible(true);
        try {
            return (DFSHedgedReadMetrics)m.invoke((Object)dfsclient, new Object[0]);
        }
        catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
            LOG.warn("Failed invoking method getHedgedReadMetrics on dfsclient; no hedged read metrics: " + e.getMessage());
            return null;
        }
    }

    public static List<Path> copyFilesParallel(FileSystem srcFS, Path src, FileSystem dstFS, Path dst, Configuration conf, int threads) throws IOException {
        List<Path> traversedPaths;
        ExecutorService pool = Executors.newFixedThreadPool(threads);
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>();
        try {
            traversedPaths = FSUtils.copyFiles(srcFS, src, dstFS, dst, conf, pool, futures);
            for (Future future : futures) {
                future.get();
            }
        }
        catch (IOException | InterruptedException | ExecutionException e) {
            throw new IOException("Copy snapshot reference files failed", e);
        }
        finally {
            pool.shutdownNow();
        }
        return traversedPaths;
    }

    private static List<Path> copyFiles(FileSystem srcFS, Path src, FileSystem dstFS, Path dst, Configuration conf, ExecutorService pool, List<Future<Void>> futures) throws IOException {
        ArrayList<Path> traversedPaths = new ArrayList<Path>();
        traversedPaths.add(dst);
        FileStatus currentFileStatus = srcFS.getFileStatus(src);
        if (currentFileStatus.isDirectory()) {
            FileStatus[] subPaths;
            if (!dstFS.mkdirs(dst)) {
                throw new IOException("Create directory failed: " + dst);
            }
            for (FileStatus subPath : subPaths = srcFS.listStatus(src)) {
                traversedPaths.addAll(FSUtils.copyFiles(srcFS, subPath.getPath(), dstFS, new Path(dst, subPath.getPath().getName()), conf, pool, futures));
            }
        } else {
            Future<Void> future = pool.submit(() -> {
                FileUtil.copy((FileSystem)srcFS, (Path)src, (FileSystem)dstFS, (Path)dst, (boolean)false, (boolean)false, (Configuration)conf);
                return null;
            });
            futures.add(future);
        }
        return traversedPaths;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static Set<InetSocketAddress> getNNAddresses(DistributedFileSystem fs, Configuration conf) {
        HashSet<InetSocketAddress> addresses = new HashSet<InetSocketAddress>();
        String serviceName = fs.getCanonicalServiceName();
        if (serviceName.startsWith("ha-hdfs")) {
            try {
                Map addressMap = DFSUtil.getNNServiceRpcAddressesForCluster((Configuration)conf);
                String nameService = serviceName.substring(serviceName.indexOf(":") + 1);
                if (!addressMap.containsKey(nameService)) return addresses;
                Map nnMap = (Map)addressMap.get(nameService);
                for (Map.Entry e2 : nnMap.entrySet()) {
                    InetSocketAddress addr = (InetSocketAddress)e2.getValue();
                    addresses.add(addr);
                }
                return addresses;
            }
            catch (Exception e) {
                LOG.warn("DFSUtil.getNNServiceRpcAddresses failed. serviceName=" + serviceName, (Throwable)e);
                return addresses;
            }
        } else {
            URI uri = fs.getUri();
            int port = uri.getPort();
            if (port < 0) {
                int idx = serviceName.indexOf(58);
                port = Integer.parseInt(serviceName.substring(idx + 1));
            }
            InetSocketAddress addr = new InetSocketAddress(uri.getHost(), port);
            addresses.add(addr);
        }
        return addresses;
    }

    public static boolean isSameHdfs(Configuration conf, FileSystem srcFs, FileSystem desFs) {
        Set<InetSocketAddress> desAddrs;
        Set<InetSocketAddress> srcAddrs;
        Collection internalNameServices;
        String srcServiceName = srcFs.getCanonicalServiceName();
        String desServiceName = desFs.getCanonicalServiceName();
        if (srcServiceName == null || desServiceName == null) {
            return false;
        }
        if (srcServiceName.equals(desServiceName)) {
            return true;
        }
        if (srcServiceName.startsWith("ha-hdfs") && desServiceName.startsWith("ha-hdfs") && !(internalNameServices = conf.getTrimmedStringCollection("dfs.internal.nameservices")).isEmpty()) {
            return internalNameServices.contains(srcServiceName.split(":")[1]);
        }
        return srcFs instanceof DistributedFileSystem && desFs instanceof DistributedFileSystem && Sets.intersection(srcAddrs = FSUtils.getNNAddresses((DistributedFileSystem)srcFs, conf), desAddrs = FSUtils.getNNAddresses((DistributedFileSystem)desFs, conf)).size() > 0;
    }

    static interface ProgressReporter {
        public void progress(FileStatus var1);
    }

    public static class ReferenceFileFilter
    extends AbstractFileStatusFilter {
        private final FileSystem fs;

        public ReferenceFileFilter(FileSystem fs) {
            this.fs = fs;
        }

        @Override
        protected boolean accept(Path p, @CheckForNull Boolean isDir) {
            if (!StoreFileInfo.isReference(p)) {
                return false;
            }
            try {
                return this.isFile(this.fs, isDir, p);
            }
            catch (IOException ioe) {
                LOG.warn("Skipping file {} due to IOException", (Object)p, (Object)ioe);
                return false;
            }
        }
    }

    public static class HFileLinkFilter
    implements PathFilter {
        public boolean accept(Path p) {
            return HFileLink.isHFileLink(p);
        }
    }

    public static class HFileFilter
    extends AbstractFileStatusFilter {
        final FileSystem fs;

        public HFileFilter(FileSystem fs) {
            this.fs = fs;
        }

        @Override
        protected boolean accept(Path p, @CheckForNull Boolean isDir) {
            if (!StoreFileInfo.isHFile(p)) {
                return false;
            }
            try {
                return this.isFile(this.fs, isDir, p);
            }
            catch (IOException ioe) {
                LOG.warn("Skipping file {} due to IOException", (Object)p, (Object)ioe);
                return false;
            }
        }
    }

    public static class FamilyDirFilter
    extends AbstractFileStatusFilter {
        final FileSystem fs;

        public FamilyDirFilter(FileSystem fs) {
            this.fs = fs;
        }

        @Override
        protected boolean accept(Path p, @CheckForNull Boolean isDir) {
            try {
                HColumnDescriptor.isLegalFamilyName((byte[])Bytes.toBytes((String)p.getName()));
            }
            catch (IllegalArgumentException iae) {
                return false;
            }
            try {
                return this.isDirectory(this.fs, isDir, p);
            }
            catch (IOException ioe) {
                LOG.warn("Skipping file {} due to IOException", (Object)p, (Object)ioe);
                return false;
            }
        }
    }

    public static class RegionDirFilter
    extends AbstractFileStatusFilter {
        public static final Pattern regionDirPattern = Pattern.compile("^[0-9a-f]*$");
        final FileSystem fs;

        public RegionDirFilter(FileSystem fs) {
            this.fs = fs;
        }

        @Override
        protected boolean accept(Path p, @CheckForNull Boolean isDir) {
            if (!regionDirPattern.matcher(p.getName()).matches()) {
                return false;
            }
            try {
                return this.isDirectory(this.fs, isDir, p);
            }
            catch (IOException ioe) {
                LOG.warn("Skipping file {} due to IOException", (Object)p, (Object)ioe);
                return false;
            }
        }
    }

    public static class UserTableDirFilter
    extends BlackListDirFilter {
        public UserTableDirFilter(FileSystem fs) {
            super(fs, HConstants.HBASE_NON_TABLE_DIRS);
        }

        @Override
        protected boolean isValidName(String name) {
            if (!super.isValidName(name)) {
                return false;
            }
            try {
                TableName.isLegalTableQualifierName((byte[])Bytes.toBytes((String)name));
            }
            catch (IllegalArgumentException e) {
                LOG.info("Invalid table name: {}", (Object)name);
                return false;
            }
            return true;
        }
    }

    public static class DirFilter
    extends BlackListDirFilter {
        public DirFilter(FileSystem fs) {
            super(fs, null);
        }
    }

    public static class BlackListDirFilter
    extends AbstractFileStatusFilter {
        private final FileSystem fs;
        private List<String> blacklist;

        public BlackListDirFilter(FileSystem fs, List<String> directoryNameBlackList) {
            this.fs = fs;
            this.blacklist = directoryNameBlackList == null ? Collections.emptyList() : directoryNameBlackList;
        }

        @Override
        protected boolean accept(Path p, @CheckForNull Boolean isDir) {
            if (!this.isValidName(p.getName())) {
                return false;
            }
            try {
                return this.isDirectory(this.fs, isDir, p);
            }
            catch (IOException e) {
                LOG.warn("An error occurred while verifying if [{}] is a valid directory. Returning 'not valid' and continuing.", (Object)p, (Object)e);
                return false;
            }
        }

        protected boolean isValidName(String name) {
            return !this.blacklist.contains(name);
        }
    }

    static class FileFilter
    extends AbstractFileStatusFilter {
        private final FileSystem fs;

        public FileFilter(FileSystem fs) {
            this.fs = fs;
        }

        @Override
        protected boolean accept(Path p, @CheckForNull Boolean isDir) {
            try {
                return this.isFile(this.fs, isDir, p);
            }
            catch (IOException e) {
                LOG.warn("Unable to verify if path={} is a regular file", (Object)p, (Object)e);
                return false;
            }
        }
    }
}

