/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.procedure;

import java.io.IOException;
import java.util.function.Function;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.hbase.HBaseClassTestRule;
import org.apache.hadoop.hbase.HBaseIOException;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptorBuilder;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.client.TableDescriptorBuilder;
import org.apache.hadoop.hbase.fs.ErasureCodingUtils;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.regionserver.CompactedHFilesDischarger;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.testclassification.MasterTests;
import org.apache.hadoop.hbase.testclassification.MediumTests;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.CommonFSUtils;
import org.apache.hadoop.hbase.util.JVMClusterUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Category(value={MasterTests.class, MediumTests.class})
public class TestManageTableErasureCodingPolicy {
    @ClassRule
    public static final HBaseClassTestRule CLASS_RULE = HBaseClassTestRule.forClass(TestManageTableErasureCodingPolicy.class);
    private static final Logger LOG = LoggerFactory.getLogger(TestManageTableErasureCodingPolicy.class);
    private static final HBaseTestingUtility UTIL = new HBaseTestingUtility();
    private static final byte[] FAMILY = Bytes.toBytes((String)"a");
    private static final TableName NON_EC_TABLE = TableName.valueOf((String)"foo");
    private static final TableDescriptor NON_EC_TABLE_DESC = TableDescriptorBuilder.newBuilder((TableName)NON_EC_TABLE).setColumnFamily(ColumnFamilyDescriptorBuilder.of((byte[])FAMILY)).build();
    private static final TableName EC_TABLE = TableName.valueOf((String)"bar");
    private static final TableDescriptor EC_TABLE_DESC = TableDescriptorBuilder.newBuilder((TableName)EC_TABLE).setErasureCodingPolicy("XOR-2-1-1024k").setColumnFamily(ColumnFamilyDescriptorBuilder.of((byte[])FAMILY)).build();
    private static boolean erasureCodingSupported;

    @BeforeClass
    public static void beforeClass() throws Exception {
        UTIL.getConfiguration().setBoolean("hbase.table.sanity.checks", true);
        UTIL.startMiniDFSCluster(3);
        UTIL.startMiniCluster(1);
        DistributedFileSystem fs = (DistributedFileSystem)FileSystem.get((Configuration)UTIL.getConfiguration());
        erasureCodingSupported = TestManageTableErasureCodingPolicy.enableErasureCoding(fs);
        Table table = UTIL.createTable(NON_EC_TABLE_DESC, (byte[][])null);
        UTIL.loadTable(table, FAMILY);
        UTIL.flush();
    }

    private static boolean enableErasureCoding(DistributedFileSystem fs) throws IOException {
        try {
            ErasureCodingUtils.enablePolicy((FileSystem)fs, (String)"XOR-2-1-1024k");
            ErasureCodingUtils.enablePolicy((FileSystem)fs, (String)"RS-6-3-1024k");
            return true;
        }
        catch (UnsupportedOperationException e) {
            LOG.info("Current hadoop version does not support erasure coding, only validation tests will run.");
            return false;
        }
    }

    @AfterClass
    public static void afterClass() throws Exception {
        UTIL.shutdownMiniCluster();
        UTIL.shutdownMiniDFSCluster();
    }

    @Test
    public void itValidatesPolicyNameForCreate() {
        this.runValidatePolicyNameTest(unused -> EC_TABLE_DESC, Admin::createTable);
    }

    @Test
    public void itValidatesPolicyNameForAlter() {
        this.runValidatePolicyNameTest(admin -> {
            try {
                return admin.getDescriptor(NON_EC_TABLE);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }, Admin::modifyTable);
    }

    private void runValidatePolicyNameTest(Function<Admin, TableDescriptor> descriptorSupplier, ThrowingTableDescriptorConsumer consumer) {
        HBaseIOException thrown = (HBaseIOException)Assert.assertThrows(HBaseIOException.class, () -> {
            try (Admin admin = UTIL.getAdmin();){
                TableDescriptor desc = (TableDescriptor)descriptorSupplier.apply(admin);
                consumer.accept(admin, TableDescriptorBuilder.newBuilder((TableDescriptor)desc).setErasureCodingPolicy("foo").build());
            }
        });
        this.assertPolicyValidationException(thrown.getMessage(), "Cannot set Erasure Coding policy: foo. Policy not found");
        thrown = (HBaseIOException)Assert.assertThrows(HBaseIOException.class, () -> {
            try (Admin admin = UTIL.getAdmin();){
                TableDescriptor desc = (TableDescriptor)descriptorSupplier.apply(admin);
                consumer.accept(admin, TableDescriptorBuilder.newBuilder((TableDescriptor)desc).setErasureCodingPolicy("RS-10-4-1024k").build());
            }
        });
        this.assertPolicyValidationException(thrown.getMessage(), "Cannot set Erasure Coding policy: RS-10-4-1024k. The policy must be enabled");
        thrown = (HBaseIOException)Assert.assertThrows(HBaseIOException.class, () -> {
            try (Admin admin = UTIL.getAdmin();){
                TableDescriptor desc = (TableDescriptor)descriptorSupplier.apply(admin);
                consumer.accept(admin, TableDescriptorBuilder.newBuilder((TableDescriptor)desc).setErasureCodingPolicy("RS-6-3-1024k").build());
            }
        });
        this.assertPolicyValidationException(thrown.getMessage(), "Failed write test for EC policy");
    }

    private void assertPolicyValidationException(String message, String expected) {
        if (erasureCodingSupported) {
            MatcherAssert.assertThat((Object)message, (Matcher)Matchers.containsString((String)expected));
        } else {
            MatcherAssert.assertThat((Object)message, (Matcher)Matchers.containsString((String)"Cannot find specified method"));
        }
    }

    @Test
    public void testCreateTableErasureCodingSync() throws IOException {
        Assume.assumeTrue((boolean)erasureCodingSupported);
        try (Admin admin = UTIL.getAdmin();){
            this.recreateTable(admin, EC_TABLE_DESC);
            UTIL.flush(EC_TABLE);
            Path rootDir = CommonFSUtils.getRootDir((Configuration)UTIL.getConfiguration());
            DistributedFileSystem dfs = (DistributedFileSystem)FileSystem.get((Configuration)UTIL.getConfiguration());
            this.checkRegionDirAndFilePolicies(dfs, rootDir, EC_TABLE, "XOR-2-1-1024k", "XOR-2-1-1024k");
        }
    }

    private void recreateTable(Admin admin, TableDescriptor desc) throws IOException {
        if (admin.tableExists(desc.getTableName())) {
            admin.disableTable(desc.getTableName());
            admin.deleteTable(desc.getTableName());
        }
        admin.createTable(desc);
        try (Table table = UTIL.getConnection().getTable(desc.getTableName());){
            UTIL.loadTable(table, FAMILY);
        }
    }

    @Test
    public void testModifyTableErasureCodingSync() throws IOException {
        Assume.assumeTrue((boolean)erasureCodingSupported);
        try (Admin admin = UTIL.getAdmin();){
            Path rootDir = CommonFSUtils.getRootDir((Configuration)UTIL.getConfiguration());
            DistributedFileSystem dfs = (DistributedFileSystem)FileSystem.get((Configuration)UTIL.getConfiguration());
            this.checkRegionDirAndFilePolicies(dfs, rootDir, NON_EC_TABLE, null, null);
            TableDescriptor desc = UTIL.getAdmin().getDescriptor(NON_EC_TABLE);
            TableDescriptor newDesc = TableDescriptorBuilder.newBuilder((TableDescriptor)desc).setErasureCodingPolicy("XOR-2-1-1024k").build();
            admin.modifyTable(newDesc);
            this.checkRegionDirAndFilePolicies(dfs, rootDir, NON_EC_TABLE, "XOR-2-1-1024k", null);
            this.compactAwayOldFiles(NON_EC_TABLE);
            this.checkRegionDirAndFilePolicies(dfs, rootDir, NON_EC_TABLE, "XOR-2-1-1024k", "XOR-2-1-1024k");
            newDesc = TableDescriptorBuilder.newBuilder((TableDescriptor)newDesc).setErasureCodingPolicy(null).build();
            admin.modifyTable(newDesc);
            this.checkRegionDirAndFilePolicies(dfs, rootDir, NON_EC_TABLE, null, "XOR-2-1-1024k");
            UTIL.compact(NON_EC_TABLE, true);
            for (JVMClusterUtil.RegionServerThread regionserver : UTIL.getHBaseCluster().getLiveRegionServerThreads()) {
                CompactedHFilesDischarger chore = regionserver.getRegionServer().getCompactedHFilesDischarger();
                chore.setUseExecutor(false);
                chore.chore();
            }
            this.checkRegionDirAndFilePolicies(dfs, rootDir, NON_EC_TABLE, null, null);
        }
    }

    private void compactAwayOldFiles(TableName tableName) throws IOException {
        LOG.info("Compacting and discharging files for {}", (Object)tableName);
        UTIL.compact(tableName, true);
        for (JVMClusterUtil.RegionServerThread regionserver : UTIL.getHBaseCluster().getLiveRegionServerThreads()) {
            CompactedHFilesDischarger chore = regionserver.getRegionServer().getCompactedHFilesDischarger();
            chore.setUseExecutor(false);
            chore.chore();
        }
    }

    @Test
    public void testRestoreSnapshot() throws IOException {
        Assume.assumeTrue((boolean)erasureCodingSupported);
        String snapshotName = "testRestoreSnapshot_snap";
        TableName tableName = TableName.valueOf((String)"testRestoreSnapshot_tbl");
        try (Admin admin = UTIL.getAdmin();){
            Path rootDir = CommonFSUtils.getRootDir((Configuration)UTIL.getConfiguration());
            DistributedFileSystem dfs = (DistributedFileSystem)FileSystem.get((Configuration)UTIL.getConfiguration());
            this.recreateTable(admin, EC_TABLE_DESC);
            admin.snapshot(snapshotName, EC_TABLE);
            admin.cloneSnapshot(snapshotName, tableName);
            this.compactAwayOldFiles(tableName);
            this.checkRegionDirAndFilePolicies(dfs, rootDir, tableName, "XOR-2-1-1024k", "XOR-2-1-1024k");
            admin.modifyTable(TableDescriptorBuilder.newBuilder((TableDescriptor)EC_TABLE_DESC).setErasureCodingPolicy(null).build());
            this.compactAwayOldFiles(EC_TABLE);
            this.checkRegionDirAndFilePolicies(dfs, rootDir, EC_TABLE, null, null);
            admin.disableTable(EC_TABLE);
            admin.restoreSnapshot(snapshotName);
            admin.enableTable(EC_TABLE);
            this.compactAwayOldFiles(EC_TABLE);
            this.checkRegionDirAndFilePolicies(dfs, rootDir, EC_TABLE, "XOR-2-1-1024k", "XOR-2-1-1024k");
        }
    }

    private void checkRegionDirAndFilePolicies(DistributedFileSystem dfs, Path rootDir, TableName testTable, String expectedDirPolicy, String expectedFilePolicy) throws IOException {
        Path tableDir = CommonFSUtils.getTableDir((Path)rootDir, (TableName)testTable);
        this.checkPolicy(dfs, tableDir, expectedDirPolicy);
        int filesMatched = 0;
        for (HRegion region : UTIL.getHBaseCluster().getRegions(testTable)) {
            Path regionDir = new Path(tableDir, region.getRegionInfo().getEncodedName());
            this.checkPolicy(dfs, regionDir, expectedDirPolicy);
            RemoteIterator itr = dfs.listFiles(regionDir, true);
            while (itr.hasNext()) {
                LocatedFileStatus fileStatus = (LocatedFileStatus)itr.next();
                Path path = fileStatus.getPath();
                if (!HFile.isHFileFormat((FileSystem)dfs, (Path)path)) {
                    LOG.info("{} is not an hfile", (Object)path);
                    continue;
                }
                ++filesMatched;
                this.checkPolicy(dfs, path, expectedFilePolicy);
            }
        }
        MatcherAssert.assertThat((Object)filesMatched, (Matcher)Matchers.greaterThan((Comparable)Integer.valueOf(0)));
    }

    private void checkPolicy(DistributedFileSystem dfs, Path path, String expectedPolicy) throws IOException {
        String policy = ErasureCodingUtils.getPolicyNameForPath((DistributedFileSystem)dfs, (Path)path);
        if (expectedPolicy == null) {
            MatcherAssert.assertThat((String)("policy for " + path), (Object)policy, (Matcher)Matchers.nullValue());
        } else {
            MatcherAssert.assertThat((String)("policy for " + path), (Object)policy, (Matcher)Matchers.equalTo((Object)expectedPolicy));
        }
    }

    @FunctionalInterface
    static interface ThrowingTableDescriptorConsumer {
        public void accept(Admin var1, TableDescriptor var2) throws IOException;
    }
}

