/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.client;

import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.StatusCode;
import io.opentelemetry.context.Scope;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.RegionLocations;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.AsyncConnectionImpl;
import org.apache.hadoop.hbase.client.AsyncMetaRegionLocator;
import org.apache.hadoop.hbase.client.AsyncNonMetaRegionLocator;
import org.apache.hadoop.hbase.client.MetricsConnection;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.RegionLocateType;
import org.apache.hadoop.hbase.client.RegionOfflineException;
import org.apache.hadoop.hbase.client.trace.ConnectionSpanBuilder;
import org.apache.hadoop.hbase.client.trace.TableSpanBuilder;
import org.apache.hadoop.hbase.exceptions.TimeoutIOException;
import org.apache.hadoop.hbase.trace.HBaseSemanticAttributes;
import org.apache.hadoop.hbase.trace.TraceUtil;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FutureUtils;
import org.apache.hbase.thirdparty.io.netty.util.HashedWheelTimer;
import org.apache.hbase.thirdparty.io.netty.util.Timeout;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
class AsyncRegionLocator {
    private static final Logger LOG = LoggerFactory.getLogger(AsyncRegionLocator.class);
    private final HashedWheelTimer retryTimer;
    private final AsyncConnectionImpl conn;
    private final AsyncMetaRegionLocator metaRegionLocator;
    private final AsyncNonMetaRegionLocator nonMetaRegionLocator;

    AsyncRegionLocator(AsyncConnectionImpl conn, HashedWheelTimer retryTimer) {
        this.conn = conn;
        this.metaRegionLocator = new AsyncMetaRegionLocator(conn.registry);
        this.nonMetaRegionLocator = new AsyncNonMetaRegionLocator(conn);
        this.retryTimer = retryTimer;
    }

    private <T> CompletableFuture<T> withTimeout(CompletableFuture<T> future, long timeoutNs, Supplier<String> timeoutMsg) {
        if (future.isDone() || timeoutNs <= 0L) {
            return future;
        }
        Timeout timeoutTask = this.retryTimer.newTimeout(t -> {
            if (future.isDone()) {
                return;
            }
            future.completeExceptionally(new TimeoutIOException((String)timeoutMsg.get()));
        }, timeoutNs, TimeUnit.NANOSECONDS);
        FutureUtils.addListener(future, (loc, error) -> {
            if (error != null && error.getClass() != TimeoutIOException.class) {
                timeoutTask.cancel();
            }
        });
        return future;
    }

    private boolean isMeta(TableName tableName) {
        return TableName.isMetaTableName(tableName);
    }

    private <T> CompletableFuture<T> tracedLocationFuture(Supplier<CompletableFuture<T>> action, Function<T, List<String>> getRegionNames, Supplier<Span> spanSupplier) {
        Span span = spanSupplier.get();
        try (Scope scope = span.makeCurrent();){
            CompletableFuture<T> future = action.get();
            FutureUtils.addListener(future, (resp, error) -> {
                if (error != null) {
                    TraceUtil.setError(span, error);
                } else {
                    List regionNames = (List)getRegionNames.apply(resp);
                    if (!regionNames.isEmpty()) {
                        span.setAttribute(HBaseSemanticAttributes.REGION_NAMES_KEY, (Object)regionNames);
                    }
                    span.setStatus(StatusCode.OK);
                }
                span.end();
            });
            CompletableFuture<T> completableFuture = future;
            return completableFuture;
        }
    }

    private static List<String> getRegionNames(RegionLocations locs) {
        if (locs == null || locs.getRegionLocations() == null) {
            return Collections.emptyList();
        }
        return Arrays.stream(locs.getRegionLocations()).filter(Objects::nonNull).map(HRegionLocation::getRegion).map(RegionInfo::getRegionNameAsString).collect(Collectors.toList());
    }

    private static List<String> getRegionNames(HRegionLocation location) {
        return Optional.ofNullable(location).map(HRegionLocation::getRegion).map(RegionInfo::getRegionNameAsString).map(Collections::singletonList).orElseGet(Collections::emptyList);
    }

    CompletableFuture<RegionLocations> getRegionLocations(TableName tableName, byte[] row, RegionLocateType type, boolean reload, long timeoutNs) {
        TableSpanBuilder supplier = new TableSpanBuilder(this.conn).setName("AsyncRegionLocator.getRegionLocations").setTableName(tableName);
        return this.tracedLocationFuture(() -> {
            CompletableFuture<RegionLocations> future = this.isMeta(tableName) ? this.metaRegionLocator.getRegionLocations(0, reload) : this.nonMetaRegionLocator.getRegionLocations(tableName, row, 0, type, reload);
            return this.withTimeout(future, timeoutNs, () -> "Timeout(" + TimeUnit.NANOSECONDS.toMillis(timeoutNs) + "ms) waiting for region locations for " + tableName + ", row='" + Bytes.toStringBinary(row) + "'");
        }, AsyncRegionLocator::getRegionNames, supplier);
    }

    CompletableFuture<HRegionLocation> getRegionLocation(TableName tableName, byte[] row, int replicaId, RegionLocateType type, boolean reload, long timeoutNs) {
        TableSpanBuilder supplier = new TableSpanBuilder(this.conn).setName("AsyncRegionLocator.getRegionLocation").setTableName(tableName);
        return this.tracedLocationFuture(() -> {
            CompletableFuture future = new CompletableFuture();
            CompletableFuture<RegionLocations> locsFuture = this.isMeta(tableName) ? this.metaRegionLocator.getRegionLocations(replicaId, reload) : this.nonMetaRegionLocator.getRegionLocations(tableName, row, replicaId, type, reload);
            FutureUtils.addListener(locsFuture, (locs, error) -> {
                if (error != null) {
                    future.completeExceptionally((Throwable)error);
                    return;
                }
                HRegionLocation loc = locs.getRegionLocation(replicaId);
                if (loc == null) {
                    future.completeExceptionally(new RegionOfflineException("No location for " + tableName + ", row='" + Bytes.toStringBinary(row) + "', locateType=" + (Object)((Object)type) + ", replicaId=" + replicaId));
                } else if (loc.getServerName() == null) {
                    future.completeExceptionally(new RegionOfflineException("No server address listed for region '" + loc.getRegion().getRegionNameAsString() + ", row='" + Bytes.toStringBinary(row) + "', locateType=" + (Object)((Object)type) + ", replicaId=" + replicaId));
                } else {
                    future.complete(loc);
                }
            });
            return this.withTimeout(future, timeoutNs, () -> "Timeout(" + TimeUnit.NANOSECONDS.toMillis(timeoutNs) + "ms) waiting for region location for " + tableName + ", row='" + Bytes.toStringBinary(row) + "', replicaId=" + replicaId);
        }, AsyncRegionLocator::getRegionNames, supplier);
    }

    CompletableFuture<HRegionLocation> getRegionLocation(TableName tableName, byte[] row, int replicaId, RegionLocateType type, long timeoutNs) {
        return this.getRegionLocation(tableName, row, replicaId, type, false, timeoutNs);
    }

    CompletableFuture<HRegionLocation> getRegionLocation(TableName tableName, byte[] row, RegionLocateType type, boolean reload, long timeoutNs) {
        return this.getRegionLocation(tableName, row, 0, type, reload, timeoutNs);
    }

    CompletableFuture<HRegionLocation> getRegionLocation(TableName tableName, byte[] row, RegionLocateType type, long timeoutNs) {
        return this.getRegionLocation(tableName, row, type, false, timeoutNs);
    }

    void updateCachedLocationOnError(HRegionLocation loc, Throwable exception) {
        if (loc.getRegion().isMetaRegion()) {
            this.metaRegionLocator.updateCachedLocationOnError(loc, exception);
        } else {
            this.nonMetaRegionLocator.updateCachedLocationOnError(loc, exception);
        }
    }

    void clearCache(TableName tableName) {
        TableSpanBuilder supplier = new TableSpanBuilder(this.conn).setName("AsyncRegionLocator.clearCache").setTableName(tableName);
        TraceUtil.trace(() -> {
            LOG.debug("Clear meta cache for {}", (Object)tableName);
            if (tableName.equals(TableName.META_TABLE_NAME)) {
                this.metaRegionLocator.clearCache();
            } else {
                this.nonMetaRegionLocator.clearCache(tableName);
            }
        }, (Supplier<Span>)supplier);
    }

    void clearCache(ServerName serverName) {
        ConnectionSpanBuilder supplier = new ConnectionSpanBuilder(this.conn).setName("AsyncRegionLocator.clearCache").addAttribute(HBaseSemanticAttributes.SERVER_NAME_KEY, serverName.getServerName());
        TraceUtil.trace(() -> {
            LOG.debug("Clear meta cache for {}", (Object)serverName);
            this.metaRegionLocator.clearCache(serverName);
            this.nonMetaRegionLocator.clearCache(serverName);
            this.conn.getConnectionMetrics().ifPresent(MetricsConnection::incrMetaCacheNumClearServer);
        }, (Supplier<Span>)supplier);
    }

    void clearCache() {
        ConnectionSpanBuilder supplier = new ConnectionSpanBuilder(this.conn).setName("AsyncRegionLocator.clearCache");
        TraceUtil.trace(() -> {
            this.metaRegionLocator.clearCache();
            this.nonMetaRegionLocator.clearCache();
        }, (Supplier<Span>)supplier);
    }

    AsyncNonMetaRegionLocator getNonMetaRegionLocator() {
        return this.nonMetaRegionLocator;
    }

    RegionLocations getRegionLocationInCache(TableName tableName, byte[] row) {
        if (TableName.isMetaTableName(tableName)) {
            return this.metaRegionLocator.getRegionLocationInCache();
        }
        return this.nonMetaRegionLocator.getRegionLocationInCache(tableName, row);
    }

    int getNumberOfCachedRegionLocations(TableName tableName) {
        if (TableName.isMetaTableName(tableName)) {
            return this.metaRegionLocator.getNumberOfCachedRegionLocations();
        }
        return this.nonMetaRegionLocator.getNumberOfCachedRegionLocations(tableName);
    }
}

