/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.fs.HFileSystem;
import org.apache.hadoop.hbase.io.ByteArrayOutputStream;
import org.apache.hadoop.hbase.io.ByteBuffInputStream;
import org.apache.hadoop.hbase.io.ByteBufferWriterDataOutputStream;
import org.apache.hadoop.hbase.io.FSDataInputStreamWrapper;
import org.apache.hadoop.hbase.io.encoding.DataBlockEncoding;
import org.apache.hadoop.hbase.io.encoding.HFileBlockDecodingContext;
import org.apache.hadoop.hbase.io.encoding.HFileBlockDefaultDecodingContext;
import org.apache.hadoop.hbase.io.encoding.HFileBlockDefaultEncodingContext;
import org.apache.hadoop.hbase.io.encoding.HFileBlockEncodingContext;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.CacheConfig;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializer;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializerIdManager;
import org.apache.hadoop.hbase.io.hfile.ChecksumUtil;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFileContext;
import org.apache.hadoop.hbase.io.hfile.HFileContextBuilder;
import org.apache.hadoop.hbase.io.hfile.HFileDataBlockEncoder;
import org.apache.hadoop.hbase.io.hfile.HFileUtil;
import org.apache.hadoop.hbase.io.hfile.NoOpDataBlockEncoder;
import org.apache.hadoop.hbase.nio.ByteBuff;
import org.apache.hadoop.hbase.nio.MultiByteBuff;
import org.apache.hadoop.hbase.nio.SingleByteBuff;
import org.apache.hadoop.hbase.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.hbase.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.ChecksumType;
import org.apache.hadoop.hbase.util.ClassSize;
import org.apache.hadoop.io.IOUtils;
import org.apache.yetus.audience.InterfaceAudience;

@InterfaceAudience.Private
public class HFileBlock
implements Cacheable {
    private static final Log LOG = LogFactory.getLog(HFileBlock.class);
    private BlockType blockType;
    private int onDiskSizeWithoutHeader;
    private int uncompressedSizeWithoutHeader;
    private long prevBlockOffset;
    private int onDiskDataSizeWithHeader;
    private ByteBuff buf;
    private HFileContext fileContext;
    private long offset = UNSET;
    private Cacheable.MemoryType memType = Cacheable.MemoryType.EXCLUSIVE;
    private int nextBlockOnDiskSize = UNSET;
    static final int CHECKSUM_VERIFICATION_NUM_IO_THRESHOLD = 3;
    private static int UNSET = -1;
    public static final boolean FILL_HEADER = true;
    public static final boolean DONT_FILL_HEADER = false;
    public static final int MULTI_BYTE_BUFFER_HEAP_SIZE = (int)ClassSize.estimateBase(MultiByteBuff.class, false);
    static final int BLOCK_METADATA_SPACE = 13;
    static final int CHECKSUM_SIZE = 4;
    static final byte[] DUMMY_HEADER_NO_CHECKSUM = new byte[24];
    static final CacheableDeserializer<Cacheable> BLOCK_DESERIALIZER = new CacheableDeserializer<Cacheable>(){

        @Override
        public HFileBlock deserialize(ByteBuff buf, boolean reuse, Cacheable.MemoryType memType) throws IOException {
            ByteBuff newByteBuff;
            buf.limit(buf.limit() - 13).rewind();
            if (reuse) {
                newByteBuff = buf.slice();
            } else {
                int len = buf.limit();
                newByteBuff = new SingleByteBuff(ByteBuffer.allocate(len));
                newByteBuff.put(0, buf, buf.position(), len);
            }
            buf.position(buf.limit());
            buf.limit(buf.limit() + 13);
            boolean usesChecksum = buf.get() == 1;
            long offset = buf.getLong();
            int nextBlockOnDiskSize = buf.getInt();
            HFileBlock hFileBlock = new HFileBlock(newByteBuff, usesChecksum, memType, offset, nextBlockOnDiskSize, null);
            return hFileBlock;
        }

        @Override
        public int getDeserialiserIdentifier() {
            return DESERIALIZER_IDENTIFIER;
        }

        @Override
        public HFileBlock deserialize(ByteBuff b) throws IOException {
            return this.deserialize(b, false, Cacheable.MemoryType.EXCLUSIVE);
        }
    };
    private static final int DESERIALIZER_IDENTIFIER = CacheableDeserializerIdManager.registerDeserializer(BLOCK_DESERIALIZER);

    private HFileBlock(HFileBlock that) {
        this(that, false);
    }

    private HFileBlock(HFileBlock that, boolean bufCopy) {
        this.init(that.blockType, that.onDiskSizeWithoutHeader, that.uncompressedSizeWithoutHeader, that.prevBlockOffset, that.offset, that.onDiskDataSizeWithHeader, that.nextBlockOnDiskSize, that.fileContext);
        this.buf = bufCopy ? new SingleByteBuff(ByteBuffer.wrap(that.buf.toBytes(0, that.buf.limit()))) : that.buf.duplicate();
    }

    HFileBlock(BlockType blockType, int onDiskSizeWithoutHeader, int uncompressedSizeWithoutHeader, long prevBlockOffset, ByteBuffer b, boolean fillHeader, long offset, int nextBlockOnDiskSize, int onDiskDataSizeWithHeader, HFileContext fileContext) {
        this.init(blockType, onDiskSizeWithoutHeader, uncompressedSizeWithoutHeader, prevBlockOffset, offset, onDiskDataSizeWithHeader, nextBlockOnDiskSize, fileContext);
        this.buf = new SingleByteBuff(b);
        if (fillHeader) {
            this.overwriteHeader();
        }
        this.buf.rewind();
    }

    HFileBlock(ByteBuff buf, boolean usesHBaseChecksum, Cacheable.MemoryType memType, long offset, int nextBlockOnDiskSize, HFileContext fileContext) throws IOException {
        int onDiskDataSizeWithHeader;
        buf.rewind();
        BlockType blockType = BlockType.read(buf);
        int onDiskSizeWithoutHeader = buf.getInt(Header.ON_DISK_SIZE_WITHOUT_HEADER_INDEX);
        int uncompressedSizeWithoutHeader = buf.getInt(Header.UNCOMPRESSED_SIZE_WITHOUT_HEADER_INDEX);
        long prevBlockOffset = buf.getLong(Header.PREV_BLOCK_OFFSET_INDEX);
        HFileContextBuilder fileContextBuilder = fileContext != null ? new HFileContextBuilder(fileContext) : new HFileContextBuilder();
        fileContextBuilder.withHBaseCheckSum(usesHBaseChecksum);
        if (usesHBaseChecksum) {
            byte checksumType = buf.get(Header.CHECKSUM_TYPE_INDEX);
            int bytesPerChecksum = buf.getInt(Header.BYTES_PER_CHECKSUM_INDEX);
            onDiskDataSizeWithHeader = buf.getInt(Header.ON_DISK_DATA_SIZE_WITH_HEADER_INDEX);
            fileContextBuilder.withChecksumType(ChecksumType.codeToType(checksumType));
            fileContextBuilder.withBytesPerCheckSum(bytesPerChecksum);
        } else {
            fileContextBuilder.withChecksumType(ChecksumType.NULL);
            fileContextBuilder.withBytesPerCheckSum(0);
            onDiskDataSizeWithHeader = onDiskSizeWithoutHeader + HFileBlock.headerSize(usesHBaseChecksum);
        }
        fileContext = fileContextBuilder.build();
        assert (usesHBaseChecksum == fileContext.isUseHBaseChecksum());
        this.init(blockType, onDiskSizeWithoutHeader, uncompressedSizeWithoutHeader, prevBlockOffset, offset, onDiskDataSizeWithHeader, nextBlockOnDiskSize, fileContext);
        this.memType = memType;
        this.offset = offset;
        this.buf = buf;
        this.buf.rewind();
    }

    private void init(BlockType blockType, int onDiskSizeWithoutHeader, int uncompressedSizeWithoutHeader, long prevBlockOffset, long offset, int onDiskDataSizeWithHeader, int nextBlockOnDiskSize, HFileContext fileContext) {
        this.blockType = blockType;
        this.onDiskSizeWithoutHeader = onDiskSizeWithoutHeader;
        this.uncompressedSizeWithoutHeader = uncompressedSizeWithoutHeader;
        this.prevBlockOffset = prevBlockOffset;
        this.offset = offset;
        this.onDiskDataSizeWithHeader = onDiskDataSizeWithHeader;
        this.nextBlockOnDiskSize = nextBlockOnDiskSize;
        this.fileContext = fileContext;
    }

    private static int getOnDiskSizeWithHeader(ByteBuffer headerBuf, boolean verifyChecksum) {
        return headerBuf.getInt(Header.ON_DISK_SIZE_WITHOUT_HEADER_INDEX) + HFileBlock.headerSize(verifyChecksum);
    }

    int getNextBlockOnDiskSize() {
        return this.nextBlockOnDiskSize;
    }

    @Override
    public BlockType getBlockType() {
        return this.blockType;
    }

    short getDataBlockEncodingId() {
        if (this.blockType != BlockType.ENCODED_DATA) {
            throw new IllegalArgumentException("Querying encoder ID of a block of type other than " + (Object)((Object)BlockType.ENCODED_DATA) + ": " + (Object)((Object)this.blockType));
        }
        return this.buf.getShort(this.headerSize());
    }

    public int getOnDiskSizeWithHeader() {
        return this.onDiskSizeWithoutHeader + this.headerSize();
    }

    int getOnDiskSizeWithoutHeader() {
        return this.onDiskSizeWithoutHeader;
    }

    int getUncompressedSizeWithoutHeader() {
        return this.uncompressedSizeWithoutHeader;
    }

    long getPrevBlockOffset() {
        return this.prevBlockOffset;
    }

    private void overwriteHeader() {
        this.buf.rewind();
        this.blockType.write(this.buf);
        this.buf.putInt(this.onDiskSizeWithoutHeader);
        this.buf.putInt(this.uncompressedSizeWithoutHeader);
        this.buf.putLong(this.prevBlockOffset);
        if (this.fileContext.isUseHBaseChecksum()) {
            this.buf.put(this.fileContext.getChecksumType().getCode());
            this.buf.putInt(this.fileContext.getBytesPerChecksum());
            this.buf.putInt(this.onDiskDataSizeWithHeader);
        }
    }

    public ByteBuff getBufferWithoutHeader() {
        ByteBuff dup = this.getBufferReadOnly();
        return dup.position(this.headerSize()).limit(this.buf.limit() - this.totalChecksumBytes()).slice();
    }

    public ByteBuff getBufferReadOnly() {
        ByteBuff dup = this.buf.duplicate();
        assert (dup.position() == 0);
        return dup;
    }

    @VisibleForTesting
    private void sanityCheckAssertion(long valueFromBuf, long valueFromField, String fieldName) throws IOException {
        if (valueFromBuf != valueFromField) {
            throw new AssertionError((Object)(fieldName + " in the buffer (" + valueFromBuf + ") is different from that in the field (" + valueFromField + ")"));
        }
    }

    @VisibleForTesting
    private void sanityCheckAssertion(BlockType valueFromBuf, BlockType valueFromField) throws IOException {
        if (valueFromBuf != valueFromField) {
            throw new IOException("Block type stored in the buffer: " + (Object)((Object)valueFromBuf) + ", block type field: " + (Object)((Object)valueFromField));
        }
    }

    @VisibleForTesting
    void sanityCheck() throws IOException {
        ByteBuff dup = this.buf.duplicate().rewind();
        this.sanityCheckAssertion(BlockType.read(dup), this.blockType);
        this.sanityCheckAssertion(dup.getInt(), this.onDiskSizeWithoutHeader, "onDiskSizeWithoutHeader");
        this.sanityCheckAssertion(dup.getInt(), this.uncompressedSizeWithoutHeader, "uncompressedSizeWithoutHeader");
        this.sanityCheckAssertion(dup.getLong(), this.prevBlockOffset, "prevBlockOffset");
        if (this.fileContext.isUseHBaseChecksum()) {
            this.sanityCheckAssertion(dup.get(), this.fileContext.getChecksumType().getCode(), "checksumType");
            this.sanityCheckAssertion(dup.getInt(), this.fileContext.getBytesPerChecksum(), "bytesPerChecksum");
            this.sanityCheckAssertion(dup.getInt(), this.onDiskDataSizeWithHeader, "onDiskDataSizeWithHeader");
        }
        int cksumBytes = this.totalChecksumBytes();
        int expectedBufLimit = this.onDiskDataSizeWithHeader + cksumBytes;
        if (dup.limit() != expectedBufLimit) {
            throw new AssertionError((Object)("Expected limit " + expectedBufLimit + ", got " + dup.limit()));
        }
        int hdrSize = this.headerSize();
        if (dup.capacity() != expectedBufLimit && dup.capacity() != expectedBufLimit + hdrSize) {
            throw new AssertionError((Object)("Invalid buffer capacity: " + dup.capacity() + ", expected " + expectedBufLimit + " or " + (expectedBufLimit + hdrSize)));
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder().append("[").append("blockType=").append((Object)this.blockType).append(", fileOffset=").append(this.offset).append(", headerSize=").append(this.headerSize()).append(", onDiskSizeWithoutHeader=").append(this.onDiskSizeWithoutHeader).append(", uncompressedSizeWithoutHeader=").append(this.uncompressedSizeWithoutHeader).append(", prevBlockOffset=").append(this.prevBlockOffset).append(", isUseHBaseChecksum=").append(this.fileContext.isUseHBaseChecksum());
        if (this.fileContext.isUseHBaseChecksum()) {
            sb.append(", checksumType=").append((Object)ChecksumType.codeToType(this.buf.get(24))).append(", bytesPerChecksum=").append(this.buf.getInt(25)).append(", onDiskDataSizeWithHeader=").append(this.onDiskDataSizeWithHeader);
        } else {
            sb.append(", onDiskDataSizeWithHeader=").append(this.onDiskDataSizeWithHeader).append("(").append(this.onDiskSizeWithoutHeader).append("+").append(24).append(")");
        }
        String dataBegin = null;
        if (this.buf.hasArray()) {
            dataBegin = Bytes.toStringBinary(this.buf.array(), this.buf.arrayOffset() + this.headerSize(), Math.min(32, this.buf.limit() - this.buf.arrayOffset() - this.headerSize()));
        } else {
            ByteBuff bufWithoutHeader = this.getBufferWithoutHeader();
            byte[] dataBeginBytes = new byte[Math.min(32, bufWithoutHeader.limit() - bufWithoutHeader.position())];
            bufWithoutHeader.get(dataBeginBytes);
            dataBegin = Bytes.toStringBinary(dataBeginBytes);
        }
        sb.append(", getOnDiskSizeWithHeader=").append(this.getOnDiskSizeWithHeader()).append(", totalChecksumBytes=").append(this.totalChecksumBytes()).append(", isUnpacked=").append(this.isUnpacked()).append(", buf=[").append(this.buf).append("]").append(", dataBeginsWith=").append(dataBegin).append(", fileContext=").append(this.fileContext).append("]");
        return sb.toString();
    }

    HFileBlock unpack(HFileContext fileContext, FSReader reader) throws IOException {
        if (!fileContext.isCompressedOrEncrypted()) {
            return this;
        }
        HFileBlock unpacked = new HFileBlock(this);
        unpacked.allocateBuffer();
        HFileBlockDecodingContext ctx = this.blockType == BlockType.ENCODED_DATA ? reader.getBlockDecodingContext() : reader.getDefaultBlockDecodingContext();
        ByteBuff dup = this.buf.duplicate();
        dup.position(this.headerSize());
        dup = dup.slice();
        ctx.prepareDecoding(unpacked.getOnDiskSizeWithoutHeader(), unpacked.getUncompressedSizeWithoutHeader(), unpacked.getBufferWithoutHeader(), dup);
        return unpacked;
    }

    private void allocateBuffer() {
        int cksumBytes = this.totalChecksumBytes();
        int headerSize = this.headerSize();
        int capacityNeeded = headerSize + this.uncompressedSizeWithoutHeader + cksumBytes;
        ByteBuffer newBuf = ByteBuffer.allocate(capacityNeeded);
        this.buf.position(0);
        this.buf.get(newBuf.array(), newBuf.arrayOffset(), headerSize);
        this.buf = new SingleByteBuff(newBuf);
        this.buf.limit(headerSize + this.uncompressedSizeWithoutHeader + cksumBytes);
    }

    public boolean isUnpacked() {
        int cksumBytes = this.totalChecksumBytes();
        int headerSize = this.headerSize();
        int expectedCapacity = headerSize + this.uncompressedSizeWithoutHeader + cksumBytes;
        int bufCapacity = this.buf.capacity();
        return bufCapacity == expectedCapacity || bufCapacity == expectedCapacity + headerSize;
    }

    @VisibleForTesting
    void sanityCheckUncompressedSize() throws IOException {
        if (this.onDiskSizeWithoutHeader != this.uncompressedSizeWithoutHeader + this.totalChecksumBytes()) {
            throw new IOException("Using no compression but onDiskSizeWithoutHeader=" + this.onDiskSizeWithoutHeader + ", uncompressedSizeWithoutHeader=" + this.uncompressedSizeWithoutHeader + ", numChecksumbytes=" + this.totalChecksumBytes());
        }
    }

    long getOffset() {
        if (this.offset < 0L) {
            throw new IllegalStateException("HFile block offset not initialized properly");
        }
        return this.offset;
    }

    DataInputStream getByteStream() {
        ByteBuff dup = this.buf.duplicate();
        dup.position(this.headerSize());
        return new DataInputStream(new ByteBuffInputStream(dup));
    }

    @Override
    public long heapSize() {
        long size = ClassSize.align((long)(ClassSize.OBJECT + 4 * ClassSize.REFERENCE + 16 + 16) + this.fileContext.heapSize());
        if (this.buf != null) {
            size += (long)ClassSize.align(this.buf.capacity() + MULTI_BYTE_BUFFER_HEAP_SIZE);
        }
        return ClassSize.align(size);
    }

    static boolean readWithExtra(InputStream in, byte[] buf, int bufOffset, int necessaryLen, int extraLen) throws IOException {
        int bytesRemaining;
        int ret;
        for (bytesRemaining = necessaryLen + extraLen; bytesRemaining > 0 && ((ret = in.read(buf, bufOffset, bytesRemaining)) != -1 || bytesRemaining > extraLen); bytesRemaining -= ret) {
            if (ret < 0) {
                throw new IOException("Premature EOF from inputStream (read returned " + ret + ", was trying to read " + necessaryLen + " necessary bytes and " + extraLen + " extra bytes, successfully read " + (necessaryLen + extraLen - bytesRemaining));
            }
            bufOffset += ret;
        }
        return bytesRemaining <= 0;
    }

    @VisibleForTesting
    static boolean positionalReadWithExtra(FSDataInputStream in, long position, byte[] buf, int bufOffset, int necessaryLen, int extraLen) throws IOException {
        int bytesRead;
        int ret;
        int bytesRemaining = necessaryLen + extraLen;
        for (bytesRead = 0; bytesRead < necessaryLen; bytesRead += ret) {
            ret = in.read(position, buf, bufOffset, bytesRemaining);
            if (ret < 0) {
                throw new IOException("Premature EOF from inputStream (positional read returned " + ret + ", was trying to read " + necessaryLen + " necessary bytes and " + extraLen + " extra bytes, successfully read " + bytesRead);
            }
            position += (long)ret;
            bufOffset += ret;
            bytesRemaining -= ret;
        }
        return bytesRead != necessaryLen && bytesRemaining <= 0;
    }

    @VisibleForTesting
    void sanityCheckUncompressed() throws IOException {
        if (this.onDiskSizeWithoutHeader != this.uncompressedSizeWithoutHeader + this.totalChecksumBytes()) {
            throw new IOException("Using no compression but onDiskSizeWithoutHeader=" + this.onDiskSizeWithoutHeader + ", uncompressedSizeWithoutHeader=" + this.uncompressedSizeWithoutHeader + ", numChecksumbytes=" + this.totalChecksumBytes());
        }
    }

    @Override
    public int getSerializedLength() {
        if (this.buf != null) {
            return this.buf.limit() + 13;
        }
        return 0;
    }

    @Override
    public void serialize(ByteBuffer destination) {
        this.buf.get(destination, 0, this.getSerializedLength() - 13);
        destination = this.addMetaData(destination);
        destination.flip();
    }

    public ByteBuffer getMetaData() {
        ByteBuffer bb = ByteBuffer.allocate(13);
        bb = this.addMetaData(bb);
        bb.flip();
        return bb;
    }

    private ByteBuffer addMetaData(ByteBuffer destination) {
        destination.put(this.fileContext.isUseHBaseChecksum() ? (byte)1 : 0);
        destination.putLong(this.offset);
        destination.putInt(this.nextBlockOnDiskSize);
        return destination;
    }

    @Override
    public CacheableDeserializer<Cacheable> getDeserializer() {
        return BLOCK_DESERIALIZER;
    }

    public int hashCode() {
        int result = 1;
        result = result * 31 + this.blockType.hashCode();
        result = result * 31 + this.nextBlockOnDiskSize;
        result = result * 31 + (int)(this.offset ^ this.offset >>> 32);
        result = result * 31 + this.onDiskSizeWithoutHeader;
        result = result * 31 + (int)(this.prevBlockOffset ^ this.prevBlockOffset >>> 32);
        result = result * 31 + this.uncompressedSizeWithoutHeader;
        result = result * 31 + this.buf.hashCode();
        return result;
    }

    public boolean equals(Object comparison) {
        if (this == comparison) {
            return true;
        }
        if (comparison == null) {
            return false;
        }
        if (comparison.getClass() != this.getClass()) {
            return false;
        }
        HFileBlock castedComparison = (HFileBlock)comparison;
        if (castedComparison.blockType != this.blockType) {
            return false;
        }
        if (castedComparison.nextBlockOnDiskSize != this.nextBlockOnDiskSize) {
            return false;
        }
        if (castedComparison.offset != this.offset) {
            return false;
        }
        if (castedComparison.onDiskSizeWithoutHeader != this.onDiskSizeWithoutHeader) {
            return false;
        }
        if (castedComparison.prevBlockOffset != this.prevBlockOffset) {
            return false;
        }
        if (castedComparison.uncompressedSizeWithoutHeader != this.uncompressedSizeWithoutHeader) {
            return false;
        }
        return ByteBuff.compareTo(this.buf, 0, this.buf.limit(), castedComparison.buf, 0, castedComparison.buf.limit()) == 0;
    }

    DataBlockEncoding getDataBlockEncoding() {
        if (this.blockType == BlockType.ENCODED_DATA) {
            return DataBlockEncoding.getEncodingById(this.getDataBlockEncodingId());
        }
        return DataBlockEncoding.NONE;
    }

    @VisibleForTesting
    byte getChecksumType() {
        return this.fileContext.getChecksumType().getCode();
    }

    int getBytesPerChecksum() {
        return this.fileContext.getBytesPerChecksum();
    }

    @VisibleForTesting
    int getOnDiskDataSizeWithHeader() {
        return this.onDiskDataSizeWithHeader;
    }

    int totalChecksumBytes() {
        if (!this.fileContext.isUseHBaseChecksum() || this.fileContext.getBytesPerChecksum() == 0) {
            return 0;
        }
        return (int)ChecksumUtil.numBytes(this.onDiskDataSizeWithHeader, this.fileContext.getBytesPerChecksum());
    }

    public int headerSize() {
        return HFileBlock.headerSize(this.fileContext.isUseHBaseChecksum());
    }

    public static int headerSize(boolean usesHBaseChecksum) {
        return usesHBaseChecksum ? 33 : 24;
    }

    @VisibleForTesting
    byte[] getDummyHeaderForVersion() {
        return HFileBlock.getDummyHeaderForVersion(this.fileContext.isUseHBaseChecksum());
    }

    private static byte[] getDummyHeaderForVersion(boolean usesHBaseChecksum) {
        return usesHBaseChecksum ? HConstants.HFILEBLOCK_DUMMY_HEADER : DUMMY_HEADER_NO_CHECKSUM;
    }

    HFileContext getHFileContext() {
        return this.fileContext;
    }

    @Override
    public Cacheable.MemoryType getMemoryType() {
        return this.memType;
    }

    boolean usesSharedMemory() {
        return this.memType == Cacheable.MemoryType.SHARED;
    }

    @VisibleForTesting
    static String toStringHeader(ByteBuff buf) throws IOException {
        byte[] magicBuf = new byte[Math.min(buf.limit() - buf.position(), 8)];
        buf.get(magicBuf);
        BlockType bt = BlockType.parse(magicBuf, 0, 8);
        int compressedBlockSizeNoHeader = buf.getInt();
        int uncompressedBlockSizeNoHeader = buf.getInt();
        long prevBlockOffset = buf.getLong();
        byte cksumtype = buf.get();
        long bytesPerChecksum = buf.getInt();
        long onDiskDataSizeWithHeader = buf.getInt();
        return " Header dump: magic: " + Bytes.toString(magicBuf) + " blockType " + (Object)((Object)bt) + " compressedBlockSizeNoHeader " + compressedBlockSizeNoHeader + " uncompressedBlockSizeNoHeader " + uncompressedBlockSizeNoHeader + " prevBlockOffset " + prevBlockOffset + " checksumType " + (Object)((Object)ChecksumType.codeToType(cksumtype)) + " bytesPerChecksum " + bytesPerChecksum + " onDiskDataSizeWithHeader " + onDiskDataSizeWithHeader;
    }

    public HFileBlock deepClone() {
        return new HFileBlock(this, true);
    }

    static class FSReaderImpl
    implements FSReader {
        private FSDataInputStreamWrapper streamWrapper;
        private HFileBlockDecodingContext encodedBlockDecodingCtx;
        private final HFileBlockDefaultDecodingContext defaultDecodingCtx;
        private AtomicReference<PrefetchedHeader> prefetchedHeader = new AtomicReference<PrefetchedHeader>(new PrefetchedHeader());
        private long fileSize;
        @VisibleForTesting
        protected final int hdrSize;
        private HFileSystem hfs;
        private HFileContext fileContext;
        private String pathName;
        private final Lock streamLock = new ReentrantLock();

        FSReaderImpl(FSDataInputStreamWrapper stream, long fileSize, HFileSystem hfs, Path path, HFileContext fileContext) throws IOException {
            this.fileSize = fileSize;
            this.hfs = hfs;
            if (path != null) {
                this.pathName = path.toString();
            }
            this.fileContext = fileContext;
            this.hdrSize = HFileBlock.headerSize(fileContext.isUseHBaseChecksum());
            this.streamWrapper = stream;
            this.streamWrapper.prepareForBlockReader(!fileContext.isUseHBaseChecksum());
            this.defaultDecodingCtx = new HFileBlockDefaultDecodingContext(fileContext);
            this.encodedBlockDecodingCtx = this.defaultDecodingCtx;
        }

        FSReaderImpl(FSDataInputStream istream, long fileSize, HFileContext fileContext) throws IOException {
            this(new FSDataInputStreamWrapper(istream), fileSize, null, null, fileContext);
        }

        @Override
        public BlockIterator blockRange(final long startOffset, final long endOffset) {
            final FSReaderImpl owner = this;
            return new BlockIterator(){
                private long offset;
                private long length;
                {
                    this.offset = startOffset;
                    this.length = -1L;
                }

                @Override
                public HFileBlock nextBlock() throws IOException {
                    if (this.offset >= endOffset) {
                        return null;
                    }
                    HFileBlock b = this.readBlockData(this.offset, this.length, false, false);
                    this.offset += (long)b.getOnDiskSizeWithHeader();
                    this.length = b.getNextBlockOnDiskSize();
                    return b.unpack(fileContext, owner);
                }

                @Override
                public HFileBlock nextBlockWithBlockType(BlockType blockType) throws IOException {
                    HFileBlock blk = this.nextBlock();
                    if (blk.getBlockType() != blockType) {
                        throw new IOException("Expected block of type " + (Object)((Object)blockType) + " but found " + (Object)((Object)blk.getBlockType()));
                    }
                    return blk;
                }
            };
        }

        @VisibleForTesting
        protected int readAtOffset(FSDataInputStream istream, byte[] dest, int destOffset, int size, boolean peekIntoNextBlock, long fileOffset, boolean pread) throws IOException {
            if (peekIntoNextBlock && destOffset + size + this.hdrSize > dest.length) {
                throw new IOException("Attempted to read " + size + " bytes and " + this.hdrSize + " bytes of next header into a " + dest.length + "-byte array at offset " + destOffset);
            }
            if (!pread) {
                HFileUtil.seekOnMultipleSources(istream, fileOffset);
                long realOffset = istream.getPos();
                if (realOffset != fileOffset) {
                    throw new IOException("Tried to seek to " + fileOffset + " to read " + size + " bytes, but pos=" + realOffset + " after seek");
                }
                if (!peekIntoNextBlock) {
                    IOUtils.readFully(istream, dest, destOffset, size);
                    return -1;
                }
                if (!HFileBlock.readWithExtra(istream, dest, destOffset, size, this.hdrSize)) {
                    return -1;
                }
            } else {
                int extraSize;
                int n = extraSize = peekIntoNextBlock ? this.hdrSize : 0;
                if (!HFileBlock.positionalReadWithExtra(istream, fileOffset, dest, destOffset, size, extraSize)) {
                    return -1;
                }
            }
            assert (peekIntoNextBlock);
            return Bytes.toInt(dest, destOffset + size + 8) + this.hdrSize;
        }

        @Override
        public HFileBlock readBlockData(long offset, long onDiskSizeWithHeaderL, boolean pread, boolean updateMetrics) throws IOException {
            boolean doVerificationThruHBaseChecksum = this.streamWrapper.shouldUseHBaseChecksum();
            FSDataInputStream is = this.streamWrapper.getStream(doVerificationThruHBaseChecksum);
            HFileBlock blk = this.readBlockDataInternal(is, offset, onDiskSizeWithHeaderL, pread, doVerificationThruHBaseChecksum, updateMetrics);
            if (blk == null) {
                HFile.LOG.warn((Object)("HBase checksum verification failed for file " + this.pathName + " at offset " + offset + " filesize " + this.fileSize + ". Retrying read with HDFS checksums turned on..."));
                if (!doVerificationThruHBaseChecksum) {
                    String msg = "HBase checksum verification failed for file " + this.pathName + " at offset " + offset + " filesize " + this.fileSize + " but this cannot happen because doVerify is " + doVerificationThruHBaseChecksum;
                    HFile.LOG.warn((Object)msg);
                    throw new IOException(msg);
                }
                HFile.CHECKSUM_FAILURES.increment();
                is = this.streamWrapper.fallbackToFsChecksum(3);
                doVerificationThruHBaseChecksum = false;
                blk = this.readBlockDataInternal(is, offset, onDiskSizeWithHeaderL, pread, doVerificationThruHBaseChecksum, updateMetrics);
                if (blk != null) {
                    HFile.LOG.warn((Object)("HDFS checksum verification succeeded for file " + this.pathName + " at offset " + offset + " filesize " + this.fileSize));
                }
            }
            if (blk == null && !doVerificationThruHBaseChecksum) {
                String msg = "readBlockData failed, possibly due to checksum verification failed for file " + this.pathName + " at offset " + offset + " filesize " + this.fileSize;
                HFile.LOG.warn((Object)msg);
                throw new IOException(msg);
            }
            this.streamWrapper.checksumOk();
            return blk;
        }

        private static int checkAndGetSizeAsInt(long onDiskSizeWithHeaderL, int hdrSize) throws IOException {
            if (onDiskSizeWithHeaderL < (long)hdrSize && onDiskSizeWithHeaderL != -1L || onDiskSizeWithHeaderL >= Integer.MAX_VALUE) {
                throw new IOException("Invalid onDisksize=" + onDiskSizeWithHeaderL + ": expected to be at least " + hdrSize + " and at most " + Integer.MAX_VALUE + ", or -1");
            }
            return (int)onDiskSizeWithHeaderL;
        }

        private void verifyOnDiskSizeMatchesHeader(int passedIn, ByteBuffer headerBuf, long offset, boolean verifyChecksum) throws IOException {
            int fromHeader = HFileBlock.getOnDiskSizeWithHeader(headerBuf, verifyChecksum);
            if (passedIn != fromHeader) {
                throw new IOException("Passed in onDiskSizeWithHeader=" + passedIn + " != " + fromHeader + ", offset=" + offset + ", fileContext=" + this.fileContext);
            }
        }

        private ByteBuffer getCachedHeader(long offset) {
            PrefetchedHeader ph = this.prefetchedHeader.get();
            return ph != null && ph.offset == offset ? ph.buf : null;
        }

        private void cacheNextBlockHeader(long offset, byte[] header, int headerOffset, int headerLength) {
            PrefetchedHeader ph = new PrefetchedHeader();
            ph.offset = offset;
            System.arraycopy(header, headerOffset, ph.header, 0, headerLength);
            this.prefetchedHeader.set(ph);
        }

        @VisibleForTesting
        protected HFileBlock readBlockDataInternal(FSDataInputStream is, long offset, long onDiskSizeWithHeaderL, boolean pread, boolean verifyChecksum, boolean updateMetrics) throws IOException {
            if (offset < 0L) {
                throw new IOException("Invalid offset=" + offset + " trying to read block (onDiskSize=" + onDiskSizeWithHeaderL + ")");
            }
            int onDiskSizeWithHeader = FSReaderImpl.checkAndGetSizeAsInt(onDiskSizeWithHeaderL, this.hdrSize);
            ByteBuffer headerBuf = this.getCachedHeader(offset);
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("Reading " + this.fileContext.getHFileName() + " at offset=" + offset + ", pread=" + pread + ", verifyChecksum=" + verifyChecksum + ", cachedHeader=" + headerBuf + ", onDiskSizeWithHeader=" + onDiskSizeWithHeader));
            }
            boolean checksumSupport = this.fileContext.isUseHBaseChecksum();
            long startTime = System.currentTimeMillis();
            if (onDiskSizeWithHeader <= 0) {
                if (headerBuf == null) {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace((Object)"Extra see to get block size!", (Throwable)new RuntimeException());
                    }
                    headerBuf = ByteBuffer.allocate(this.hdrSize);
                    this.readAtOffset(is, headerBuf.array(), headerBuf.arrayOffset(), this.hdrSize, false, offset, pread);
                }
                onDiskSizeWithHeader = HFileBlock.getOnDiskSizeWithHeader(headerBuf, checksumSupport);
            }
            int preReadHeaderSize = headerBuf == null ? 0 : this.hdrSize;
            byte[] onDiskBlock = new byte[onDiskSizeWithHeader + this.hdrSize];
            int nextBlockOnDiskSize = this.readAtOffset(is, onDiskBlock, preReadHeaderSize, onDiskSizeWithHeader - preReadHeaderSize, true, offset + (long)preReadHeaderSize, pread);
            if (headerBuf != null) {
                System.arraycopy(headerBuf.array(), headerBuf.arrayOffset(), onDiskBlock, 0, this.hdrSize);
            } else {
                headerBuf = ByteBuffer.wrap(onDiskBlock, 0, this.hdrSize);
            }
            assert (onDiskSizeWithHeader > this.hdrSize);
            this.verifyOnDiskSizeMatchesHeader(onDiskSizeWithHeader, headerBuf, offset, checksumSupport);
            ByteBuffer onDiskBlockByteBuffer = ByteBuffer.wrap(onDiskBlock, 0, onDiskSizeWithHeader);
            if (verifyChecksum && !this.validateChecksum(offset, onDiskBlockByteBuffer, this.hdrSize)) {
                return null;
            }
            long duration = System.currentTimeMillis() - startTime;
            if (updateMetrics) {
                HFile.updateReadLatency(duration, pread);
            }
            HFileBlock hFileBlock = new HFileBlock(new SingleByteBuff(onDiskBlockByteBuffer), checksumSupport, Cacheable.MemoryType.EXCLUSIVE, offset, nextBlockOnDiskSize, this.fileContext);
            if (!this.fileContext.isCompressedOrEncrypted()) {
                hFileBlock.sanityCheckUncompressed();
            }
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("Read " + hFileBlock + " in " + duration + " ns"));
            }
            if (nextBlockOnDiskSize != -1) {
                this.cacheNextBlockHeader(offset + (long)hFileBlock.getOnDiskSizeWithHeader(), onDiskBlock, onDiskSizeWithHeader, this.hdrSize);
            }
            return hFileBlock;
        }

        @Override
        public void setIncludesMemstoreTS(boolean includesMemstoreTS) {
            this.fileContext.setIncludesMvcc(includesMemstoreTS);
        }

        @Override
        public void setDataBlockEncoder(HFileDataBlockEncoder encoder) {
            this.encodedBlockDecodingCtx = encoder.newDataBlockDecodingContext(this.fileContext);
        }

        @Override
        public HFileBlockDecodingContext getBlockDecodingContext() {
            return this.encodedBlockDecodingCtx;
        }

        @Override
        public HFileBlockDecodingContext getDefaultBlockDecodingContext() {
            return this.defaultDecodingCtx;
        }

        private boolean validateChecksum(long offset, ByteBuffer data, int hdrSize) throws IOException {
            if (!this.fileContext.isUseHBaseChecksum()) {
                return false;
            }
            return ChecksumUtil.validateChecksum(data, this.pathName, offset, hdrSize);
        }

        @Override
        public void closeStreams() throws IOException {
            this.streamWrapper.close();
        }

        @Override
        public void unbufferStream() {
            if (this.streamLock.tryLock()) {
                try {
                    this.streamWrapper.unbuffer();
                }
                finally {
                    this.streamLock.unlock();
                }
            }
        }

        public String toString() {
            return "hfs=" + this.hfs + ", path=" + this.pathName + ", fileContext=" + this.fileContext;
        }
    }

    private static class PrefetchedHeader {
        long offset = -1L;
        byte[] header = new byte[33];
        final ByteBuffer buf = ByteBuffer.wrap(this.header, 0, 33);

        private PrefetchedHeader() {
        }

        public String toString() {
            return "offset=" + this.offset + ", header=" + Bytes.toStringBinary(this.header);
        }
    }

    static interface FSReader {
        public HFileBlock readBlockData(long var1, long var3, boolean var5, boolean var6) throws IOException;

        public BlockIterator blockRange(long var1, long var3);

        public void closeStreams() throws IOException;

        public HFileBlockDecodingContext getBlockDecodingContext();

        public HFileBlockDecodingContext getDefaultBlockDecodingContext();

        public void setIncludesMemstoreTS(boolean var1);

        public void setDataBlockEncoder(HFileDataBlockEncoder var1);

        public void unbufferStream();
    }

    static interface BlockIterator {
        public HFileBlock nextBlock() throws IOException;

        public HFileBlock nextBlockWithBlockType(BlockType var1) throws IOException;
    }

    static interface BlockWritable {
        public BlockType getBlockType();

        public void writeToBlock(DataOutput var1) throws IOException;
    }

    static class Writer {
        private State state = State.INIT;
        private final HFileDataBlockEncoder dataBlockEncoder;
        private HFileBlockEncodingContext dataBlockEncodingCtx;
        private HFileBlockDefaultEncodingContext defaultBlockEncodingCtx;
        private ByteArrayOutputStream baosInMemory;
        private BlockType blockType;
        private DataOutputStream userDataStream;
        private int unencodedDataSizeWritten;
        private int encodedDataSizeWritten;
        private ByteArrayOutputStream onDiskBlockBytesWithHeader;
        private byte[] onDiskChecksum = HConstants.EMPTY_BYTE_ARRAY;
        private long startOffset;
        private long[] prevOffsetByType;
        private long prevOffset;
        private HFileContext fileContext;

        public Writer(HFileDataBlockEncoder dataBlockEncoder, HFileContext fileContext) {
            if (fileContext.getBytesPerChecksum() < 33) {
                throw new RuntimeException("Unsupported value of bytesPerChecksum.  Minimum is 33 but the configured value is " + fileContext.getBytesPerChecksum());
            }
            this.dataBlockEncoder = dataBlockEncoder != null ? dataBlockEncoder : NoOpDataBlockEncoder.INSTANCE;
            this.dataBlockEncodingCtx = this.dataBlockEncoder.newDataBlockEncodingContext(HConstants.HFILEBLOCK_DUMMY_HEADER, fileContext);
            this.defaultBlockEncodingCtx = new HFileBlockDefaultEncodingContext(null, HConstants.HFILEBLOCK_DUMMY_HEADER, fileContext);
            this.baosInMemory = new ByteArrayOutputStream();
            this.prevOffsetByType = new long[BlockType.values().length];
            for (int i = 0; i < this.prevOffsetByType.length; ++i) {
                this.prevOffsetByType[i] = UNSET;
            }
            this.fileContext = fileContext;
        }

        DataOutputStream startWriting(BlockType newBlockType) throws IOException {
            if (this.state == State.BLOCK_READY && this.startOffset != -1L) {
                this.prevOffsetByType[this.blockType.getId()] = this.startOffset;
            }
            this.startOffset = -1L;
            this.blockType = newBlockType;
            this.baosInMemory.reset();
            this.baosInMemory.write(HConstants.HFILEBLOCK_DUMMY_HEADER);
            this.state = State.WRITING;
            this.userDataStream = new ByteBufferWriterDataOutputStream(this.baosInMemory);
            if (newBlockType == BlockType.DATA) {
                this.dataBlockEncoder.startBlockEncoding(this.dataBlockEncodingCtx, this.userDataStream);
            }
            this.unencodedDataSizeWritten = 0;
            this.encodedDataSizeWritten = 0;
            return this.userDataStream;
        }

        void write(Cell cell) throws IOException {
            this.expectState(State.WRITING);
            int posBeforeEncode = this.userDataStream.size();
            this.unencodedDataSizeWritten += this.dataBlockEncoder.encode(cell, this.dataBlockEncodingCtx, this.userDataStream);
            this.encodedDataSizeWritten += this.userDataStream.size() - posBeforeEncode;
        }

        DataOutputStream getUserDataStream() {
            this.expectState(State.WRITING);
            return this.userDataStream;
        }

        void ensureBlockReady() throws IOException {
            Preconditions.checkState(this.state != State.INIT, "Unexpected state: " + (Object)((Object)this.state));
            if (this.state == State.BLOCK_READY) {
                return;
            }
            this.finishBlock();
        }

        private void finishBlock() throws IOException {
            if (this.blockType == BlockType.DATA) {
                this.dataBlockEncoder.endBlockEncoding(this.dataBlockEncodingCtx, this.userDataStream, this.baosInMemory.getBuffer(), this.blockType);
                this.blockType = this.dataBlockEncodingCtx.getBlockType();
            }
            this.userDataStream.flush();
            this.prevOffset = this.prevOffsetByType[this.blockType.getId()];
            this.state = State.BLOCK_READY;
            Bytes compressAndEncryptDat = this.blockType == BlockType.DATA || this.blockType == BlockType.ENCODED_DATA ? this.dataBlockEncodingCtx.compressAndEncrypt(this.baosInMemory.getBuffer(), 0, this.baosInMemory.size()) : this.defaultBlockEncodingCtx.compressAndEncrypt(this.baosInMemory.getBuffer(), 0, this.baosInMemory.size());
            if (compressAndEncryptDat == null) {
                compressAndEncryptDat = new Bytes(this.baosInMemory.getBuffer(), 0, this.baosInMemory.size());
            }
            if (this.onDiskBlockBytesWithHeader == null) {
                this.onDiskBlockBytesWithHeader = new ByteArrayOutputStream(compressAndEncryptDat.getLength());
            }
            this.onDiskBlockBytesWithHeader.reset();
            this.onDiskBlockBytesWithHeader.write(compressAndEncryptDat.get(), compressAndEncryptDat.getOffset(), compressAndEncryptDat.getLength());
            int numBytes = (int)ChecksumUtil.numBytes(this.onDiskBlockBytesWithHeader.size(), this.fileContext.getBytesPerChecksum());
            this.putHeader(this.onDiskBlockBytesWithHeader, this.onDiskBlockBytesWithHeader.size() + numBytes, this.baosInMemory.size(), this.onDiskBlockBytesWithHeader.size());
            if (this.onDiskChecksum.length != numBytes) {
                this.onDiskChecksum = new byte[numBytes];
            }
            ChecksumUtil.generateChecksums(this.onDiskBlockBytesWithHeader.getBuffer(), 0, this.onDiskBlockBytesWithHeader.size(), this.onDiskChecksum, 0, this.fileContext.getChecksumType(), this.fileContext.getBytesPerChecksum());
        }

        private void putHeader(byte[] dest, int offset, int onDiskSize, int uncompressedSize, int onDiskDataSize) {
            offset = this.blockType.put(dest, offset);
            offset = Bytes.putInt(dest, offset, onDiskSize - 33);
            offset = Bytes.putInt(dest, offset, uncompressedSize - 33);
            offset = Bytes.putLong(dest, offset, this.prevOffset);
            offset = Bytes.putByte(dest, offset, this.fileContext.getChecksumType().getCode());
            offset = Bytes.putInt(dest, offset, this.fileContext.getBytesPerChecksum());
            Bytes.putInt(dest, offset, onDiskDataSize);
        }

        private void putHeader(ByteArrayOutputStream dest, int onDiskSize, int uncompressedSize, int onDiskDataSize) {
            this.putHeader(dest.getBuffer(), 0, onDiskSize, uncompressedSize, onDiskDataSize);
        }

        void writeHeaderAndData(FSDataOutputStream out) throws IOException {
            long offset = out.getPos();
            if (this.startOffset != (long)UNSET && offset != this.startOffset) {
                throw new IOException("A " + (Object)((Object)this.blockType) + " block written to a stream twice, first at offset " + this.startOffset + ", then at " + offset);
            }
            this.startOffset = offset;
            this.finishBlockAndWriteHeaderAndData(out);
        }

        protected void finishBlockAndWriteHeaderAndData(DataOutputStream out) throws IOException {
            this.ensureBlockReady();
            long startTime = System.currentTimeMillis();
            out.write(this.onDiskBlockBytesWithHeader.getBuffer(), 0, this.onDiskBlockBytesWithHeader.size());
            out.write(this.onDiskChecksum);
            HFile.updateWriteLatency(System.currentTimeMillis() - startTime);
        }

        byte[] getHeaderAndDataForTest() throws IOException {
            this.ensureBlockReady();
            byte[] output = new byte[this.onDiskBlockBytesWithHeader.size() + this.onDiskChecksum.length];
            System.arraycopy(this.onDiskBlockBytesWithHeader.getBuffer(), 0, output, 0, this.onDiskBlockBytesWithHeader.size());
            System.arraycopy(this.onDiskChecksum, 0, output, this.onDiskBlockBytesWithHeader.size(), this.onDiskChecksum.length);
            return output;
        }

        void release() {
            if (this.dataBlockEncodingCtx != null) {
                this.dataBlockEncodingCtx.close();
                this.dataBlockEncodingCtx = null;
            }
            if (this.defaultBlockEncodingCtx != null) {
                this.defaultBlockEncodingCtx.close();
                this.defaultBlockEncodingCtx = null;
            }
        }

        int getOnDiskSizeWithoutHeader() {
            this.expectState(State.BLOCK_READY);
            return this.onDiskBlockBytesWithHeader.size() + this.onDiskChecksum.length - 33;
        }

        int getOnDiskSizeWithHeader() {
            this.expectState(State.BLOCK_READY);
            return this.onDiskBlockBytesWithHeader.size() + this.onDiskChecksum.length;
        }

        int getUncompressedSizeWithoutHeader() {
            this.expectState(State.BLOCK_READY);
            return this.baosInMemory.size() - 33;
        }

        int getUncompressedSizeWithHeader() {
            this.expectState(State.BLOCK_READY);
            return this.baosInMemory.size();
        }

        boolean isWriting() {
            return this.state == State.WRITING;
        }

        public int encodedBlockSizeWritten() {
            if (this.state != State.WRITING) {
                return 0;
            }
            return this.encodedDataSizeWritten;
        }

        int blockSizeWritten() {
            if (this.state != State.WRITING) {
                return 0;
            }
            return this.unencodedDataSizeWritten;
        }

        @VisibleForTesting
        ByteBuffer cloneUncompressedBufferWithHeader() {
            this.expectState(State.BLOCK_READY);
            byte[] uncompressedBlockBytesWithHeader = this.baosInMemory.toByteArray();
            int numBytes = (int)ChecksumUtil.numBytes(this.onDiskBlockBytesWithHeader.size(), this.fileContext.getBytesPerChecksum());
            this.putHeader(uncompressedBlockBytesWithHeader, 0, this.onDiskBlockBytesWithHeader.size() + numBytes, this.baosInMemory.size(), this.onDiskBlockBytesWithHeader.size());
            return ByteBuffer.wrap(uncompressedBlockBytesWithHeader);
        }

        private ByteBuffer cloneOnDiskBufferWithHeader() {
            this.expectState(State.BLOCK_READY);
            return ByteBuffer.wrap(this.onDiskBlockBytesWithHeader.toByteArray());
        }

        private void expectState(State expectedState) {
            if (this.state != expectedState) {
                throw new IllegalStateException("Expected state: " + (Object)((Object)expectedState) + ", actual state: " + (Object)((Object)this.state));
            }
        }

        void writeBlock(BlockWritable bw, FSDataOutputStream out) throws IOException {
            bw.writeToBlock(this.startWriting(bw.getBlockType()));
            this.writeHeaderAndData(out);
        }

        HFileBlock getBlockForCaching(CacheConfig cacheConf) {
            HFileContext newContext = new HFileContextBuilder().withBlockSize(this.fileContext.getBlocksize()).withBytesPerCheckSum(0).withChecksumType(ChecksumType.NULL).withCompression(this.fileContext.getCompression()).withDataBlockEncoding(this.fileContext.getDataBlockEncoding()).withHBaseCheckSum(this.fileContext.isUseHBaseChecksum()).withCompressTags(this.fileContext.isCompressTags()).withIncludesMvcc(this.fileContext.isIncludesMvcc()).withIncludesTags(this.fileContext.isIncludesTags()).build();
            return new HFileBlock(this.blockType, this.getOnDiskSizeWithoutHeader(), this.getUncompressedSizeWithoutHeader(), this.prevOffset, cacheConf.shouldCacheCompressed(this.blockType.getCategory()) ? this.cloneOnDiskBufferWithHeader() : this.cloneUncompressedBufferWithHeader(), true, this.startOffset, UNSET, this.onDiskBlockBytesWithHeader.size() + this.onDiskChecksum.length, newContext);
        }

        private static enum State {
            INIT,
            WRITING,
            BLOCK_READY;

        }
    }

    static class Header {
        static int BLOCK_MAGIC_INDEX = 0;
        static int ON_DISK_SIZE_WITHOUT_HEADER_INDEX = 8;
        static int UNCOMPRESSED_SIZE_WITHOUT_HEADER_INDEX = 12;
        static int PREV_BLOCK_OFFSET_INDEX = 16;
        static int CHECKSUM_TYPE_INDEX = 24;
        static int BYTES_PER_CHECKSUM_INDEX = 25;
        static int ON_DISK_DATA_SIZE_WITH_HEADER_INDEX = 29;

        Header() {
        }
    }
}

