/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile.bucket;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Arrays;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializer;
import org.apache.hadoop.hbase.io.hfile.bucket.IOEngine;
import org.apache.hadoop.hbase.nio.ByteBuff;
import org.apache.hadoop.hbase.nio.SingleByteBuff;
import org.apache.hadoop.hbase.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.util.StringUtils;
import org.apache.yetus.audience.InterfaceAudience;

@InterfaceAudience.Private
public class FileIOEngine
implements IOEngine {
    private static final Log LOG = LogFactory.getLog(FileIOEngine.class);
    public static final String FILE_DELIMITER = ",";
    private final String[] filePaths;
    private final FileChannel[] fileChannels;
    private final RandomAccessFile[] rafs;
    private final long sizePerFile;
    private final long capacity;
    private FileReadAccessor readAccessor = new FileReadAccessor();
    private FileWriteAccessor writeAccessor = new FileWriteAccessor();

    public FileIOEngine(long capacity, boolean maintainPersistence, String ... filePaths) throws IOException {
        this.sizePerFile = capacity / (long)filePaths.length;
        this.capacity = this.sizePerFile * (long)filePaths.length;
        this.filePaths = filePaths;
        this.fileChannels = new FileChannel[filePaths.length];
        if (!maintainPersistence) {
            for (String filePath : filePaths) {
                File file = new File(filePath);
                if (!file.exists()) continue;
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("File " + filePath + " already exists. Deleting!!"));
                }
                file.delete();
            }
        }
        this.rafs = new RandomAccessFile[filePaths.length];
        for (int i = 0; i < filePaths.length; ++i) {
            String filePath = filePaths[i];
            try {
                this.rafs[i] = new RandomAccessFile(filePath, "rw");
                long totalSpace = new File(filePath).getTotalSpace();
                if (totalSpace < this.sizePerFile) {
                    String msg = "Only " + StringUtils.byteDesc(totalSpace) + " total space under " + filePath + ", not enough for requested " + StringUtils.byteDesc(this.sizePerFile);
                    LOG.warn((Object)msg);
                }
                this.rafs[i].setLength(this.sizePerFile);
                this.fileChannels[i] = this.rafs[i].getChannel();
                LOG.info((Object)("Allocating cache " + StringUtils.byteDesc(this.sizePerFile) + ", on the path:" + filePath));
                continue;
            }
            catch (IOException fex) {
                LOG.error((Object)("Failed allocating cache on " + filePath), (Throwable)fex);
                this.shutdown();
                throw fex;
            }
        }
    }

    public String toString() {
        return "ioengine=" + this.getClass().getSimpleName() + ", paths=" + Arrays.asList(this.filePaths) + ", capacity=" + String.format("%,d", this.capacity);
    }

    @Override
    public boolean isPersistent() {
        return true;
    }

    @Override
    public Cacheable read(long offset, int length, CacheableDeserializer<Cacheable> deserializer) throws IOException {
        Preconditions.checkArgument(length >= 0, "Length of read can not be less than 0.");
        ByteBuffer dstBuffer = ByteBuffer.allocate(length);
        if (length != 0) {
            this.accessFile(this.readAccessor, dstBuffer, offset);
            if (dstBuffer.limit() != length) {
                throw new RuntimeException("Only " + dstBuffer.limit() + " bytes read, " + length + " expected");
            }
        }
        return deserializer.deserialize(new SingleByteBuff(dstBuffer), true, Cacheable.MemoryType.EXCLUSIVE);
    }

    @Override
    public void write(ByteBuffer srcBuffer, long offset) throws IOException {
        if (!srcBuffer.hasRemaining()) {
            return;
        }
        this.accessFile(this.writeAccessor, srcBuffer, offset);
    }

    @Override
    public void sync() throws IOException {
        for (int i = 0; i < this.fileChannels.length; ++i) {
            try {
                if (this.fileChannels[i] == null) continue;
                this.fileChannels[i].force(true);
                continue;
            }
            catch (IOException ie) {
                LOG.warn((Object)("Failed syncing data to " + this.filePaths[i]));
                throw ie;
            }
        }
    }

    @Override
    public void shutdown() {
        for (int i = 0; i < this.filePaths.length; ++i) {
            try {
                if (this.fileChannels[i] != null) {
                    this.fileChannels[i].close();
                }
                if (this.rafs[i] == null) continue;
                this.rafs[i].close();
                continue;
            }
            catch (IOException ex) {
                LOG.error((Object)("Failed closing " + this.filePaths[i] + " when shudown the IOEngine"), (Throwable)ex);
            }
        }
    }

    @Override
    public void write(ByteBuff srcBuffer, long offset) throws IOException {
        assert (srcBuffer.hasArray());
        this.write(ByteBuffer.wrap(srcBuffer.array(), srcBuffer.arrayOffset(), srcBuffer.remaining()), offset);
    }

    private void accessFile(FileAccessor accessor, ByteBuffer buffer, long globalOffset) throws IOException {
        block2: {
            int startFileNum = this.getFileNum(globalOffset);
            int remainingAccessDataLen = buffer.remaining();
            int endFileNum = this.getFileNum(globalOffset + (long)remainingAccessDataLen - 1L);
            int accessFileNum = startFileNum;
            long accessOffset = this.getAbsoluteOffsetInFile(accessFileNum, globalOffset);
            int bufLimit = buffer.limit();
            do {
                FileChannel fileChannel = this.fileChannels[accessFileNum];
                if (endFileNum > accessFileNum) {
                    buffer.limit((int)((long)(buffer.limit() - remainingAccessDataLen) + this.sizePerFile - accessOffset));
                }
                int accessLen = accessor.access(fileChannel, buffer, accessOffset);
                buffer.limit(bufLimit);
                if (accessLen >= remainingAccessDataLen) break block2;
                remainingAccessDataLen -= accessLen;
                accessOffset = 0L;
            } while (++accessFileNum < this.fileChannels.length);
            throw new IOException("Required data len " + StringUtils.byteDesc(buffer.remaining()) + " exceed the engine's capacity " + StringUtils.byteDesc(this.capacity) + " where offset=" + globalOffset);
        }
    }

    private long getAbsoluteOffsetInFile(int fileNum, long globalOffset) {
        return globalOffset - (long)fileNum * this.sizePerFile;
    }

    private int getFileNum(long offset) {
        if (offset < 0L) {
            throw new IllegalArgumentException("Unexpected offset " + offset);
        }
        int fileNum = (int)(offset / this.sizePerFile);
        if (fileNum >= this.fileChannels.length) {
            throw new RuntimeException("Not expected offset " + offset + " where capacity=" + this.capacity);
        }
        return fileNum;
    }

    private static class FileWriteAccessor
    implements FileAccessor {
        private FileWriteAccessor() {
        }

        @Override
        public int access(FileChannel fileChannel, ByteBuffer byteBuffer, long accessOffset) throws IOException {
            return fileChannel.write(byteBuffer, accessOffset);
        }
    }

    private static class FileReadAccessor
    implements FileAccessor {
        private FileReadAccessor() {
        }

        @Override
        public int access(FileChannel fileChannel, ByteBuffer byteBuffer, long accessOffset) throws IOException {
            return fileChannel.read(byteBuffer, accessOffset);
        }
    }

    private static interface FileAccessor {
        public int access(FileChannel var1, ByteBuffer var2, long var3) throws IOException;
    }
}

