/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.balancer;

import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.ClusterStatus;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.RegionLoad;
import org.apache.hadoop.hbase.ServerLoad;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.RegionPlan;
import org.apache.hadoop.hbase.master.balancer.BalancerRegionLoad;
import org.apache.hadoop.hbase.master.balancer.BaseLoadBalancer;
import org.apache.hadoop.hbase.master.balancer.ClusterLoadState;
import org.apache.hadoop.hbase.master.balancer.MetricsStochasticBalancer;
import org.apache.hadoop.hbase.master.balancer.RegionLocationFinder;
import org.apache.hadoop.hbase.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.hbase.shaded.com.google.common.base.Optional;
import org.apache.hadoop.hbase.shaded.com.google.common.collect.Lists;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.yetus.audience.InterfaceAudience;

@InterfaceAudience.LimitedPrivate(value={"Configuration"})
@SuppressWarnings(value={"IS2_INCONSISTENT_SYNC"}, justification="Complaint is about costFunctions not being synchronized; not end of the world")
public class StochasticLoadBalancer
extends BaseLoadBalancer {
    protected static final String STEPS_PER_REGION_KEY = "hbase.master.balancer.stochastic.stepsPerRegion";
    protected static final String MAX_STEPS_KEY = "hbase.master.balancer.stochastic.maxSteps";
    protected static final String RUN_MAX_STEPS_KEY = "hbase.master.balancer.stochastic.runMaxSteps";
    protected static final String MAX_RUNNING_TIME_KEY = "hbase.master.balancer.stochastic.maxRunningTime";
    protected static final String KEEP_REGION_LOADS = "hbase.master.balancer.stochastic.numRegionLoadsToRemember";
    private static final String TABLE_FUNCTION_SEP = "_";
    protected static final String MIN_COST_NEED_BALANCE_KEY = "hbase.master.balancer.stochastic.minCostNeedBalance";
    protected static final Random RANDOM = new Random(System.currentTimeMillis());
    private static final Log LOG = LogFactory.getLog(StochasticLoadBalancer.class);
    Map<String, Deque<BalancerRegionLoad>> loads = new HashMap<String, Deque<BalancerRegionLoad>>();
    private int maxSteps = 1000000;
    private boolean runMaxSteps = false;
    private int stepsPerRegion = 800;
    private long maxRunningTime = 30000L;
    private int numRegionLoadsToRemember = 15;
    private float minCostNeedBalance = 0.05f;
    private List<CandidateGenerator> candidateGenerators;
    private CostFromRegionLoadFunction[] regionLoadFunctions;
    private CostFunction[] costFunctions;
    private Double curOverallCost = 0.0;
    private Double[] tempFunctionCosts;
    private Double[] curFunctionCosts;
    private LocalityBasedCandidateGenerator localityCandidateGenerator;
    private ServerLocalityCostFunction localityCost;
    private RackLocalityCostFunction rackLocalityCost;
    private RegionReplicaHostCostFunction regionReplicaHostCostFunction;
    private RegionReplicaRackCostFunction regionReplicaRackCostFunction;
    private boolean isByTable = false;
    private TableName tableName = null;

    public StochasticLoadBalancer() {
        super(new MetricsStochasticBalancer());
    }

    @Override
    public void onConfigurationChange(Configuration conf) {
        this.setConf(conf);
    }

    @Override
    public synchronized void setConf(Configuration conf) {
        super.setConf(conf);
        this.maxSteps = conf.getInt(MAX_STEPS_KEY, this.maxSteps);
        this.stepsPerRegion = conf.getInt(STEPS_PER_REGION_KEY, this.stepsPerRegion);
        this.maxRunningTime = conf.getLong(MAX_RUNNING_TIME_KEY, this.maxRunningTime);
        this.runMaxSteps = conf.getBoolean(RUN_MAX_STEPS_KEY, this.runMaxSteps);
        this.numRegionLoadsToRemember = conf.getInt(KEEP_REGION_LOADS, this.numRegionLoadsToRemember);
        this.isByTable = conf.getBoolean("hbase.master.loadbalance.bytable", this.isByTable);
        this.minCostNeedBalance = conf.getFloat(MIN_COST_NEED_BALANCE_KEY, this.minCostNeedBalance);
        if (this.localityCandidateGenerator == null) {
            this.localityCandidateGenerator = new LocalityBasedCandidateGenerator(this.services);
        }
        this.localityCost = new ServerLocalityCostFunction(conf, this.services);
        this.rackLocalityCost = new RackLocalityCostFunction(conf, this.services);
        if (this.candidateGenerators == null) {
            this.candidateGenerators = Lists.newArrayList();
            this.candidateGenerators.add(new RandomCandidateGenerator());
            this.candidateGenerators.add(new LoadCandidateGenerator());
            this.candidateGenerators.add(this.localityCandidateGenerator);
            this.candidateGenerators.add(new RegionReplicaRackCandidateGenerator());
        }
        this.regionLoadFunctions = new CostFromRegionLoadFunction[]{new ReadRequestCostFunction(conf), new WriteRequestCostFunction(conf), new MemstoreSizeCostFunction(conf), new StoreFileCostFunction(conf)};
        this.regionReplicaHostCostFunction = new RegionReplicaHostCostFunction(conf);
        this.regionReplicaRackCostFunction = new RegionReplicaRackCostFunction(conf);
        this.costFunctions = new CostFunction[]{new RegionCountSkewCostFunction(conf), new PrimaryRegionCountSkewCostFunction(conf), new MoveCostFunction(conf), this.localityCost, this.rackLocalityCost, new TableSkewCostFunction(conf), this.regionReplicaHostCostFunction, this.regionReplicaRackCostFunction, this.regionLoadFunctions[0], this.regionLoadFunctions[1], this.regionLoadFunctions[2], this.regionLoadFunctions[3]};
        this.curFunctionCosts = new Double[this.costFunctions.length];
        this.tempFunctionCosts = new Double[this.costFunctions.length];
        LOG.info((Object)("Loaded config; maxSteps=" + this.maxSteps + ", stepsPerRegion=" + this.stepsPerRegion + ", maxRunningTime=" + this.maxRunningTime + ", isByTable=" + this.isByTable + ", etc."));
    }

    protected void setCandidateGenerators(List<CandidateGenerator> customCandidateGenerators) {
        this.candidateGenerators = customCandidateGenerators;
    }

    @Override
    protected void setSlop(Configuration conf) {
        this.slop = conf.getFloat("hbase.regions.slop", 0.001f);
    }

    @Override
    public synchronized void setClusterStatus(ClusterStatus st) {
        super.setClusterStatus(st);
        this.updateRegionLoad();
        for (CostFromRegionLoadFunction cost : this.regionLoadFunctions) {
            cost.setClusterStatus(st);
        }
        try {
            int tablesCount = this.isByTable ? this.services.getTableDescriptors().getAll().size() : 1;
            int functionsCount = this.getCostFunctionNames().length;
            this.updateMetricsSize(tablesCount * (functionsCount + 1));
        }
        catch (Exception e) {
            LOG.error((Object)"failed to get the size of all tables", (Throwable)e);
        }
    }

    public void updateMetricsSize(int size) {
        if (this.metricsBalancer instanceof MetricsStochasticBalancer) {
            ((MetricsStochasticBalancer)this.metricsBalancer).updateMetricsSize(size);
        }
    }

    @Override
    public synchronized void setMasterServices(MasterServices masterServices) {
        super.setMasterServices(masterServices);
        this.localityCost.setServices(masterServices);
        this.rackLocalityCost.setServices(masterServices);
        this.localityCandidateGenerator.setServices(masterServices);
    }

    @Override
    protected synchronized boolean areSomeRegionReplicasColocated(BaseLoadBalancer.Cluster c) {
        this.regionReplicaHostCostFunction.init(c);
        if (this.regionReplicaHostCostFunction.cost() > 0.0) {
            return true;
        }
        this.regionReplicaRackCostFunction.init(c);
        return this.regionReplicaRackCostFunction.cost() > 0.0;
    }

    @Override
    protected boolean needsBalance(BaseLoadBalancer.Cluster cluster) {
        ClusterLoadState cs = new ClusterLoadState(cluster.clusterState);
        if (cs.getNumServers() < 2) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Not running balancer because only " + cs.getNumServers() + " active regionserver(s)"));
            }
            return false;
        }
        if (this.areSomeRegionReplicasColocated(cluster)) {
            return true;
        }
        double total = 0.0;
        float sumMultiplier = 0.0f;
        for (CostFunction c : this.costFunctions) {
            float multiplier = c.getMultiplier();
            if (multiplier <= 0.0f) continue;
            if (!c.isNeeded()) {
                LOG.debug((Object)(c.getClass().getName() + " indicated that its cost should not be considered"));
                continue;
            }
            sumMultiplier += multiplier;
            total += c.cost() * (double)multiplier;
        }
        if (total <= 0.0 || sumMultiplier <= 0.0f || sumMultiplier > 0.0f && total / (double)sumMultiplier < (double)this.minCostNeedBalance) {
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("Skipping load balancing because balanced cluster; total cost is " + total + ", sum multiplier is " + sumMultiplier + " min cost which need balance is " + this.minCostNeedBalance));
            }
            return false;
        }
        return true;
    }

    @Override
    public synchronized List<RegionPlan> balanceCluster(TableName tableName, Map<ServerName, List<HRegionInfo>> clusterState) {
        this.tableName = tableName;
        return this.balanceCluster(clusterState);
    }

    @VisibleForTesting
    BaseLoadBalancer.Cluster.Action nextAction(BaseLoadBalancer.Cluster cluster) {
        return this.candidateGenerators.get(RANDOM.nextInt(this.candidateGenerators.size())).generate(cluster);
    }

    @Override
    public synchronized List<RegionPlan> balanceCluster(Map<ServerName, List<HRegionInfo>> clusterState) {
        long step;
        List<RegionPlan> plans = this.balanceMasterRegions(clusterState);
        if (plans != null || clusterState == null || clusterState.size() <= 1) {
            return plans;
        }
        if (this.masterServerName != null && clusterState.containsKey(this.masterServerName)) {
            if (clusterState.size() <= 2) {
                return null;
            }
            clusterState = new HashMap<ServerName, List<HRegionInfo>>(clusterState);
            clusterState.remove(this.masterServerName);
        }
        RegionLocationFinder finder = null;
        if (this.localityCost != null && this.localityCost.getMultiplier() > 0.0f || this.rackLocalityCost != null && this.rackLocalityCost.getMultiplier() > 0.0f) {
            finder = this.regionFinder;
        }
        BaseLoadBalancer.Cluster cluster = new BaseLoadBalancer.Cluster(clusterState, this.loads, finder, this.rackManager);
        long startTime = EnvironmentEdgeManager.currentTime();
        this.initCosts(cluster);
        if (!this.needsBalance(cluster)) {
            return null;
        }
        double currentCost = this.computeCost(cluster, Double.MAX_VALUE);
        this.curOverallCost = currentCost;
        for (int i = 0; i < this.curFunctionCosts.length; ++i) {
            this.curFunctionCosts[i] = this.tempFunctionCosts[i];
        }
        LOG.info((Object)("start StochasticLoadBalancer.balancer, initCost=" + currentCost + ", functionCost=" + this.functionCost()));
        double initCost = currentCost;
        double newCost = currentCost;
        long computedMaxSteps = this.runMaxSteps ? Math.max((long)this.maxSteps, (long)cluster.numRegions * (long)this.stepsPerRegion * (long)cluster.numServers) : Math.min((long)this.maxSteps, (long)cluster.numRegions * (long)this.stepsPerRegion * (long)cluster.numServers);
        for (step = 0L; step < computedMaxSteps; ++step) {
            BaseLoadBalancer.Cluster.Action action = this.nextAction(cluster);
            if (action.type == BaseLoadBalancer.Cluster.Action.Type.NULL) continue;
            cluster.doAction(action);
            this.updateCostsWithAction(cluster, action);
            newCost = this.computeCost(cluster, currentCost);
            if (newCost < currentCost) {
                currentCost = newCost;
                this.curOverallCost = currentCost;
                for (int i = 0; i < this.curFunctionCosts.length; ++i) {
                    this.curFunctionCosts[i] = this.tempFunctionCosts[i];
                }
            } else {
                BaseLoadBalancer.Cluster.Action undoAction = action.undoAction();
                cluster.doAction(undoAction);
                this.updateCostsWithAction(cluster, undoAction);
            }
            if (EnvironmentEdgeManager.currentTime() - startTime > this.maxRunningTime) break;
        }
        long endTime = EnvironmentEdgeManager.currentTime();
        this.metricsBalancer.balanceCluster(endTime - startTime);
        this.updateStochasticCosts(this.tableName, this.curOverallCost, this.curFunctionCosts);
        if (initCost > currentCost) {
            plans = this.createRegionPlans(cluster);
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Finished computing new load balance plan.  Computation took " + (endTime - startTime) + "ms to try " + step + " different iterations.  Found a solution that moves " + plans.size() + " regions; Going from a computed cost of " + initCost + " to a new cost of " + currentCost));
            }
            return plans;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Could not find a better load balance plan.  Tried " + step + " different configurations in " + (endTime - startTime) + "ms, and did not find anything with a computed cost less than " + initCost));
        }
        return null;
    }

    private void updateStochasticCosts(TableName tableName, Double overall, Double[] subCosts) {
        if (tableName == null) {
            return;
        }
        if (this.metricsBalancer instanceof MetricsStochasticBalancer) {
            MetricsStochasticBalancer balancer = (MetricsStochasticBalancer)this.metricsBalancer;
            balancer.updateStochasticCost(tableName.getNameAsString(), "Overall", "Overall cost", overall);
            for (int i = 0; i < this.costFunctions.length; ++i) {
                CostFunction costFunction = this.costFunctions[i];
                String costFunctionName = costFunction.getClass().getSimpleName();
                Double costPercent = overall == 0.0 ? 0.0 : subCosts[i] / overall;
                balancer.updateStochasticCost(tableName.getNameAsString(), costFunctionName, "The percent of " + costFunctionName, costPercent);
            }
        }
    }

    private String functionCost() {
        StringBuilder builder = new StringBuilder();
        for (CostFunction c : this.costFunctions) {
            builder.append(c.getClass().getSimpleName());
            builder.append(" : (");
            builder.append(c.getMultiplier());
            builder.append(", ");
            builder.append(c.cost());
            builder.append("); ");
        }
        return builder.toString();
    }

    private List<RegionPlan> createRegionPlans(BaseLoadBalancer.Cluster cluster) {
        LinkedList<RegionPlan> plans = new LinkedList<RegionPlan>();
        for (int regionIndex = 0; regionIndex < cluster.regionIndexToServerIndex.length; ++regionIndex) {
            int initialServerIndex = cluster.initialRegionIndexToServerIndex[regionIndex];
            int newServerIndex = cluster.regionIndexToServerIndex[regionIndex];
            if (initialServerIndex == newServerIndex) continue;
            HRegionInfo region = cluster.regions[regionIndex];
            ServerName initialServer = cluster.servers[initialServerIndex];
            ServerName newServer = cluster.servers[newServerIndex];
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("Moving Region " + region.getEncodedName() + " from server " + initialServer.getHostname() + " to " + newServer.getHostname()));
            }
            RegionPlan rp = new RegionPlan(region, initialServer, newServer);
            plans.add(rp);
        }
        return plans;
    }

    private synchronized void updateRegionLoad() {
        Map<String, Deque<BalancerRegionLoad>> oldLoads = this.loads;
        this.loads = new HashMap<String, Deque<BalancerRegionLoad>>();
        for (ServerName sn : this.clusterStatus.getServers()) {
            ServerLoad sl = this.clusterStatus.getLoad(sn);
            if (sl == null) continue;
            for (Map.Entry<byte[], RegionLoad> entry : sl.getRegionsLoad().entrySet()) {
                Deque<BalancerRegionLoad> rLoads = oldLoads.get(Bytes.toString(entry.getKey()));
                if (rLoads == null) {
                    rLoads = new ArrayDeque<BalancerRegionLoad>();
                } else if (rLoads.size() >= this.numRegionLoadsToRemember) {
                    rLoads.remove();
                }
                rLoads.add(new BalancerRegionLoad(entry.getValue()));
                this.loads.put(Bytes.toString(entry.getKey()), rLoads);
            }
        }
        for (CostFromRegionLoadFunction cost : this.regionLoadFunctions) {
            cost.setLoads(this.loads);
        }
    }

    protected void initCosts(BaseLoadBalancer.Cluster cluster) {
        for (CostFunction c : this.costFunctions) {
            c.init(cluster);
        }
    }

    protected void updateCostsWithAction(BaseLoadBalancer.Cluster cluster, BaseLoadBalancer.Cluster.Action action) {
        for (CostFunction c : this.costFunctions) {
            c.postAction(action);
        }
    }

    public String[] getCostFunctionNames() {
        if (this.costFunctions == null) {
            return null;
        }
        String[] ret = new String[this.costFunctions.length];
        for (int i = 0; i < this.costFunctions.length; ++i) {
            CostFunction c = this.costFunctions[i];
            ret[i] = c.getClass().getSimpleName();
        }
        return ret;
    }

    protected double computeCost(BaseLoadBalancer.Cluster cluster, double previousCost) {
        double total = 0.0;
        for (int i = 0; i < this.costFunctions.length; ++i) {
            CostFunction c = this.costFunctions[i];
            this.tempFunctionCosts[i] = 0.0;
            if (c.getMultiplier() <= 0.0f) continue;
            Float multiplier = Float.valueOf(c.getMultiplier());
            Double cost = c.cost();
            this.tempFunctionCosts[i] = (double)multiplier.floatValue() * cost;
            if ((total += this.tempFunctionCosts[i].doubleValue()) > previousCost) break;
        }
        return total;
    }

    public static String composeAttributeName(String tableName, String costFunctionName) {
        return tableName + TABLE_FUNCTION_SEP + costFunctionName;
    }

    static class StoreFileCostFunction
    extends CostFromRegionLoadFunction {
        private static final String STOREFILE_SIZE_COST_KEY = "hbase.master.balancer.stochastic.storefileSizeCost";
        private static final float DEFAULT_STOREFILE_SIZE_COST = 5.0f;

        StoreFileCostFunction(Configuration conf) {
            super(conf);
            this.setMultiplier(conf.getFloat(STOREFILE_SIZE_COST_KEY, 5.0f));
        }

        @Override
        protected double getCostFromRl(BalancerRegionLoad rl) {
            return rl.getStorefileSizeMB();
        }
    }

    static class MemstoreSizeCostFunction
    extends CostFromRegionLoadAsRateFunction {
        private static final String MEMSTORE_SIZE_COST_KEY = "hbase.master.balancer.stochastic.memstoreSizeCost";
        private static final float DEFAULT_MEMSTORE_SIZE_COST = 5.0f;

        MemstoreSizeCostFunction(Configuration conf) {
            super(conf);
            this.setMultiplier(conf.getFloat(MEMSTORE_SIZE_COST_KEY, 5.0f));
        }

        @Override
        protected double getCostFromRl(BalancerRegionLoad rl) {
            return rl.getMemStoreSizeMB();
        }
    }

    static class RegionReplicaRackCostFunction
    extends RegionReplicaHostCostFunction {
        private static final String REGION_REPLICA_RACK_COST_KEY = "hbase.master.balancer.stochastic.regionReplicaRackCostKey";
        private static final float DEFAULT_REGION_REPLICA_RACK_COST_KEY = 10000.0f;

        public RegionReplicaRackCostFunction(Configuration conf) {
            super(conf);
            this.setMultiplier(conf.getFloat(REGION_REPLICA_RACK_COST_KEY, 10000.0f));
        }

        @Override
        void init(BaseLoadBalancer.Cluster cluster) {
            this.cluster = cluster;
            if (cluster.numRacks <= 1) {
                this.maxCost = 0L;
                return;
            }
            this.maxCost = this.getMaxCost(cluster);
            this.costsPerGroup = new long[cluster.numRacks];
            for (int i = 0; i < cluster.primariesOfRegionsPerRack.length; ++i) {
                this.costsPerGroup[i] = this.costPerGroup(cluster.primariesOfRegionsPerRack[i]);
            }
        }

        @Override
        protected void regionMoved(int region, int oldServer, int newServer) {
            if (this.maxCost <= 0L) {
                return;
            }
            int newRack = this.cluster.serverIndexToRackIndex[newServer];
            int oldRack = this.cluster.serverIndexToRackIndex[oldServer];
            if (newRack != oldRack) {
                this.costsPerGroup[oldRack] = this.costPerGroup(this.cluster.primariesOfRegionsPerRack[oldRack]);
                this.costsPerGroup[newRack] = this.costPerGroup(this.cluster.primariesOfRegionsPerRack[newRack]);
            }
        }
    }

    static class RegionReplicaHostCostFunction
    extends CostFunction {
        private static final String REGION_REPLICA_HOST_COST_KEY = "hbase.master.balancer.stochastic.regionReplicaHostCostKey";
        private static final float DEFAULT_REGION_REPLICA_HOST_COST_KEY = 100000.0f;
        long maxCost = 0L;
        long[] costsPerGroup;
        int[][] primariesOfRegionsPerGroup;

        public RegionReplicaHostCostFunction(Configuration conf) {
            super(conf);
            this.setMultiplier(conf.getFloat(REGION_REPLICA_HOST_COST_KEY, 100000.0f));
        }

        @Override
        void init(BaseLoadBalancer.Cluster cluster) {
            super.init(cluster);
            this.maxCost = cluster.numHosts > 1 ? this.getMaxCost(cluster) : 0L;
            this.costsPerGroup = new long[cluster.numHosts];
            this.primariesOfRegionsPerGroup = cluster.multiServersPerHost ? cluster.primariesOfRegionsPerHost : cluster.primariesOfRegionsPerServer;
            for (int i = 0; i < this.primariesOfRegionsPerGroup.length; ++i) {
                this.costsPerGroup[i] = this.costPerGroup(this.primariesOfRegionsPerGroup[i]);
            }
        }

        long getMaxCost(BaseLoadBalancer.Cluster cluster) {
            if (!cluster.hasRegionReplicas) {
                return 0L;
            }
            int[] primariesOfRegions = new int[cluster.numRegions];
            System.arraycopy(cluster.regionIndexToPrimaryIndex, 0, primariesOfRegions, 0, cluster.regions.length);
            Arrays.sort(primariesOfRegions);
            return this.costPerGroup(primariesOfRegions);
        }

        @Override
        boolean isNeeded() {
            return this.cluster.hasRegionReplicas;
        }

        @Override
        double cost() {
            if (this.maxCost <= 0L) {
                return 0.0;
            }
            long totalCost = 0L;
            for (int i = 0; i < this.costsPerGroup.length; ++i) {
                totalCost += this.costsPerGroup[i];
            }
            return this.scale(0.0, this.maxCost, totalCost);
        }

        protected long costPerGroup(int[] primariesOfRegions) {
            long cost = 0L;
            int currentPrimary = -1;
            int currentPrimaryIndex = -1;
            for (int j = 0; j <= primariesOfRegions.length; ++j) {
                int primary;
                int n = primary = j < primariesOfRegions.length ? primariesOfRegions[j] : -1;
                if (primary == currentPrimary) continue;
                int numReplicas = j - currentPrimaryIndex;
                if (numReplicas > 1) {
                    cost += (long)((numReplicas - 1) * (numReplicas - 1));
                }
                currentPrimary = primary;
                currentPrimaryIndex = j;
            }
            return cost;
        }

        @Override
        protected void regionMoved(int region, int oldServer, int newServer) {
            if (this.maxCost <= 0L) {
                return;
            }
            if (this.cluster.multiServersPerHost) {
                int newHost = this.cluster.serverIndexToHostIndex[newServer];
                int oldHost = this.cluster.serverIndexToHostIndex[oldServer];
                if (newHost != oldHost) {
                    this.costsPerGroup[oldHost] = this.costPerGroup(this.cluster.primariesOfRegionsPerHost[oldHost]);
                    this.costsPerGroup[newHost] = this.costPerGroup(this.cluster.primariesOfRegionsPerHost[newHost]);
                }
            } else {
                this.costsPerGroup[oldServer] = this.costPerGroup(this.cluster.primariesOfRegionsPerServer[oldServer]);
                this.costsPerGroup[newServer] = this.costPerGroup(this.cluster.primariesOfRegionsPerServer[newServer]);
            }
        }
    }

    static class WriteRequestCostFunction
    extends CostFromRegionLoadAsRateFunction {
        private static final String WRITE_REQUEST_COST_KEY = "hbase.master.balancer.stochastic.writeRequestCost";
        private static final float DEFAULT_WRITE_REQUEST_COST = 5.0f;

        WriteRequestCostFunction(Configuration conf) {
            super(conf);
            this.setMultiplier(conf.getFloat(WRITE_REQUEST_COST_KEY, 5.0f));
        }

        @Override
        protected double getCostFromRl(BalancerRegionLoad rl) {
            return rl.getWriteRequestsCount();
        }
    }

    static class ReadRequestCostFunction
    extends CostFromRegionLoadAsRateFunction {
        private static final String READ_REQUEST_COST_KEY = "hbase.master.balancer.stochastic.readRequestCost";
        private static final float DEFAULT_READ_REQUEST_COST = 5.0f;

        ReadRequestCostFunction(Configuration conf) {
            super(conf);
            this.setMultiplier(conf.getFloat(READ_REQUEST_COST_KEY, 5.0f));
        }

        @Override
        protected double getCostFromRl(BalancerRegionLoad rl) {
            return rl.getReadRequestsCount();
        }
    }

    static abstract class CostFromRegionLoadAsRateFunction
    extends CostFromRegionLoadFunction {
        CostFromRegionLoadAsRateFunction(Configuration conf) {
            super(conf);
        }

        @Override
        protected double getRegionLoadCost(Collection<BalancerRegionLoad> regionLoadList) {
            double cost = 0.0;
            double previous = 0.0;
            boolean isFirst = true;
            for (BalancerRegionLoad rl : regionLoadList) {
                double current = this.getCostFromRl(rl);
                if (isFirst) {
                    isFirst = false;
                } else {
                    cost += current - previous;
                }
                previous = current;
            }
            return Math.max(0.0, cost / (double)(regionLoadList.size() - 1));
        }
    }

    static abstract class CostFromRegionLoadFunction
    extends CostFunction {
        private ClusterStatus clusterStatus = null;
        private Map<String, Deque<BalancerRegionLoad>> loads = null;
        private double[] stats = null;

        CostFromRegionLoadFunction(Configuration conf) {
            super(conf);
        }

        void setClusterStatus(ClusterStatus status) {
            this.clusterStatus = status;
        }

        void setLoads(Map<String, Deque<BalancerRegionLoad>> l) {
            this.loads = l;
        }

        @Override
        double cost() {
            if (this.clusterStatus == null || this.loads == null) {
                return 0.0;
            }
            if (this.stats == null || this.stats.length != this.cluster.numServers) {
                this.stats = new double[this.cluster.numServers];
            }
            for (int i = 0; i < this.stats.length; ++i) {
                long cost = 0L;
                for (int regionIndex : this.cluster.regionsPerServer[i]) {
                    Deque<BalancerRegionLoad> regionLoadList = this.cluster.regionLoads[regionIndex];
                    if (regionLoadList == null) continue;
                    cost = (long)((double)cost + this.getRegionLoadCost(regionLoadList));
                }
                this.stats[i] = cost;
            }
            return this.costFromArray(this.stats);
        }

        protected double getRegionLoadCost(Collection<BalancerRegionLoad> regionLoadList) {
            double cost = 0.0;
            for (BalancerRegionLoad rl : regionLoadList) {
                cost += this.getCostFromRl(rl);
            }
            return cost / (double)regionLoadList.size();
        }

        protected abstract double getCostFromRl(BalancerRegionLoad var1);
    }

    static class RackLocalityCostFunction
    extends LocalityBasedCostFunction {
        private static final String RACK_LOCALITY_COST_KEY = "hbase.master.balancer.stochastic.rackLocalityCost";
        private static final float DEFAULT_RACK_LOCALITY_COST = 15.0f;

        public RackLocalityCostFunction(Configuration conf, MasterServices services) {
            super(conf, services, BaseLoadBalancer.Cluster.LocalityType.RACK, RACK_LOCALITY_COST_KEY, 15.0f);
        }

        @Override
        int regionIndexToEntityIndex(int region) {
            return this.cluster.getRackForRegion(region);
        }
    }

    static class ServerLocalityCostFunction
    extends LocalityBasedCostFunction {
        private static final String LOCALITY_COST_KEY = "hbase.master.balancer.stochastic.localityCost";
        private static final float DEFAULT_LOCALITY_COST = 25.0f;

        ServerLocalityCostFunction(Configuration conf, MasterServices srv) {
            super(conf, srv, BaseLoadBalancer.Cluster.LocalityType.SERVER, LOCALITY_COST_KEY, 25.0f);
        }

        @Override
        int regionIndexToEntityIndex(int region) {
            return this.cluster.regionIndexToServerIndex[region];
        }
    }

    static abstract class LocalityBasedCostFunction
    extends CostFunction {
        private final BaseLoadBalancer.Cluster.LocalityType type;
        private double bestLocality;
        private double locality;
        private MasterServices services;

        LocalityBasedCostFunction(Configuration conf, MasterServices srv, BaseLoadBalancer.Cluster.LocalityType type, String localityCostKey, float defaultLocalityCost) {
            super(conf);
            this.type = type;
            this.setMultiplier(conf.getFloat(localityCostKey, defaultLocalityCost));
            this.services = srv;
            this.locality = 0.0;
            this.bestLocality = 0.0;
        }

        abstract int regionIndexToEntityIndex(int var1);

        public void setServices(MasterServices srvc) {
            this.services = srvc;
        }

        @Override
        void init(BaseLoadBalancer.Cluster cluster) {
            super.init(cluster);
            this.locality = 0.0;
            this.bestLocality = 0.0;
            if (this.services == null) {
                return;
            }
            for (int region = 0; region < cluster.numRegions; ++region) {
                this.locality += this.getWeightedLocality(region, this.regionIndexToEntityIndex(region));
                this.bestLocality += this.getWeightedLocality(region, this.getMostLocalEntityForRegion(region));
            }
            this.locality = this.bestLocality == 0.0 ? 1.0 : this.locality / this.bestLocality;
        }

        @Override
        protected void regionMoved(int region, int oldServer, int newServer) {
            int newEntity;
            int oldEntity = this.type == BaseLoadBalancer.Cluster.LocalityType.SERVER ? oldServer : this.cluster.serverIndexToRackIndex[oldServer];
            int n = newEntity = this.type == BaseLoadBalancer.Cluster.LocalityType.SERVER ? newServer : this.cluster.serverIndexToRackIndex[newServer];
            if (this.services == null) {
                return;
            }
            double localityDelta = this.getWeightedLocality(region, newEntity) - this.getWeightedLocality(region, oldEntity);
            double normalizedDelta = this.bestLocality == 0.0 ? 0.0 : localityDelta / this.bestLocality;
            this.locality += normalizedDelta;
        }

        @Override
        double cost() {
            return 1.0 - this.locality;
        }

        private int getMostLocalEntityForRegion(int region) {
            return this.cluster.getOrComputeRegionsToMostLocalEntities(this.type)[region];
        }

        private double getWeightedLocality(int region, int entity) {
            return this.cluster.getOrComputeWeightedLocality(region, entity, this.type);
        }
    }

    static class TableSkewCostFunction
    extends CostFunction {
        private static final String TABLE_SKEW_COST_KEY = "hbase.master.balancer.stochastic.tableSkewCost";
        private static final float DEFAULT_TABLE_SKEW_COST = 35.0f;

        TableSkewCostFunction(Configuration conf) {
            super(conf);
            this.setMultiplier(conf.getFloat(TABLE_SKEW_COST_KEY, 35.0f));
        }

        @Override
        double cost() {
            double max = this.cluster.numRegions;
            double min = (double)this.cluster.numRegions / (double)this.cluster.numServers;
            double value = 0.0;
            for (int i = 0; i < this.cluster.numMaxRegionsPerTable.length; ++i) {
                value += (double)this.cluster.numMaxRegionsPerTable[i];
            }
            return this.scale(min, max, value);
        }
    }

    static class PrimaryRegionCountSkewCostFunction
    extends CostFunction {
        private static final String PRIMARY_REGION_COUNT_SKEW_COST_KEY = "hbase.master.balancer.stochastic.primaryRegionCountCost";
        private static final float DEFAULT_PRIMARY_REGION_COUNT_SKEW_COST = 500.0f;
        private double[] stats = null;

        PrimaryRegionCountSkewCostFunction(Configuration conf) {
            super(conf);
            this.setMultiplier(conf.getFloat(PRIMARY_REGION_COUNT_SKEW_COST_KEY, 500.0f));
        }

        @Override
        double cost() {
            if (!this.cluster.hasRegionReplicas) {
                return 0.0;
            }
            if (this.stats == null || this.stats.length != this.cluster.numServers) {
                this.stats = new double[this.cluster.numServers];
            }
            for (int i = 0; i < this.cluster.numServers; ++i) {
                this.stats[i] = 0.0;
                for (int regionIdx : this.cluster.regionsPerServer[i]) {
                    if (regionIdx != this.cluster.regionIndexToPrimaryIndex[regionIdx]) continue;
                    int n = i;
                    this.stats[n] = this.stats[n] + 1.0;
                }
            }
            return this.costFromArray(this.stats);
        }
    }

    static class RegionCountSkewCostFunction
    extends CostFunction {
        private static final String REGION_COUNT_SKEW_COST_KEY = "hbase.master.balancer.stochastic.regionCountCost";
        private static final float DEFAULT_REGION_COUNT_SKEW_COST = 500.0f;
        private double[] stats = null;

        RegionCountSkewCostFunction(Configuration conf) {
            super(conf);
            this.setMultiplier(conf.getFloat(REGION_COUNT_SKEW_COST_KEY, 500.0f));
        }

        @Override
        double cost() {
            if (this.stats == null || this.stats.length != this.cluster.numServers) {
                this.stats = new double[this.cluster.numServers];
            }
            for (int i = 0; i < this.cluster.numServers; ++i) {
                this.stats[i] = this.cluster.regionsPerServer[i].length;
            }
            return this.costFromArray(this.stats);
        }
    }

    static class MoveCostFunction
    extends CostFunction {
        private static final String MOVE_COST_KEY = "hbase.master.balancer.stochastic.moveCost";
        private static final String MAX_MOVES_PERCENT_KEY = "hbase.master.balancer.stochastic.maxMovePercent";
        private static final float DEFAULT_MOVE_COST = 7.0f;
        private static final int DEFAULT_MAX_MOVES = 600;
        private static final float DEFAULT_MAX_MOVE_PERCENT = 0.25f;
        private final float maxMovesPercent;

        MoveCostFunction(Configuration conf) {
            super(conf);
            this.setMultiplier(conf.getFloat(MOVE_COST_KEY, 7.0f));
            this.maxMovesPercent = conf.getFloat(MAX_MOVES_PERCENT_KEY, 0.25f);
        }

        @Override
        double cost() {
            double moveCost = this.cluster.numMovedRegions;
            int maxMoves = Math.max((int)((float)this.cluster.numRegions * this.maxMovesPercent), 600);
            if (moveCost > (double)maxMoves) {
                return 1000000.0;
            }
            return this.scale(0.0, Math.min(this.cluster.numRegions, maxMoves), moveCost);
        }
    }

    static abstract class CostFunction {
        private float multiplier = 0.0f;
        protected BaseLoadBalancer.Cluster cluster;

        CostFunction(Configuration c) {
        }

        boolean isNeeded() {
            return true;
        }

        float getMultiplier() {
            return this.multiplier;
        }

        void setMultiplier(float m) {
            this.multiplier = m;
        }

        void init(BaseLoadBalancer.Cluster cluster) {
            this.cluster = cluster;
        }

        void postAction(BaseLoadBalancer.Cluster.Action action) {
            switch (action.type) {
                case NULL: {
                    break;
                }
                case ASSIGN_REGION: {
                    BaseLoadBalancer.Cluster.AssignRegionAction ar = (BaseLoadBalancer.Cluster.AssignRegionAction)action;
                    this.regionMoved(ar.region, -1, ar.server);
                    break;
                }
                case MOVE_REGION: {
                    BaseLoadBalancer.Cluster.MoveRegionAction mra = (BaseLoadBalancer.Cluster.MoveRegionAction)action;
                    this.regionMoved(mra.region, mra.fromServer, mra.toServer);
                    break;
                }
                case SWAP_REGIONS: {
                    BaseLoadBalancer.Cluster.SwapRegionsAction a = (BaseLoadBalancer.Cluster.SwapRegionsAction)action;
                    this.regionMoved(a.fromRegion, a.fromServer, a.toServer);
                    this.regionMoved(a.toRegion, a.toServer, a.fromServer);
                    break;
                }
                default: {
                    throw new RuntimeException("Uknown action:" + (Object)((Object)action.type));
                }
            }
        }

        protected void regionMoved(int region, int oldServer, int newServer) {
        }

        abstract double cost();

        protected double costFromArray(double[] stats) {
            double min;
            double totalCost = 0.0;
            double total = this.getSum(stats);
            double count = stats.length;
            double mean = total / count;
            double max = (count - 1.0) * mean + (total - mean);
            if (count > total) {
                min = (count - total) * mean + (1.0 - mean) * total;
            } else {
                int numHigh = (int)(total - Math.floor(mean) * count);
                int numLow = (int)(count - (double)numHigh);
                min = (double)numHigh * (Math.ceil(mean) - mean) + (double)numLow * (mean - Math.floor(mean));
            }
            min = Math.max(0.0, min);
            for (int i = 0; i < stats.length; ++i) {
                double n = stats[i];
                double diff = Math.abs(mean - n);
                totalCost += diff;
            }
            double scaled = this.scale(min, max, totalCost);
            return scaled;
        }

        private double getSum(double[] stats) {
            double total = 0.0;
            for (double s : stats) {
                total += s;
            }
            return total;
        }

        protected double scale(double min, double max, double value) {
            if (max <= min || value <= min) {
                return 0.0;
            }
            if (max - min == 0.0) {
                return 0.0;
            }
            return Math.max(0.0, Math.min(1.0, (value - min) / (max - min)));
        }
    }

    static class RegionReplicaRackCandidateGenerator
    extends RegionReplicaCandidateGenerator {
        RegionReplicaRackCandidateGenerator() {
        }

        @Override
        BaseLoadBalancer.Cluster.Action generate(BaseLoadBalancer.Cluster cluster) {
            int rackIndex = this.pickRandomRack(cluster);
            if (cluster.numRacks <= 1 || rackIndex == -1) {
                return super.generate(cluster);
            }
            int regionIndex = this.selectCoHostedRegionPerGroup(cluster.primariesOfRegionsPerRack[rackIndex], cluster.regionsPerRack[rackIndex], cluster.regionIndexToPrimaryIndex);
            if (regionIndex == -1) {
                return this.randomGenerator.generate(cluster);
            }
            int serverIndex = cluster.regionIndexToServerIndex[regionIndex];
            int toRackIndex = this.pickOtherRandomRack(cluster, rackIndex);
            int rand = RANDOM.nextInt(cluster.serversPerRack[toRackIndex].length);
            int toServerIndex = cluster.serversPerRack[toRackIndex][rand];
            int toRegionIndex = this.pickRandomRegion(cluster, toServerIndex, 0.9f);
            return this.getAction(serverIndex, regionIndex, toServerIndex, toRegionIndex);
        }
    }

    static class RegionReplicaCandidateGenerator
    extends CandidateGenerator {
        RandomCandidateGenerator randomGenerator = new RandomCandidateGenerator();

        RegionReplicaCandidateGenerator() {
        }

        int selectCoHostedRegionPerGroup(int[] primariesOfRegionsPerGroup, int[] regionsPerGroup, int[] regionIndexToPrimaryIndex) {
            int j;
            int currentPrimary = -1;
            int currentPrimaryIndex = -1;
            int selectedPrimaryIndex = -1;
            double currentLargestRandom = -1.0;
            for (j = 0; j <= primariesOfRegionsPerGroup.length; ++j) {
                double currentRandom;
                int primary;
                int n = primary = j < primariesOfRegionsPerGroup.length ? primariesOfRegionsPerGroup[j] : -1;
                if (primary == currentPrimary) continue;
                int numReplicas = j - currentPrimaryIndex;
                if (numReplicas > 1 && (currentRandom = RANDOM.nextDouble()) > currentLargestRandom) {
                    selectedPrimaryIndex = currentPrimary;
                    currentLargestRandom = currentRandom;
                }
                currentPrimary = primary;
                currentPrimaryIndex = j;
            }
            for (j = 0; j < regionsPerGroup.length; ++j) {
                int regionIndex = regionsPerGroup[j];
                if (selectedPrimaryIndex != regionIndexToPrimaryIndex[regionIndex] || selectedPrimaryIndex == regionIndex) continue;
                return regionIndex;
            }
            return -1;
        }

        @Override
        BaseLoadBalancer.Cluster.Action generate(BaseLoadBalancer.Cluster cluster) {
            int serverIndex = this.pickRandomServer(cluster);
            if (cluster.numServers <= 1 || serverIndex == -1) {
                return BaseLoadBalancer.Cluster.NullAction;
            }
            int regionIndex = this.selectCoHostedRegionPerGroup(cluster.primariesOfRegionsPerServer[serverIndex], cluster.regionsPerServer[serverIndex], cluster.regionIndexToPrimaryIndex);
            if (regionIndex == -1) {
                return this.randomGenerator.generate(cluster);
            }
            int toServerIndex = this.pickOtherRandomServer(cluster, serverIndex);
            int toRegionIndex = this.pickRandomRegion(cluster, toServerIndex, 0.9f);
            return this.getAction(serverIndex, regionIndex, toServerIndex, toRegionIndex);
        }
    }

    static class LocalityBasedCandidateGenerator
    extends CandidateGenerator {
        private MasterServices masterServices;

        LocalityBasedCandidateGenerator(MasterServices masterServices) {
            this.masterServices = masterServices;
        }

        @Override
        BaseLoadBalancer.Cluster.Action generate(BaseLoadBalancer.Cluster cluster) {
            if (this.masterServices == null) {
                int thisServer = this.pickRandomServer(cluster);
                int otherServer = this.pickOtherRandomServer(cluster, thisServer);
                return this.pickRandomRegions(cluster, thisServer, otherServer);
            }
            for (int region : this.getRandomIterationOrder(cluster.numRegions)) {
                Optional<BaseLoadBalancer.Cluster.Action> potential;
                int currentServer = cluster.regionIndexToServerIndex[region];
                if (currentServer == cluster.getOrComputeRegionsToMostLocalEntities(BaseLoadBalancer.Cluster.LocalityType.SERVER)[region] || !(potential = this.tryMoveOrSwap(cluster, currentServer, region, cluster.getOrComputeRegionsToMostLocalEntities(BaseLoadBalancer.Cluster.LocalityType.SERVER)[region])).isPresent()) continue;
                return potential.get();
            }
            return BaseLoadBalancer.Cluster.NullAction;
        }

        private Optional<BaseLoadBalancer.Cluster.Action> tryMoveOrSwap(BaseLoadBalancer.Cluster cluster, int fromServer, int fromRegion, int toServer) {
            if (cluster.serverHasTooFewRegions(toServer)) {
                return Optional.of(this.getAction(fromServer, fromRegion, toServer, -1));
            }
            double fromRegionLocalityDelta = this.getWeightedLocality(cluster, fromRegion, toServer) - this.getWeightedLocality(cluster, fromRegion, fromServer);
            for (int toRegionIndex : this.getRandomIterationOrder(cluster.regionsPerServer[toServer].length)) {
                int toRegion = cluster.regionsPerServer[toServer][toRegionIndex];
                double toRegionLocalityDelta = this.getWeightedLocality(cluster, toRegion, fromServer) - this.getWeightedLocality(cluster, toRegion, toServer);
                if (!(fromRegionLocalityDelta + toRegionLocalityDelta >= 0.0)) continue;
                return Optional.of(this.getAction(fromServer, fromRegion, toServer, toRegion));
            }
            return Optional.absent();
        }

        private double getWeightedLocality(BaseLoadBalancer.Cluster cluster, int region, int server) {
            return cluster.getOrComputeWeightedLocality(region, server, BaseLoadBalancer.Cluster.LocalityType.SERVER);
        }

        void setServices(MasterServices services) {
            this.masterServices = services;
        }
    }

    static class LoadCandidateGenerator
    extends CandidateGenerator {
        LoadCandidateGenerator() {
        }

        @Override
        BaseLoadBalancer.Cluster.Action generate(BaseLoadBalancer.Cluster cluster) {
            cluster.sortServersByRegionCount();
            int thisServer = this.pickMostLoadedServer(cluster, -1);
            int otherServer = this.pickLeastLoadedServer(cluster, thisServer);
            return this.pickRandomRegions(cluster, thisServer, otherServer);
        }

        private int pickLeastLoadedServer(BaseLoadBalancer.Cluster cluster, int thisServer) {
            Integer[] servers = cluster.serverIndicesSortedByRegionCount;
            int index = 0;
            while (servers[index] == null || servers[index] == thisServer) {
                if (++index != servers.length) continue;
                return -1;
            }
            return servers[index];
        }

        private int pickMostLoadedServer(BaseLoadBalancer.Cluster cluster, int thisServer) {
            Integer[] servers = cluster.serverIndicesSortedByRegionCount;
            int index = servers.length - 1;
            while (servers[index] == null || servers[index] == thisServer) {
                if (--index >= 0) continue;
                return -1;
            }
            return servers[index];
        }
    }

    static class RandomCandidateGenerator
    extends CandidateGenerator {
        RandomCandidateGenerator() {
        }

        @Override
        BaseLoadBalancer.Cluster.Action generate(BaseLoadBalancer.Cluster cluster) {
            int thisServer = this.pickRandomServer(cluster);
            int otherServer = this.pickOtherRandomServer(cluster, thisServer);
            return this.pickRandomRegions(cluster, thisServer, otherServer);
        }
    }

    static abstract class CandidateGenerator {
        CandidateGenerator() {
        }

        abstract BaseLoadBalancer.Cluster.Action generate(BaseLoadBalancer.Cluster var1);

        protected int pickRandomRegion(BaseLoadBalancer.Cluster cluster, int server, double chanceOfNoSwap) {
            if (cluster.regionsPerServer[server].length == 0 || (double)RANDOM.nextFloat() < chanceOfNoSwap) {
                return -1;
            }
            int rand = RANDOM.nextInt(cluster.regionsPerServer[server].length);
            return cluster.regionsPerServer[server][rand];
        }

        protected int pickRandomServer(BaseLoadBalancer.Cluster cluster) {
            if (cluster.numServers < 1) {
                return -1;
            }
            return RANDOM.nextInt(cluster.numServers);
        }

        protected int pickRandomRack(BaseLoadBalancer.Cluster cluster) {
            if (cluster.numRacks < 1) {
                return -1;
            }
            return RANDOM.nextInt(cluster.numRacks);
        }

        protected int pickOtherRandomServer(BaseLoadBalancer.Cluster cluster, int serverIndex) {
            int otherServerIndex;
            if (cluster.numServers < 2) {
                return -1;
            }
            while ((otherServerIndex = this.pickRandomServer(cluster)) == serverIndex) {
            }
            return otherServerIndex;
        }

        protected int pickOtherRandomRack(BaseLoadBalancer.Cluster cluster, int rackIndex) {
            int otherRackIndex;
            if (cluster.numRacks < 2) {
                return -1;
            }
            while ((otherRackIndex = this.pickRandomRack(cluster)) == rackIndex) {
            }
            return otherRackIndex;
        }

        protected BaseLoadBalancer.Cluster.Action pickRandomRegions(BaseLoadBalancer.Cluster cluster, int thisServer, int otherServer) {
            int otherRegionCount;
            if (thisServer < 0 || otherServer < 0) {
                return BaseLoadBalancer.Cluster.NullAction;
            }
            int thisRegionCount = cluster.getNumRegions(thisServer);
            double thisChance = thisRegionCount > (otherRegionCount = cluster.getNumRegions(otherServer)) ? 0.0 : 0.5;
            double otherChance = thisRegionCount <= otherRegionCount ? 0.0 : 0.5;
            int thisRegion = this.pickRandomRegion(cluster, thisServer, thisChance);
            int otherRegion = this.pickRandomRegion(cluster, otherServer, otherChance);
            return this.getAction(thisServer, thisRegion, otherServer, otherRegion);
        }

        protected BaseLoadBalancer.Cluster.Action getAction(int fromServer, int fromRegion, int toServer, int toRegion) {
            if (fromServer < 0 || toServer < 0) {
                return BaseLoadBalancer.Cluster.NullAction;
            }
            if (fromRegion > 0 && toRegion > 0) {
                return new BaseLoadBalancer.Cluster.SwapRegionsAction(fromServer, fromRegion, toServer, toRegion);
            }
            if (fromRegion > 0) {
                return new BaseLoadBalancer.Cluster.MoveRegionAction(fromRegion, fromServer, toServer);
            }
            if (toRegion > 0) {
                return new BaseLoadBalancer.Cluster.MoveRegionAction(toRegion, toServer, fromServer);
            }
            return BaseLoadBalancer.Cluster.NullAction;
        }

        protected List<Integer> getRandomIterationOrder(int length) {
            ArrayList<Integer> order = new ArrayList<Integer>(length);
            for (int i = 0; i < length; ++i) {
                order.add(i);
            }
            Collections.shuffle(order);
            return order;
        }
    }
}

