/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.monitoring;

import java.io.PrintWriter;
import java.lang.ref.WeakReference;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.monitoring.MonitoredRPCHandler;
import org.apache.hadoop.hbase.monitoring.MonitoredRPCHandlerImpl;
import org.apache.hadoop.hbase.monitoring.MonitoredTask;
import org.apache.hadoop.hbase.monitoring.MonitoredTaskImpl;
import org.apache.hadoop.hbase.shaded.com.google.common.collect.Lists;
import org.apache.hadoop.hbase.shaded.org.apache.commons.collections4.queue.CircularFifoQueue;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.yetus.audience.InterfaceAudience;

@InterfaceAudience.Private
public class TaskMonitor {
    private static final Log LOG = LogFactory.getLog(TaskMonitor.class);
    public static final String MAX_TASKS_KEY = "hbase.taskmonitor.max.tasks";
    public static final int DEFAULT_MAX_TASKS = 1000;
    public static final String RPC_WARN_TIME_KEY = "hbase.taskmonitor.rpc.warn.time";
    public static final long DEFAULT_RPC_WARN_TIME = 0L;
    public static final String EXPIRATION_TIME_KEY = "hbase.taskmonitor.expiration.time";
    public static final long DEFAULT_EXPIRATION_TIME = 60000L;
    public static final String MONITOR_INTERVAL_KEY = "hbase.taskmonitor.monitor.interval";
    public static final long DEFAULT_MONITOR_INTERVAL = 10000L;
    private static TaskMonitor instance;
    private final int maxTasks;
    private final long rpcWarnTime;
    private final long expirationTime;
    private final CircularFifoQueue tasks;
    private final List<TaskAndWeakRefPair> rpcTasks;
    private final long monitorInterval;
    private Thread monitorThread;

    TaskMonitor(Configuration conf) {
        this.maxTasks = conf.getInt(MAX_TASKS_KEY, 1000);
        this.expirationTime = conf.getLong(EXPIRATION_TIME_KEY, 60000L);
        this.rpcWarnTime = conf.getLong(RPC_WARN_TIME_KEY, 0L);
        this.tasks = new CircularFifoQueue(this.maxTasks);
        this.rpcTasks = Lists.newArrayList();
        this.monitorInterval = conf.getLong(MONITOR_INTERVAL_KEY, 10000L);
        this.monitorThread = new Thread(new MonitorRunnable());
        Threads.setDaemonThreadRunning(this.monitorThread, "Monitor thread for TaskMonitor");
    }

    public static synchronized TaskMonitor get() {
        if (instance == null) {
            instance = new TaskMonitor(HBaseConfiguration.create());
        }
        return instance;
    }

    public synchronized MonitoredTask createStatus(String description) {
        MonitoredTaskImpl stat = new MonitoredTaskImpl();
        stat.setDescription(description);
        MonitoredTask proxy = (MonitoredTask)Proxy.newProxyInstance(stat.getClass().getClassLoader(), new Class[]{MonitoredTask.class}, new PassthroughInvocationHandler<MonitoredTaskImpl>(stat));
        TaskAndWeakRefPair pair = new TaskAndWeakRefPair(stat, proxy);
        if (this.tasks.isFull()) {
            this.purgeExpiredTasks();
        }
        this.tasks.add(pair);
        return proxy;
    }

    public synchronized MonitoredRPCHandler createRPCStatus(String description) {
        MonitoredRPCHandlerImpl stat = new MonitoredRPCHandlerImpl();
        stat.setDescription(description);
        MonitoredRPCHandler proxy = (MonitoredRPCHandler)Proxy.newProxyInstance(stat.getClass().getClassLoader(), new Class[]{MonitoredRPCHandler.class}, new PassthroughInvocationHandler<MonitoredRPCHandlerImpl>(stat));
        TaskAndWeakRefPair pair = new TaskAndWeakRefPair(stat, proxy);
        this.rpcTasks.add(pair);
        return proxy;
    }

    private synchronized void warnStuckTasks() {
        if (this.rpcWarnTime > 0L) {
            long now = EnvironmentEdgeManager.currentTime();
            for (TaskAndWeakRefPair pair : this.rpcTasks) {
                MonitoredTask stat = pair.get();
                if (stat.getState() != MonitoredTask.State.RUNNING || now < stat.getWarnTime() + this.rpcWarnTime) continue;
                LOG.warn((Object)("Task may be stuck: " + stat));
                stat.setWarnTime(now);
            }
        }
    }

    private synchronized void purgeExpiredTasks() {
        Iterator it = this.tasks.iterator();
        while (it.hasNext()) {
            TaskAndWeakRefPair pair = (TaskAndWeakRefPair)it.next();
            MonitoredTask stat = pair.get();
            if (pair.isDead() && stat.getState() == MonitoredTask.State.RUNNING) {
                LOG.warn((Object)("Status " + stat + " appears to have been leaked"));
                stat.cleanup();
            }
            if (!this.canPurge(stat)) continue;
            it.remove();
        }
    }

    public List<MonitoredTask> getTasks() {
        return this.getTasks(null);
    }

    public synchronized List<MonitoredTask> getTasks(String filter) {
        this.purgeExpiredTasks();
        TaskFilter taskFilter = TaskMonitor.createTaskFilter(filter);
        ArrayList<MonitoredTask> results = Lists.newArrayListWithCapacity(this.tasks.size() + this.rpcTasks.size());
        TaskMonitor.processTasks(this.tasks, taskFilter, results);
        TaskMonitor.processTasks(this.rpcTasks, taskFilter, results);
        return results;
    }

    private static TaskFilter createTaskFilter(String filter) {
        switch (TaskFilter.TaskType.getTaskType(filter)) {
            case GENERAL: {
                return task -> task instanceof MonitoredRPCHandler;
            }
            case HANDLER: {
                return task -> !(task instanceof MonitoredRPCHandler);
            }
            case RPC: {
                return task -> !(task instanceof MonitoredRPCHandler) || !((MonitoredRPCHandler)task).isRPCRunning();
            }
            case OPERATION: {
                return task -> !(task instanceof MonitoredRPCHandler) || !((MonitoredRPCHandler)task).isOperationRunning();
            }
        }
        return task -> false;
    }

    private static void processTasks(Iterable<TaskAndWeakRefPair> tasks, TaskFilter filter, List<MonitoredTask> results) {
        for (TaskAndWeakRefPair task : tasks) {
            MonitoredTask t = task.get();
            if (filter.filter(t)) continue;
            results.add(t.clone());
        }
    }

    private boolean canPurge(MonitoredTask stat) {
        long cts = stat.getCompletionTimestamp();
        return cts > 0L && EnvironmentEdgeManager.currentTime() - cts > this.expirationTime;
    }

    public void dumpAsText(PrintWriter out) {
        long now = EnvironmentEdgeManager.currentTime();
        List<MonitoredTask> tasks = this.getTasks();
        for (MonitoredTask task : tasks) {
            out.println("Task: " + task.getDescription());
            out.println("Status: " + (Object)((Object)task.getState()) + ":" + task.getStatus());
            long running = (now - task.getStartTime()) / 1000L;
            if (task.getCompletionTimestamp() != -1L) {
                long completed = (now - task.getCompletionTimestamp()) / 1000L;
                out.println("Completed " + completed + "s ago");
                out.println("Ran for " + (task.getCompletionTimestamp() - task.getStartTime()) / 1000L + "s");
            } else {
                out.println("Running for " + running + "s");
            }
            out.println();
        }
    }

    public synchronized void shutdown() {
        if (this.monitorThread != null) {
            this.monitorThread.interrupt();
        }
    }

    private static interface TaskFilter {
        public boolean filter(MonitoredTask var1);

        public static enum TaskType {
            GENERAL("general"),
            HANDLER("handler"),
            RPC("rpc"),
            OPERATION("operation"),
            ALL("all");

            private String type;

            private TaskType(String type) {
                this.type = type.toLowerCase();
            }

            static TaskType getTaskType(String type) {
                if (type == null || type.isEmpty()) {
                    return ALL;
                }
                type = type.toLowerCase();
                for (TaskType taskType : TaskType.values()) {
                    if (!taskType.toString().equals(type)) continue;
                    return taskType;
                }
                return ALL;
            }

            public String toString() {
                return this.type;
            }
        }
    }

    private class MonitorRunnable
    implements Runnable {
        private boolean running = true;

        private MonitorRunnable() {
        }

        @Override
        public void run() {
            while (this.running) {
                try {
                    Thread.sleep(TaskMonitor.this.monitorInterval);
                    if (TaskMonitor.this.tasks.isFull()) {
                        TaskMonitor.this.purgeExpiredTasks();
                    }
                    TaskMonitor.this.warnStuckTasks();
                }
                catch (InterruptedException e) {
                    this.running = false;
                }
            }
        }
    }

    private static class PassthroughInvocationHandler<T>
    implements InvocationHandler {
        private T delegatee;

        public PassthroughInvocationHandler(T delegatee) {
            this.delegatee = delegatee;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            return method.invoke(this.delegatee, args);
        }
    }

    private static class TaskAndWeakRefPair {
        private MonitoredTask impl;
        private WeakReference<MonitoredTask> weakProxy;

        public TaskAndWeakRefPair(MonitoredTask stat, MonitoredTask proxy) {
            this.impl = stat;
            this.weakProxy = new WeakReference<MonitoredTask>(proxy);
        }

        public MonitoredTask get() {
            return this.impl;
        }

        public boolean isDead() {
            return this.weakProxy.get() == null;
        }
    }
}

