/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.procedure2;

import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.exceptions.IllegalArgumentIOException;
import org.apache.hadoop.hbase.procedure2.Procedure;
import org.apache.hadoop.hbase.procedure2.ProcedureInMemoryChore;
import org.apache.hadoop.hbase.procedure2.ProcedureScheduler;
import org.apache.hadoop.hbase.procedure2.ProcedureStateSerializer;
import org.apache.hadoop.hbase.procedure2.ProcedureSuspendedException;
import org.apache.hadoop.hbase.procedure2.ProcedureYieldException;
import org.apache.hadoop.hbase.procedure2.RemoteProcedureException;
import org.apache.hadoop.hbase.procedure2.RootProcedureState;
import org.apache.hadoop.hbase.procedure2.SimpleProcedureScheduler;
import org.apache.hadoop.hbase.procedure2.store.ProcedureStore;
import org.apache.hadoop.hbase.procedure2.util.DelayedUtil;
import org.apache.hadoop.hbase.procedure2.util.StringUtils;
import org.apache.hadoop.hbase.security.User;
import org.apache.hadoop.hbase.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.hbase.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.hbase.shaded.protobuf.generated.ProcedureProtos;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.NonceKey;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class ProcedureExecutor<TEnvironment> {
    private static final Log LOG = LogFactory.getLog(ProcedureExecutor.class);
    public static final String CHECK_OWNER_SET_CONF_KEY = "hbase.procedure.check.owner.set";
    private static final boolean DEFAULT_CHECK_OWNER_SET = false;
    public static final String WORKER_KEEP_ALIVE_TIME_CONF_KEY = "hbase.procedure.worker.keep.alive.time.msec";
    private static final long DEFAULT_WORKER_KEEP_ALIVE_TIME = Long.MAX_VALUE;
    Testing testing = null;
    private final ConcurrentHashMap<Long, CompletedProcedureRetainer> completed = new ConcurrentHashMap();
    private final ConcurrentHashMap<Long, RootProcedureState> rollbackStack = new ConcurrentHashMap();
    private final ConcurrentHashMap<Long, Procedure> procedures = new ConcurrentHashMap();
    private final ConcurrentHashMap<NonceKey, Long> nonceKeysToProcIdsMap = new ConcurrentHashMap();
    private final CopyOnWriteArrayList<ProcedureExecutorListener> listeners = new CopyOnWriteArrayList();
    private Configuration conf;
    private ThreadGroup threadGroup;
    private CopyOnWriteArrayList<WorkerThread> workerThreads;
    private TimeoutExecutorThread timeoutExecutor;
    private int corePoolSize;
    private volatile long keepAliveTime = Long.MAX_VALUE;
    private final ProcedureScheduler scheduler;
    private final AtomicLong lastProcId = new AtomicLong(-1L);
    private final AtomicLong workerId = new AtomicLong(0L);
    private final AtomicInteger activeExecutorCount = new AtomicInteger(0);
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final TEnvironment environment;
    private final ProcedureStore store;
    private final boolean checkOwnerSet;

    public ProcedureExecutor(Configuration conf, TEnvironment environment, ProcedureStore store) {
        this(conf, environment, store, new SimpleProcedureScheduler());
    }

    public ProcedureExecutor(Configuration conf, TEnvironment environment, ProcedureStore store, ProcedureScheduler scheduler) {
        this.environment = environment;
        this.scheduler = scheduler;
        this.store = store;
        this.conf = conf;
        this.checkOwnerSet = conf.getBoolean(CHECK_OWNER_SET_CONF_KEY, false);
        this.refreshConfiguration(conf);
    }

    private void load(final boolean abortOnCorruption) throws IOException {
        Preconditions.checkArgument(this.completed.isEmpty(), "completed not empty");
        Preconditions.checkArgument(this.rollbackStack.isEmpty(), "rollback state not empty");
        Preconditions.checkArgument(this.procedures.isEmpty(), "procedure map not empty");
        Preconditions.checkArgument(this.scheduler.size() == 0, "run queue not empty");
        this.store.load(new ProcedureStore.ProcedureLoader(){

            @Override
            public void setMaxProcId(long maxProcId) {
                assert (ProcedureExecutor.this.lastProcId.get() < 0L) : "expected only one call to setMaxProcId()";
                LOG.debug((Object)("Load max pid=" + maxProcId));
                ProcedureExecutor.this.lastProcId.set(maxProcId);
            }

            @Override
            public void load(ProcedureStore.ProcedureIterator procIter) throws IOException {
                ProcedureExecutor.this.loadProcedures(procIter, abortOnCorruption);
            }

            @Override
            public void handleCorrupted(ProcedureStore.ProcedureIterator procIter) throws IOException {
                int corruptedCount = 0;
                while (procIter.hasNext()) {
                    Procedure proc = procIter.next();
                    LOG.error((Object)("Corrupt " + proc));
                    ++corruptedCount;
                }
                if (abortOnCorruption && corruptedCount > 0) {
                    throw new IOException("found " + corruptedCount + " corrupted procedure(s) on replay");
                }
            }
        });
    }

    private void loadProcedures(ProcedureStore.ProcedureIterator procIter, boolean abortOnCorruption) throws IOException {
        boolean debugEnabled = LOG.isDebugEnabled();
        int runnablesCount = 0;
        while (procIter.hasNext()) {
            boolean finished = procIter.isNextFinished();
            Procedure proc = procIter.next();
            NonceKey nonceKey = proc.getNonceKey();
            long procId = proc.getProcId();
            if (finished) {
                this.completed.put(proc.getProcId(), new CompletedProcedureRetainer(proc));
                if (debugEnabled) {
                    LOG.debug((Object)("Completed " + proc));
                }
            } else {
                if (!proc.hasParent()) {
                    assert (!proc.isFinished()) : "unexpected finished procedure";
                    this.rollbackStack.put(proc.getProcId(), new RootProcedureState());
                }
                proc.beforeReplay(this.getEnvironment());
                this.procedures.put(proc.getProcId(), proc);
                if (proc.getState() == ProcedureProtos.ProcedureState.RUNNABLE) {
                    ++runnablesCount;
                }
            }
            if (nonceKey == null) continue;
            this.nonceKeysToProcIdsMap.put(nonceKey, procId);
        }
        ArrayList<Procedure> runnableList = new ArrayList<Procedure>(runnablesCount);
        HashSet<Procedure> waitingSet = null;
        procIter.reset();
        while (procIter.hasNext()) {
            Procedure parent;
            Long rootProcId;
            if (procIter.isNextFinished()) {
                procIter.skipNext();
                continue;
            }
            Procedure proc = procIter.next();
            assert (!proc.isFinished() || proc.hasParent()) : "unexpected completed proc=" + proc;
            if (debugEnabled) {
                LOG.debug((Object)String.format("Loading %s", proc));
            }
            if ((rootProcId = this.getRootProcedureId(proc)) == null) {
                this.scheduler.addBack(proc);
                continue;
            }
            if (proc.hasParent() && (parent = this.procedures.get(proc.getParentProcId())) != null && !proc.isFinished()) {
                parent.incChildrenLatch();
            }
            RootProcedureState procStack = this.rollbackStack.get(rootProcId);
            procStack.loadStack(proc);
            proc.setRootProcId(rootProcId);
            switch (proc.getState()) {
                case RUNNABLE: {
                    runnableList.add(proc);
                    break;
                }
                case WAITING: {
                    if (proc.hasChildren()) break;
                    runnableList.add(proc);
                    break;
                }
                case WAITING_TIMEOUT: {
                    if (waitingSet == null) {
                        waitingSet = new HashSet<Procedure>();
                    }
                    waitingSet.add(proc);
                    break;
                }
                case FAILED: {
                    this.scheduler.addBack(proc);
                    break;
                }
                case ROLLEDBACK: 
                case INITIALIZING: {
                    String msg = "Unexpected " + proc.getState() + " state for " + proc;
                    LOG.error((Object)msg);
                    throw new UnsupportedOperationException(msg);
                }
            }
        }
        int corruptedCount = 0;
        Iterator<Map.Entry<Long, RootProcedureState>> itStack = this.rollbackStack.entrySet().iterator();
        while (itStack.hasNext()) {
            Map.Entry<Long, RootProcedureState> entry = itStack.next();
            RootProcedureState procStack = entry.getValue();
            if (procStack.isValid()) continue;
            for (Procedure proc : procStack.getSubproceduresStack()) {
                LOG.error((Object)("Corrupted " + proc));
                this.procedures.remove(proc.getProcId());
                runnableList.remove(proc);
                if (waitingSet != null) {
                    waitingSet.remove(proc);
                }
                ++corruptedCount;
            }
            itStack.remove();
        }
        if (abortOnCorruption && corruptedCount > 0) {
            throw new IOException("found " + corruptedCount + " procedures on replay");
        }
        if (waitingSet != null && !waitingSet.isEmpty()) {
            for (Procedure proc : waitingSet) {
                proc.afterReplay(this.getEnvironment());
                this.timeoutExecutor.add(proc);
            }
        }
        if (!runnableList.isEmpty()) {
            for (int i = runnableList.size() - 1; i >= 0; --i) {
                Procedure proc;
                proc = (Procedure)runnableList.get(i);
                proc.afterReplay(this.getEnvironment());
                if (!proc.hasParent()) {
                    this.sendProcedureLoadedNotification(proc.getProcId());
                }
                if (proc.wasExecuted()) {
                    this.scheduler.addFront(proc);
                    continue;
                }
                this.scheduler.addBack(proc);
            }
        }
    }

    public void start(int numThreads, boolean abortOnCorruption) throws IOException {
        if (this.running.getAndSet(true)) {
            LOG.warn((Object)"Already running");
            return;
        }
        this.corePoolSize = numThreads;
        LOG.info((Object)("Starting ProcedureExecutor Worker threads (ProcExecWrkr)=" + this.corePoolSize));
        this.threadGroup = new ThreadGroup("ProcExecThrdGrp");
        this.timeoutExecutor = new TimeoutExecutorThread(this.threadGroup);
        this.workerId.set(0L);
        this.workerThreads = new CopyOnWriteArrayList();
        for (int i = 0; i < this.corePoolSize; ++i) {
            this.workerThreads.add(new WorkerThread(this.threadGroup));
        }
        long st = EnvironmentEdgeManager.currentTime();
        this.store.recoverLease();
        long et = EnvironmentEdgeManager.currentTime();
        LOG.info((Object)String.format("Recover store (%s) lease: %s", this.store.getClass().getSimpleName(), StringUtils.humanTimeDiff(et - st)));
        this.scheduler.start();
        st = EnvironmentEdgeManager.currentTime();
        this.load(abortOnCorruption);
        et = EnvironmentEdgeManager.currentTime();
        LOG.info((Object)String.format("Load store (%s): %s", this.store.getClass().getSimpleName(), StringUtils.humanTimeDiff(et - st)));
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Start workers " + this.workerThreads.size()));
        }
        this.timeoutExecutor.start();
        for (WorkerThread worker : this.workerThreads) {
            worker.start();
        }
        this.timeoutExecutor.add(new WorkerMonitor());
        this.addChore(new CompletedProcedureCleaner(this.conf, this.store, this.completed, this.nonceKeysToProcIdsMap));
    }

    public void stop() {
        if (!this.running.getAndSet(false)) {
            return;
        }
        LOG.info((Object)"Stopping");
        this.scheduler.stop();
        this.timeoutExecutor.sendStopSignal();
    }

    @VisibleForTesting
    public void join() {
        assert (!this.isRunning()) : "expected not running";
        this.timeoutExecutor.awaitTermination();
        this.timeoutExecutor = null;
        for (WorkerThread worker : this.workerThreads) {
            worker.awaitTermination();
        }
        this.workerThreads = null;
        try {
            this.threadGroup.destroy();
        }
        catch (IllegalThreadStateException e) {
            LOG.error((Object)("Thread group " + this.threadGroup + " contains running threads"));
            this.threadGroup.list();
        }
        finally {
            this.threadGroup = null;
        }
        this.completed.clear();
        this.rollbackStack.clear();
        this.procedures.clear();
        this.nonceKeysToProcIdsMap.clear();
        this.scheduler.clear();
        this.lastProcId.set(-1L);
    }

    public void refreshConfiguration(Configuration conf) {
        this.conf = conf;
        this.setKeepAliveTime(conf.getLong(WORKER_KEEP_ALIVE_TIME_CONF_KEY, Long.MAX_VALUE), TimeUnit.MILLISECONDS);
    }

    public boolean isRunning() {
        return this.running.get();
    }

    public int getWorkerThreadCount() {
        return this.workerThreads.size();
    }

    public int getCorePoolSize() {
        return this.corePoolSize;
    }

    public int getActiveExecutorCount() {
        return this.activeExecutorCount.get();
    }

    public TEnvironment getEnvironment() {
        return this.environment;
    }

    public ProcedureStore getStore() {
        return this.store;
    }

    protected ProcedureScheduler getScheduler() {
        return this.scheduler;
    }

    public void setKeepAliveTime(long keepAliveTime, TimeUnit timeUnit) {
        this.keepAliveTime = timeUnit.toMillis(keepAliveTime);
        this.scheduler.signalAll();
    }

    public long getKeepAliveTime(TimeUnit timeUnit) {
        return timeUnit.convert(this.keepAliveTime, TimeUnit.MILLISECONDS);
    }

    public void addChore(ProcedureInMemoryChore chore) {
        chore.setState(ProcedureProtos.ProcedureState.WAITING_TIMEOUT);
        this.timeoutExecutor.add(chore);
    }

    public boolean removeChore(ProcedureInMemoryChore chore) {
        chore.setState(ProcedureProtos.ProcedureState.SUCCESS);
        return this.timeoutExecutor.remove(chore);
    }

    public NonceKey createNonceKey(long nonceGroup, long nonce) {
        return nonce == 0L ? null : new NonceKey(nonceGroup, nonce);
    }

    public long registerNonce(NonceKey nonceKey) {
        long newProcId;
        if (nonceKey == null) {
            return -1L;
        }
        Long oldProcId = this.nonceKeysToProcIdsMap.get(nonceKey);
        if (oldProcId == null && (oldProcId = this.nonceKeysToProcIdsMap.putIfAbsent(nonceKey, newProcId = this.nextProcId())) == null) {
            return -1L;
        }
        boolean traceEnabled = LOG.isTraceEnabled();
        while (this.isRunning() && !this.procedures.containsKey(oldProcId) && !this.completed.containsKey(oldProcId) && this.nonceKeysToProcIdsMap.containsKey(nonceKey)) {
            if (traceEnabled) {
                LOG.trace((Object)("Waiting for pid=" + oldProcId + " to be submitted"));
            }
            Threads.sleep(100L);
        }
        return oldProcId;
    }

    public void unregisterNonceIfProcedureWasNotSubmitted(NonceKey nonceKey) {
        if (nonceKey == null) {
            return;
        }
        Long procId = this.nonceKeysToProcIdsMap.get(nonceKey);
        if (procId == null) {
            return;
        }
        if (!this.procedures.containsKey(procId) && !this.completed.containsKey(procId)) {
            this.nonceKeysToProcIdsMap.remove(nonceKey);
        }
    }

    public void setFailureResultForNonce(NonceKey nonceKey, String procName, User procOwner, IOException exception) {
        if (nonceKey == null) {
            return;
        }
        Long procId = this.nonceKeysToProcIdsMap.get(nonceKey);
        if (procId == null || this.completed.containsKey(procId)) {
            return;
        }
        FailedProcedure proc = new FailedProcedure(procId, procName, procOwner, nonceKey, exception);
        this.completed.putIfAbsent(procId, new CompletedProcedureRetainer(proc));
    }

    public long submitProcedure(Procedure proc) {
        return this.submitProcedure(proc, null);
    }

    @SuppressWarnings(value={"NP_NULL_ON_SOME_PATH"}, justification="FindBugs is blind to the check-for-null")
    public long submitProcedure(Procedure proc, NonceKey nonceKey) {
        Long currentProcId;
        Preconditions.checkArgument(this.lastProcId.get() >= 0L);
        Preconditions.checkArgument(this.isRunning(), "executor not running");
        this.prepareProcedure(proc);
        if (nonceKey != null) {
            currentProcId = this.nonceKeysToProcIdsMap.get(nonceKey);
            Preconditions.checkArgument(currentProcId != null, "Expected nonceKey=" + nonceKey + " to be reserved, use registerNonce(); proc=" + proc);
        } else {
            currentProcId = this.nextProcId();
        }
        proc.setNonceKey(nonceKey);
        proc.setProcId(currentProcId);
        this.store.insert(proc, null);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Stored " + proc));
        }
        return this.pushProcedure(proc);
    }

    public void submitProcedures(Procedure[] procs) {
        int i;
        Preconditions.checkArgument(this.lastProcId.get() >= 0L);
        Preconditions.checkArgument(this.isRunning(), "executor not running");
        for (i = 0; i < procs.length; ++i) {
            this.prepareProcedure(procs[i]).setProcId(this.nextProcId());
        }
        this.store.insert(procs);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Stored " + Arrays.toString(procs)));
        }
        for (i = 0; i < procs.length; ++i) {
            this.pushProcedure(procs[i]);
        }
    }

    private Procedure prepareProcedure(Procedure proc) {
        Preconditions.checkArgument(proc.getState() == ProcedureProtos.ProcedureState.INITIALIZING);
        Preconditions.checkArgument(this.isRunning(), "executor not running");
        Preconditions.checkArgument(!proc.hasParent(), "unexpected parent", (Object)proc);
        if (this.checkOwnerSet) {
            Preconditions.checkArgument(proc.hasOwner(), "missing owner");
        }
        return proc;
    }

    private long pushProcedure(Procedure proc) {
        long currentProcId = proc.getProcId();
        proc.updateMetricsOnSubmit(this.getEnvironment());
        RootProcedureState stack = new RootProcedureState();
        this.rollbackStack.put(currentProcId, stack);
        assert (!this.procedures.containsKey(currentProcId));
        this.procedures.put(currentProcId, proc);
        this.sendProcedureAddedNotification(currentProcId);
        this.scheduler.addBack(proc);
        return proc.getProcId();
    }

    public boolean abort(long procId) {
        return this.abort(procId, true);
    }

    public boolean abort(long procId, boolean mayInterruptIfRunning) {
        Procedure proc = this.procedures.get(procId);
        if (proc != null) {
            if (!mayInterruptIfRunning && proc.wasExecuted()) {
                return false;
            }
            return proc.abort(this.getEnvironment());
        }
        return false;
    }

    public Procedure getProcedure(long procId) {
        return this.procedures.get(procId);
    }

    public <T extends Procedure> T getProcedure(Class<T> clazz, long procId) {
        Procedure proc = this.getProcedure(procId);
        if (clazz.isInstance(proc)) {
            return (T)proc;
        }
        return null;
    }

    public Procedure getResult(long procId) {
        CompletedProcedureRetainer retainer = this.completed.get(procId);
        if (retainer == null) {
            return null;
        }
        return retainer.getProcedure();
    }

    public boolean isFinished(long procId) {
        return !this.procedures.containsKey(procId);
    }

    public boolean isStarted(long procId) {
        Procedure proc = this.procedures.get(procId);
        if (proc == null) {
            return this.completed.get(procId) != null;
        }
        return proc.wasExecuted();
    }

    public void removeResult(long procId) {
        CompletedProcedureRetainer retainer = this.completed.get(procId);
        if (retainer == null) {
            assert (!this.procedures.containsKey(procId)) : "pid=" + procId + " is still running";
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("pid=" + procId + " already removed by the cleaner."));
            }
            return;
        }
        retainer.setClientAckTime(EnvironmentEdgeManager.currentTime());
    }

    public Procedure getResultOrProcedure(long procId) {
        CompletedProcedureRetainer retainer = this.completed.get(procId);
        if (retainer == null) {
            return this.procedures.get(procId);
        }
        return retainer.getProcedure();
    }

    public boolean isProcedureOwner(long procId, User user) {
        if (user == null) {
            return false;
        }
        Procedure runningProc = this.procedures.get(procId);
        if (runningProc != null) {
            return runningProc.getOwner().equals(user.getShortName());
        }
        CompletedProcedureRetainer retainer = this.completed.get(procId);
        if (retainer != null) {
            return retainer.getProcedure().getOwner().equals(user.getShortName());
        }
        return false;
    }

    public List<Procedure<?>> getProcedures() {
        ArrayList procedureLists = new ArrayList(this.procedures.size() + this.completed.size());
        for (Procedure procedure : this.procedures.values()) {
            procedureLists.add(procedure);
        }
        for (CompletedProcedureRetainer retainer : this.completed.values()) {
            procedureLists.add(retainer.getProcedure());
        }
        return procedureLists;
    }

    public void registerListener(ProcedureExecutorListener listener) {
        this.listeners.add(listener);
    }

    public boolean unregisterListener(ProcedureExecutorListener listener) {
        return this.listeners.remove(listener);
    }

    private void sendProcedureLoadedNotification(long procId) {
        if (!this.listeners.isEmpty()) {
            for (ProcedureExecutorListener listener : this.listeners) {
                try {
                    listener.procedureLoaded(procId);
                }
                catch (Throwable e) {
                    LOG.error((Object)("Listener " + listener + " had an error: " + e.getMessage()), e);
                }
            }
        }
    }

    private void sendProcedureAddedNotification(long procId) {
        if (!this.listeners.isEmpty()) {
            for (ProcedureExecutorListener listener : this.listeners) {
                try {
                    listener.procedureAdded(procId);
                }
                catch (Throwable e) {
                    LOG.error((Object)("Listener " + listener + " had an error: " + e.getMessage()), e);
                }
            }
        }
    }

    private void sendProcedureFinishedNotification(long procId) {
        if (!this.listeners.isEmpty()) {
            for (ProcedureExecutorListener listener : this.listeners) {
                try {
                    listener.procedureFinished(procId);
                }
                catch (Throwable e) {
                    LOG.error((Object)("Listener " + listener + " had an error: " + e.getMessage()), e);
                }
            }
        }
    }

    private long nextProcId() {
        long procId = this.lastProcId.incrementAndGet();
        if (procId < 0L) {
            while (!this.lastProcId.compareAndSet(procId, 0L) && (procId = this.lastProcId.get()) < 0L) {
            }
            while (this.procedures.containsKey(procId)) {
                procId = this.lastProcId.incrementAndGet();
            }
        }
        assert (procId >= 0L) : "Invalid procId " + procId;
        return procId;
    }

    @VisibleForTesting
    protected long getLastProcId() {
        return this.lastProcId.get();
    }

    @VisibleForTesting
    public Set<Long> getActiveProcIds() {
        return this.procedures.keySet();
    }

    private Long getRootProcedureId(Procedure proc) {
        return Procedure.getRootProcedureId(this.procedures, proc);
    }

    private void executeProcedure(Procedure proc) {
        Long rootProcId = this.getRootProcedureId(proc);
        if (rootProcId == null) {
            LOG.warn((Object)("Rollback because parent is done/rolledback proc=" + proc));
            this.executeRollback(proc);
            return;
        }
        RootProcedureState procStack = this.rollbackStack.get(rootProcId);
        if (procStack == null) {
            LOG.warn((Object)("RootProcedureState is null for " + proc.getProcId()));
            return;
        }
        block15: do {
            if (!procStack.acquire(proc)) {
                if (procStack.setRollback()) {
                    switch (this.executeRollback(rootProcId, procStack)) {
                        case LOCK_ACQUIRED: {
                            break block15;
                        }
                        case LOCK_YIELD_WAIT: {
                            procStack.unsetRollback();
                            this.scheduler.yield(proc);
                            break block15;
                        }
                        case LOCK_EVENT_WAIT: {
                            LOG.info((Object)("LOCK_EVENT_WAIT rollback..." + proc));
                            procStack.unsetRollback();
                            break block15;
                        }
                        default: {
                            throw new UnsupportedOperationException();
                        }
                    }
                }
                if (proc.wasExecuted()) break;
                switch (this.executeRollback(proc)) {
                    case LOCK_ACQUIRED: {
                        break block15;
                    }
                    case LOCK_YIELD_WAIT: {
                        this.scheduler.yield(proc);
                        break block15;
                    }
                    case LOCK_EVENT_WAIT: {
                        LOG.info((Object)("LOCK_EVENT_WAIT can't rollback child running?..." + proc));
                        break block15;
                    }
                    default: {
                        throw new UnsupportedOperationException();
                    }
                }
            }
            assert (proc.getState() == ProcedureProtos.ProcedureState.RUNNABLE) : proc;
            Procedure.LockState lockState = this.acquireLock(proc);
            switch (lockState) {
                case LOCK_ACQUIRED: {
                    this.execProcedure(procStack, proc);
                    this.releaseLock(proc, false);
                    break;
                }
                case LOCK_YIELD_WAIT: {
                    LOG.info((Object)((Object)((Object)lockState) + " " + proc));
                    this.scheduler.yield(proc);
                    break;
                }
                case LOCK_EVENT_WAIT: {
                    LOG.debug((Object)((Object)((Object)lockState) + " " + proc));
                    break;
                }
                default: {
                    throw new UnsupportedOperationException();
                }
            }
            procStack.release(proc);
            if (this.testing != null && !this.isRunning()) break;
            if (!proc.isSuccess()) continue;
            proc.updateMetricsOnFinish(this.getEnvironment(), proc.elapsedTime(), true);
            LOG.info((Object)("Finished " + proc + " in " + StringUtils.humanTimeDiff(proc.elapsedTime())));
            if (proc.getProcId() == rootProcId.longValue()) {
                this.procedureFinished(proc);
                break;
            }
            this.execCompletionCleanup(proc);
            break;
        } while (procStack.isFailed());
    }

    private Procedure.LockState acquireLock(Procedure proc) {
        TEnvironment env = this.getEnvironment();
        if (proc.holdLock(env) && proc.hasLock(env)) {
            return Procedure.LockState.LOCK_ACQUIRED;
        }
        return proc.doAcquireLock(env);
    }

    private void releaseLock(Procedure proc, boolean force) {
        TEnvironment env = this.getEnvironment();
        if (force || !proc.holdLock(env)) {
            proc.doReleaseLock(env);
        }
    }

    private Procedure.LockState executeRollback(long rootProcId, RootProcedureState procStack) {
        Procedure rootProc = this.procedures.get(rootProcId);
        RemoteProcedureException exception = rootProc.getException();
        if (exception == null) {
            exception = procStack.getException();
            rootProc.setFailure(exception);
            this.store.update(rootProc);
        }
        List<Procedure> subprocStack = procStack.getSubproceduresStack();
        assert (subprocStack != null) : "Called rollback with no steps executed rootProc=" + rootProc;
        int stackTail = subprocStack.size();
        boolean reuseLock = false;
        while (stackTail-- > 0) {
            Procedure.LockState lockState;
            Procedure proc = subprocStack.get(stackTail);
            if (!reuseLock && (lockState = this.acquireLock(proc)) != Procedure.LockState.LOCK_ACQUIRED) {
                return lockState;
            }
            lockState = this.executeRollback(proc);
            boolean abortRollback = lockState != Procedure.LockState.LOCK_ACQUIRED;
            boolean bl = reuseLock = stackTail > 0 && subprocStack.get(stackTail - 1) == proc && !(abortRollback |= !this.isRunning() || !this.store.isRunning());
            if (!reuseLock) {
                this.releaseLock(proc, false);
            }
            if (abortRollback) {
                return lockState;
            }
            subprocStack.remove(stackTail);
            if (proc.isYieldAfterExecutionStep(this.getEnvironment())) {
                return Procedure.LockState.LOCK_YIELD_WAIT;
            }
            if (proc == rootProc) continue;
            this.execCompletionCleanup(proc);
        }
        LOG.info((Object)("Rolled back " + rootProc + " exec-time=" + StringUtils.humanTimeDiff(rootProc.elapsedTime())));
        this.procedureFinished(rootProc);
        return Procedure.LockState.LOCK_ACQUIRED;
    }

    private Procedure.LockState executeRollback(Procedure proc) {
        try {
            proc.doRollback(this.getEnvironment());
        }
        catch (IOException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Roll back attempt failed for " + proc), (Throwable)e);
            }
            return Procedure.LockState.LOCK_YIELD_WAIT;
        }
        catch (InterruptedException e) {
            this.handleInterruptedException(proc, e);
            return Procedure.LockState.LOCK_YIELD_WAIT;
        }
        catch (Throwable e) {
            LOG.fatal((Object)("CODE-BUG: Uncaught runtime exception for " + proc), e);
        }
        if (this.testing != null && this.testing.shouldKillBeforeStoreUpdate()) {
            LOG.debug((Object)"TESTING: Kill before store update");
            this.stop();
            return Procedure.LockState.LOCK_YIELD_WAIT;
        }
        if (proc.removeStackIndex()) {
            proc.setState(ProcedureProtos.ProcedureState.ROLLEDBACK);
            proc.updateMetricsOnFinish(this.getEnvironment(), proc.elapsedTime(), false);
            if (proc.hasParent()) {
                this.store.delete(proc.getProcId());
                this.procedures.remove(proc.getProcId());
            } else {
                long[] childProcIds = this.rollbackStack.get(proc.getProcId()).getSubprocedureIds();
                if (childProcIds != null) {
                    this.store.delete(proc, childProcIds);
                } else {
                    this.store.update(proc);
                }
            }
        } else {
            this.store.update(proc);
        }
        return Procedure.LockState.LOCK_ACQUIRED;
    }

    private void execProcedure(RootProcedureState procStack, Procedure<TEnvironment> procedure) {
        Preconditions.checkArgument(procedure.getState() == ProcedureProtos.ProcedureState.RUNNABLE);
        boolean suspended = false;
        boolean reExecute = false;
        Procedure[] subprocs = null;
        do {
            reExecute = false;
            try {
                subprocs = procedure.doExecute(this.getEnvironment());
                if (subprocs != null && subprocs.length == 0) {
                    subprocs = null;
                }
            }
            catch (ProcedureSuspendedException e2) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)("Suspend " + procedure));
                }
                suspended = true;
            }
            catch (ProcedureYieldException e3) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)("Yield " + procedure + ": " + e3.getMessage()), (Throwable)e3);
                }
                this.scheduler.yield(procedure);
                return;
            }
            catch (InterruptedException e4) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)("Yield interrupt " + procedure + ": " + e4.getMessage()), (Throwable)e4);
                }
                this.handleInterruptedException(procedure, e4);
                this.scheduler.yield(procedure);
                return;
            }
            catch (Throwable e5) {
                String msg = "CODE-BUG: Uncaught runtime exception: " + procedure;
                LOG.error((Object)msg, e5);
                procedure.setFailure(new RemoteProcedureException(msg, e5));
            }
            if (!procedure.isFailed()) {
                if (subprocs != null) {
                    if (subprocs.length == 1 && subprocs[0] == procedure) {
                        subprocs = null;
                        reExecute = true;
                        if (LOG.isTraceEnabled()) {
                            LOG.trace((Object)("Short-circuit to next step on pid=" + procedure.getProcId()));
                        }
                    } else {
                        subprocs = this.initializeChildren(procStack, procedure, subprocs);
                        LOG.info((Object)("Initialized subprocedures=" + (subprocs == null ? null : Stream.of(subprocs).map(e -> "{" + e.toString() + "}").collect(Collectors.toList()).toString())));
                    }
                } else if (procedure.getState() == ProcedureProtos.ProcedureState.WAITING_TIMEOUT) {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace((Object)("Added to timeoutExecutor " + procedure));
                    }
                    this.timeoutExecutor.add(procedure);
                } else if (!suspended) {
                    procedure.setState(ProcedureProtos.ProcedureState.SUCCESS);
                }
            }
            procStack.addRollbackStep(procedure);
            if (this.testing != null && this.testing.shouldKillBeforeStoreUpdate(suspended)) {
                LOG.debug((Object)("TESTING: Kill before store update: " + procedure));
                this.stop();
                return;
            }
            this.updateStoreOnExec(procStack, procedure, subprocs);
            if (!this.store.isRunning()) {
                return;
            }
            if (procedure.isRunnable() && !suspended && procedure.isYieldAfterExecutionStep(this.getEnvironment())) {
                this.scheduler.yield(procedure);
                return;
            }
            assert (reExecute && subprocs == null || !reExecute);
        } while (reExecute);
        if (subprocs != null && !procedure.isFailed()) {
            this.submitChildrenProcedures(subprocs);
        }
        if (!suspended && procedure.isFinished() && procedure.hasParent()) {
            this.countDownChildren(procStack, procedure);
        }
    }

    private Procedure[] initializeChildren(RootProcedureState procStack, Procedure procedure, Procedure[] subprocs) {
        assert (subprocs != null) : "expected subprocedures";
        long rootProcId = this.getRootProcedureId(procedure);
        for (int i = 0; i < subprocs.length; ++i) {
            Procedure subproc = subprocs[i];
            if (subproc == null) {
                String msg = "subproc[" + i + "] is null, aborting the procedure";
                procedure.setFailure(new RemoteProcedureException(msg, new IllegalArgumentIOException(msg)));
                return null;
            }
            assert (subproc.getState() == ProcedureProtos.ProcedureState.INITIALIZING) : subproc;
            subproc.setParentProcId(procedure.getProcId());
            subproc.setRootProcId(rootProcId);
            subproc.setProcId(this.nextProcId());
            procStack.addSubProcedure(subproc);
        }
        if (!procedure.isFailed()) {
            procedure.setChildrenLatch(subprocs.length);
            switch (procedure.getState()) {
                case RUNNABLE: {
                    procedure.setState(ProcedureProtos.ProcedureState.WAITING);
                    break;
                }
                case WAITING_TIMEOUT: {
                    this.timeoutExecutor.add(procedure);
                    break;
                }
            }
        }
        return subprocs;
    }

    private void submitChildrenProcedures(Procedure[] subprocs) {
        for (int i = 0; i < subprocs.length; ++i) {
            Procedure subproc = subprocs[i];
            subproc.updateMetricsOnSubmit(this.getEnvironment());
            assert (!this.procedures.containsKey(subproc.getProcId()));
            this.procedures.put(subproc.getProcId(), subproc);
            this.scheduler.addFront(subproc);
        }
    }

    private void countDownChildren(RootProcedureState procStack, Procedure procedure) {
        Procedure parent = this.procedures.get(procedure.getParentProcId());
        if (parent == null) {
            assert (procStack.isRollingback());
            return;
        }
        LOG.info((Object)("Finish suprocedure " + procedure));
        if (parent.tryRunnable()) {
            this.store.update(parent);
            this.scheduler.addFront(parent);
            LOG.info((Object)("Finished subprocedure(s) of " + parent + "; resume parent processing."));
            return;
        }
    }

    private void updateStoreOnExec(RootProcedureState procStack, Procedure procedure, Procedure[] subprocs) {
        if (subprocs != null && !procedure.isFailed()) {
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("Stored " + procedure + ", children " + Arrays.toString(subprocs)));
            }
            this.store.insert(procedure, subprocs);
        } else {
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("Store update " + procedure));
            }
            if (procedure.isFinished() && !procedure.hasParent()) {
                long[] childProcIds = procStack.getSubprocedureIds();
                if (childProcIds != null) {
                    this.store.delete(procedure, childProcIds);
                    for (int i = 0; i < childProcIds.length; ++i) {
                        this.procedures.remove(childProcIds[i]);
                    }
                } else {
                    this.store.update(procedure);
                }
            } else {
                this.store.update(procedure);
            }
        }
    }

    private void handleInterruptedException(Procedure proc, InterruptedException e) {
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Interrupt during " + proc + ". suspend and retry it later."), (Throwable)e);
        }
    }

    private void execCompletionCleanup(Procedure proc) {
        TEnvironment env = this.getEnvironment();
        if (proc.holdLock(env) && proc.hasLock(env)) {
            this.releaseLock(proc, true);
        }
        try {
            proc.completionCleanup(env);
        }
        catch (Throwable e) {
            LOG.error((Object)("CODE-BUG: uncatched runtime exception for procedure: " + proc), e);
        }
    }

    private void procedureFinished(Procedure proc) {
        this.execCompletionCleanup(proc);
        CompletedProcedureRetainer retainer = new CompletedProcedureRetainer(proc);
        if (!proc.shouldWaitClientAck(this.getEnvironment())) {
            retainer.setClientAckTime(0L);
        }
        this.completed.put(proc.getProcId(), retainer);
        this.rollbackStack.remove(proc.getProcId());
        this.procedures.remove(proc.getProcId());
        try {
            this.scheduler.completionCleanup(proc);
        }
        catch (Throwable e) {
            LOG.error((Object)("CODE-BUG: uncatched runtime exception for completion cleanup: " + proc), e);
        }
        this.sendProcedureFinishedNotification(proc.getProcId());
    }

    private final class WorkerMonitor
    extends InlineChore {
        public static final String WORKER_MONITOR_INTERVAL_CONF_KEY = "hbase.procedure.worker.monitor.interval.msec";
        private static final int DEFAULT_WORKER_MONITOR_INTERVAL = 5000;
        public static final String WORKER_STUCK_THRESHOLD_CONF_KEY = "hbase.procedure.worker.stuck.threshold.msec";
        private static final int DEFAULT_WORKER_STUCK_THRESHOLD = 10000;
        public static final String WORKER_ADD_STUCK_PERCENTAGE_CONF_KEY = "hbase.procedure.worker.add.stuck.percentage";
        private static final float DEFAULT_WORKER_ADD_STUCK_PERCENTAGE = 0.5f;
        private float addWorkerStuckPercentage;
        private int timeoutInterval;
        private int stuckThreshold;

        public WorkerMonitor() {
            this.addWorkerStuckPercentage = 0.5f;
            this.timeoutInterval = 5000;
            this.stuckThreshold = 10000;
            this.refreshConfig();
        }

        @Override
        public void run() {
            int stuckCount = this.checkForStuckWorkers();
            this.checkThreadCount(stuckCount);
            this.refreshConfig();
        }

        private int checkForStuckWorkers() {
            int stuckCount = 0;
            for (WorkerThread worker : ProcedureExecutor.this.workerThreads) {
                if (worker.getCurrentRunTime() < (long)this.stuckThreshold) continue;
                ++stuckCount;
                LOG.warn((Object)("Worker stuck " + worker + " run time " + StringUtils.humanTimeDiff(worker.getCurrentRunTime())));
            }
            return stuckCount;
        }

        private void checkThreadCount(int stuckCount) {
            if (stuckCount < 1 || !ProcedureExecutor.this.scheduler.hasRunnables()) {
                return;
            }
            float stuckPerc = (float)stuckCount / (float)ProcedureExecutor.this.workerThreads.size();
            if (stuckPerc >= this.addWorkerStuckPercentage && ProcedureExecutor.this.activeExecutorCount.get() == ProcedureExecutor.this.workerThreads.size()) {
                WorkerThread worker = new WorkerThread(ProcedureExecutor.this.threadGroup);
                ProcedureExecutor.this.workerThreads.add(worker);
                worker.start();
                LOG.debug((Object)("Added new worker thread " + worker));
            }
        }

        private void refreshConfig() {
            this.addWorkerStuckPercentage = ProcedureExecutor.this.conf.getFloat(WORKER_ADD_STUCK_PERCENTAGE_CONF_KEY, 0.5f);
            this.timeoutInterval = ProcedureExecutor.this.conf.getInt(WORKER_MONITOR_INTERVAL_CONF_KEY, 5000);
            this.stuckThreshold = ProcedureExecutor.this.conf.getInt(WORKER_STUCK_THRESHOLD_CONF_KEY, 10000);
        }

        @Override
        public int getTimeoutInterval() {
            return this.timeoutInterval;
        }
    }

    private static abstract class InlineChore
    extends DelayedUtil.DelayedObject
    implements Runnable {
        private long timeout;

        private InlineChore() {
        }

        public abstract int getTimeoutInterval();

        protected void refreshTimeout() {
            this.timeout = EnvironmentEdgeManager.currentTime() + (long)this.getTimeoutInterval();
        }

        @Override
        public long getTimeout() {
            return this.timeout;
        }
    }

    private static abstract class StoppableThread
    extends Thread {
        public StoppableThread(ThreadGroup group, String name) {
            super(group, name);
        }

        public abstract void sendStopSignal();

        public void awaitTermination() {
            try {
                long startTime = EnvironmentEdgeManager.currentTime();
                int i = 0;
                while (this.isAlive()) {
                    this.sendStopSignal();
                    this.join(250L);
                    if (i > 0 && i % 8 == 0) {
                        LOG.warn((Object)("Waiting termination of thread " + this.getName() + ", " + StringUtils.humanTimeDiff(EnvironmentEdgeManager.currentTime() - startTime)));
                    }
                    ++i;
                }
            }
            catch (InterruptedException e) {
                LOG.warn((Object)(this.getName() + " join wait got interrupted"), (Throwable)e);
            }
        }
    }

    private static final class DelayedProcedure
    extends DelayedUtil.DelayedContainerWithTimestamp<Procedure> {
        public DelayedProcedure(Procedure procedure) {
            super(procedure, procedure.getTimeoutTimestamp());
        }
    }

    private final class TimeoutExecutorThread
    extends StoppableThread {
        private final DelayQueue<DelayedUtil.DelayedWithTimeout> queue;

        public TimeoutExecutorThread(ThreadGroup group) {
            super(group, "ProcExecTimeout");
            this.queue = new DelayQueue();
        }

        @Override
        public void sendStopSignal() {
            this.queue.add(DelayedUtil.DELAYED_POISON);
        }

        @Override
        public void run() {
            boolean traceEnabled = LOG.isTraceEnabled();
            while (ProcedureExecutor.this.isRunning()) {
                DelayedUtil.DelayedWithTimeout task = DelayedUtil.takeWithoutInterrupt(this.queue);
                if (task == null || task == DelayedUtil.DELAYED_POISON) continue;
                if (traceEnabled) {
                    LOG.trace((Object)("Executing " + task));
                }
                if (task instanceof InlineChore) {
                    this.execInlineChore((InlineChore)task);
                    continue;
                }
                if (task instanceof DelayedProcedure) {
                    this.execDelayedProcedure((DelayedProcedure)task);
                    continue;
                }
                LOG.error((Object)("CODE-BUG unknown timeout task type " + task));
            }
        }

        public void add(InlineChore chore) {
            chore.refreshTimeout();
            this.queue.add(chore);
        }

        public void add(Procedure procedure) {
            assert (procedure.getState() == ProcedureProtos.ProcedureState.WAITING_TIMEOUT);
            LOG.info((Object)("ADDED " + procedure + "; timeout=" + procedure.getTimeout() + ", timestamp=" + procedure.getTimeoutTimestamp()));
            this.queue.add(new DelayedProcedure(procedure));
        }

        public boolean remove(Procedure procedure) {
            return this.queue.remove(new DelayedProcedure(procedure));
        }

        private void execInlineChore(InlineChore chore) {
            chore.run();
            this.add(chore);
        }

        private void execDelayedProcedure(DelayedProcedure delayed) {
            Procedure procedure = (Procedure)delayed.getObject();
            if (procedure instanceof ProcedureInMemoryChore) {
                this.executeInMemoryChore((ProcedureInMemoryChore)procedure);
                procedure.updateTimestamp();
                if (procedure.isWaiting()) {
                    delayed.setTimeout(procedure.getTimeoutTimestamp());
                    this.queue.add(delayed);
                }
            } else {
                this.executeTimedoutProcedure(procedure);
            }
        }

        private void executeInMemoryChore(ProcedureInMemoryChore chore) {
            if (!chore.isWaiting()) {
                return;
            }
            try {
                chore.periodicExecute(ProcedureExecutor.this.getEnvironment());
            }
            catch (Throwable e) {
                LOG.error((Object)("Ignoring " + chore + " exception: " + e.getMessage()), e);
            }
        }

        private void executeTimedoutProcedure(Procedure proc) {
            if (proc.setTimeoutFailure(ProcedureExecutor.this.getEnvironment())) {
                long rootProcId = Procedure.getRootProcedureId(ProcedureExecutor.this.procedures, proc);
                RootProcedureState procStack = (RootProcedureState)ProcedureExecutor.this.rollbackStack.get(rootProcId);
                procStack.abort();
                ProcedureExecutor.this.store.update(proc);
                ProcedureExecutor.this.scheduler.addFront(proc);
            }
        }
    }

    private final class WorkerThread
    extends StoppableThread {
        private final AtomicLong executionStartTime;
        private Procedure activeProcedure;

        public WorkerThread(ThreadGroup group) {
            super(group, "ProcExecWrkr-" + ProcedureExecutor.this.workerId.incrementAndGet());
            this.executionStartTime = new AtomicLong(Long.MAX_VALUE);
        }

        @Override
        public void sendStopSignal() {
            ProcedureExecutor.this.scheduler.signalAll();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            long lastUpdate = EnvironmentEdgeManager.currentTime();
            try {
                while (ProcedureExecutor.this.isRunning() && this.keepAlive(lastUpdate)) {
                    this.activeProcedure = ProcedureExecutor.this.scheduler.poll(ProcedureExecutor.this.keepAliveTime, TimeUnit.MILLISECONDS);
                    if (this.activeProcedure == null) continue;
                    int activeCount = ProcedureExecutor.this.activeExecutorCount.incrementAndGet();
                    int runningCount = ProcedureExecutor.this.store.setRunningProcedureCount(activeCount);
                    if (LOG.isTraceEnabled()) {
                        LOG.trace((Object)("Execute pid=" + this.activeProcedure.getProcId() + " runningCount=" + runningCount + ", activeCount=" + activeCount));
                    }
                    this.executionStartTime.set(EnvironmentEdgeManager.currentTime());
                    try {
                        ProcedureExecutor.this.executeProcedure(this.activeProcedure);
                    }
                    catch (AssertionError e) {
                        LOG.info((Object)("ASSERT pid=" + this.activeProcedure.getProcId()), (Throwable)((Object)e));
                        throw e;
                    }
                    finally {
                        activeCount = ProcedureExecutor.this.activeExecutorCount.decrementAndGet();
                        runningCount = ProcedureExecutor.this.store.setRunningProcedureCount(activeCount);
                        if (LOG.isTraceEnabled()) {
                            LOG.trace((Object)("Halt pid=" + this.activeProcedure.getProcId() + " runningCount=" + runningCount + ", activeCount=" + activeCount));
                        }
                        this.activeProcedure = null;
                        lastUpdate = EnvironmentEdgeManager.currentTime();
                        this.executionStartTime.set(Long.MAX_VALUE);
                    }
                }
            }
            catch (Throwable t) {
                LOG.warn((Object)("Worker terminating UNNATURALLY " + this.activeProcedure), t);
            }
            finally {
                LOG.debug((Object)"Worker terminated.");
            }
            ProcedureExecutor.this.workerThreads.remove(this);
        }

        @Override
        public String toString() {
            Procedure p = this.activeProcedure;
            return this.getName() + "(pid=" + (p == null ? Long.valueOf(-1L) : p.getProcId() + ")");
        }

        public long getCurrentRunTime() {
            return EnvironmentEdgeManager.currentTime() - this.executionStartTime.get();
        }

        private boolean keepAlive(long lastUpdate) {
            if (ProcedureExecutor.this.workerThreads.size() <= ProcedureExecutor.this.corePoolSize) {
                return true;
            }
            return EnvironmentEdgeManager.currentTime() - lastUpdate < ProcedureExecutor.this.keepAliveTime;
        }
    }

    public static class FailedProcedure<TEnvironment>
    extends Procedure<TEnvironment> {
        private String procName;

        public FailedProcedure() {
        }

        public FailedProcedure(long procId, String procName, User owner, NonceKey nonceKey, IOException exception) {
            this.procName = procName;
            this.setProcId(procId);
            this.setState(ProcedureProtos.ProcedureState.ROLLEDBACK);
            this.setOwner(owner);
            this.setNonceKey(nonceKey);
            long currentTime = EnvironmentEdgeManager.currentTime();
            this.setSubmittedTime(currentTime);
            this.setLastUpdate(currentTime);
            this.setFailure(Objects.toString(exception.getMessage(), ""), exception);
        }

        @Override
        public String getProcName() {
            return this.procName;
        }

        @Override
        protected Procedure<TEnvironment>[] execute(TEnvironment env) throws ProcedureYieldException, ProcedureSuspendedException, InterruptedException {
            throw new UnsupportedOperationException();
        }

        @Override
        protected void rollback(TEnvironment env) throws IOException, InterruptedException {
            throw new UnsupportedOperationException();
        }

        @Override
        protected boolean abort(TEnvironment env) {
            throw new UnsupportedOperationException();
        }

        @Override
        protected void serializeStateData(ProcedureStateSerializer serializer) throws IOException {
        }

        @Override
        protected void deserializeStateData(ProcedureStateSerializer serializer) throws IOException {
        }
    }

    private static class CompletedProcedureCleaner<TEnvironment>
    extends ProcedureInMemoryChore<TEnvironment> {
        private static final Log LOG = LogFactory.getLog(CompletedProcedureCleaner.class);
        private static final String CLEANER_INTERVAL_CONF_KEY = "hbase.procedure.cleaner.interval";
        private static final int DEFAULT_CLEANER_INTERVAL = 30000;
        private static final String EVICT_TTL_CONF_KEY = "hbase.procedure.cleaner.evict.ttl";
        private static final int DEFAULT_EVICT_TTL = 900000;
        private static final String EVICT_ACKED_TTL_CONF_KEY = "hbase.procedure.cleaner.acked.evict.ttl";
        private static final int DEFAULT_ACKED_EVICT_TTL = 300000;
        private static final String BATCH_SIZE_CONF_KEY = "hbase.procedure.cleaner.evict.batch.size";
        private static final int DEFAULT_BATCH_SIZE = 32;
        private final Map<Long, CompletedProcedureRetainer> completed;
        private final Map<NonceKey, Long> nonceKeysToProcIdsMap;
        private final ProcedureStore store;
        private Configuration conf;

        public CompletedProcedureCleaner(Configuration conf, ProcedureStore store, Map<Long, CompletedProcedureRetainer> completedMap, Map<NonceKey, Long> nonceKeysToProcIdsMap) {
            super(conf.getInt(CLEANER_INTERVAL_CONF_KEY, 30000));
            this.completed = completedMap;
            this.nonceKeysToProcIdsMap = nonceKeysToProcIdsMap;
            this.store = store;
            this.conf = conf;
        }

        @Override
        protected void periodicExecute(TEnvironment env) {
            if (this.completed.isEmpty()) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)"No completed procedures to cleanup.");
                }
                return;
            }
            long evictTtl = this.conf.getInt(EVICT_TTL_CONF_KEY, 900000);
            long evictAckTtl = this.conf.getInt(EVICT_ACKED_TTL_CONF_KEY, 300000);
            int batchSize = this.conf.getInt(BATCH_SIZE_CONF_KEY, 32);
            long[] batchIds = new long[batchSize];
            int batchCount = 0;
            long now = EnvironmentEdgeManager.currentTime();
            Iterator<Map.Entry<Long, CompletedProcedureRetainer>> it = this.completed.entrySet().iterator();
            boolean debugEnabled = LOG.isDebugEnabled();
            while (it.hasNext() && this.store.isRunning()) {
                Map.Entry<Long, CompletedProcedureRetainer> entry = it.next();
                CompletedProcedureRetainer retainer = entry.getValue();
                Procedure<?> proc = retainer.getProcedure();
                if (!retainer.isExpired(now, evictTtl, evictAckTtl)) continue;
                if (debugEnabled) {
                    LOG.debug((Object)("Evict completed " + proc));
                }
                batchIds[batchCount++] = entry.getKey();
                if (batchCount == batchIds.length) {
                    this.store.delete(batchIds, 0, batchCount);
                    batchCount = 0;
                }
                it.remove();
                NonceKey nonceKey = proc.getNonceKey();
                if (nonceKey == null) continue;
                this.nonceKeysToProcIdsMap.remove(nonceKey);
            }
            if (batchCount > 0) {
                this.store.delete(batchIds, 0, batchCount);
            }
        }
    }

    private static class CompletedProcedureRetainer {
        private final Procedure<?> procedure;
        private long clientAckTime;

        public CompletedProcedureRetainer(Procedure<?> procedure) {
            this.procedure = procedure;
            this.clientAckTime = -1L;
        }

        public Procedure<?> getProcedure() {
            return this.procedure;
        }

        public boolean hasClientAckTime() {
            return this.clientAckTime != -1L;
        }

        public long getClientAckTime() {
            return this.clientAckTime;
        }

        public void setClientAckTime(long clientAckTime) {
            this.clientAckTime = clientAckTime;
        }

        public boolean isExpired(long now, long evictTtl, long evictAckTtl) {
            return this.hasClientAckTime() && now - this.getClientAckTime() >= evictAckTtl || now - this.procedure.getLastUpdate() >= evictTtl;
        }
    }

    public static interface ProcedureExecutorListener {
        public void procedureLoaded(long var1);

        public void procedureAdded(long var1);

        public void procedureFinished(long var1);
    }

    public static class Testing {
        protected boolean killIfSuspended = false;
        protected boolean killBeforeStoreUpdate = false;
        protected boolean toggleKillBeforeStoreUpdate = false;

        protected boolean shouldKillBeforeStoreUpdate() {
            boolean kill = this.killBeforeStoreUpdate;
            if (this.toggleKillBeforeStoreUpdate) {
                this.killBeforeStoreUpdate = !kill;
                LOG.warn((Object)("Toggle KILL before store update to: " + this.killBeforeStoreUpdate));
            }
            return kill;
        }

        protected boolean shouldKillBeforeStoreUpdate(boolean isSuspended) {
            return isSuspended && !this.killIfSuspended ? false : this.shouldKillBeforeStoreUpdate();
        }
    }
}

