/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.NavigableSet;
import java.util.NoSuchElementException;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.hbase.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.hbase.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
public class LeaseManager {
    public static final Log LOG = LogFactory.getLog(LeaseManager.class);
    private final FSNamesystem fsnamesystem;
    private long softLimit = 60000L;
    private long hardLimit = 3600000L;
    private final SortedMap<String, Lease> leases = new TreeMap<String, Lease>();
    private final NavigableSet<Lease> sortedLeases = new TreeSet<Lease>();
    private final SortedMap<String, Lease> sortedLeasesByPath = new TreeMap<String, Lease>();
    private Daemon lmthread;
    private volatile boolean shouldRunMonitor;

    LeaseManager(FSNamesystem fsnamesystem) {
        this.fsnamesystem = fsnamesystem;
    }

    Lease getLease(String holder) {
        return (Lease)this.leases.get(holder);
    }

    @VisibleForTesting
    int getNumSortedLeases() {
        return this.sortedLeases.size();
    }

    synchronized long getNumUnderConstructionBlocks() {
        assert (this.fsnamesystem.hasReadLock()) : "The FSNamesystem read lock wasn'tacquired before counting under construction blocks";
        long numUCBlocks = 0L;
        for (Lease lease : this.sortedLeases) {
            for (String path : lease.getPaths()) {
                INodeFile cons;
                try {
                    cons = this.fsnamesystem.getFSDirectory().getINode(path).asFile();
                    Preconditions.checkState(cons.isUnderConstruction());
                }
                catch (UnresolvedLinkException e) {
                    throw new AssertionError((Object)"Lease files should reside on this FS");
                }
                BlockInfoContiguous[] blocks = cons.getBlocks();
                if (blocks == null) continue;
                for (BlockInfoContiguous b : blocks) {
                    if (b.isComplete()) continue;
                    ++numUCBlocks;
                }
            }
        }
        LOG.info((Object)("Number of blocks under construction: " + numUCBlocks));
        return numUCBlocks;
    }

    public Lease getLeaseByPath(String src) {
        return (Lease)this.sortedLeasesByPath.get(src);
    }

    public synchronized int countLease() {
        return this.sortedLeases.size();
    }

    synchronized int countPath() {
        int count = 0;
        for (Lease lease : this.sortedLeases) {
            count += lease.getPaths().size();
        }
        return count;
    }

    synchronized Lease addLease(String holder, String src) {
        Lease lease = this.getLease(holder);
        if (lease == null) {
            lease = new Lease(holder);
            this.leases.put(holder, lease);
            this.sortedLeases.add(lease);
        } else {
            this.renewLease(lease);
        }
        this.sortedLeasesByPath.put(src, lease);
        lease.paths.add(src);
        return lease;
    }

    synchronized void removeLease(Lease lease, String src) {
        this.sortedLeasesByPath.remove(src);
        if (!lease.removePath(src) && LOG.isDebugEnabled()) {
            LOG.debug((Object)(src + " not found in lease.paths (=" + lease.paths + ")"));
        }
        if (!lease.hasPath()) {
            this.leases.remove(lease.holder);
            if (!this.sortedLeases.remove(lease)) {
                LOG.error((Object)(lease + " not found in sortedLeases"));
            }
        }
    }

    synchronized void removeLease(String holder, String src) {
        Lease lease = this.getLease(holder);
        if (lease != null) {
            this.removeLease(lease, src);
        } else {
            LOG.warn((Object)("Removing non-existent lease! holder=" + holder + " src=" + src));
        }
    }

    synchronized void removeAllLeases() {
        this.sortedLeases.clear();
        this.sortedLeasesByPath.clear();
        this.leases.clear();
    }

    synchronized Lease reassignLease(Lease lease, String src, String newHolder) {
        assert (newHolder != null) : "new lease holder is null";
        if (lease != null) {
            this.removeLease(lease, src);
        }
        return this.addLease(newHolder, src);
    }

    synchronized void renewLease(String holder) {
        this.renewLease(this.getLease(holder));
    }

    synchronized void renewLease(Lease lease) {
        if (lease != null) {
            this.sortedLeases.remove(lease);
            lease.renew();
            this.sortedLeases.add(lease);
        }
    }

    synchronized void renewAllLeases() {
        for (Lease l : this.leases.values()) {
            this.renewLease(l);
        }
    }

    synchronized void changeLease(String src, String dst) {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(this.getClass().getSimpleName() + ".changelease: " + " src=" + src + ", dest=" + dst));
        }
        int len = src.length();
        for (Map.Entry<String, Lease> entry : LeaseManager.findLeaseWithPrefixPath(src, this.sortedLeasesByPath).entrySet()) {
            String oldpath = entry.getKey();
            Lease lease = entry.getValue();
            String newpath = dst + oldpath.substring(len);
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("changeLease: replacing " + oldpath + " with " + newpath));
            }
            lease.replacePath(oldpath, newpath);
            this.sortedLeasesByPath.remove(oldpath);
            this.sortedLeasesByPath.put(newpath, lease);
        }
    }

    synchronized void removeLeaseWithPrefixPath(String prefix) {
        for (Map.Entry<String, Lease> entry : LeaseManager.findLeaseWithPrefixPath(prefix, this.sortedLeasesByPath).entrySet()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)(LeaseManager.class.getSimpleName() + ".removeLeaseWithPrefixPath: entry=" + entry));
            }
            this.removeLease(entry.getValue(), entry.getKey());
        }
    }

    private static Map<String, Lease> findLeaseWithPrefixPath(String prefix, SortedMap<String, Lease> path2lease) {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(LeaseManager.class.getSimpleName() + ".findLease: prefix=" + prefix));
        }
        HashMap<String, Lease> entries = new HashMap<String, Lease>();
        int srclen = prefix.length();
        if (prefix.charAt(srclen - 1) == '/') {
            --srclen;
        }
        for (Map.Entry<String, Lease> entry : path2lease.tailMap(prefix).entrySet()) {
            String p = entry.getKey();
            if (!p.startsWith(prefix)) {
                return entries;
            }
            if (p.length() != srclen && p.charAt(srclen) != '/') continue;
            entries.put(entry.getKey(), entry.getValue());
        }
        return entries;
    }

    public void setLeasePeriod(long softLimit, long hardLimit) {
        this.softLimit = softLimit;
        this.hardLimit = hardLimit;
    }

    Map<String, INodeFile> getINodesUnderConstruction() {
        TreeMap<String, INodeFile> inodes = new TreeMap<String, INodeFile>();
        for (String p : this.sortedLeasesByPath.keySet()) {
            try {
                INodeFile node = INodeFile.valueOf(this.fsnamesystem.dir.getINode(p), p);
                Preconditions.checkState(node.isUnderConstruction());
                inodes.put(p, node);
            }
            catch (IOException ioe) {
                LOG.error((Object)ioe);
            }
        }
        return inodes;
    }

    @VisibleForTesting
    synchronized boolean checkLeases() {
        boolean needSync = false;
        assert (this.fsnamesystem.hasWriteLock());
        Lease leaseToCheck = null;
        try {
            leaseToCheck = (Lease)this.sortedLeases.first();
        }
        catch (NoSuchElementException noSuchElementException) {
            // empty catch block
        }
        while (leaseToCheck != null && leaseToCheck.expiredHardLimit()) {
            LOG.info((Object)(leaseToCheck + " has expired hard limit"));
            ArrayList<String> removing = new ArrayList<String>();
            String[] leasePaths = new String[leaseToCheck.getPaths().size()];
            leaseToCheck.getPaths().toArray(leasePaths);
            for (String p : leasePaths) {
                try {
                    INodesInPath iip = this.fsnamesystem.getFSDirectory().getINodesInPath(p, true);
                    boolean completed = this.fsnamesystem.internalReleaseLease(leaseToCheck, p, iip, "HDFS_NameNode");
                    if (LOG.isDebugEnabled()) {
                        if (completed) {
                            LOG.debug((Object)("Lease recovery for " + p + " is complete. File closed."));
                        } else {
                            LOG.debug((Object)("Started block recovery " + p + " lease " + leaseToCheck));
                        }
                    }
                    if (needSync || completed) continue;
                    needSync = true;
                }
                catch (IOException e) {
                    LOG.error((Object)("Cannot release the path " + p + " in the lease " + leaseToCheck), (Throwable)e);
                    removing.add(p);
                }
            }
            for (String p : removing) {
                this.removeLease(leaseToCheck, p);
            }
            leaseToCheck = this.sortedLeases.higher(leaseToCheck);
        }
        try {
            if (leaseToCheck != this.sortedLeases.first()) {
                LOG.warn((Object)("Unable to release hard-limit expired lease: " + this.sortedLeases.first()));
            }
        }
        catch (NoSuchElementException noSuchElementException) {
            // empty catch block
        }
        return needSync;
    }

    public synchronized String toString() {
        return this.getClass().getSimpleName() + "= {" + "\n leases=" + this.leases + "\n sortedLeases=" + this.sortedLeases + "\n sortedLeasesByPath=" + this.sortedLeasesByPath + "\n}";
    }

    void startMonitor() {
        Preconditions.checkState(this.lmthread == null, "Lease Monitor already running");
        this.shouldRunMonitor = true;
        this.lmthread = new Daemon(new Monitor());
        this.lmthread.start();
    }

    void stopMonitor() {
        if (this.lmthread != null) {
            this.shouldRunMonitor = false;
            try {
                this.lmthread.interrupt();
                this.lmthread.join(3000L);
            }
            catch (InterruptedException ie) {
                LOG.warn((Object)"Encountered exception ", (Throwable)ie);
            }
            this.lmthread = null;
        }
    }

    @VisibleForTesting
    void triggerMonitorCheckNow() {
        Preconditions.checkState(this.lmthread != null, "Lease monitor is not running");
        this.lmthread.interrupt();
    }

    class Monitor
    implements Runnable {
        final String name = this.getClass().getSimpleName();

        Monitor() {
        }

        @Override
        public void run() {
            while (LeaseManager.this.shouldRunMonitor && LeaseManager.this.fsnamesystem.isRunning()) {
                boolean needSync = false;
                try {
                    LeaseManager.this.fsnamesystem.writeLockInterruptibly();
                    try {
                        if (!LeaseManager.this.fsnamesystem.isInSafeMode()) {
                            needSync = LeaseManager.this.checkLeases();
                        }
                    }
                    finally {
                        LeaseManager.this.fsnamesystem.writeUnlock();
                        if (needSync) {
                            LeaseManager.this.fsnamesystem.getEditLog().logSync();
                        }
                    }
                    Thread.sleep(2000L);
                }
                catch (InterruptedException ie) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug((Object)(this.name + " is interrupted"), (Throwable)ie);
                }
            }
        }
    }

    class Lease
    implements Comparable<Lease> {
        private final String holder;
        private long lastUpdate;
        private final Collection<String> paths = new TreeSet<String>();

        private Lease(String holder) {
            this.holder = holder;
            this.renew();
        }

        private void renew() {
            this.lastUpdate = Time.monotonicNow();
        }

        public boolean expiredHardLimit() {
            return Time.monotonicNow() - this.lastUpdate > LeaseManager.this.hardLimit;
        }

        public boolean expiredSoftLimit() {
            return Time.monotonicNow() - this.lastUpdate > LeaseManager.this.softLimit;
        }

        boolean hasPath() {
            return !this.paths.isEmpty();
        }

        boolean removePath(String src) {
            return this.paths.remove(src);
        }

        public String toString() {
            return "[Lease.  Holder: " + this.holder + ", pendingcreates: " + this.paths.size() + "]";
        }

        @Override
        public int compareTo(Lease o) {
            Lease l1 = this;
            Lease l2 = o;
            long lu1 = l1.lastUpdate;
            long lu2 = l2.lastUpdate;
            if (lu1 < lu2) {
                return -1;
            }
            if (lu1 > lu2) {
                return 1;
            }
            return l1.holder.compareTo(l2.holder);
        }

        public boolean equals(Object o) {
            if (!(o instanceof Lease)) {
                return false;
            }
            Lease obj = (Lease)o;
            return this.lastUpdate == obj.lastUpdate && this.holder.equals(obj.holder);
        }

        public int hashCode() {
            return this.holder.hashCode();
        }

        Collection<String> getPaths() {
            return this.paths;
        }

        String getHolder() {
            return this.holder;
        }

        void replacePath(String oldpath, String newpath) {
            this.paths.remove(oldpath);
            this.paths.add(newpath);
        }

        @VisibleForTesting
        long getLastUpdate() {
            return this.lastUpdate;
        }
    }
}

