/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.mob.mapreduce;

import java.io.IOException;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashSet;
import java.util.UUID;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptor;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.ConnectionFactory;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.io.HFileLink;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapReduceUtil;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.mob.MobUtils;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.HFileArchiveUtil;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.hadoop.mapreduce.Reducer;
import org.apache.hadoop.mapreduce.lib.output.TextOutputFormat;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class MobRefReporter
extends Configured
implements Tool {
    private static Logger LOG = LoggerFactory.getLogger(MobRefReporter.class);
    public static final String NAME = "mobrefs";
    static final String REPORT_JOB_ID = "mob.report.job.id";
    static final String REPORT_START_DATETIME = "mob.report.job.start";

    static String log10GroupedString(long number) {
        return String.format("%,d", (long)Math.pow(10.0, Math.floor(Math.log10(number))));
    }

    public int run(String[] args) throws IOException, InterruptedException {
        if (args.length != 3) {
            this.printUsage();
            return 1;
        }
        String output = args[0];
        String tableName = args[1];
        String familyName = args[2];
        long reportStartTime = EnvironmentEdgeManager.currentTime();
        Configuration conf = this.getConf();
        try {
            int maxVersions;
            byte[] family;
            FileSystem fs = FileSystem.get((Configuration)conf);
            String currentUserName = UserGroupInformation.getCurrentUser().getShortUserName();
            FileStatus[] hbaseRootFileStat = fs.listStatus(new Path(conf.get("hbase.rootdir")));
            if (hbaseRootFileStat.length > 0) {
                String owner = hbaseRootFileStat[0].getOwner();
                if (!owner.equals(currentUserName)) {
                    String errorMsg = "The current user[" + currentUserName + "] does not have hbase root credentials. If this job fails due to an inability to read HBase's internal directories, you will need to rerun as a user with sufficient permissions. The HBase superuser is a safe choice.";
                    LOG.warn(errorMsg);
                }
            } else {
                LOG.error("The passed configs point to an HBase dir does not exist: {}", (Object)conf.get("hbase.rootdir"));
                throw new IOException("The target HBase does not exist");
            }
            TableName tn = TableName.valueOf(tableName);
            try (Connection connection = ConnectionFactory.createConnection(conf);
                 Admin admin = connection.getAdmin();){
                TableDescriptor htd = admin.getDescriptor(tn);
                ColumnFamilyDescriptor hcd = htd.getColumnFamily(Bytes.toBytes(familyName));
                if (hcd == null || !hcd.isMobEnabled()) {
                    throw new IOException("Column family " + familyName + " is not a MOB column family");
                }
                family = hcd.getName();
                maxVersions = hcd.getMaxVersions();
            }
            String id = ((Object)((Object)this)).getClass().getSimpleName() + UUID.randomUUID().toString().replace("-", "");
            Job job = null;
            Scan scan = new Scan();
            scan.addFamily(family);
            scan.setAttribute("hbase.mob.scan.raw", Bytes.toBytes(Boolean.TRUE));
            scan.setAttribute("hbase.mob.scan.ref.only", Bytes.toBytes(Boolean.TRUE));
            scan.setCaching(conf.getInt("hbase.client.scanner.caching", 10000));
            scan.setCacheBlocks(false);
            scan.setMaxVersions(maxVersions);
            conf.set(REPORT_JOB_ID, id);
            job = Job.getInstance((Configuration)conf);
            job.setJarByClass(((Object)((Object)this)).getClass());
            TableMapReduceUtil.initTableMapperJob(tn, scan, MobRefMapper.class, Text.class, ImmutableBytesWritable.class, job);
            job.setReducerClass(MobRefReducer.class);
            job.setOutputFormatClass(TextOutputFormat.class);
            TextOutputFormat.setOutputPath((Job)job, (Path)new Path(output));
            job.setJobName(((Object)((Object)this)).getClass().getSimpleName() + "-" + tn + "-" + familyName);
            job.getConfiguration().set("hbase.mapreduce.scan.column.family", familyName);
            job.getConfiguration().setLong(REPORT_START_DATETIME, reportStartTime);
            if (!job.waitForCompletion(true)) {
                System.err.println("Job was not successful");
                return 3;
            }
            LOG.info("Finished creating report for '{}', family='{}'", (Object)tn, (Object)familyName);
            return 0;
        }
        catch (IOException | ClassNotFoundException | InterruptedException | RuntimeException e) {
            System.err.println("Job aborted due to exception " + e);
            return 2;
        }
    }

    public static void main(String[] args) throws Exception {
        Configuration conf = HBaseConfiguration.create();
        int ret = ToolRunner.run((Configuration)conf, (Tool)new MobRefReporter(), (String[])args);
        System.exit(ret);
    }

    private void printUsage() {
        System.err.println("Usage:\n--------------------------\n" + MobRefReporter.class.getName() + " output-dir tableName familyName");
        System.err.println(" output-dir       Where to write output report.");
        System.err.println(" tableName        The table name");
        System.err.println(" familyName       The column family name");
    }

    public static class MobRefReducer
    extends Reducer<Text, ImmutableBytesWritable, Text, Text> {
        TableName table;
        String mobRegion;
        Path mob;
        Path archive;
        String seperator;
        final Text OK_MOB_DIR = new Text("MOB DIR");
        final Text OK_HLINK_RESTORE = new Text("HLINK TO ARCHIVE FOR SAME TABLE");
        final Text OK_HLINK_CLONE = new Text("HLINK TO ARCHIVE FOR OTHER TABLE");
        final Text INCONSISTENT_ARCHIVE_BAD_LINK = new Text("ARCHIVE WITH HLINK BUT NOT FROM OUR TABLE");
        final Text INCONSISTENT_ARCHIVE_STALE = new Text("ARCHIVE BUT NO HLINKS");
        final Text INCONSISTENT_ARCHIVE_IOE = new Text("ARCHIVE BUT FAILURE WHILE CHECKING HLINKS");
        final Text DATALOSS_MISSING = new Text("MISSING FILE");
        final Text DATALOSS_HLINK_DANGLING = new Text("HLINK BUT POINTS TO MISSING FILE");
        final Text DATALOSS_MISSING_IOE = new Text("MISSING FILE BUT FAILURE WHILE CHECKING HLINKS");
        final Base64.Encoder base64 = Base64.getEncoder();

        public void setup(Reducer.Context context) throws IOException, InterruptedException {
            Configuration conf = context.getConfiguration();
            String tableName = conf.get("hbase.mapreduce.inputtable");
            if (null == tableName) {
                throw new IOException("Job configuration did not include table.");
            }
            this.table = TableName.valueOf(tableName);
            this.mobRegion = MobUtils.getMobRegionInfo(this.table).getEncodedName();
            String family = conf.get("hbase.mapreduce.scan.column.family");
            if (null == family) {
                throw new IOException("Job configuration did not include column family");
            }
            this.mob = MobUtils.getMobFamilyPath(conf, this.table, family);
            LOG.info("Using active mob area '{}'", (Object)this.mob);
            this.archive = HFileArchiveUtil.getStoreArchivePath(conf, this.table, MobUtils.getMobRegionInfo(this.table).getEncodedName(), family);
            LOG.info("Using archive mob area '{}'", (Object)this.archive);
            this.seperator = conf.get(TextOutputFormat.SEPERATOR, "\t");
        }

        public void reduce(Text key, Iterable<ImmutableBytesWritable> rows, Reducer.Context context) throws IOException, InterruptedException {
            block20: {
                Configuration conf = context.getConfiguration();
                String file = key.toString();
                if (this.mob.getFileSystem(conf).exists(new Path(this.mob, file))) {
                    LOG.debug("Found file '{}' in mob area", (Object)file);
                    context.write((Object)this.OK_MOB_DIR, (Object)key);
                } else if (this.archive.getFileSystem(conf).exists(new Path(this.archive, file))) {
                    Path backRefDir = HFileLink.getBackReferencesDir(this.archive, file);
                    try {
                        FileStatus[] backRefs = FSUtils.listStatus(this.archive.getFileSystem(conf), backRefDir);
                        if (backRefs != null) {
                            boolean found = false;
                            for (FileStatus backRef : backRefs) {
                                Pair<TableName, String> refParts = HFileLink.parseBackReferenceName(backRef.getPath().getName());
                                if (!this.table.equals(refParts.getFirst()) || !this.mobRegion.equals(refParts.getSecond())) continue;
                                Path hlinkPath = HFileLink.getHFileFromBackReference(MobUtils.getMobHome(conf), backRef.getPath());
                                if (hlinkPath.getFileSystem(conf).exists(hlinkPath)) {
                                    found = true;
                                    continue;
                                }
                                LOG.warn("Found file '{}' in archive area with a back reference to the mob area for our table, but the mob area does not have a corresponding hfilelink.", (Object)file);
                            }
                            if (found) {
                                LOG.debug("Found file '{}' in archive area. has proper hlink back references to suggest it is from a restored snapshot for this table.", (Object)file);
                                context.write((Object)this.OK_HLINK_RESTORE, (Object)key);
                            } else {
                                LOG.warn("Found file '{}' in archive area, but the hlink back references do not properly point to the mob area for our table.", (Object)file);
                                context.write((Object)this.INCONSISTENT_ARCHIVE_BAD_LINK, (Object)this.encodeRows(context, key, rows));
                            }
                            break block20;
                        }
                        LOG.warn("Found file '{}' in archive area, but there are no hlinks pointing to it. Not yet used snapshot or an error.", (Object)file);
                        context.write((Object)this.INCONSISTENT_ARCHIVE_STALE, (Object)this.encodeRows(context, key, rows));
                    }
                    catch (IOException e) {
                        LOG.warn("Found file '{}' in archive area, but got an error while checking on back references.", (Object)file, (Object)e);
                        context.write((Object)this.INCONSISTENT_ARCHIVE_IOE, (Object)this.encodeRows(context, key, rows));
                    }
                } else {
                    try {
                        Object[] hlinks = this.mob.getFileSystem(conf).globStatus(new Path(this.mob + "/*=*-" + file));
                        if (hlinks != null && hlinks.length != 0) {
                            if (hlinks.length != 1) {
                                LOG.warn("Found file '{}' as hfilelinks in the mob area, but there are more than one: {}", (Object)file, (Object)Arrays.deepToString(hlinks));
                            }
                            HFileLink found = null;
                            for (Object hlink : hlinks) {
                                HFileLink tmp = HFileLink.buildFromHFileLinkPattern(conf, hlink.getPath());
                                if (tmp.exists(this.archive.getFileSystem(conf))) {
                                    found = tmp;
                                    break;
                                }
                                LOG.debug("Target file does not exist for ref {}", (Object)tmp);
                            }
                            if (found != null) {
                                LOG.debug("Found file '{}' as a ref in the mob area: {}", (Object)file, found);
                                context.write((Object)this.OK_HLINK_CLONE, (Object)key);
                            } else {
                                LOG.warn("Found file '{}' as ref(s) in the mob area but they do not point to an hfile that exists.", (Object)file);
                                context.write((Object)this.DATALOSS_HLINK_DANGLING, (Object)this.encodeRows(context, key, rows));
                            }
                        } else {
                            LOG.error("Could not find referenced file '{}'. See the docs on this tool.", (Object)file);
                            LOG.debug("Note that we don't have the server-side tag from the mob cells that says what table the reference is originally from. So if the HFileLink in this table is missing but the referenced file is still in the table from that tag, then lookups of these impacted rows will work. Do a scan of the reference details of the cell for the hfile name and then check the entire hbase install if this table was made from a snapshot of another table. see the ref guide section on mob for details.");
                            context.write((Object)this.DATALOSS_MISSING, (Object)this.encodeRows(context, key, rows));
                        }
                    }
                    catch (IOException e) {
                        LOG.error("Exception while checking mob area of our table for HFileLinks that point to {}", (Object)file, (Object)e);
                        context.write((Object)this.DATALOSS_MISSING_IOE, (Object)this.encodeRows(context, key, rows));
                    }
                }
            }
        }

        private Text encodeRows(Reducer.Context context, Text key, Iterable<ImmutableBytesWritable> rows) throws IOException {
            StringBuilder sb = new StringBuilder(key.toString());
            sb.append(this.seperator);
            boolean moreThanOne = false;
            long count = 0L;
            for (ImmutableBytesWritable row : rows) {
                if (moreThanOne) {
                    sb.append(",");
                }
                sb.append(this.base64.encodeToString(row.copyBytes()));
                moreThanOne = true;
                ++count;
            }
            context.getCounter("PROBLEM", "Problem MOB files").increment(1L);
            context.getCounter("PROBLEM", "Affected rows").increment(count);
            context.getCounter("ROWS WITH PROBLEMS PER FILE", "Number of HFiles with " + MobRefReporter.log10GroupedString(count) + "s of affected rows").increment(1L);
            key.set(sb.toString());
            return key;
        }
    }

    public static class MobRefMapper
    extends TableMapper<Text, ImmutableBytesWritable> {
        public void map(ImmutableBytesWritable r, Result columns, Mapper.Context context) throws IOException, InterruptedException {
            if (columns == null) {
                return;
            }
            Cell[] cells = columns.rawCells();
            if (cells == null || cells.length == 0) {
                return;
            }
            HashSet<String> files = new HashSet<String>();
            long count = 0L;
            long size = 0L;
            for (Cell c : cells) {
                if (MobUtils.hasValidMobRefCellValue(c)) {
                    String fileName = MobUtils.getMobFileName(c);
                    if (!files.contains(fileName)) {
                        context.write((Object)new Text(fileName), (Object)r);
                        files.add(fileName);
                    }
                    int cellsize = MobUtils.getMobValueLength(c);
                    context.getCounter("SIZES OF CELLS", "Number of cells with size in the " + MobRefReporter.log10GroupedString(cellsize) + "s of bytes").increment(1L);
                    size += (long)cellsize;
                    ++count;
                    continue;
                }
                LOG.debug("cell is not a mob ref, even though we asked for only refs. cell={}", (Object)c);
            }
            context.getCounter("CELLS PER ROW", "Number of rows with " + MobRefReporter.log10GroupedString(count) + "s of cells per row").increment(1L);
            context.getCounter("SIZES OF ROWS", "Number of rows with total size in the " + MobRefReporter.log10GroupedString(size) + "s of bytes").increment(1L);
            context.getCounter("MOB", "NUM_CELLS").increment(count);
        }
    }
}

