/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.cleaner;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.conf.ConfigurationObserver;
import org.apache.hadoop.hbase.io.HFileLink;
import org.apache.hadoop.hbase.master.cleaner.BaseHFileCleanerDelegate;
import org.apache.hadoop.hbase.master.cleaner.CleanerChore;
import org.apache.hadoop.hbase.master.cleaner.DirScanPool;
import org.apache.hadoop.hbase.regionserver.StoreFileInfo;
import org.apache.hadoop.hbase.util.StealJobQueue;
import org.apache.hbase.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class HFileCleaner
extends CleanerChore<BaseHFileCleanerDelegate>
implements ConfigurationObserver {
    public static final String MASTER_HFILE_CLEANER_PLUGINS = "hbase.master.hfilecleaner.plugins";
    public static final String HFILE_DELETE_THROTTLE_THRESHOLD = "hbase.regionserver.thread.hfilecleaner.throttle";
    public static final int DEFAULT_HFILE_DELETE_THROTTLE_THRESHOLD = 0x4000000;
    public static final String LARGE_HFILE_QUEUE_INIT_SIZE = "hbase.regionserver.hfilecleaner.large.queue.size";
    public static final int DEFAULT_LARGE_HFILE_QUEUE_INIT_SIZE = 10240;
    public static final String SMALL_HFILE_QUEUE_INIT_SIZE = "hbase.regionserver.hfilecleaner.small.queue.size";
    public static final int DEFAULT_SMALL_HFILE_QUEUE_INIT_SIZE = 10240;
    public static final String LARGE_HFILE_DELETE_THREAD_NUMBER = "hbase.regionserver.hfilecleaner.large.thread.count";
    public static final int DEFAULT_LARGE_HFILE_DELETE_THREAD_NUMBER = 1;
    public static final String SMALL_HFILE_DELETE_THREAD_NUMBER = "hbase.regionserver.hfilecleaner.small.thread.count";
    public static final int DEFAULT_SMALL_HFILE_DELETE_THREAD_NUMBER = 1;
    public static final String HFILE_DELETE_THREAD_TIMEOUT_MSEC = "hbase.regionserver.hfilecleaner.thread.timeout.msec";
    @VisibleForTesting
    static final long DEFAULT_HFILE_DELETE_THREAD_TIMEOUT_MSEC = 60000L;
    public static final String HFILE_DELETE_THREAD_CHECK_INTERVAL_MSEC = "hbase.regionserver.hfilecleaner.thread.check.interval.msec";
    @VisibleForTesting
    static final long DEFAULT_HFILE_DELETE_THREAD_CHECK_INTERVAL_MSEC = 1000L;
    private static final Logger LOG = LoggerFactory.getLogger(HFileCleaner.class);
    StealJobQueue<HFileDeleteTask> largeFileQueue;
    BlockingQueue<HFileDeleteTask> smallFileQueue;
    private int throttlePoint;
    private int largeQueueInitSize;
    private int smallQueueInitSize;
    private int largeFileDeleteThreadNumber;
    private int smallFileDeleteThreadNumber;
    private long cleanerThreadTimeoutMsec;
    private long cleanerThreadCheckIntervalMsec;
    private List<Thread> threads = new ArrayList<Thread>();
    private volatile boolean running;
    private AtomicLong deletedLargeFiles = new AtomicLong();
    private AtomicLong deletedSmallFiles = new AtomicLong();
    private static final Comparator<HFileDeleteTask> COMPARATOR = new Comparator<HFileDeleteTask>(){

        @Override
        public int compare(HFileDeleteTask o1, HFileDeleteTask o2) {
            int cmp = Long.compare(o2.fileLength, o1.fileLength);
            if (cmp != 0) {
                return cmp;
            }
            return System.identityHashCode(o1) - System.identityHashCode(o2);
        }
    };

    public HFileCleaner(int period, Stoppable stopper, Configuration conf, FileSystem fs, Path directory, DirScanPool pool) {
        this(period, stopper, conf, fs, directory, pool, null);
    }

    public HFileCleaner(int period, Stoppable stopper, Configuration conf, FileSystem fs, Path directory, DirScanPool pool, Map<String, Object> params) {
        this("HFileCleaner", period, stopper, conf, fs, directory, MASTER_HFILE_CLEANER_PLUGINS, pool, params);
    }

    public HFileCleaner(String name, int period, Stoppable stopper, Configuration conf, FileSystem fs, Path directory, String confKey, DirScanPool pool, Map<String, Object> params) {
        super(name, period, stopper, conf, fs, directory, confKey, pool, params);
        this.throttlePoint = conf.getInt(HFILE_DELETE_THROTTLE_THRESHOLD, 0x4000000);
        this.largeQueueInitSize = conf.getInt(LARGE_HFILE_QUEUE_INIT_SIZE, 10240);
        this.smallQueueInitSize = conf.getInt(SMALL_HFILE_QUEUE_INIT_SIZE, 10240);
        this.largeFileQueue = new StealJobQueue<HFileDeleteTask>(this.largeQueueInitSize, this.smallQueueInitSize, COMPARATOR);
        this.smallFileQueue = this.largeFileQueue.getStealFromQueue();
        this.largeFileDeleteThreadNumber = conf.getInt(LARGE_HFILE_DELETE_THREAD_NUMBER, 1);
        this.smallFileDeleteThreadNumber = conf.getInt(SMALL_HFILE_DELETE_THREAD_NUMBER, 1);
        this.cleanerThreadTimeoutMsec = conf.getLong(HFILE_DELETE_THREAD_TIMEOUT_MSEC, 60000L);
        this.cleanerThreadCheckIntervalMsec = conf.getLong(HFILE_DELETE_THREAD_CHECK_INTERVAL_MSEC, 1000L);
        this.startHFileDeleteThreads();
    }

    @Override
    protected boolean validate(Path file) {
        return HFileLink.isBackReferencesDir(file) || HFileLink.isBackReferencesDir(file.getParent()) || StoreFileInfo.validateStoreFileName(file.getName()) || file.getName().endsWith("$-masterlocalhfile-$");
    }

    public List<BaseHFileCleanerDelegate> getDelegatesForTesting() {
        return this.cleanersChain;
    }

    @Override
    public int deleteFiles(Iterable<FileStatus> filesToDelete) {
        int deletedFiles = 0;
        ArrayList<HFileDeleteTask> tasks = new ArrayList<HFileDeleteTask>();
        for (FileStatus file : filesToDelete) {
            HFileDeleteTask task = this.deleteFile(file);
            if (task == null) continue;
            tasks.add(task);
        }
        for (HFileDeleteTask task : tasks) {
            if (!task.getResult(this.cleanerThreadCheckIntervalMsec)) continue;
            ++deletedFiles;
        }
        return deletedFiles;
    }

    private HFileDeleteTask deleteFile(FileStatus file) {
        HFileDeleteTask task = new HFileDeleteTask(file, this.cleanerThreadTimeoutMsec);
        boolean enqueued = this.dispatch(task);
        return enqueued ? task : null;
    }

    private boolean dispatch(HFileDeleteTask task) {
        if (task.fileLength >= (long)this.throttlePoint) {
            if (!this.largeFileQueue.offer(task)) {
                LOG.trace("Large file deletion queue is full");
                return false;
            }
        } else if (!this.smallFileQueue.offer(task)) {
            LOG.trace("Small file deletion queue is full");
            return false;
        }
        return true;
    }

    @Override
    public synchronized void cleanup() {
        super.cleanup();
        this.stopHFileDeleteThreads();
    }

    private void startHFileDeleteThreads() {
        int i;
        String n = Thread.currentThread().getName();
        this.running = true;
        for (i = 0; i < this.largeFileDeleteThreadNumber; ++i) {
            Thread large = new Thread(){

                @Override
                public void run() {
                    HFileCleaner.this.consumerLoop(HFileCleaner.this.largeFileQueue);
                }
            };
            large.setDaemon(true);
            large.setName(n + "-HFileCleaner.large." + i + "-" + System.currentTimeMillis());
            large.start();
            LOG.debug("Starting for large file={}", (Object)large);
            this.threads.add(large);
        }
        for (i = 0; i < this.smallFileDeleteThreadNumber; ++i) {
            Thread small = new Thread(){

                @Override
                public void run() {
                    HFileCleaner.this.consumerLoop(HFileCleaner.this.smallFileQueue);
                }
            };
            small.setDaemon(true);
            small.setName(n + "-HFileCleaner.small." + i + "-" + System.currentTimeMillis());
            small.start();
            LOG.debug("Starting for small files={}", (Object)small);
            this.threads.add(small);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void consumerLoop(BlockingQueue<HFileDeleteTask> queue) {
        try {
            while (this.running) {
                boolean succeed;
                HFileDeleteTask task = null;
                try {
                    task = queue.take();
                }
                catch (InterruptedException e) {
                    LOG.trace("Interrupted while trying to take a task from queue", (Throwable)e);
                    break;
                }
                if (task == null) continue;
                LOG.trace("Removing {}", (Object)task.filePath);
                try {
                    succeed = this.fs.delete(task.filePath, false);
                }
                catch (IOException e) {
                    LOG.warn("Failed to delete {}", (Object)task.filePath, (Object)e);
                    succeed = false;
                }
                task.setResult(succeed);
                if (!succeed) continue;
                this.countDeletedFiles(task.fileLength >= (long)this.throttlePoint, queue == this.largeFileQueue);
            }
        }
        finally {
            LOG.debug("Exit {}", (Object)Thread.currentThread());
        }
    }

    private void countDeletedFiles(boolean isLargeFile, boolean fromLargeQueue) {
        if (isLargeFile) {
            if (this.deletedLargeFiles.get() == Long.MAX_VALUE) {
                LOG.debug("Deleted more than Long.MAX_VALUE large files, reset counter to 0");
                this.deletedLargeFiles.set(0L);
            }
            this.deletedLargeFiles.incrementAndGet();
        } else {
            if (this.deletedSmallFiles.get() == Long.MAX_VALUE) {
                LOG.debug("Deleted more than Long.MAX_VALUE small files, reset counter to 0");
                this.deletedSmallFiles.set(0L);
            }
            if (fromLargeQueue) {
                LOG.trace("Stolen a small file deletion task in large file thread");
            }
            this.deletedSmallFiles.incrementAndGet();
        }
    }

    private void stopHFileDeleteThreads() {
        this.running = false;
        LOG.debug("Stopping file delete threads");
        for (Thread thread : this.threads) {
            thread.interrupt();
        }
    }

    @VisibleForTesting
    public List<Thread> getCleanerThreads() {
        return this.threads;
    }

    @VisibleForTesting
    public long getNumOfDeletedLargeFiles() {
        return this.deletedLargeFiles.get();
    }

    @VisibleForTesting
    public long getNumOfDeletedSmallFiles() {
        return this.deletedSmallFiles.get();
    }

    @VisibleForTesting
    public long getLargeQueueInitSize() {
        return this.largeQueueInitSize;
    }

    @VisibleForTesting
    public long getSmallQueueInitSize() {
        return this.smallQueueInitSize;
    }

    @VisibleForTesting
    public long getThrottlePoint() {
        return this.throttlePoint;
    }

    @VisibleForTesting
    long getCleanerThreadTimeoutMsec() {
        return this.cleanerThreadTimeoutMsec;
    }

    @VisibleForTesting
    long getCleanerThreadCheckIntervalMsec() {
        return this.cleanerThreadCheckIntervalMsec;
    }

    @Override
    public void onConfigurationChange(Configuration conf) {
        if (!this.checkAndUpdateConfigurations(conf)) {
            LOG.debug("Update configuration triggered but nothing changed for this cleaner");
            return;
        }
        this.stopHFileDeleteThreads();
        ArrayList<HFileDeleteTask> leftOverTasks = new ArrayList<HFileDeleteTask>(this.largeFileQueue.size() + this.smallFileQueue.size());
        leftOverTasks.addAll(this.largeFileQueue);
        leftOverTasks.addAll(this.smallFileQueue);
        this.largeFileQueue = new StealJobQueue<HFileDeleteTask>(this.largeQueueInitSize, this.smallQueueInitSize, COMPARATOR);
        this.smallFileQueue = this.largeFileQueue.getStealFromQueue();
        this.threads.clear();
        this.startHFileDeleteThreads();
        for (HFileDeleteTask task : leftOverTasks) {
            this.dispatch(task);
        }
    }

    private boolean checkAndUpdateConfigurations(Configuration conf) {
        long cleanerThreadCheckIntervalMsec;
        long cleanerThreadTimeoutMsec;
        int smallFileDeleteThreadNumber;
        int largeFileDeleteThreadNumber;
        int smallQueueInitSize;
        int largeQueueInitSize;
        boolean updated = false;
        int throttlePoint = conf.getInt(HFILE_DELETE_THROTTLE_THRESHOLD, 0x4000000);
        if (throttlePoint != this.throttlePoint) {
            LOG.debug("Updating throttle point, from {} to {}", (Object)this.throttlePoint, (Object)throttlePoint);
            this.throttlePoint = throttlePoint;
            updated = true;
        }
        if ((largeQueueInitSize = conf.getInt(LARGE_HFILE_QUEUE_INIT_SIZE, 10240)) != this.largeQueueInitSize) {
            LOG.debug("Updating largeQueueInitSize, from {} to {}", (Object)this.largeQueueInitSize, (Object)largeQueueInitSize);
            this.largeQueueInitSize = largeQueueInitSize;
            updated = true;
        }
        if ((smallQueueInitSize = conf.getInt(SMALL_HFILE_QUEUE_INIT_SIZE, 10240)) != this.smallQueueInitSize) {
            LOG.debug("Updating smallQueueInitSize, from {} to {}", (Object)this.smallQueueInitSize, (Object)smallQueueInitSize);
            this.smallQueueInitSize = smallQueueInitSize;
            updated = true;
        }
        if ((largeFileDeleteThreadNumber = conf.getInt(LARGE_HFILE_DELETE_THREAD_NUMBER, 1)) != this.largeFileDeleteThreadNumber) {
            LOG.debug("Updating largeFileDeleteThreadNumber, from {} to {}", (Object)this.largeFileDeleteThreadNumber, (Object)largeFileDeleteThreadNumber);
            this.largeFileDeleteThreadNumber = largeFileDeleteThreadNumber;
            updated = true;
        }
        if ((smallFileDeleteThreadNumber = conf.getInt(SMALL_HFILE_DELETE_THREAD_NUMBER, 1)) != this.smallFileDeleteThreadNumber) {
            LOG.debug("Updating smallFileDeleteThreadNumber, from {} to {}", (Object)this.smallFileDeleteThreadNumber, (Object)smallFileDeleteThreadNumber);
            this.smallFileDeleteThreadNumber = smallFileDeleteThreadNumber;
            updated = true;
        }
        if ((cleanerThreadTimeoutMsec = conf.getLong(HFILE_DELETE_THREAD_TIMEOUT_MSEC, 60000L)) != this.cleanerThreadTimeoutMsec) {
            this.cleanerThreadTimeoutMsec = cleanerThreadTimeoutMsec;
            updated = true;
        }
        if ((cleanerThreadCheckIntervalMsec = conf.getLong(HFILE_DELETE_THREAD_CHECK_INTERVAL_MSEC, 1000L)) != this.cleanerThreadCheckIntervalMsec) {
            this.cleanerThreadCheckIntervalMsec = cleanerThreadCheckIntervalMsec;
            updated = true;
        }
        return updated;
    }

    @Override
    public synchronized void cancel(boolean mayInterruptIfRunning) {
        super.cancel(mayInterruptIfRunning);
        for (Thread t : this.threads) {
            t.interrupt();
        }
    }

    private static final class HFileDeleteTask {
        boolean done = false;
        boolean result;
        final Path filePath;
        final long fileLength;
        final long timeoutMsec;

        public HFileDeleteTask(FileStatus file, long timeoutMsec) {
            this.filePath = file.getPath();
            this.fileLength = file.getLen();
            this.timeoutMsec = timeoutMsec;
        }

        public synchronized void setResult(boolean result) {
            this.done = true;
            this.result = result;
            this.notify();
        }

        public synchronized boolean getResult(long waitIfNotFinished) {
            long waitTimeMsec = 0L;
            try {
                while (!this.done) {
                    long startTimeNanos = System.nanoTime();
                    this.wait(waitIfNotFinished);
                    waitTimeMsec += TimeUnit.MILLISECONDS.convert(System.nanoTime() - startTimeNanos, TimeUnit.NANOSECONDS);
                    if (this.done) {
                        return this.result;
                    }
                    if (waitTimeMsec <= this.timeoutMsec) continue;
                    LOG.warn("Wait more than " + this.timeoutMsec + " ms for deleting " + this.filePath + ", exit...");
                    return false;
                }
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupted while waiting for result of deleting " + this.filePath + ", will return false", (Throwable)e);
                return false;
            }
            return this.result;
        }
    }
}

