/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.client;

import java.io.Closeable;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.AsyncProcess;
import org.apache.hadoop.hbase.client.AsyncProcessTask;
import org.apache.hadoop.hbase.client.AsyncRequestFuture;
import org.apache.hadoop.hbase.client.BufferedMutator;
import org.apache.hadoop.hbase.client.BufferedMutatorParams;
import org.apache.hadoop.hbase.client.ClusterConnection;
import org.apache.hadoop.hbase.client.ConnectionConfiguration;
import org.apache.hadoop.hbase.client.ConnectionUtils;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Mutation;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.RetriesExhaustedWithDetailsException;
import org.apache.hadoop.hbase.client.Row;
import org.apache.hadoop.hbase.client.RowAccess;
import org.apache.hadoop.hbase.client.RpcRetryingCallerFactory;
import org.apache.hadoop.hbase.ipc.RpcControllerFactory;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class BufferedMutatorImpl
implements BufferedMutator {
    private static final Logger LOG = LoggerFactory.getLogger(BufferedMutatorImpl.class);
    private final BufferedMutator.ExceptionListener listener;
    private final TableName tableName;
    private final Configuration conf;
    private final ConcurrentLinkedQueue<Mutation> writeAsyncBuffer = new ConcurrentLinkedQueue();
    private final AtomicLong currentWriteBufferSize = new AtomicLong(0L);
    private final AtomicInteger undealtMutationCount = new AtomicInteger(0);
    private final long writeBufferSize;
    private final AtomicLong writeBufferPeriodicFlushTimeoutMs = new AtomicLong(0L);
    private final AtomicLong writeBufferPeriodicFlushTimerTickMs = new AtomicLong(100L);
    private Timer writeBufferPeriodicFlushTimer = null;
    private final int maxKeyValueSize;
    private final ExecutorService pool;
    private final AtomicInteger rpcTimeout;
    private final AtomicInteger operationTimeout;
    private final boolean cleanupPoolOnClose;
    private volatile boolean closed = false;
    private final AsyncProcess ap;
    private final AtomicLong firstRecordInBufferTimestamp = new AtomicLong(0L);
    private final AtomicLong executedWriteBufferPeriodicFlushes = new AtomicLong(0L);

    BufferedMutatorImpl(ClusterConnection conn, BufferedMutatorParams params, AsyncProcess ap) {
        if (conn == null || conn.isClosed()) {
            throw new IllegalArgumentException("Connection is null or closed.");
        }
        this.tableName = params.getTableName();
        this.conf = conn.getConfiguration();
        this.listener = params.getListener();
        if (params.getPool() == null) {
            this.pool = HTable.getDefaultExecutor(this.conf);
            this.cleanupPoolOnClose = true;
        } else {
            this.pool = params.getPool();
            this.cleanupPoolOnClose = false;
        }
        ConnectionConfiguration tableConf = new ConnectionConfiguration(this.conf);
        this.writeBufferSize = params.getWriteBufferSize() != -1L ? params.getWriteBufferSize() : tableConf.getWriteBufferSize();
        long newWriteBufferPeriodicFlushTimeoutMs = params.getWriteBufferPeriodicFlushTimeoutMs() != -1L ? params.getWriteBufferPeriodicFlushTimeoutMs() : tableConf.getWriteBufferPeriodicFlushTimeoutMs();
        long newWriteBufferPeriodicFlushTimerTickMs = params.getWriteBufferPeriodicFlushTimerTickMs() != -1L ? params.getWriteBufferPeriodicFlushTimerTickMs() : tableConf.getWriteBufferPeriodicFlushTimerTickMs();
        this.setWriteBufferPeriodicFlush(newWriteBufferPeriodicFlushTimeoutMs, newWriteBufferPeriodicFlushTimerTickMs);
        this.maxKeyValueSize = params.getMaxKeyValueSize() != -1 ? params.getMaxKeyValueSize() : tableConf.getMaxKeyValueSize();
        this.rpcTimeout = new AtomicInteger(params.getRpcTimeout() != -1 ? params.getRpcTimeout() : conn.getConnectionConfiguration().getWriteRpcTimeout());
        this.operationTimeout = new AtomicInteger(params.getOperationTimeout() != -1 ? params.getOperationTimeout() : conn.getConnectionConfiguration().getOperationTimeout());
        this.ap = ap;
    }

    BufferedMutatorImpl(ClusterConnection conn, RpcRetryingCallerFactory rpcCallerFactory, RpcControllerFactory rpcFactory, BufferedMutatorParams params) {
        this(conn, params, new AsyncProcess(conn, conn.getConfiguration(), rpcCallerFactory, rpcFactory));
    }

    private void checkClose() {
        if (this.closed) {
            throw new IllegalStateException("Cannot put when the BufferedMutator is closed.");
        }
    }

    ExecutorService getPool() {
        return this.pool;
    }

    AsyncProcess getAsyncProcess() {
        return this.ap;
    }

    @Override
    public TableName getName() {
        return this.tableName;
    }

    @Override
    public Configuration getConfiguration() {
        return this.conf;
    }

    @Override
    public void mutate(Mutation m3) throws InterruptedIOException, RetriesExhaustedWithDetailsException {
        this.mutate(Collections.singletonList(m3));
    }

    @Override
    public void mutate(List<? extends Mutation> ms) throws InterruptedIOException, RetriesExhaustedWithDetailsException {
        this.checkClose();
        long toAddSize = 0L;
        int toAddCount = 0;
        for (Mutation mutation : ms) {
            if (mutation instanceof Put) {
                ConnectionUtils.validatePut((Put)mutation, this.maxKeyValueSize);
            }
            toAddSize += mutation.heapSize();
            ++toAddCount;
        }
        if (this.currentWriteBufferSize.get() == 0L) {
            this.firstRecordInBufferTimestamp.set(System.currentTimeMillis());
        }
        this.currentWriteBufferSize.addAndGet(toAddSize);
        this.writeAsyncBuffer.addAll(ms);
        this.undealtMutationCount.addAndGet(toAddCount);
        this.doFlush(false);
    }

    protected long getExecutedWriteBufferPeriodicFlushes() {
        return this.executedWriteBufferPeriodicFlushes.get();
    }

    private void timerCallbackForWriteBufferPeriodicFlush() {
        if (this.currentWriteBufferSize.get() == 0L) {
            return;
        }
        long now = System.currentTimeMillis();
        if (this.firstRecordInBufferTimestamp.get() + this.writeBufferPeriodicFlushTimeoutMs.get() > now) {
            return;
        }
        try {
            this.executedWriteBufferPeriodicFlushes.incrementAndGet();
            this.flush();
        }
        catch (InterruptedIOException | RetriesExhaustedWithDetailsException e) {
            LOG.error("Exception during timerCallbackForWriteBufferPeriodicFlush --> " + e.getMessage());
        }
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.closed) {
            return;
        }
        this.disableWriteBufferPeriodicFlush();
        try {
            this.doFlush(true);
        }
        finally {
            if (this.cleanupPoolOnClose) {
                this.pool.shutdown();
                try {
                    if (!this.pool.awaitTermination(600L, TimeUnit.SECONDS)) {
                        LOG.warn("close() failed to terminate pool after 10 minutes. Abandoning pool.");
                    }
                }
                catch (InterruptedException e) {
                    LOG.warn("waitForTermination interrupted");
                    Thread.currentThread().interrupt();
                }
            }
            this.closed = true;
        }
    }

    private AsyncProcessTask createTask(QueueRowAccess access) {
        return new AsyncProcessTask(AsyncProcessTask.newBuilder().setPool(this.pool).setTableName(this.tableName).setRowAccess(access).setSubmittedRows(AsyncProcessTask.SubmittedRows.AT_LEAST_ONE).build()){

            @Override
            public int getRpcTimeout() {
                return BufferedMutatorImpl.this.rpcTimeout.get();
            }

            @Override
            public int getOperationTimeout() {
                return BufferedMutatorImpl.this.operationTimeout.get();
            }
        };
    }

    @Override
    public void flush() throws InterruptedIOException, RetriesExhaustedWithDetailsException {
        this.checkClose();
        this.doFlush(true);
    }

    private void doFlush(boolean flushAll) throws InterruptedIOException, RetriesExhaustedWithDetailsException {
        ArrayList<RetriesExhaustedWithDetailsException> errors = new ArrayList<RetriesExhaustedWithDetailsException>();
        while (flushAll || this.currentWriteBufferSize.get() > this.writeBufferSize) {
            AsyncRequestFuture asf;
            try (QueueRowAccess access = this.createQueueRowAccess();){
                if (access.isEmpty()) break;
                asf = this.ap.submit(this.createTask(access));
            }
            asf.waitUntilDone();
            if (!asf.hasError()) continue;
            errors.add(asf.getErrors());
        }
        RetriesExhaustedWithDetailsException exception = BufferedMutatorImpl.makeException(errors);
        if (exception == null) {
            return;
        }
        if (this.listener == null) {
            throw exception;
        }
        this.listener.onException(exception, this);
    }

    private static RetriesExhaustedWithDetailsException makeException(List<RetriesExhaustedWithDetailsException> errors) {
        switch (errors.size()) {
            case 0: {
                return null;
            }
            case 1: {
                return errors.get(0);
            }
        }
        ArrayList<Throwable> exceptions = new ArrayList<Throwable>();
        ArrayList<Row> actions = new ArrayList<Row>();
        ArrayList<String> hostnameAndPort = new ArrayList<String>();
        errors.forEach(e -> {
            exceptions.addAll(e.exceptions);
            actions.addAll(e.actions);
            hostnameAndPort.addAll(e.hostnameAndPort);
        });
        return new RetriesExhaustedWithDetailsException(exceptions, actions, hostnameAndPort);
    }

    @Override
    public long getWriteBufferSize() {
        return this.writeBufferSize;
    }

    @Override
    public synchronized void setWriteBufferPeriodicFlush(long timeoutMs, long timerTickMs) {
        long originalTimeoutMs = this.writeBufferPeriodicFlushTimeoutMs.get();
        long originalTimerTickMs = this.writeBufferPeriodicFlushTimerTickMs.get();
        this.writeBufferPeriodicFlushTimeoutMs.set(Math.max(0L, timeoutMs));
        this.writeBufferPeriodicFlushTimerTickMs.set(Math.max(100L, timerTickMs));
        if ((this.writeBufferPeriodicFlushTimeoutMs.get() != originalTimeoutMs || this.writeBufferPeriodicFlushTimerTickMs.get() != originalTimerTickMs) && this.writeBufferPeriodicFlushTimer != null) {
            this.writeBufferPeriodicFlushTimer.cancel();
            this.writeBufferPeriodicFlushTimer = null;
        }
        if (this.writeBufferPeriodicFlushTimer == null && this.writeBufferPeriodicFlushTimeoutMs.get() > 0L) {
            this.writeBufferPeriodicFlushTimer = new Timer(true);
            this.writeBufferPeriodicFlushTimer.schedule(new TimerTask(){

                @Override
                public void run() {
                    BufferedMutatorImpl.this.timerCallbackForWriteBufferPeriodicFlush();
                }
            }, this.writeBufferPeriodicFlushTimerTickMs.get(), this.writeBufferPeriodicFlushTimerTickMs.get());
        }
    }

    @Override
    public long getWriteBufferPeriodicFlushTimeoutMs() {
        return this.writeBufferPeriodicFlushTimeoutMs.get();
    }

    @Override
    public long getWriteBufferPeriodicFlushTimerTickMs() {
        return this.writeBufferPeriodicFlushTimerTickMs.get();
    }

    @Override
    public void setRpcTimeout(int rpcTimeout) {
        this.rpcTimeout.set(rpcTimeout);
    }

    @Override
    public void setOperationTimeout(int operationTimeout) {
        this.operationTimeout.set(operationTimeout);
    }

    long getCurrentWriteBufferSize() {
        return this.currentWriteBufferSize.get();
    }

    int size() {
        return this.undealtMutationCount.get();
    }

    int getUnflushedSize() {
        return this.writeAsyncBuffer.size();
    }

    QueueRowAccess createQueueRowAccess() {
        return new QueueRowAccess();
    }

    class QueueRowAccess
    implements RowAccess<Row>,
    Closeable {
        private int remainder;
        private Mutation last;

        QueueRowAccess() {
            this.remainder = BufferedMutatorImpl.this.undealtMutationCount.getAndSet(0);
            this.last = null;
        }

        private void restoreLastMutation() {
            if (this.last != null) {
                BufferedMutatorImpl.this.writeAsyncBuffer.add(this.last);
                BufferedMutatorImpl.this.currentWriteBufferSize.addAndGet(this.last.heapSize());
                this.last = null;
            }
        }

        @Override
        public void close() {
            this.restoreLastMutation();
            if (this.remainder > 0) {
                BufferedMutatorImpl.this.undealtMutationCount.addAndGet(this.remainder);
                this.remainder = 0;
            }
        }

        @Override
        public Iterator<Row> iterator() {
            return new Iterator<Row>(){
                private int countDown;
                {
                    this.countDown = QueueRowAccess.this.remainder;
                }

                @Override
                public boolean hasNext() {
                    return this.countDown > 0;
                }

                @Override
                public Row next() {
                    QueueRowAccess.this.restoreLastMutation();
                    if (!this.hasNext()) {
                        throw new NoSuchElementException();
                    }
                    QueueRowAccess.this.last = (Mutation)BufferedMutatorImpl.this.writeAsyncBuffer.poll();
                    if (QueueRowAccess.this.last == null) {
                        throw new NoSuchElementException();
                    }
                    BufferedMutatorImpl.this.currentWriteBufferSize.addAndGet(-QueueRowAccess.this.last.heapSize());
                    --this.countDown;
                    return QueueRowAccess.this.last;
                }

                @Override
                public void remove() {
                    if (QueueRowAccess.this.last == null) {
                        throw new IllegalStateException();
                    }
                    --QueueRowAccess.this.remainder;
                    QueueRowAccess.this.last = null;
                }
            };
        }

        @Override
        public int size() {
            return this.remainder;
        }

        @Override
        public boolean isEmpty() {
            return this.remainder <= 0;
        }
    }
}

