/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.ScheduledChore;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.io.HFileLink;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.regionserver.HRegionServer;
import org.apache.hadoop.hbase.regionserver.HStore;
import org.apache.hadoop.hbase.regionserver.StoreFileInfo;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class BrokenStoreFileCleaner
extends ScheduledChore {
    private static final Logger LOG = LoggerFactory.getLogger(BrokenStoreFileCleaner.class);
    public static final String BROKEN_STOREFILE_CLEANER_ENABLED = "hbase.region.broken.storefilecleaner.enabled";
    public static final boolean DEFAULT_BROKEN_STOREFILE_CLEANER_ENABLED = false;
    public static final String BROKEN_STOREFILE_CLEANER_TTL = "hbase.region.broken.storefilecleaner.ttl";
    public static final long DEFAULT_BROKEN_STOREFILE_CLEANER_TTL = 43200000L;
    public static final String BROKEN_STOREFILE_CLEANER_DELAY = "hbase.region.broken.storefilecleaner.delay";
    public static final int DEFAULT_BROKEN_STOREFILE_CLEANER_DELAY = 0x6DDD00;
    public static final String BROKEN_STOREFILE_CLEANER_DELAY_JITTER = "hbase.region.broken.storefilecleaner.delay.jitter";
    public static final double DEFAULT_BROKEN_STOREFILE_CLEANER_DELAY_JITTER = 0.25;
    public static final String BROKEN_STOREFILE_CLEANER_PERIOD = "hbase.region.broken.storefilecleaner.period";
    public static final int DEFAULT_BROKEN_STOREFILE_CLEANER_PERIOD = 21600000;
    private HRegionServer regionServer;
    private final AtomicBoolean enabled = new AtomicBoolean(true);
    private long fileTtl;

    public BrokenStoreFileCleaner(int delay, int period, Stoppable stopper, Configuration conf, HRegionServer regionServer) {
        super("BrokenStoreFileCleaner", stopper, period, delay);
        this.regionServer = regionServer;
        this.setEnabled(conf.getBoolean(BROKEN_STOREFILE_CLEANER_ENABLED, false));
        this.fileTtl = conf.getLong(BROKEN_STOREFILE_CLEANER_TTL, 43200000L);
    }

    public boolean setEnabled(boolean enabled) {
        return this.enabled.getAndSet(enabled);
    }

    public boolean getEnabled() {
        return this.enabled.get();
    }

    @Override
    public void chore() {
        if (this.getEnabled()) {
            long start = EnvironmentEdgeManager.currentTime();
            AtomicLong deletedFiles = new AtomicLong(0L);
            AtomicLong failedDeletes = new AtomicLong(0L);
            for (HRegion region : this.regionServer.getRegions()) {
                for (HStore store : region.getStores()) {
                    if (store.getStoreEngine().requireWritingToTmpDirFirst()) continue;
                    Path storePath = new Path(region.getRegionFileSystem().getRegionDir(), store.getColumnFamilyName());
                    try {
                        List<FileStatus> fsStoreFiles = Arrays.asList(region.getRegionFileSystem().fs.listStatus(storePath));
                        fsStoreFiles.forEach(file -> this.cleanFileIfNeeded((FileStatus)file, store, deletedFiles, failedDeletes));
                    }
                    catch (IOException e) {
                        LOG.warn("Failed to list files in {}, cleanup is skipped there", (Object)storePath);
                    }
                }
            }
            LOG.debug("BrokenStoreFileCleaner on {} run for: {}ms. It deleted {} files and tried but failed to delete {}", new Object[]{this.regionServer.getServerName().getServerName(), EnvironmentEdgeManager.currentTime() - start, deletedFiles.get(), failedDeletes.get()});
        } else {
            LOG.trace("Broken storefile Cleaner chore disabled! Not cleaning.");
        }
    }

    private void cleanFileIfNeeded(FileStatus file, HStore store, AtomicLong deletedFiles, AtomicLong failedDeletes) {
        if (file.isDirectory()) {
            LOG.trace("This is a Directory {}, skip cleanup", (Object)file.getPath());
            return;
        }
        if (!this.validate(file.getPath())) {
            LOG.trace("Invalid file {}, skip cleanup", (Object)file.getPath());
            return;
        }
        if (!this.isOldEnough(file)) {
            LOG.trace("Fresh file {}, skip cleanup", (Object)file.getPath());
            return;
        }
        if (this.isActiveStorefile(file, store)) {
            LOG.trace("Actively used storefile file {}, skip cleanup", (Object)file.getPath());
            return;
        }
        if (this.isCompactedFile(file, store)) {
            LOG.trace("Cleanup is done by a different chore for file {}, skip cleanup", (Object)file.getPath());
            return;
        }
        if (this.isCompactionResultFile(file, store)) {
            LOG.trace("The file is the result of an ongoing compaction {}, skip cleanup", (Object)file.getPath());
            return;
        }
        this.deleteFile(file, store, deletedFiles, failedDeletes);
    }

    private boolean isCompactionResultFile(FileStatus file, HStore store) {
        return store.getStoreFilesBeingWritten().contains(file.getPath());
    }

    private boolean isCompactedFile(FileStatus file, HStore store) {
        return store.getStoreEngine().getStoreFileManager().getCompactedfiles().stream().anyMatch(sf -> sf.getPath().equals((Object)file.getPath()));
    }

    private boolean isActiveStorefile(FileStatus file, HStore store) {
        return store.getStoreEngine().getStoreFileManager().getStoreFiles().stream().anyMatch(sf -> sf.getPath().equals((Object)file.getPath()));
    }

    boolean validate(Path file) {
        if (HFileLink.isBackReferencesDir(file) || HFileLink.isBackReferencesDir(file.getParent())) {
            return true;
        }
        return StoreFileInfo.validateStoreFileName(file.getName());
    }

    boolean isOldEnough(FileStatus file) {
        return file.getModificationTime() + this.fileTtl < EnvironmentEdgeManager.currentTime();
    }

    private void deleteFile(FileStatus file, HStore store, AtomicLong deletedFiles, AtomicLong failedDeletes) {
        Path filePath = file.getPath();
        LOG.debug("Removing {} from store", (Object)filePath);
        try {
            boolean success = store.getFileSystem().delete(filePath, false);
            if (!success) {
                failedDeletes.incrementAndGet();
                LOG.warn("Attempted to delete:" + filePath + ", but couldn't. Attempt to delete on next pass.");
            } else {
                deletedFiles.incrementAndGet();
            }
        }
        catch (IOException e) {
            e = e instanceof RemoteException ? ((RemoteException)((Object)e)).unwrapRemoteException() : e;
            LOG.warn("Error while deleting: " + filePath, (Throwable)e);
        }
    }
}

