/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.io.HeapSize;
import org.apache.hadoop.hbase.util.ClassSize;

@InterfaceAudience.Private
public class LruHashMap<K extends HeapSize, V extends HeapSize>
implements HeapSize,
Map<K, V> {
    static final Log LOG = LogFactory.getLog(LruHashMap.class);
    private static final long DEFAULT_MAX_MEM_USAGE = 50000L;
    private static final int DEFAULT_INITIAL_CAPACITY = 16;
    private static final int MAXIMUM_CAPACITY = 0x40000000;
    private static final float DEFAULT_LOAD_FACTOR = 0.75f;
    private static final int OVERHEAD = 56 + 3 * ClassSize.REFERENCE + 1 * ClassSize.ARRAY;
    private final float loadFactor;
    private int size;
    private int threshold;
    private Entry[] entries;
    private Entry<K, V> headPtr;
    private Entry<K, V> tailPtr;
    private long memTotal = 0L;
    private long memFree = 0L;
    private long hitCount = 0L;
    private long missCount = 0L;

    public LruHashMap(int initialCapacity, float loadFactor, long maxMemUsage) {
        if (initialCapacity < 1) {
            throw new IllegalArgumentException("Initial capacity must be > 0");
        }
        if (initialCapacity > 0x40000000) {
            throw new IllegalArgumentException("Initial capacity is too large");
        }
        if (loadFactor <= 0.0f || Float.isNaN(loadFactor)) {
            throw new IllegalArgumentException("Load factor must be > 0");
        }
        if (maxMemUsage <= (long)(OVERHEAD + initialCapacity * ClassSize.REFERENCE)) {
            throw new IllegalArgumentException("Max memory usage too small to support base overhead");
        }
        int capacity = this.calculateCapacity(initialCapacity);
        this.loadFactor = loadFactor;
        this.threshold = this.calculateThreshold(capacity, loadFactor);
        this.entries = new Entry[capacity];
        this.memFree = maxMemUsage;
        this.memTotal = maxMemUsage;
        this.init();
    }

    public LruHashMap(int initialCapacity, float loadFactor) {
        this(initialCapacity, loadFactor, 50000L);
    }

    public LruHashMap(int initialCapacity) {
        this(initialCapacity, 0.75f, 50000L);
    }

    public LruHashMap(long maxMemUsage) {
        this(16, 0.75f, maxMemUsage);
    }

    public LruHashMap() {
        this(16, 0.75f, 50000L);
    }

    public long getMemFree() {
        return this.memFree;
    }

    public long getMemMax() {
        return this.memTotal;
    }

    public long getMemUsed() {
        return this.memTotal - this.memFree;
    }

    public long getHitCount() {
        return this.hitCount;
    }

    public long getMissCount() {
        return this.missCount;
    }

    public double getHitRatio() {
        return (double)this.hitCount / (double)(this.hitCount + this.missCount);
    }

    public synchronized long freeMemory(long requestedAmount) throws Exception {
        long freedMemory;
        if (requestedAmount > this.getMemUsed() - this.getMinimumUsage()) {
            return this.clearAll();
        }
        for (freedMemory = 0L; freedMemory < requestedAmount; freedMemory += this.evictFromLru()) {
        }
        return freedMemory;
    }

    @Override
    public long heapSize() {
        return this.memTotal - this.memFree;
    }

    @Override
    public synchronized V get(Object key) {
        this.checkKey((HeapSize)key);
        int hash = this.hash(key);
        int i = this.hashIndex(hash, this.entries.length);
        Entry e = this.entries[i];
        while (true) {
            if (e == null) {
                ++this.missCount;
                return null;
            }
            if (e.hash == hash && this.isEqual(key, e.key)) {
                ++this.hitCount;
                this.updateLru(e);
                return e.value;
            }
            e = e.next;
        }
    }

    @Override
    public synchronized V put(K key, V value) {
        this.checkKey(key);
        this.checkValue(value);
        int hash = this.hash(key);
        int i = this.hashIndex(hash, this.entries.length);
        Entry e = this.entries[i];
        while (e != null) {
            if (e.hash == hash && this.isEqual(key, e.key)) {
                Object oldValue = e.value;
                long memChange = e.replaceValue(value);
                this.checkAndFreeMemory(memChange);
                this.updateLru(e);
                return oldValue;
            }
            e = e.next;
        }
        long memChange = this.addEntry(hash, key, value, i);
        this.checkAndFreeMemory(memChange);
        return null;
    }

    @Override
    public synchronized V remove(Object key) {
        Entry<HeapSize, V> e = this.removeEntryForKey((HeapSize)key);
        if (e == null) {
            return null;
        }
        this.memFree += e.heapSize();
        return e.value;
    }

    @Override
    public int size() {
        return this.size;
    }

    @Override
    public boolean isEmpty() {
        return this.size == 0;
    }

    @Override
    public synchronized void clear() {
        this.memFree += this.clearAll();
    }

    @Override
    public synchronized boolean containsKey(Object key) {
        this.checkKey((HeapSize)key);
        int hash = this.hash(key);
        int i = this.hashIndex(hash, this.entries.length);
        Entry e = this.entries[i];
        while (e != null) {
            if (e.hash == hash && this.isEqual(key, e.key)) {
                return true;
            }
            e = e.next;
        }
        return false;
    }

    @Override
    public synchronized boolean containsValue(Object value) {
        this.checkValue((HeapSize)value);
        Entry[] tab = this.entries;
        for (int i = 0; i < tab.length; ++i) {
            Entry e = tab[i];
            while (e != null) {
                if (value.equals(e.value)) {
                    return true;
                }
                e = e.next;
            }
        }
        return false;
    }

    private void checkKey(K key) {
        if (key == null) {
            throw new NullPointerException("null keys are not allowed");
        }
    }

    private void checkValue(V value) {
        if (value == null) {
            throw new NullPointerException("null values are not allowed");
        }
    }

    private long getMinimumUsage() {
        return OVERHEAD + this.entries.length * ClassSize.REFERENCE;
    }

    private void checkAndFreeMemory(long memNeeded) {
        while (this.memFree < memNeeded) {
            this.evictFromLru();
        }
        this.memFree -= memNeeded;
    }

    private long evictFromLru() {
        long freed = this.headPtr.heapSize();
        this.memFree += freed;
        this.removeEntry(this.headPtr);
        return freed;
    }

    private void updateLru(Entry<K, V> e) {
        Entry<K, V> prev = e.getPrevPtr();
        Entry<K, V> next = e.getNextPtr();
        if (next != null) {
            if (prev != null) {
                prev.setNextPtr(next);
                next.setPrevPtr(prev);
            } else {
                this.headPtr = next;
                this.headPtr.setPrevPtr(null);
            }
            e.setNextPtr(null);
            e.setPrevPtr(this.tailPtr);
            this.tailPtr.setNextPtr(e);
            this.tailPtr = e;
        }
    }

    private void removeEntry(Entry<K, V> entry) {
        Entry prev;
        Object k = entry.key;
        int hash = entry.hash;
        int i = this.hashIndex(hash, this.entries.length);
        Entry e = prev = this.entries[i];
        while (e != null) {
            Entry next = e.next;
            if (e.hash == hash && this.isEqual(k, e.key)) {
                --this.size;
                if (prev == e) {
                    this.entries[i] = next;
                } else {
                    prev.next = next;
                }
                Entry prevPtr = e.getPrevPtr();
                Entry nextPtr = e.getNextPtr();
                if (prevPtr != null && nextPtr != null) {
                    prevPtr.setNextPtr(nextPtr);
                    nextPtr.setPrevPtr(prevPtr);
                } else if (prevPtr != null) {
                    this.tailPtr = prevPtr;
                    prevPtr.setNextPtr(null);
                } else if (nextPtr != null) {
                    this.headPtr = nextPtr;
                    nextPtr.setPrevPtr(null);
                }
                return;
            }
            prev = e;
            e = next;
        }
    }

    private Entry<K, V> removeEntryForKey(K key) {
        Entry prev;
        int hash = this.hash(key);
        int i = this.hashIndex(hash, this.entries.length);
        Entry e = prev = this.entries[i];
        while (e != null) {
            Entry next = e.next;
            if (e.hash == hash && this.isEqual(key, e.key)) {
                --this.size;
                if (prev == e) {
                    this.entries[i] = next;
                } else {
                    prev.next = next;
                }
                Entry prevPtr = e.getPrevPtr();
                Entry nextPtr = e.getNextPtr();
                if (prevPtr != null && nextPtr != null) {
                    prevPtr.setNextPtr(nextPtr);
                    nextPtr.setPrevPtr(prevPtr);
                } else if (prevPtr != null) {
                    this.tailPtr = prevPtr;
                    prevPtr.setNextPtr(null);
                } else if (nextPtr != null) {
                    this.headPtr = nextPtr;
                    nextPtr.setPrevPtr(null);
                }
                return e;
            }
            prev = e;
            e = next;
        }
        return e;
    }

    private long addEntry(int hash, K key, V value, int bucketIndex) {
        Entry<K, V> newE;
        Entry e = this.entries[bucketIndex];
        this.entries[bucketIndex] = newE = new Entry<K, V>(hash, key, value, e, this.tailPtr);
        if (this.size == 0) {
            this.headPtr = newE;
            this.tailPtr = newE;
        } else {
            newE.setPrevPtr(this.tailPtr);
            this.tailPtr.setNextPtr(newE);
            this.tailPtr = newE;
        }
        if (this.size++ >= this.threshold) {
            this.growTable(2 * this.entries.length);
        }
        return newE.heapSize();
    }

    private long clearAll() {
        long freedMemory = 0L;
        for (int i = 0; i < this.entries.length; ++i) {
            Entry cur = this.entries[i];
            while (cur != null) {
                freedMemory += cur.heapSize();
                cur = cur.next;
            }
            this.entries[i] = null;
        }
        this.headPtr = null;
        this.tailPtr = null;
        this.size = 0;
        return freedMemory;
    }

    private void growTable(int newCapacity) {
        Entry[] oldTable = this.entries;
        int oldCapacity = oldTable.length;
        if (oldCapacity == 0x40000000) {
            this.threshold = Integer.MAX_VALUE;
            return;
        }
        long requiredSpace = (newCapacity - oldCapacity) * ClassSize.REFERENCE;
        this.checkAndFreeMemory(requiredSpace);
        Entry[] newTable = new Entry[newCapacity];
        for (int i = 0; i < oldCapacity; ++i) {
            Entry next;
            Entry entry = oldTable[i];
            if (entry == null) continue;
            oldTable[i] = null;
            do {
                next = entry.next;
                int idx = this.hashIndex(entry.hash, newCapacity);
                entry.next = newTable[idx];
                newTable[idx] = entry;
            } while ((entry = next) != null);
        }
        this.entries = newTable;
        this.threshold = (int)((float)newCapacity * this.loadFactor);
    }

    private int hash(Object key) {
        int h = key.hashCode();
        h += ~(h << 9);
        h ^= h >>> 14;
        h += h << 4;
        h ^= h >>> 10;
        return h;
    }

    private boolean isEqual(Object x, Object y) {
        return x == y || x.equals(y);
    }

    private int hashIndex(int hashValue, int length) {
        return hashValue & length - 1;
    }

    private int calculateCapacity(int proposedCapacity) {
        int newCapacity;
        if (proposedCapacity > 0x40000000) {
            newCapacity = 0x40000000;
        } else {
            for (newCapacity = 1; newCapacity < proposedCapacity; newCapacity <<= 1) {
            }
            if (newCapacity > 0x40000000) {
                newCapacity = 0x40000000;
            }
        }
        return newCapacity;
    }

    private int calculateThreshold(int capacity, float factor) {
        return (int)((float)capacity * factor);
    }

    private void init() {
        this.memFree -= (long)OVERHEAD;
        this.memFree -= (long)(this.entries.length * ClassSize.REFERENCE);
    }

    public List<Entry<K, V>> entryLruList() {
        ArrayList<Entry<K, V>> entryList = new ArrayList<Entry<K, V>>();
        for (Entry<K, V> entry = this.headPtr; entry != null; entry = entry.getNextPtr()) {
            entryList.add(entry);
        }
        return entryList;
    }

    public Set<Entry<K, V>> entryTableSet() {
        HashSet<Entry<K, V>> entrySet = new HashSet<Entry<K, V>>();
        Entry[] table = this.entries;
        for (int i = 0; i < table.length; ++i) {
            Entry e = table[i];
            while (e != null) {
                entrySet.add(e);
                e = e.next;
            }
        }
        return entrySet;
    }

    public Entry getHeadPtr() {
        return this.headPtr;
    }

    public Entry getTailPtr() {
        return this.tailPtr;
    }

    @Override
    public Set<Map.Entry<K, V>> entrySet() {
        throw new UnsupportedOperationException("entrySet() is intentionally unimplemented");
    }

    @Override
    public boolean equals(Object o) {
        throw new UnsupportedOperationException("equals(Object) is intentionally unimplemented");
    }

    @Override
    public int hashCode() {
        throw new UnsupportedOperationException("hashCode(Object) is intentionally unimplemented");
    }

    @Override
    public Set<K> keySet() {
        throw new UnsupportedOperationException("keySet() is intentionally unimplemented");
    }

    @Override
    public void putAll(Map<? extends K, ? extends V> m) {
        throw new UnsupportedOperationException("putAll() is intentionally unimplemented");
    }

    @Override
    public Collection<V> values() {
        throw new UnsupportedOperationException("values() is intentionally unimplemented");
    }

    protected static class Entry<K extends HeapSize, V extends HeapSize>
    implements Map.Entry<K, V>,
    HeapSize {
        static final int OVERHEAD = 8 + 5 * ClassSize.REFERENCE + 8;
        protected final K key;
        protected V value;
        protected final int hash;
        protected Entry<K, V> next;
        protected Entry<K, V> prevPtr;
        protected Entry<K, V> nextPtr;
        protected long heapSize;

        Entry(int h, K k, V v, Entry<K, V> nextChainPtr, Entry<K, V> prevLruPtr) {
            this.value = v;
            this.next = nextChainPtr;
            this.key = k;
            this.hash = h;
            this.prevPtr = prevLruPtr;
            this.nextPtr = null;
            this.heapSize = (long)OVERHEAD + k.heapSize() + v.heapSize();
        }

        @Override
        public K getKey() {
            return this.key;
        }

        @Override
        public V getValue() {
            return this.value;
        }

        @Override
        public V setValue(V newValue) {
            V oldValue = this.value;
            this.value = newValue;
            return oldValue;
        }

        protected long replaceValue(V newValue) {
            long sizeDiff = newValue.heapSize() - this.value.heapSize();
            this.value = newValue;
            this.heapSize += sizeDiff;
            return sizeDiff;
        }

        @Override
        public boolean equals(Object o) {
            Object v2;
            Object v1;
            Object k2;
            if (!(o instanceof Map.Entry)) {
                return false;
            }
            Map.Entry e = (Map.Entry)o;
            Object k1 = this.getKey();
            return (k1 == (k2 = e.getKey()) || k1 != null && k1.equals(k2)) && ((v1 = this.getValue()) == (v2 = e.getValue()) || v1 != null && v1.equals(v2));
        }

        @Override
        public int hashCode() {
            return this.key.hashCode() ^ this.value.hashCode();
        }

        public String toString() {
            return this.getKey() + "=" + this.getValue();
        }

        protected void setPrevPtr(Entry<K, V> prevPtr) {
            this.prevPtr = prevPtr;
        }

        protected Entry<K, V> getPrevPtr() {
            return this.prevPtr;
        }

        protected void setNextPtr(Entry<K, V> nextPtr) {
            this.nextPtr = nextPtr;
        }

        protected Entry<K, V> getNextPtr() {
            return this.nextPtr;
        }

        @Override
        public long heapSize() {
            return this.heapSize;
        }
    }
}

