/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.DataInput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BatchedRemoteIterator;
import org.apache.hadoop.fs.CacheFlag;
import org.apache.hadoop.fs.InvalidRequestException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hbase.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.hbase.shaded.com.google.common.collect.Lists;
import org.apache.hadoop.hbase.shaded.com.google.protobuf.GeneratedMessage;
import org.apache.hadoop.hbase.shaded.org.apache.commons.io.IOUtils;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.protocol.CacheDirective;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveEntry;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveInfo;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveStats;
import org.apache.hadoop.hdfs.protocol.CachePoolEntry;
import org.apache.hadoop.hdfs.protocol.CachePoolInfo;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.proto.ClientNamenodeProtocolProtos;
import org.apache.hadoop.hdfs.protocolPB.PBHelper;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.CacheReplicationMonitor;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.namenode.CachePool;
import org.apache.hadoop.hdfs.server.namenode.CachedBlock;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.FSImageSerialization;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.FSPermissionChecker;
import org.apache.hadoop.hdfs.server.namenode.FsImageProto;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.metrics.NameNodeMetrics;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.Phase;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.StartupProgress;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.Step;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.StepType;
import org.apache.hadoop.hdfs.util.ReadOnlyList;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.util.GSet;
import org.apache.hadoop.util.LightWeightGSet;
import org.apache.hadoop.util.Time;

@InterfaceAudience.LimitedPrivate(value={"HDFS"})
public final class CacheManager {
    public static final Log LOG = LogFactory.getLog(CacheManager.class);
    private static final float MIN_CACHED_BLOCKS_PERCENT = 0.001f;
    private final FSNamesystem namesystem;
    private final BlockManager blockManager;
    private final TreeMap<Long, CacheDirective> directivesById = new TreeMap();
    private long nextDirectiveId;
    private final TreeMap<String, List<CacheDirective>> directivesByPath = new TreeMap();
    private final TreeMap<String, CachePool> cachePools = new TreeMap();
    private final int maxListCachePoolsResponses;
    private final int maxListCacheDirectivesNumResponses;
    private final long scanIntervalMs;
    private final GSet<CachedBlock, CachedBlock> cachedBlocks;
    private final ReentrantLock crmLock = new ReentrantLock();
    private final SerializerCompat serializerCompat = new SerializerCompat();
    private CacheReplicationMonitor monitor;

    CacheManager(FSNamesystem namesystem, Configuration conf, BlockManager blockManager) {
        this.namesystem = namesystem;
        this.blockManager = blockManager;
        this.nextDirectiveId = 1L;
        this.maxListCachePoolsResponses = conf.getInt("dfs.namenode.list.cache.pools.num.responses", 100);
        this.maxListCacheDirectivesNumResponses = conf.getInt("dfs.namenode.list.cache.directives.num.responses", 100);
        this.scanIntervalMs = conf.getLong("dfs.namenode.path.based.cache.refresh.interval.ms", 30000L);
        float cachedBlocksPercent = conf.getFloat("dfs.namenode.path.based.cache.block.map.allocation.percent", 0.25f);
        if (cachedBlocksPercent < 0.001f) {
            LOG.info((Object)"Using minimum value 0.001 for dfs.namenode.path.based.cache.block.map.allocation.percent");
            cachedBlocksPercent = 0.001f;
        }
        this.cachedBlocks = new LightWeightGSet<CachedBlock, CachedBlock>(LightWeightGSet.computeCapacity(cachedBlocksPercent, "cachedBlocks"));
    }

    void clear() {
        this.directivesById.clear();
        this.directivesByPath.clear();
        this.cachePools.clear();
        this.nextDirectiveId = 1L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void startMonitorThread() {
        this.crmLock.lock();
        try {
            if (this.monitor == null) {
                this.monitor = new CacheReplicationMonitor(this.namesystem, this, this.scanIntervalMs, this.crmLock);
                this.monitor.start();
            }
        }
        finally {
            this.crmLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stopMonitorThread() {
        this.crmLock.lock();
        try {
            if (this.monitor != null) {
                CacheReplicationMonitor prevMonitor = this.monitor;
                this.monitor = null;
                IOUtils.closeQuietly(prevMonitor);
            }
        }
        finally {
            this.crmLock.unlock();
        }
    }

    public void clearDirectiveStats() {
        assert (this.namesystem.hasWriteLock());
        for (CacheDirective directive : this.directivesById.values()) {
            directive.resetStatistics();
        }
    }

    public Collection<CachePool> getCachePools() {
        assert (this.namesystem.hasReadLock());
        return Collections.unmodifiableCollection(this.cachePools.values());
    }

    public Collection<CacheDirective> getCacheDirectives() {
        assert (this.namesystem.hasReadLock());
        return Collections.unmodifiableCollection(this.directivesById.values());
    }

    @VisibleForTesting
    public GSet<CachedBlock, CachedBlock> getCachedBlocks() {
        assert (this.namesystem.hasReadLock());
        return this.cachedBlocks;
    }

    private long getNextDirectiveId() throws IOException {
        assert (this.namesystem.hasWriteLock());
        if (this.nextDirectiveId >= 0x7FFFFFFFFFFFFFFEL) {
            throw new IOException("No more available IDs.");
        }
        return this.nextDirectiveId++;
    }

    private static void checkWritePermission(FSPermissionChecker pc, CachePool pool) throws AccessControlException {
        if (pc != null) {
            pc.checkPermission(pool, FsAction.WRITE);
        }
    }

    private static String validatePoolName(CacheDirectiveInfo directive) throws InvalidRequestException {
        String pool = directive.getPool();
        if (pool == null) {
            throw new InvalidRequestException("No pool specified.");
        }
        if (pool.isEmpty()) {
            throw new InvalidRequestException("Invalid empty pool name.");
        }
        return pool;
    }

    private static String validatePath(CacheDirectiveInfo directive) throws InvalidRequestException {
        if (directive.getPath() == null) {
            throw new InvalidRequestException("No path specified.");
        }
        String path = directive.getPath().toUri().getPath();
        if (!DFSUtil.isValidName(path)) {
            throw new InvalidRequestException("Invalid path '" + path + "'.");
        }
        return path;
    }

    private static short validateReplication(CacheDirectiveInfo directive, short defaultValue) throws InvalidRequestException {
        short repl;
        short s = repl = directive.getReplication() != null ? directive.getReplication() : defaultValue;
        if (repl <= 0) {
            throw new InvalidRequestException("Invalid replication factor " + repl + " <= 0");
        }
        return repl;
    }

    private static long validateExpiryTime(CacheDirectiveInfo info, long maxRelativeExpiryTime) throws InvalidRequestException {
        long absExpiryTime;
        long relExpiryTime;
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Validating directive " + info + " pool maxRelativeExpiryTime " + maxRelativeExpiryTime));
        }
        long now = new Date().getTime();
        long maxAbsoluteExpiryTime = now + maxRelativeExpiryTime;
        if (info == null || info.getExpiration() == null) {
            return maxAbsoluteExpiryTime;
        }
        CacheDirectiveInfo.Expiration expiry = info.getExpiration();
        if (expiry.getMillis() < 0L) {
            throw new InvalidRequestException("Cannot set a negative expiration: " + expiry.getMillis());
        }
        if (expiry.isRelative()) {
            relExpiryTime = expiry.getMillis();
            absExpiryTime = now + relExpiryTime;
        } else {
            absExpiryTime = expiry.getMillis();
            relExpiryTime = absExpiryTime - now;
        }
        if (relExpiryTime > 0x1FFFFFFFFFFFFFFFL) {
            throw new InvalidRequestException("Expiration " + expiry.toString() + " is too far in the future!");
        }
        if (relExpiryTime > maxRelativeExpiryTime) {
            throw new InvalidRequestException("Expiration " + expiry.toString() + " exceeds the max relative expiration time of " + maxRelativeExpiryTime + " ms.");
        }
        return absExpiryTime;
    }

    private void checkLimit(CachePool pool, String path, short replication) throws InvalidRequestException {
        CacheDirectiveStats stats = this.computeNeeded(path, replication);
        if (pool.getLimit() == Long.MAX_VALUE) {
            return;
        }
        if (pool.getBytesNeeded() + stats.getBytesNeeded() * (long)replication > pool.getLimit()) {
            throw new InvalidRequestException("Caching path " + path + " of size " + stats.getBytesNeeded() / (long)replication + " bytes at replication " + replication + " would exceed pool " + pool.getPoolName() + "'s remaining capacity of " + (pool.getLimit() - pool.getBytesNeeded()) + " bytes.");
        }
    }

    private CacheDirectiveStats computeNeeded(String path, short replication) {
        INode node;
        FSDirectory fsDir = this.namesystem.getFSDirectory();
        long requestedBytes = 0L;
        long requestedFiles = 0L;
        CacheDirectiveStats.Builder builder = new CacheDirectiveStats.Builder();
        try {
            node = fsDir.getINode(path);
        }
        catch (UnresolvedLinkException e) {
            return builder.build();
        }
        if (node == null) {
            return builder.build();
        }
        if (node.isFile()) {
            requestedFiles = 1L;
            INodeFile file = node.asFile();
            requestedBytes = file.computeFileSize();
        } else if (node.isDirectory()) {
            INodeDirectory dir = node.asDirectory();
            ReadOnlyList<INode> children = dir.getChildrenList(0x7FFFFFFE);
            requestedFiles = children.size();
            for (INode child : children) {
                if (!child.isFile()) continue;
                requestedBytes += child.asFile().computeFileSize();
            }
        }
        return new CacheDirectiveStats.Builder().setBytesNeeded(requestedBytes).setFilesCached(requestedFiles).build();
    }

    private CacheDirective getById(long id) throws InvalidRequestException {
        if (id <= 0L) {
            throw new InvalidRequestException("Invalid negative ID.");
        }
        CacheDirective directive = this.directivesById.get(id);
        if (directive == null) {
            throw new InvalidRequestException("No directive with ID " + id + " found.");
        }
        return directive;
    }

    private CachePool getCachePool(String poolName) throws InvalidRequestException {
        CachePool pool = this.cachePools.get(poolName);
        if (pool == null) {
            throw new InvalidRequestException("Unknown pool " + poolName);
        }
        return pool;
    }

    private void addInternal(CacheDirective directive, CachePool pool) {
        boolean addedDirective = pool.getDirectiveList().add(directive);
        assert (addedDirective);
        this.directivesById.put(directive.getId(), directive);
        String path = directive.getPath();
        List<CacheDirective> directives = this.directivesByPath.get(path);
        if (directives == null) {
            directives = new ArrayList<CacheDirective>(1);
            this.directivesByPath.put(path, directives);
        }
        directives.add(directive);
        CacheDirectiveStats stats = this.computeNeeded(directive.getPath(), directive.getReplication());
        directive.addBytesNeeded(stats.getBytesNeeded());
        directive.addFilesNeeded(directive.getFilesNeeded());
        this.setNeedsRescan();
    }

    CacheDirectiveInfo addDirectiveFromEditLog(CacheDirectiveInfo directive) throws InvalidRequestException {
        long id = directive.getId();
        CacheDirective entry = new CacheDirective(directive);
        CachePool pool = this.cachePools.get(directive.getPool());
        this.addInternal(entry, pool);
        if (this.nextDirectiveId <= id) {
            this.nextDirectiveId = id + 1L;
        }
        return entry.toInfo();
    }

    public CacheDirectiveInfo addDirective(CacheDirectiveInfo info, FSPermissionChecker pc, EnumSet<CacheFlag> flags) throws IOException {
        CacheDirective directive;
        assert (this.namesystem.hasWriteLock());
        try {
            CachePool pool = this.getCachePool(CacheManager.validatePoolName(info));
            CacheManager.checkWritePermission(pc, pool);
            String path = CacheManager.validatePath(info);
            short replication = CacheManager.validateReplication(info, (short)1);
            long expiryTime = CacheManager.validateExpiryTime(info, pool.getMaxRelativeExpiryMs());
            if (!flags.contains((Object)CacheFlag.FORCE)) {
                this.checkLimit(pool, path, replication);
            }
            long id = this.getNextDirectiveId();
            directive = new CacheDirective(id, path, replication, expiryTime);
            this.addInternal(directive, pool);
        }
        catch (IOException e) {
            LOG.warn((Object)("addDirective of " + info + " failed: "), (Throwable)e);
            throw e;
        }
        LOG.info((Object)("addDirective of " + info + " successful."));
        return directive.toInfo();
    }

    private static CacheDirectiveInfo createFromInfoAndDefaults(CacheDirectiveInfo info, CacheDirective defaults) {
        CacheDirectiveInfo.Builder builder = new CacheDirectiveInfo.Builder(defaults.toInfo());
        if (info.getPath() != null) {
            builder.setPath(info.getPath());
        }
        if (info.getReplication() != null) {
            builder.setReplication(info.getReplication());
        }
        if (info.getPool() != null) {
            builder.setPool(info.getPool());
        }
        if (info.getExpiration() != null) {
            builder.setExpiration(info.getExpiration());
        }
        return builder.build();
    }

    void modifyDirectiveFromEditLog(CacheDirectiveInfo info) throws InvalidRequestException {
        Long id = info.getId();
        if (id == null) {
            throw new InvalidRequestException("Must supply an ID.");
        }
        CacheDirective prevEntry = this.getById(id);
        CacheDirectiveInfo newInfo = CacheManager.createFromInfoAndDefaults(info, prevEntry);
        this.removeInternal(prevEntry);
        this.addInternal(new CacheDirective(newInfo), this.getCachePool(newInfo.getPool()));
    }

    public void modifyDirective(CacheDirectiveInfo info, FSPermissionChecker pc, EnumSet<CacheFlag> flags) throws IOException {
        assert (this.namesystem.hasWriteLock());
        String idString = info.getId() == null ? "(null)" : info.getId().toString();
        try {
            Long id = info.getId();
            if (id == null) {
                throw new InvalidRequestException("Must supply an ID.");
            }
            CacheDirective prevEntry = this.getById(id);
            CacheManager.checkWritePermission(pc, prevEntry.getPool());
            CacheDirectiveInfo infoWithDefaults = CacheManager.createFromInfoAndDefaults(info, prevEntry);
            CacheDirectiveInfo.Builder builder = new CacheDirectiveInfo.Builder(infoWithDefaults);
            CacheManager.validatePath(infoWithDefaults);
            CacheManager.validateReplication(infoWithDefaults, (short)-1);
            CachePool srcPool = prevEntry.getPool();
            CachePool destPool = this.getCachePool(CacheManager.validatePoolName(infoWithDefaults));
            if (!srcPool.getPoolName().equals(destPool.getPoolName())) {
                CacheManager.checkWritePermission(pc, destPool);
                if (!flags.contains((Object)CacheFlag.FORCE)) {
                    this.checkLimit(destPool, infoWithDefaults.getPath().toUri().getPath(), infoWithDefaults.getReplication());
                }
            }
            CacheManager.validateExpiryTime(infoWithDefaults, destPool.getMaxRelativeExpiryMs());
            this.setNeedsRescan();
            this.removeInternal(prevEntry);
            this.addInternal(new CacheDirective(builder.build()), destPool);
        }
        catch (IOException e) {
            LOG.warn((Object)("modifyDirective of " + idString + " failed: "), (Throwable)e);
            throw e;
        }
        LOG.info((Object)("modifyDirective of " + idString + " successfully applied " + info + "."));
    }

    private void removeInternal(CacheDirective directive) throws InvalidRequestException {
        assert (this.namesystem.hasWriteLock());
        String path = directive.getPath();
        List<CacheDirective> directives = this.directivesByPath.get(path);
        if (directives == null || !directives.remove(directive)) {
            throw new InvalidRequestException("Failed to locate entry " + directive.getId() + " by path " + directive.getPath());
        }
        if (directives.size() == 0) {
            this.directivesByPath.remove(path);
        }
        CachePool pool = directive.getPool();
        directive.addBytesNeeded(-directive.getBytesNeeded());
        directive.addFilesNeeded(-directive.getFilesNeeded());
        this.directivesById.remove(directive.getId());
        pool.getDirectiveList().remove(directive);
        assert (directive.getPool() == null);
        this.setNeedsRescan();
    }

    public void removeDirective(long id, FSPermissionChecker pc) throws IOException {
        assert (this.namesystem.hasWriteLock());
        try {
            CacheDirective directive = this.getById(id);
            CacheManager.checkWritePermission(pc, directive.getPool());
            this.removeInternal(directive);
        }
        catch (IOException e) {
            LOG.warn((Object)("removeDirective of " + id + " failed: "), (Throwable)e);
            throw e;
        }
        LOG.info((Object)("removeDirective of " + id + " successful."));
    }

    public BatchedRemoteIterator.BatchedListEntries<CacheDirectiveEntry> listCacheDirectives(long prevId, CacheDirectiveInfo filter, FSPermissionChecker pc) throws IOException {
        assert (this.namesystem.hasReadLock());
        int NUM_PRE_ALLOCATED_ENTRIES = 16;
        String filterPath = null;
        if (filter.getPath() != null) {
            filterPath = CacheManager.validatePath(filter);
        }
        if (filter.getReplication() != null) {
            throw new InvalidRequestException("Filtering by replication is unsupported.");
        }
        Long id = filter.getId();
        if (id != null) {
            if (!this.directivesById.containsKey(id)) {
                throw new InvalidRequestException("Did not find requested id " + id);
            }
            prevId = id - 1L;
        }
        ArrayList<CacheDirectiveEntry> replies = new ArrayList<CacheDirectiveEntry>(16);
        int numReplies = 0;
        SortedMap<Long, CacheDirective> tailMap = this.directivesById.tailMap(prevId + 1L);
        for (Map.Entry<Long, CacheDirective> cur : tailMap.entrySet()) {
            if (numReplies >= this.maxListCacheDirectivesNumResponses) {
                return new BatchedRemoteIterator.BatchedListEntries<CacheDirectiveEntry>(replies, true);
            }
            CacheDirective curDirective = cur.getValue();
            CacheDirectiveInfo info = cur.getValue().toInfo();
            if (id != null && !info.getId().equals(id)) break;
            if (filter.getPool() != null && !info.getPool().equals(filter.getPool()) || filterPath != null && !info.getPath().toUri().getPath().equals(filterPath)) continue;
            boolean hasPermission = true;
            if (pc != null) {
                try {
                    pc.checkPermission(curDirective.getPool(), FsAction.READ);
                }
                catch (AccessControlException e) {
                    hasPermission = false;
                }
            }
            if (!hasPermission) continue;
            replies.add(new CacheDirectiveEntry(info, cur.getValue().toStats()));
            ++numReplies;
        }
        return new BatchedRemoteIterator.BatchedListEntries<CacheDirectiveEntry>(replies, false);
    }

    public CachePoolInfo addCachePool(CachePoolInfo info) throws IOException {
        CachePool pool;
        assert (this.namesystem.hasWriteLock());
        try {
            CachePoolInfo.validate(info);
            String poolName = info.getPoolName();
            pool = this.cachePools.get(poolName);
            if (pool != null) {
                throw new InvalidRequestException("Cache pool " + poolName + " already exists.");
            }
            pool = CachePool.createFromInfoAndDefaults(info);
            this.cachePools.put(pool.getPoolName(), pool);
        }
        catch (IOException e) {
            LOG.info((Object)("addCachePool of " + info + " failed: "), (Throwable)e);
            throw e;
        }
        LOG.info((Object)("addCachePool of " + info + " successful."));
        return pool.getInfo(true);
    }

    public void modifyCachePool(CachePoolInfo info) throws IOException {
        assert (this.namesystem.hasWriteLock());
        StringBuilder bld = new StringBuilder();
        try {
            CachePoolInfo.validate(info);
            String poolName = info.getPoolName();
            CachePool pool = this.cachePools.get(poolName);
            if (pool == null) {
                throw new InvalidRequestException("Cache pool " + poolName + " does not exist.");
            }
            String prefix = "";
            if (info.getOwnerName() != null) {
                pool.setOwnerName(info.getOwnerName());
                bld.append(prefix).append("set owner to ").append(info.getOwnerName());
                prefix = "; ";
            }
            if (info.getGroupName() != null) {
                pool.setGroupName(info.getGroupName());
                bld.append(prefix).append("set group to ").append(info.getGroupName());
                prefix = "; ";
            }
            if (info.getMode() != null) {
                pool.setMode(info.getMode());
                bld.append(prefix).append("set mode to " + info.getMode());
                prefix = "; ";
            }
            if (info.getLimit() != null) {
                pool.setLimit(info.getLimit());
                bld.append(prefix).append("set limit to " + info.getLimit());
                prefix = "; ";
                this.setNeedsRescan();
            }
            if (info.getMaxRelativeExpiryMs() != null) {
                Long maxRelativeExpiry = info.getMaxRelativeExpiryMs();
                pool.setMaxRelativeExpiryMs(maxRelativeExpiry);
                bld.append(prefix).append("set maxRelativeExpiry to " + maxRelativeExpiry);
                prefix = "; ";
            }
            if (prefix.isEmpty()) {
                bld.append("no changes.");
            }
        }
        catch (IOException e) {
            LOG.info((Object)("modifyCachePool of " + info + " failed: "), (Throwable)e);
            throw e;
        }
        LOG.info((Object)("modifyCachePool of " + info.getPoolName() + " successful; " + bld.toString()));
    }

    public void removeCachePool(String poolName) throws IOException {
        assert (this.namesystem.hasWriteLock());
        try {
            CachePoolInfo.validateName(poolName);
            CachePool pool = this.cachePools.remove(poolName);
            if (pool == null) {
                throw new InvalidRequestException("Cannot remove non-existent cache pool " + poolName);
            }
            Iterator iter = pool.getDirectiveList().iterator();
            while (iter.hasNext()) {
                CacheDirective directive = (CacheDirective)iter.next();
                this.directivesByPath.remove(directive.getPath());
                this.directivesById.remove(directive.getId());
                iter.remove();
            }
            this.setNeedsRescan();
        }
        catch (IOException e) {
            LOG.info((Object)("removeCachePool of " + poolName + " failed: "), (Throwable)e);
            throw e;
        }
        LOG.info((Object)("removeCachePool of " + poolName + " successful."));
    }

    public BatchedRemoteIterator.BatchedListEntries<CachePoolEntry> listCachePools(FSPermissionChecker pc, String prevKey) {
        assert (this.namesystem.hasReadLock());
        int NUM_PRE_ALLOCATED_ENTRIES = 16;
        ArrayList<CachePoolEntry> results = new ArrayList<CachePoolEntry>(16);
        NavigableMap<String, CachePool> tailMap = this.cachePools.tailMap(prevKey, false);
        int numListed = 0;
        for (Map.Entry cur : tailMap.entrySet()) {
            if (numListed++ >= this.maxListCachePoolsResponses) {
                return new BatchedRemoteIterator.BatchedListEntries<CachePoolEntry>(results, true);
            }
            results.add(((CachePool)cur.getValue()).getEntry(pc));
        }
        return new BatchedRemoteIterator.BatchedListEntries<CachePoolEntry>(results, false);
    }

    public void setCachedLocations(LocatedBlock block) {
        CachedBlock cachedBlock = new CachedBlock(block.getBlock().getBlockId(), 0, false);
        if ((cachedBlock = this.cachedBlocks.get(cachedBlock)) == null) {
            return;
        }
        List<DatanodeDescriptor> datanodes = cachedBlock.getDatanodes(DatanodeDescriptor.CachedBlocksList.Type.CACHED);
        for (DatanodeDescriptor datanode : datanodes) {
            block.addCachedLoc(datanode);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final void processCacheReport(DatanodeID datanodeID, List<Long> blockIds) throws IOException {
        long endTime;
        this.namesystem.writeLock();
        long startTime = Time.monotonicNow();
        try {
            DatanodeDescriptor datanode = this.blockManager.getDatanodeManager().getDatanode(datanodeID);
            if (datanode == null || !datanode.isAlive) {
                throw new IOException("processCacheReport from dead or unregistered datanode: " + datanode);
            }
            this.processCacheReportImpl(datanode, blockIds);
        }
        finally {
            endTime = Time.monotonicNow();
            this.namesystem.writeUnlock();
        }
        NameNodeMetrics metrics = NameNode.getNameNodeMetrics();
        if (metrics != null) {
            metrics.addCacheBlockReport((int)(endTime - startTime));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Processed cache report from " + datanodeID + ", blocks: " + blockIds.size() + ", processing time: " + (endTime - startTime) + " msecs"));
        }
    }

    private void processCacheReportImpl(DatanodeDescriptor datanode, List<Long> blockIds) {
        DatanodeDescriptor.CachedBlocksList cached = datanode.getCached();
        cached.clear();
        DatanodeDescriptor.CachedBlocksList cachedList = datanode.getCached();
        DatanodeDescriptor.CachedBlocksList pendingCachedList = datanode.getPendingCached();
        for (long blockId : blockIds) {
            CachedBlock cachedBlock = new CachedBlock(blockId, 0, false);
            CachedBlock prevCachedBlock = this.cachedBlocks.get(cachedBlock);
            if (prevCachedBlock != null) {
                cachedBlock = prevCachedBlock;
            } else {
                this.cachedBlocks.put(cachedBlock);
            }
            if (!cachedBlock.isPresent(cachedList)) {
                cachedList.add(cachedBlock);
            }
            if (!cachedBlock.isPresent(pendingCachedList)) continue;
            pendingCachedList.remove(cachedBlock);
        }
    }

    public void saveStateCompat(DataOutputStream out, String sdPath) throws IOException {
        this.serializerCompat.save(out, sdPath);
    }

    public PersistState saveState() throws IOException {
        GeneratedMessage.Builder b;
        ArrayList<ClientNamenodeProtocolProtos.CachePoolInfoProto> pools = Lists.newArrayListWithCapacity(this.cachePools.size());
        ArrayList<ClientNamenodeProtocolProtos.CacheDirectiveInfoProto> directives = Lists.newArrayListWithCapacity(this.directivesById.size());
        for (CachePool pool : this.cachePools.values()) {
            CachePoolInfo p = pool.getInfo(true);
            b = ClientNamenodeProtocolProtos.CachePoolInfoProto.newBuilder().setPoolName(p.getPoolName());
            if (p.getOwnerName() != null) {
                ((ClientNamenodeProtocolProtos.CachePoolInfoProto.Builder)b).setOwnerName(p.getOwnerName());
            }
            if (p.getGroupName() != null) {
                ((ClientNamenodeProtocolProtos.CachePoolInfoProto.Builder)b).setGroupName(p.getGroupName());
            }
            if (p.getMode() != null) {
                ((ClientNamenodeProtocolProtos.CachePoolInfoProto.Builder)b).setMode(p.getMode().toShort());
            }
            if (p.getLimit() != null) {
                ((ClientNamenodeProtocolProtos.CachePoolInfoProto.Builder)b).setLimit(p.getLimit());
            }
            pools.add(((ClientNamenodeProtocolProtos.CachePoolInfoProto.Builder)b).build());
        }
        for (CacheDirective directive : this.directivesById.values()) {
            CacheDirectiveInfo.Expiration expiry;
            CacheDirectiveInfo info = directive.toInfo();
            b = ClientNamenodeProtocolProtos.CacheDirectiveInfoProto.newBuilder().setId(info.getId());
            if (info.getPath() != null) {
                ((ClientNamenodeProtocolProtos.CacheDirectiveInfoProto.Builder)b).setPath(info.getPath().toUri().getPath());
            }
            if (info.getReplication() != null) {
                ((ClientNamenodeProtocolProtos.CacheDirectiveInfoProto.Builder)b).setReplication(info.getReplication().shortValue());
            }
            if (info.getPool() != null) {
                ((ClientNamenodeProtocolProtos.CacheDirectiveInfoProto.Builder)b).setPool(info.getPool());
            }
            if ((expiry = info.getExpiration()) != null) {
                assert (!expiry.isRelative());
                ((ClientNamenodeProtocolProtos.CacheDirectiveInfoProto.Builder)b).setExpiration(PBHelper.convert(expiry));
            }
            directives.add(((ClientNamenodeProtocolProtos.CacheDirectiveInfoProto.Builder)b).build());
        }
        FsImageProto.CacheManagerSection s = FsImageProto.CacheManagerSection.newBuilder().setNextDirectiveId(this.nextDirectiveId).setNumPools(pools.size()).setNumDirectives(directives.size()).build();
        return new PersistState(s, pools, directives);
    }

    public void loadStateCompat(DataInput in) throws IOException {
        this.serializerCompat.load(in);
    }

    public void loadState(PersistState s) throws IOException {
        this.nextDirectiveId = s.section.getNextDirectiveId();
        for (ClientNamenodeProtocolProtos.CachePoolInfoProto cachePoolInfoProto : s.pools) {
            CachePoolInfo info = new CachePoolInfo(cachePoolInfoProto.getPoolName());
            if (cachePoolInfoProto.hasOwnerName()) {
                info.setOwnerName(cachePoolInfoProto.getOwnerName());
            }
            if (cachePoolInfoProto.hasGroupName()) {
                info.setGroupName(cachePoolInfoProto.getGroupName());
            }
            if (cachePoolInfoProto.hasMode()) {
                info.setMode(new FsPermission((short)cachePoolInfoProto.getMode()));
            }
            if (cachePoolInfoProto.hasLimit()) {
                info.setLimit(cachePoolInfoProto.getLimit());
            }
            this.addCachePool(info);
        }
        for (ClientNamenodeProtocolProtos.CacheDirectiveInfoProto cacheDirectiveInfoProto : s.directives) {
            String poolName = cacheDirectiveInfoProto.getPool();
            CacheDirective directive = new CacheDirective(cacheDirectiveInfoProto.getId(), new Path(cacheDirectiveInfoProto.getPath()).toUri().getPath(), (short)cacheDirectiveInfoProto.getReplication(), cacheDirectiveInfoProto.getExpiration().getMillis());
            this.addCacheDirective(poolName, directive);
        }
    }

    private void addCacheDirective(String poolName, CacheDirective directive) throws IOException {
        CachePool pool = this.cachePools.get(poolName);
        if (pool == null) {
            throw new IOException("Directive refers to pool " + poolName + ", which does not exist.");
        }
        boolean addedDirective = pool.getDirectiveList().add(directive);
        assert (addedDirective);
        if (this.directivesById.put(directive.getId(), directive) != null) {
            throw new IOException("A directive with ID " + directive.getId() + " already exists");
        }
        List<CacheDirective> directives = this.directivesByPath.get(directive.getPath());
        if (directives == null) {
            directives = new LinkedList<CacheDirective>();
            this.directivesByPath.put(directive.getPath(), directives);
        }
        directives.add(directive);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void waitForRescanIfNeeded() {
        this.crmLock.lock();
        try {
            if (this.monitor != null) {
                this.monitor.waitForRescanIfNeeded();
            }
        }
        finally {
            this.crmLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void setNeedsRescan() {
        this.crmLock.lock();
        try {
            if (this.monitor != null) {
                this.monitor.setNeedsRescan();
            }
        }
        finally {
            this.crmLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public Thread getCacheReplicationMonitor() {
        this.crmLock.lock();
        try {
            CacheReplicationMonitor cacheReplicationMonitor = this.monitor;
            return cacheReplicationMonitor;
        }
        finally {
            this.crmLock.unlock();
        }
    }

    private final class SerializerCompat {
        private SerializerCompat() {
        }

        private void save(DataOutputStream out, String sdPath) throws IOException {
            out.writeLong(CacheManager.this.nextDirectiveId);
            this.savePools(out, sdPath);
            this.saveDirectives(out, sdPath);
        }

        private void load(DataInput in) throws IOException {
            CacheManager.this.nextDirectiveId = in.readLong();
            this.loadPools(in);
            this.loadDirectives(in);
        }

        private void savePools(DataOutputStream out, String sdPath) throws IOException {
            StartupProgress prog = NameNode.getStartupProgress();
            Step step = new Step(StepType.CACHE_POOLS, sdPath);
            prog.beginStep(Phase.SAVING_CHECKPOINT, step);
            prog.setTotal(Phase.SAVING_CHECKPOINT, step, CacheManager.this.cachePools.size());
            StartupProgress.Counter counter = prog.getCounter(Phase.SAVING_CHECKPOINT, step);
            out.writeInt(CacheManager.this.cachePools.size());
            for (CachePool pool : CacheManager.this.cachePools.values()) {
                FSImageSerialization.writeCachePoolInfo(out, pool.getInfo(true));
                counter.increment();
            }
            prog.endStep(Phase.SAVING_CHECKPOINT, step);
        }

        private void saveDirectives(DataOutputStream out, String sdPath) throws IOException {
            StartupProgress prog = NameNode.getStartupProgress();
            Step step = new Step(StepType.CACHE_ENTRIES, sdPath);
            prog.beginStep(Phase.SAVING_CHECKPOINT, step);
            prog.setTotal(Phase.SAVING_CHECKPOINT, step, CacheManager.this.directivesById.size());
            StartupProgress.Counter counter = prog.getCounter(Phase.SAVING_CHECKPOINT, step);
            out.writeInt(CacheManager.this.directivesById.size());
            for (CacheDirective directive : CacheManager.this.directivesById.values()) {
                FSImageSerialization.writeCacheDirectiveInfo(out, directive.toInfo());
                counter.increment();
            }
            prog.endStep(Phase.SAVING_CHECKPOINT, step);
        }

        private void loadPools(DataInput in) throws IOException {
            StartupProgress prog = NameNode.getStartupProgress();
            Step step = new Step(StepType.CACHE_POOLS);
            prog.beginStep(Phase.LOADING_FSIMAGE, step);
            int numberOfPools = in.readInt();
            prog.setTotal(Phase.LOADING_FSIMAGE, step, numberOfPools);
            StartupProgress.Counter counter = prog.getCounter(Phase.LOADING_FSIMAGE, step);
            for (int i = 0; i < numberOfPools; ++i) {
                CacheManager.this.addCachePool(FSImageSerialization.readCachePoolInfo(in));
                counter.increment();
            }
            prog.endStep(Phase.LOADING_FSIMAGE, step);
        }

        private void loadDirectives(DataInput in) throws IOException {
            StartupProgress prog = NameNode.getStartupProgress();
            Step step = new Step(StepType.CACHE_ENTRIES);
            prog.beginStep(Phase.LOADING_FSIMAGE, step);
            int numDirectives = in.readInt();
            prog.setTotal(Phase.LOADING_FSIMAGE, step, numDirectives);
            StartupProgress.Counter counter = prog.getCounter(Phase.LOADING_FSIMAGE, step);
            for (int i = 0; i < numDirectives; ++i) {
                CacheDirectiveInfo info = FSImageSerialization.readCacheDirectiveInfo(in);
                String poolName = info.getPool();
                CacheDirective directive = new CacheDirective(info.getId(), info.getPath().toUri().getPath(), info.getReplication(), info.getExpiration().getAbsoluteMillis());
                CacheManager.this.addCacheDirective(poolName, directive);
                counter.increment();
            }
            prog.endStep(Phase.LOADING_FSIMAGE, step);
        }
    }

    public static final class PersistState {
        public final FsImageProto.CacheManagerSection section;
        public final List<ClientNamenodeProtocolProtos.CachePoolInfoProto> pools;
        public final List<ClientNamenodeProtocolProtos.CacheDirectiveInfoProto> directives;

        public PersistState(FsImageProto.CacheManagerSection section, List<ClientNamenodeProtocolProtos.CachePoolInfoProto> pools, List<ClientNamenodeProtocolProtos.CacheDirectiveInfoProto> directives) {
            this.section = section;
            this.pools = pools;
            this.directives = directives;
        }
    }
}

