/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile.bucket;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.util.Arrays;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.io.hfile.bucket.IOEngine;
import org.apache.hadoop.hbase.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.util.StringUtils;

@InterfaceAudience.Private
public class FileIOEngine
implements IOEngine {
    private static final Log LOG = LogFactory.getLog(FileIOEngine.class);
    public static final String FILE_DELIMITER = ",";
    private final String[] filePaths;
    private final FileChannel[] fileChannels;
    private final RandomAccessFile[] rafs;
    private final long sizePerFile;
    private final long capacity;
    private FileReadAccessor readAccessor = new FileReadAccessor();
    private FileWriteAccessor writeAccessor = new FileWriteAccessor();

    public FileIOEngine(long capacity, String ... filePaths) throws IOException {
        this.sizePerFile = capacity / (long)filePaths.length;
        this.capacity = this.sizePerFile * (long)filePaths.length;
        this.filePaths = filePaths;
        this.fileChannels = new FileChannel[filePaths.length];
        this.rafs = new RandomAccessFile[filePaths.length];
        for (int i = 0; i < filePaths.length; ++i) {
            String filePath = filePaths[i];
            try {
                this.rafs[i] = new RandomAccessFile(filePath, "rw");
                long totalSpace = new File(filePath).getTotalSpace();
                if (totalSpace < this.sizePerFile) {
                    String msg = "Only " + StringUtils.byteDesc(totalSpace) + " total space under " + filePath + ", not enough for requested " + StringUtils.byteDesc(this.sizePerFile);
                    LOG.warn((Object)msg);
                }
                this.rafs[i].setLength(this.sizePerFile);
                this.fileChannels[i] = this.rafs[i].getChannel();
                LOG.info((Object)("Allocating cache " + StringUtils.byteDesc(this.sizePerFile) + ", on the path:" + filePath));
                continue;
            }
            catch (IOException fex) {
                LOG.error((Object)("Failed allocating cache on " + filePath), (Throwable)fex);
                this.shutdown();
                throw fex;
            }
        }
    }

    public String toString() {
        return "ioengine=" + this.getClass().getSimpleName() + ", paths=" + Arrays.asList(this.filePaths) + ", capacity=" + String.format("%,d", this.capacity);
    }

    @Override
    public boolean isPersistent() {
        return true;
    }

    @Override
    public int read(ByteBuffer dstBuffer, long offset) throws IOException {
        if (dstBuffer.remaining() != 0) {
            return this.accessFile(this.readAccessor, dstBuffer, offset);
        }
        return 0;
    }

    @VisibleForTesting
    void closeFileChannels() {
        for (FileChannel fileChannel : this.fileChannels) {
            try {
                fileChannel.close();
            }
            catch (IOException e) {
                LOG.warn((Object)"Failed to close FileChannel", (Throwable)e);
            }
        }
    }

    @Override
    public void write(ByteBuffer srcBuffer, long offset) throws IOException {
        if (!srcBuffer.hasRemaining()) {
            return;
        }
        this.accessFile(this.writeAccessor, srcBuffer, offset);
    }

    @Override
    public void sync() throws IOException {
        for (int i = 0; i < this.fileChannels.length; ++i) {
            try {
                if (this.fileChannels[i] == null) continue;
                this.fileChannels[i].force(true);
                continue;
            }
            catch (IOException ie) {
                LOG.warn((Object)("Failed syncing data to " + this.filePaths[i]));
                throw ie;
            }
        }
    }

    @Override
    public void shutdown() {
        for (int i = 0; i < this.filePaths.length; ++i) {
            try {
                if (this.fileChannels[i] != null) {
                    this.fileChannels[i].close();
                }
                if (this.rafs[i] == null) continue;
                this.rafs[i].close();
                continue;
            }
            catch (IOException ex) {
                LOG.error((Object)("Failed closing " + this.filePaths[i] + " when shudown the IOEngine"), (Throwable)ex);
            }
        }
    }

    private int accessFile(FileAccessor accessor, ByteBuffer buffer, long globalOffset) throws IOException {
        int bufLimit;
        block4: {
            int startFileNum = this.getFileNum(globalOffset);
            int remainingAccessDataLen = buffer.remaining();
            int endFileNum = this.getFileNum(globalOffset + (long)remainingAccessDataLen - 1L);
            int accessFileNum = startFileNum;
            long accessOffset = this.getAbsoluteOffsetInFile(accessFileNum, globalOffset);
            bufLimit = buffer.limit();
            while (true) {
                FileChannel fileChannel = this.fileChannels[accessFileNum];
                int accessLen = 0;
                if (endFileNum > accessFileNum) {
                    buffer.limit((int)((long)(buffer.limit() - remainingAccessDataLen) + this.sizePerFile - accessOffset));
                }
                try {
                    accessLen = accessor.access(fileChannel, buffer, accessOffset);
                }
                catch (ClosedChannelException e) {
                    LOG.warn((Object)"Caught ClosedChannelException accessing BucketCache, reopening file. ", (Throwable)e);
                    this.refreshFileConnection(accessFileNum);
                    continue;
                }
                buffer.limit(bufLimit);
                if (accessLen >= remainingAccessDataLen) break block4;
                remainingAccessDataLen -= accessLen;
                accessOffset = 0L;
                if (++accessFileNum >= this.fileChannels.length) break;
            }
            throw new IOException("Required data len " + StringUtils.byteDesc(buffer.remaining()) + " exceed the engine's capacity " + StringUtils.byteDesc(this.capacity) + " where offset=" + globalOffset);
        }
        return bufLimit;
    }

    private long getAbsoluteOffsetInFile(int fileNum, long globalOffset) {
        return globalOffset - (long)fileNum * this.sizePerFile;
    }

    private int getFileNum(long offset) {
        if (offset < 0L) {
            throw new IllegalArgumentException("Unexpected offset " + offset);
        }
        int fileNum = (int)(offset / this.sizePerFile);
        if (fileNum >= this.fileChannels.length) {
            throw new RuntimeException("Not expected offset " + offset + " where capacity=" + this.capacity);
        }
        return fileNum;
    }

    @VisibleForTesting
    FileChannel[] getFileChannels() {
        return this.fileChannels;
    }

    @VisibleForTesting
    void refreshFileConnection(int accessFileNum) throws IOException {
        FileChannel fileChannel = this.fileChannels[accessFileNum];
        if (fileChannel != null) {
            fileChannel.close();
        }
        this.rafs[accessFileNum] = new RandomAccessFile(this.filePaths[accessFileNum], "rw");
        this.fileChannels[accessFileNum] = this.rafs[accessFileNum].getChannel();
    }

    private static class FileWriteAccessor
    implements FileAccessor {
        private FileWriteAccessor() {
        }

        @Override
        public int access(FileChannel fileChannel, ByteBuffer byteBuffer, long accessOffset) throws IOException {
            return fileChannel.write(byteBuffer, accessOffset);
        }
    }

    private static class FileReadAccessor
    implements FileAccessor {
        private FileReadAccessor() {
        }

        @Override
        public int access(FileChannel fileChannel, ByteBuffer byteBuffer, long accessOffset) throws IOException {
            return fileChannel.read(byteBuffer, accessOffset);
        }
    }

    private static interface FileAccessor {
        public int access(FileChannel var1, ByteBuffer var2, long var3) throws IOException;
    }
}

