/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.normalizer;

import java.io.IOException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HBaseIOException;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.RegionLoad;
import org.apache.hadoop.hbase.ServerLoad;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.master.MasterRpcServices;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.RegionState;
import org.apache.hadoop.hbase.master.RegionStates;
import org.apache.hadoop.hbase.master.normalizer.MergeNormalizationPlan;
import org.apache.hadoop.hbase.master.normalizer.NormalizationPlan;
import org.apache.hadoop.hbase.master.normalizer.RegionNormalizer;
import org.apache.hadoop.hbase.master.normalizer.SplitNormalizationPlan;
import org.apache.hadoop.hbase.protobuf.RequestConverter;
import org.apache.hadoop.hbase.shaded.com.google.protobuf.ServiceException;
import org.apache.hadoop.hbase.shaded.org.apache.commons.collections.CollectionUtils;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;

@InterfaceAudience.LimitedPrivate(value={"Configuration"})
public class SimpleRegionNormalizer
implements RegionNormalizer {
    private static final Log LOG = LogFactory.getLog(SimpleRegionNormalizer.class);
    static final String SPLIT_ENABLED_KEY = "hbase.normalizer.split.enabled";
    static final boolean DEFAULT_SPLIT_ENABLED = true;
    static final String MERGE_ENABLED_KEY = "hbase.normalizer.merge.enabled";
    static final boolean DEFAULT_MERGE_ENABLED = true;
    static final String MIN_REGION_COUNT_KEY = "hbase.normalizer.min.region.count";
    static final int DEFAULT_MIN_REGION_COUNT = 3;
    static final String MERGE_MIN_REGION_AGE_DAYS_KEY = "hbase.normalizer.merge.min_region_age.days";
    static final int DEFAULT_MERGE_MIN_REGION_AGE_DAYS = 3;
    static final String MERGE_MIN_REGION_SIZE_MB_KEY = "hbase.normalizer.merge.min_region_size.mb";
    static final int DEFAULT_MERGE_MIN_REGION_SIZE_MB = 1;
    private final long[] skippedCount;
    private Configuration conf;
    private MasterServices masterServices;
    private MasterRpcServices masterRpcServices;
    private boolean splitEnabled = true;
    private boolean mergeEnabled = true;
    private int minRegionCount = 3;
    private int mergeMinRegionAge = 3;
    private int mergeMinRegionSizeMb = 1;
    private Comparator<NormalizationPlan> planComparator = new Comparator<NormalizationPlan>(){

        @Override
        public int compare(NormalizationPlan plan, NormalizationPlan plan2) {
            if (plan instanceof SplitNormalizationPlan) {
                return -1;
            }
            if (plan2 instanceof SplitNormalizationPlan) {
                return 1;
            }
            return 0;
        }
    };

    public SimpleRegionNormalizer() {
        this.skippedCount = new long[NormalizationPlan.PlanType.values().length];
    }

    @Override
    public void setMasterRpcServices(MasterRpcServices masterRpcServices) {
        this.masterRpcServices = masterRpcServices;
    }

    @Override
    public Configuration getConf() {
        return this.conf;
    }

    @Override
    public void setConf(Configuration conf) {
        if (conf == null) {
            return;
        }
        this.conf = conf;
        this.splitEnabled = conf.getBoolean(SPLIT_ENABLED_KEY, true);
        this.mergeEnabled = conf.getBoolean(MERGE_ENABLED_KEY, true);
        this.minRegionCount = this.parseMinRegionCount(conf);
        this.mergeMinRegionAge = this.parseMergeMinRegionAge(conf);
        this.mergeMinRegionSizeMb = this.parseMergeMinRegionSizeMb(conf);
    }

    private int parseMergeMinRegionSizeMb(Configuration conf) {
        int settledValue;
        int parsedValue = conf.getInt(MERGE_MIN_REGION_SIZE_MB_KEY, 1);
        if (parsedValue != (settledValue = Math.max(0, parsedValue))) {
            this.warnInvalidValue(MERGE_MIN_REGION_SIZE_MB_KEY, parsedValue, settledValue);
        }
        return settledValue;
    }

    private int parseMinRegionCount(Configuration conf) {
        int settledValue;
        int parsedValue = conf.getInt(MIN_REGION_COUNT_KEY, 3);
        if (parsedValue != (settledValue = Math.max(1, parsedValue))) {
            this.warnInvalidValue(MIN_REGION_COUNT_KEY, parsedValue, settledValue);
        }
        return settledValue;
    }

    private int parseMergeMinRegionAge(Configuration conf) {
        int settledValue;
        int parsedValue = conf.getInt(MERGE_MIN_REGION_AGE_DAYS_KEY, 3);
        if (parsedValue != (settledValue = Math.max(0, parsedValue))) {
            this.warnInvalidValue(MERGE_MIN_REGION_AGE_DAYS_KEY, parsedValue, settledValue);
        }
        return settledValue;
    }

    private void warnInvalidValue(String key, int parsedValue, int settledValue) {
        LOG.warn((Object)("Configured value " + key + "=" + parsedValue + " is invalid. Setting value to" + settledValue));
    }

    public boolean isSplitEnabled() {
        return this.splitEnabled;
    }

    public boolean isMergeEnabled() {
        return this.mergeEnabled;
    }

    private boolean isMasterSwitchEnabled(Admin.MasterSwitchType masterSwitchType) {
        boolean enabled = false;
        try {
            enabled = this.masterRpcServices.isSplitOrMergeEnabled(null, RequestConverter.buildIsSplitOrMergeEnabledRequest(masterSwitchType)).getEnabled();
        }
        catch (ServiceException e) {
            LOG.debug((Object)"Unable to determine whether split or merge is enabled", (Throwable)e);
        }
        return enabled;
    }

    public int getMinRegionCount() {
        return this.minRegionCount;
    }

    public int getMergeMinRegionAge() {
        return this.mergeMinRegionAge;
    }

    public int getMergeMinRegionSizeMb() {
        return this.mergeMinRegionSizeMb;
    }

    @Override
    public void setMasterServices(MasterServices masterServices) {
        this.masterServices = masterServices;
    }

    @Override
    public List<NormalizationPlan> computePlansForTable(TableName table) throws HBaseIOException {
        if (table == null) {
            return Collections.emptyList();
        }
        if (table.isSystemTable()) {
            LOG.debug((Object)("Normalization of system table " + table + " isn't allowed"));
            return Collections.emptyList();
        }
        boolean proceedWithSplitPlanning = this.proceedWithSplitPlanning();
        boolean proceedWithMergePlanning = this.proceedWithMergePlanning();
        if (!proceedWithMergePlanning && !proceedWithSplitPlanning) {
            LOG.debug((Object)("Both split and merge are disabled. Skipping normalization of table: " + table));
            return Collections.emptyList();
        }
        NormalizeContext ctx = new NormalizeContext(table);
        if (CollectionUtils.isEmpty(ctx.getTableRegions())) {
            return Collections.emptyList();
        }
        LOG.debug((Object)("Computing normalization plan for table:  " + table + ", number of regions: " + ctx.getTableRegions().size()));
        ArrayList<NormalizationPlan> plans = new ArrayList<NormalizationPlan>();
        if (proceedWithSplitPlanning) {
            plans.addAll(this.computeSplitNormalizationPlans(ctx));
        }
        if (proceedWithMergePlanning) {
            plans.addAll(this.computeMergeNormalizationPlans(ctx));
        }
        LOG.debug((Object)("Computed " + plans.size() + " normalization plans for table" + table));
        return plans;
    }

    private boolean proceedWithMergePlanning() {
        return this.isMergeEnabled() && this.isMasterSwitchEnabled(Admin.MasterSwitchType.MERGE);
    }

    private boolean proceedWithSplitPlanning() {
        return this.isSplitEnabled() && this.isMasterSwitchEnabled(Admin.MasterSwitchType.SPLIT);
    }

    private long getRegionSizeMB(HRegionInfo hri) {
        ServerName sn = this.masterServices.getAssignmentManager().getRegionStates().getRegionServerOfRegion(hri);
        if (sn == null) {
            LOG.debug((Object)(hri.getRegionNameAsString() + " region was not found on any Server"));
            return -1L;
        }
        ServerLoad load = this.masterServices.getServerManager().getLoad(sn);
        if (load == null) {
            LOG.debug((Object)(sn.getServerName() + " was not found in online servers"));
            return -1L;
        }
        RegionLoad regionLoad = load.getRegionsLoad().get(hri.getRegionName());
        if (regionLoad == null) {
            LOG.debug((Object)(hri.getRegionNameAsString() + " was not found in RegionsLoad"));
            return -1L;
        }
        return regionLoad.getStorefileSizeMB();
    }

    private double getAverageRegionSizeMb(List<HRegionInfo> tableRegions) {
        if (CollectionUtils.isEmpty(tableRegions)) {
            throw new IllegalStateException("Cannot calculate average size of a table without any regions.");
        }
        int regionCount = tableRegions.size();
        long totalSizeMb = 0L;
        for (HRegionInfo rinfo : tableRegions) {
            totalSizeMb += this.getRegionSizeMB(rinfo);
        }
        TableName table = tableRegions.get(0).getTable();
        int targetRegionCount = -1;
        long targetRegionSize = -1L;
        try {
            HTableDescriptor tableDescriptor = this.masterServices.getTableDescriptors().get(table);
            if (tableDescriptor != null && LOG.isDebugEnabled()) {
                targetRegionCount = tableDescriptor.getNormalizerTargetRegionCount();
                targetRegionSize = tableDescriptor.getNormalizerTargetRegionSize();
                LOG.debug((Object)("Table " + table + " configured with target region count" + targetRegionCount + ", target region size " + targetRegionSize));
            }
        }
        catch (IOException e) {
            LOG.warn((Object)("TableDescriptor for " + table + " unavailable, table-level target region count and size" + " configurations cannot be considered."), (Throwable)e);
        }
        double avgRegionSize = targetRegionSize > 0L ? (double)targetRegionSize : (targetRegionCount > 0 ? (double)totalSizeMb / (double)targetRegionCount : (double)totalSizeMb / (double)regionCount);
        LOG.debug((Object)("Table " + table + ", total aggregated regions size: " + totalSizeMb + " and average region size " + avgRegionSize));
        return avgRegionSize;
    }

    private boolean skipForMerge(RegionStates regionStates, HRegionInfo regionInfo) {
        boolean regionIsOpen = regionStates.isRegionInState(regionInfo, RegionState.State.OPEN);
        String name = regionInfo.getEncodedName();
        if (!regionIsOpen) {
            LOG.trace((Object)("skipping merge of region " + name + " because it is not open"));
            return true;
        }
        if (!this.isOldEnoughForMerge(regionInfo)) {
            LOG.trace((Object)("skipping merge of region " + name + " because it is not old enough."));
            return true;
        }
        if (!this.isLargeEnoughForMerge(regionInfo)) {
            LOG.trace((Object)("skipping merge region " + name + " because it is not large enough."));
            return true;
        }
        return false;
    }

    private List<NormalizationPlan> computeMergeNormalizationPlans(NormalizeContext ctx) {
        if (ctx.getTableRegions().size() < this.minRegionCount) {
            LOG.debug((Object)("Table " + ctx.getTableName() + " has " + ctx.getTableRegions().size() + " regions, required min number of regions for normalizer to run" + " is " + this.minRegionCount + ", not computing merge plans."));
            return Collections.emptyList();
        }
        double avgRegionSizeMb = ctx.getAverageRegionSizeMb();
        LOG.debug((Object)("Computing normalization plan for table " + ctx.getTableName() + ". average region size: " + avgRegionSizeMb + ", number of" + " regions: " + ctx.getTableRegions().size()));
        ArrayList<NormalizationPlan> plans = new ArrayList<NormalizationPlan>();
        for (int candidateIdx = 0; candidateIdx < ctx.getTableRegions().size() - 1; ++candidateIdx) {
            long nextSizeMb;
            long currentSizeMb;
            HRegionInfo current = ctx.getTableRegions().get(candidateIdx);
            HRegionInfo next = ctx.getTableRegions().get(candidateIdx + 1);
            if (this.skipForMerge(ctx.getRegionStates(), current) || this.skipForMerge(ctx.getRegionStates(), next) || !((double)((currentSizeMb = this.getRegionSizeMB(current)) + (nextSizeMb = this.getRegionSizeMB(next))) < avgRegionSizeMb)) continue;
            plans.add(new MergeNormalizationPlan(current, next));
            ++candidateIdx;
        }
        return plans;
    }

    private List<NormalizationPlan> computeSplitNormalizationPlans(NormalizeContext ctx) {
        double avgRegionSize = ctx.getAverageRegionSizeMb();
        TableName tableName = ctx.getTableName();
        LOG.debug((Object)("Table " + tableName + ", average region size: " + avgRegionSize));
        ArrayList<NormalizationPlan> plans = new ArrayList<NormalizationPlan>();
        for (HRegionInfo hri : ctx.getTableRegions()) {
            long regionSize;
            boolean regionIsOpen = ctx.getRegionStates().isRegionInState(hri, RegionState.State.OPEN);
            if (!regionIsOpen || !((double)(regionSize = this.getRegionSizeMB(hri)) > 2.0 * avgRegionSize)) continue;
            LOG.info((Object)("Table " + tableName + ", large region " + hri.getRegionNameAsString() + " has size " + regionSize + ", more than twice avg size " + avgRegionSize + ", splitting"));
            plans.add(new SplitNormalizationPlan(hri, null));
        }
        return plans;
    }

    private boolean isOldEnoughForMerge(HRegionInfo regionInfo) {
        Timestamp currentTime = new Timestamp(EnvironmentEdgeManager.currentTime());
        Timestamp regionCreateTime = new Timestamp(regionInfo.getRegionId());
        return new Timestamp(regionCreateTime.getTime() + TimeUnit.DAYS.toMillis(this.mergeMinRegionAge)).before(currentTime);
    }

    private boolean isLargeEnoughForMerge(HRegionInfo regionInfo) {
        return this.getRegionSizeMB(regionInfo) >= (long)this.mergeMinRegionSizeMb;
    }

    private class NormalizeContext {
        private final TableName tableName;
        private final RegionStates regionStates;
        private final List<HRegionInfo> tableRegions;
        private final double averageRegionSizeMb;

        public NormalizeContext(TableName tableName) {
            this.tableName = tableName;
            this.regionStates = SimpleRegionNormalizer.this.masterServices.getAssignmentManager().getRegionStates();
            this.tableRegions = this.regionStates.getRegionsOfTable(tableName);
            Collections.sort(this.tableRegions);
            this.averageRegionSizeMb = SimpleRegionNormalizer.this.getAverageRegionSizeMb(this.tableRegions);
        }

        public TableName getTableName() {
            return this.tableName;
        }

        public RegionStates getRegionStates() {
            return this.regionStates;
        }

        public List<HRegionInfo> getTableRegions() {
            return this.tableRegions;
        }

        public double getAverageRegionSizeMb() {
            return this.averageRegionSizeMb;
        }
    }
}

