/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.metadata;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.common.FileUtils;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.Warehouse;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.ql.metadata.CheckResult;
import org.apache.hadoop.hive.ql.metadata.Hive;
import org.apache.hadoop.hive.ql.metadata.HiveException;
import org.apache.hadoop.hive.ql.metadata.Partition;
import org.apache.hadoop.hive.ql.metadata.Table;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HiveMetaStoreChecker {
    public static final Logger LOG = LoggerFactory.getLogger(HiveMetaStoreChecker.class);
    private final Hive hive;
    private final HiveConf conf;

    public HiveMetaStoreChecker(Hive hive) {
        this.hive = hive;
        this.conf = hive.getConf();
    }

    public void checkMetastore(String dbName, String tableName, List<? extends Map<String, String>> partitions, CheckResult result) throws HiveException, IOException {
        if (dbName == null || "".equalsIgnoreCase(dbName)) {
            dbName = "default";
        }
        try {
            if (tableName == null || "".equals(tableName)) {
                List<String> tables = this.hive.getTablesForDb(dbName, ".*");
                for (String currentTableName : tables) {
                    this.checkTable(dbName, currentTableName, null, result);
                }
                this.findUnknownTables(dbName, tables, result);
            } else if (partitions == null || partitions.isEmpty()) {
                this.checkTable(dbName, tableName, null, result);
            } else {
                this.checkTable(dbName, tableName, partitions, result);
            }
            Collections.sort(result.getPartitionsNotInMs());
            Collections.sort(result.getPartitionsNotOnFs());
            Collections.sort(result.getTablesNotInMs());
            Collections.sort(result.getTablesNotOnFs());
        }
        catch (MetaException e) {
            throw new HiveException(e);
        }
        catch (TException e) {
            throw new HiveException(e);
        }
    }

    void findUnknownTables(String dbName, List<String> tables, CheckResult result) throws IOException, MetaException, TException, HiveException {
        HashSet<Path> dbPaths = new HashSet<Path>();
        HashSet<String> tableNames = new HashSet<String>(tables);
        for (String tableName : tables) {
            Table table = this.hive.getTable(dbName, tableName);
            String isExternal = table.getParameters().get("EXTERNAL");
            if (isExternal != null && "TRUE".equalsIgnoreCase(isExternal)) continue;
            dbPaths.add(table.getPath().getParent());
        }
        for (Path dbPath : dbPaths) {
            FileStatus[] statuses;
            FileSystem fs = dbPath.getFileSystem((Configuration)this.conf);
            for (FileStatus status : statuses = fs.listStatus(dbPath, FileUtils.HIDDEN_FILES_PATH_FILTER)) {
                if (!status.isDir() || tableNames.contains(status.getPath().getName())) continue;
                result.getTablesNotInMs().add(status.getPath().getName());
            }
        }
    }

    void checkTable(String dbName, String tableName, List<? extends Map<String, String>> partitions, CheckResult result) throws MetaException, IOException, HiveException {
        Table table = null;
        try {
            table = this.hive.getTable(dbName, tableName);
        }
        catch (HiveException e) {
            result.getTablesNotInMs().add(tableName);
            return;
        }
        List<Partition> parts = new ArrayList<Partition>();
        boolean findUnknownPartitions = true;
        if (table.isPartitioned()) {
            if (partitions == null || partitions.isEmpty()) {
                parts = this.hive.getPartitions(table);
            } else {
                findUnknownPartitions = false;
                for (Map<String, String> map : partitions) {
                    Partition part = this.hive.getPartition(table, map, false);
                    if (part == null) {
                        CheckResult.PartitionResult pr = new CheckResult.PartitionResult();
                        pr.setTableName(tableName);
                        pr.setPartitionName(Warehouse.makePartPath(map));
                        result.getPartitionsNotInMs().add(pr);
                        continue;
                    }
                    parts.add(part);
                }
            }
        }
        this.checkTable(table, parts, findUnknownPartitions, result);
    }

    void checkTable(Table table, List<Partition> parts, boolean findUnknownPartitions, CheckResult result) throws IOException, HiveException {
        Path tablePath = table.getPath();
        FileSystem fs = tablePath.getFileSystem((Configuration)this.conf);
        if (!fs.exists(tablePath)) {
            result.getTablesNotOnFs().add(table.getTableName());
            return;
        }
        HashSet<Path> partPaths = new HashSet<Path>();
        for (Partition partition : parts) {
            if (partition == null) continue;
            Path partPath = partition.getDataLocation();
            fs = partPath.getFileSystem((Configuration)this.conf);
            if (!fs.exists(partPath)) {
                CheckResult.PartitionResult pr = new CheckResult.PartitionResult();
                pr.setPartitionName(partition.getName());
                pr.setTableName(partition.getTable().getTableName());
                result.getPartitionsNotOnFs().add(pr);
            }
            for (int i = 0; i < partition.getSpec().size(); ++i) {
                partPaths.add(partPath.makeQualified(fs));
                partPath = partPath.getParent();
            }
        }
        if (findUnknownPartitions) {
            this.findUnknownPartitions(table, partPaths, result);
        }
    }

    void findUnknownPartitions(Table table, Set<Path> partPaths, CheckResult result) throws IOException, HiveException {
        Path tablePath = table.getPath();
        HashSet<Path> allPartDirs = new HashSet<Path>();
        this.getAllLeafDirs(tablePath, allPartDirs);
        allPartDirs.remove(tablePath);
        allPartDirs.removeAll(partPaths);
        for (Path partPath : allPartDirs) {
            FileSystem fs = partPath.getFileSystem((Configuration)this.conf);
            String partitionName = this.getPartitionName(fs.makeQualified(tablePath), partPath);
            if (partitionName == null) continue;
            CheckResult.PartitionResult pr = new CheckResult.PartitionResult();
            pr.setPartitionName(partitionName);
            pr.setTableName(table.getTableName());
            result.getPartitionsNotInMs().add(pr);
        }
    }

    private String getPartitionName(Path tablePath, Path partitionPath) {
        String result = null;
        for (Path currPath = partitionPath; currPath != null && !tablePath.equals((Object)currPath); currPath = currPath.getParent()) {
            result = result == null ? currPath.getName() : currPath.getName() + "/" + result;
        }
        return result;
    }

    private void getAllLeafDirs(Path basePath, Set<Path> allDirs) throws IOException, HiveException {
        ExecutorService pool;
        ConcurrentLinkedQueue<Path> basePaths = new ConcurrentLinkedQueue<Path>();
        basePaths.add(basePath);
        ConcurrentHashMap<Path, Object> dirSet = new ConcurrentHashMap<Path, Object>();
        ExecutorService executorService = pool = this.conf.getInt(HiveConf.ConfVars.HIVE_MOVE_FILES_THREAD_COUNT.varname, 25) > 0 ? Executors.newFixedThreadPool(this.conf.getInt(HiveConf.ConfVars.HIVE_MOVE_FILES_THREAD_COUNT.varname, 25), new ThreadFactoryBuilder().setDaemon(true).setNameFormat("MSCK-GetPaths-%d").build()) : null;
        if (pool == null) {
            LOG.debug("Not-using threaded version of MSCK-GetPaths");
        } else {
            LOG.debug("Using threaded version of MSCK-GetPaths with number of threads " + ((ThreadPoolExecutor)pool).getPoolSize());
        }
        this.getAllLeafDirs(pool, basePaths, dirSet, basePath.getFileSystem((Configuration)this.conf));
        pool.shutdown();
        allDirs.addAll(dirSet.keySet());
    }

    private void getAllLeafDirs(ExecutorService pool, ConcurrentLinkedQueue<Path> basePaths, final Map<Path, Object> allDirs, final FileSystem fs) throws IOException, HiveException {
        final ConcurrentLinkedQueue<Path> nextLevel = new ConcurrentLinkedQueue<Path>();
        if (null == pool) {
            for (Path path : basePaths) {
                FileStatus[] fileStatusArray = fs.listStatus(path, FileUtils.HIDDEN_FILES_PATH_FILTER);
                boolean directoryFound = false;
                for (FileStatus status : fileStatusArray) {
                    if (!status.isDir()) continue;
                    directoryFound = true;
                    nextLevel.add(status.getPath());
                }
                if (!directoryFound) {
                    allDirs.put(path, null);
                }
                if (nextLevel.isEmpty()) continue;
                this.getAllLeafDirs(pool, nextLevel, allDirs, fs);
            }
        } else {
            LinkedList<Future<Void>> futures = new LinkedList<Future<Void>>();
            for (final Path path : basePaths) {
                futures.add(pool.submit(new Callable<Void>(){

                    @Override
                    public Void call() throws Exception {
                        FileStatus[] statuses = fs.listStatus(path, FileUtils.HIDDEN_FILES_PATH_FILTER);
                        boolean directoryFound = false;
                        for (FileStatus status : statuses) {
                            if (!status.isDir()) continue;
                            directoryFound = true;
                            nextLevel.add(status.getPath());
                        }
                        if (!directoryFound) {
                            allDirs.put(path, null);
                        }
                        return null;
                    }
                }));
            }
            for (Future future : futures) {
                try {
                    future.get();
                }
                catch (Exception e) {
                    LOG.error(e.getMessage());
                    pool.shutdownNow();
                    throw new HiveException(e.getCause());
                }
            }
            if (!nextLevel.isEmpty()) {
                this.getAllLeafDirs(pool, nextLevel, allDirs, fs);
            }
        }
    }
}

