/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hive.ql.exec.vector.expressions.gen;

import org.apache.hadoop.hive.ql.exec.vector.expressions.VectorExpression;
import org.apache.hadoop.hive.ql.exec.vector.DecimalColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.VectorizedRowBatch;
import org.apache.hadoop.hive.ql.exec.vector.VectorExpressionDescriptor;
import org.apache.hadoop.hive.serde2.io.HiveDecimalWritable;
import org.apache.hadoop.hive.ql.metadata.HiveException;

/**
 * Generated from template FilterDecimalColumnCompareColumn.txt, which covers binary comparison
 * filter expressions between two columns. Output is not produced in a separate column.
 * The selected vector of the input {@link org.apache.hadoop.hive.ql.exec.vector.VectorizedRowBatch} is updated for in-place filtering.
 */
public class FilterDecimalColGreaterDecimalColumn extends VectorExpression {

  private static final long serialVersionUID = 1L;

  public FilterDecimalColGreaterDecimalColumn(int colNum1, int colNum2) {
    super(colNum1, colNum2, -1);
  }

  public FilterDecimalColGreaterDecimalColumn() {
    super();
  }

  @Override
  public void evaluate(VectorizedRowBatch batch) throws HiveException {

    // return immediately if batch is empty
    final int n = batch.size;
    if (n == 0) {
      return;
    }

    if (childExpressions != null) {
      super.evaluateChildren(batch);
    }

    DecimalColumnVector inputColVector1 = (DecimalColumnVector) batch.cols[inputColumnNum[0]];
    DecimalColumnVector inputColVector2 = (DecimalColumnVector) batch.cols[inputColumnNum[1]];
    int[] sel = batch.selected;
    boolean[] nullPos1 = inputColVector1.isNull;
    boolean[] nullPos2 = inputColVector2.isNull;

    HiveDecimalWritable[] vector1 = inputColVector1.vector;
    HiveDecimalWritable[] vector2 = inputColVector2.vector;

    // handle case where neither input has nulls
    if (inputColVector1.noNulls && inputColVector2.noNulls) {
      if (inputColVector1.isRepeating && inputColVector2.isRepeating) {

        /* Either all must remain selected or all will be eliminated.
         * Repeating property will not change.
         */
        if (!(vector1[0].compareTo(vector2[0]) > 0)) {
          batch.size = 0;
        }
      } else if (inputColVector1.isRepeating) {
        if (batch.selectedInUse) {
          int newSize = 0;
          for(int j = 0; j != n; j++) {
            int i = sel[j];
            if (vector1[0].compareTo(vector2[i]) > 0) {
              sel[newSize++] = i;
            }
          }
          batch.size = newSize;
        } else {
          int newSize = 0;
          for(int i = 0; i != n; i++) {
            if (vector1[0].compareTo(vector2[i]) > 0) {
              sel[newSize++] = i;
            }
          }
          if (newSize < batch.size) {
            batch.size = newSize;
            batch.selectedInUse = true;
          }
        }
      } else if (inputColVector2.isRepeating) {
        if (batch.selectedInUse) {
          int newSize = 0;
          for(int j = 0; j != n; j++) {
            int i = sel[j];
            if (vector1[i].compareTo(vector2[0]) > 0) {
              sel[newSize++] = i;
            }
          }
          batch.size = newSize;
        } else {
          int newSize = 0;
          for(int i = 0; i != n; i++) {
            if (vector1[i].compareTo(vector2[0]) > 0) {
              sel[newSize++] = i;
            }
          }
          if (newSize < batch.size) {
            batch.size = newSize;
            batch.selectedInUse = true;
          }
        }
      } else if (batch.selectedInUse) {
        int newSize = 0;
        for(int j = 0; j != n; j++) {
          int i = sel[j];
          if (vector1[i].compareTo(vector2[i]) > 0) {
            sel[newSize++] = i;
          }
        }
        batch.size = newSize;
      } else {
        int newSize = 0;
        for(int i = 0; i != n; i++) {
          if (vector1[i].compareTo(vector2[i]) > 0) {
            sel[newSize++] = i;
          }
        }
        if (newSize < batch.size) {
          batch.size = newSize;
          batch.selectedInUse = true;
        }
      }

    // handle case where only input 2 has nulls
    } else if (inputColVector1.noNulls) {
      if (inputColVector1.isRepeating && inputColVector2.isRepeating) {
        if (nullPos2[0] ||
            !(vector1[0].compareTo(vector2[0]) > 0)) {
          batch.size = 0;
        }
      } else if (inputColVector1.isRepeating) {

         // no need to check for nulls in input 1
         if (batch.selectedInUse) {
          int newSize = 0;
          for(int j = 0; j != n; j++) {
            int i = sel[j];
            if (!nullPos2[i]) {
              if (vector1[0].compareTo(vector2[i]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          batch.size = newSize;
        } else {
          int newSize = 0;
          for(int i = 0; i != n; i++) {
            if (!nullPos2[i]) {
              if (vector1[0].compareTo(vector2[i]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          if (newSize < batch.size) {
            batch.size = newSize;
            batch.selectedInUse = true;
          }
        }
      } else if (inputColVector2.isRepeating) {
        if (nullPos2[0]) {

          // no values will qualify because every comparison will be with NULL
          batch.size = 0;
          return;
        }
        if (batch.selectedInUse) {
          int newSize = 0;
          for(int j = 0; j != n; j++) {
            int i = sel[j];
            if (vector1[i].compareTo(vector2[0]) > 0) {
              sel[newSize++] = i;
            }
          }
          batch.size = newSize;
        } else {
          int newSize = 0;
          for(int i = 0; i != n; i++) {
            if (vector1[i].compareTo(vector2[0]) > 0) {
              sel[newSize++] = i;
            }
          }
          if (newSize < batch.size) {
            batch.size = newSize;
            batch.selectedInUse = true;
          }
        }
      } else { // neither input is repeating
        if (batch.selectedInUse) {
          int newSize = 0;
          for(int j = 0; j != n; j++) {
            int i = sel[j];
            if (!nullPos2[i]) {
              if (vector1[i].compareTo(vector2[i]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          batch.size = newSize;
        } else {
          int newSize = 0;
          for(int i = 0; i != n; i++) {
            if (!nullPos2[i]) {
              if (vector1[i].compareTo(vector2[i]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          if (newSize < batch.size) {
            batch.size = newSize;
            batch.selectedInUse = true;
          }
        }
      }

    // handle case where only input 1 has nulls
    } else if (inputColVector2.noNulls) {
      if (inputColVector1.isRepeating && inputColVector2.isRepeating) {
        if (nullPos1[0] ||
            !(vector1[0].compareTo(vector2[0]) > 0)) {
          batch.size = 0;
          return;
        }
      } else if (inputColVector1.isRepeating) {
        if (nullPos1[0]) {

          // if repeating value is null then every comparison will fail so nothing qualifies
          batch.size = 0;
          return;
        }
        if (batch.selectedInUse) {
          int newSize = 0;
          for(int j = 0; j != n; j++) {
            int i = sel[j];
            if (vector1[0].compareTo(vector2[i]) > 0) {
              sel[newSize++] = i;
            }
          }
          batch.size = newSize;
        } else {
          int newSize = 0;
          for(int i = 0; i != n; i++) {
            if (vector1[0].compareTo(vector2[i]) > 0) {
              sel[newSize++] = i;
            }
          }
          if (newSize < batch.size) {
            batch.size = newSize;
            batch.selectedInUse = true;
          }
        }
      } else if (inputColVector2.isRepeating) {
         if (batch.selectedInUse) {
          int newSize = 0;
          for(int j = 0; j != n; j++) {
            int i = sel[j];
            if (!nullPos1[i]) {
              if (vector1[i].compareTo(vector2[0]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          batch.size = newSize;
        } else {
          int newSize = 0;
          for(int i = 0; i != n; i++) {
            if (!nullPos1[i]) {
              if (vector1[i].compareTo(vector2[0]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          if (newSize < batch.size) {
            batch.size = newSize;
            batch.selectedInUse = true;
          }
        }
      } else { // neither input is repeating
         if (batch.selectedInUse) {
          int newSize = 0;
          for(int j = 0; j != n; j++) {
            int i = sel[j];
            if (!nullPos1[i]) {
              if (vector1[i].compareTo(vector2[i]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          batch.size = newSize;
        } else {
          int newSize = 0;
          for(int i = 0; i != n; i++) {
            if (!nullPos1[i]) {
              if (vector1[i].compareTo(vector2[i]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          if (newSize < batch.size) {
            batch.size = newSize;
            batch.selectedInUse = true;
          }
        }
      }

    // handle case where both inputs have nulls
    } else {
      if (inputColVector1.isRepeating && inputColVector2.isRepeating) {
        if (nullPos1[0] || nullPos2[0] ||
            !(vector1[0].compareTo(vector2[0]) > 0)) {
          batch.size = 0;
        }
      } else if (inputColVector1.isRepeating) {
         if (nullPos1[0]) {
           batch.size = 0;
           return;
         }
         if (batch.selectedInUse) {
          int newSize = 0;
          for(int j = 0; j != n; j++) {
            int i = sel[j];
            if (!nullPos2[i]) {
              if (vector1[0].compareTo(vector2[i]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          batch.size = newSize;
        } else {
          int newSize = 0;
          for(int i = 0; i != n; i++) {
            if (!nullPos2[i]) {
              if (vector1[0].compareTo(vector2[i]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          if (newSize < batch.size) {
            batch.size = newSize;
            batch.selectedInUse = true;
          }
        }
      } else if (inputColVector2.isRepeating) {
        if (nullPos2[0]) {
          batch.size = 0;
          return;
        }
        if (batch.selectedInUse) {
          int newSize = 0;
          for(int j = 0; j != n; j++) {
            int i = sel[j];
            if (!nullPos1[i]) {
              if (vector1[i].compareTo(vector2[0]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          batch.size = newSize;
        } else {
          int newSize = 0;
          for(int i = 0; i != n; i++) {
            if (!nullPos1[i]) {
              if (vector1[i].compareTo(vector2[0]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          if (newSize < batch.size) {
            batch.size = newSize;
            batch.selectedInUse = true;
          }
        }
      } else { // neither input is repeating
         if (batch.selectedInUse) {
          int newSize = 0;
          for(int j = 0; j != n; j++) {
            int i = sel[j];
            if (!nullPos1[i] && !nullPos2[i]) {
              if (vector1[i].compareTo(vector2[i]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          batch.size = newSize;
        } else {
          int newSize = 0;
          for(int i = 0; i != n; i++) {
            if (!nullPos1[i] && !nullPos2[i]) {
              if (vector1[i].compareTo(vector2[i]) > 0) {
                sel[newSize++] = i;
              }
            }
          }
          if (newSize < batch.size) {
            batch.size = newSize;
            batch.selectedInUse = true;
          }
        }
      }
    }
  }

  @Override
  public String vectorExpressionParameters() {
    return getColumnParamString(0, inputColumnNum[0]) + ", " + getColumnParamString(1, inputColumnNum[1]);
  }

  @Override
  public VectorExpressionDescriptor.Descriptor getDescriptor() {
    return (new VectorExpressionDescriptor.Builder())
        .setMode(
            VectorExpressionDescriptor.Mode.FILTER)
        .setNumArguments(2)
        .setArgumentTypes(
            VectorExpressionDescriptor.ArgumentType.getType("decimal"),
            VectorExpressionDescriptor.ArgumentType.getType("decimal"))
        .setInputExpressionTypes(
            VectorExpressionDescriptor.InputExpressionType.COLUMN,
            VectorExpressionDescriptor.InputExpressionType.COLUMN).build();
  }
}
