/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.llap.cache;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.common.io.Allocator;
import org.apache.hadoop.hive.common.io.encoded.MemoryBuffer;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.llap.cache.BuddyAllocatorMXBean;
import org.apache.hadoop.hive.llap.cache.EvictionAwareAllocator;
import org.apache.hadoop.hive.llap.cache.LlapDataBuffer;
import org.apache.hadoop.hive.llap.cache.MemoryManager;
import org.apache.hadoop.hive.llap.io.api.impl.LlapIoImpl;
import org.apache.hadoop.hive.llap.metrics.LlapDaemonCacheMetrics;

public final class BuddyAllocator
implements EvictionAwareAllocator,
BuddyAllocatorMXBean {
    private final Arena[] arenas;
    private final AtomicInteger allocatedArenas = new AtomicInteger(0);
    private final MemoryManager memoryManager;
    private final int minAllocLog2;
    private final int maxAllocLog2;
    private final int arenaSizeLog2;
    private final int maxArenas;
    private final int minAllocation;
    private final int maxAllocation;
    private final int arenaSize;
    private final long maxSize;
    private final boolean isDirect;
    private final boolean isMapped;
    private final Path cacheDir;
    private final LlapDaemonCacheMetrics metrics;
    private static final int MAX_ARENA_SIZE = 0x40000000;
    private static final int MIN_TOTAL_MEMORY_SIZE = 0x4000000;
    private static final FileAttribute<Set<PosixFilePermission>> RWX = PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString("rwx------"));
    private static final FileAttribute<Set<PosixFilePermission>> RW_ = PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString("rw-------"));

    public BuddyAllocator(Configuration conf, MemoryManager mm, LlapDaemonCacheMetrics metrics) {
        this(HiveConf.getBoolVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_DIRECT), HiveConf.getBoolVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_MAPPED), (int)HiveConf.getSizeVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_MIN_ALLOC), (int)HiveConf.getSizeVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_MAX_ALLOC), HiveConf.getIntVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_ARENA_COUNT), BuddyAllocator.getMaxTotalMemorySize(conf), HiveConf.getVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_MAPPED_PATH), mm, metrics);
    }

    private static long getMaxTotalMemorySize(Configuration conf) {
        long maxSize = HiveConf.getSizeVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_IO_MEMORY_MAX_SIZE);
        if (maxSize > 0x4000000L || HiveConf.getBoolVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_IN_TEST)) {
            return maxSize;
        }
        throw new RuntimeException("Allocator space is too small for reasonable operation; " + HiveConf.ConfVars.LLAP_IO_MEMORY_MAX_SIZE.varname + "=" + maxSize + ", but at least " + 0x4000000 + " is required. If you cannot spare any memory, you can disable LLAP IO entirely via " + HiveConf.ConfVars.LLAP_IO_ENABLED.varname + "; or set " + HiveConf.ConfVars.LLAP_IO_MEMORY_MODE.varname + " to 'none'");
    }

    @VisibleForTesting
    public BuddyAllocator(boolean isDirectVal, int minAllocVal, int maxAllocVal, int arenaCount, long maxSizeVal, MemoryManager memoryManager, LlapDaemonCacheMetrics metrics) {
        this(isDirectVal, false, minAllocVal, maxAllocVal, arenaCount, maxSizeVal, null, memoryManager, metrics);
    }

    @VisibleForTesting
    public BuddyAllocator(boolean isDirectVal, boolean isMappedVal, int minAllocVal, int maxAllocVal, int arenaCount, long maxSizeVal, String mapPath, MemoryManager memoryManager, LlapDaemonCacheMetrics metrics) {
        this.isDirect = isDirectVal;
        this.isMapped = isMappedVal;
        this.minAllocation = minAllocVal;
        this.maxAllocation = maxAllocVal;
        if (this.isMapped) {
            try {
                this.cacheDir = Files.createTempDirectory(FileSystems.getDefault().getPath(mapPath, new String[0]), "llap-", RWX);
            }
            catch (IOException ioe) {
                throw new AssertionError("Configured mmap directory should be writable", ioe);
            }
        } else {
            this.cacheDir = null;
        }
        long arenaSizeVal = arenaCount == 0 ? 0x40000000L : maxSizeVal / (long)arenaCount;
        arenaSizeVal = Math.max((long)this.maxAllocation, Math.min(arenaSizeVal, 0x40000000L));
        if (LlapIoImpl.LOG.isInfoEnabled()) {
            LlapIoImpl.LOG.info("Buddy allocator with " + (this.isDirect ? "direct" : "byte") + " buffers;" + (this.isMapped ? " memory mapped off " + this.cacheDir.toString() + "; " : "") + "allocation sizes " + this.minAllocation + " - " + this.maxAllocation + ", arena size " + arenaSizeVal + ". total size " + maxSizeVal);
        }
        String minName = HiveConf.ConfVars.LLAP_ALLOCATOR_MIN_ALLOC.varname;
        String maxName = HiveConf.ConfVars.LLAP_ALLOCATOR_MAX_ALLOC.varname;
        if (this.minAllocation < 8) {
            throw new RuntimeException(minName + " must be at least 8 bytes: " + this.minAllocation);
        }
        if (maxSizeVal < (long)this.maxAllocation || this.maxAllocation < this.minAllocation) {
            throw new RuntimeException("Inconsistent sizes; expecting " + minName + " <= " + maxName + " <= " + HiveConf.ConfVars.LLAP_IO_MEMORY_MAX_SIZE.varname + "; configured with min=" + this.minAllocation + ", max=" + this.maxAllocation + " and total=" + maxSizeVal);
        }
        if (Integer.bitCount(this.minAllocation) != 1 || Integer.bitCount(this.maxAllocation) != 1) {
            throw new RuntimeException("Allocation sizes must be powers of two; configured with " + minName + "=" + this.minAllocation + ", " + maxName + "=" + this.maxAllocation);
        }
        if (arenaSizeVal % (long)this.maxAllocation > 0L) {
            long oldArenaSize = arenaSizeVal;
            arenaSizeVal = arenaSizeVal / (long)this.maxAllocation * (long)this.maxAllocation;
            LlapIoImpl.LOG.warn("Rounding arena size to " + arenaSizeVal + " from " + oldArenaSize + " to be divisible by allocation size " + this.maxAllocation);
        }
        this.arenaSize = (int)arenaSizeVal;
        if (maxSizeVal % (long)this.arenaSize > 0L) {
            long oldMaxSize = maxSizeVal;
            maxSizeVal = maxSizeVal / (long)this.arenaSize * (long)this.arenaSize;
            LlapIoImpl.LOG.warn("Rounding cache size to " + maxSizeVal + " from " + oldMaxSize + " to be divisible by arena size " + this.arenaSize);
        }
        if (maxSizeVal / (long)this.arenaSize > Integer.MAX_VALUE) {
            throw new RuntimeException("Too many arenas needed to allocate the cache: " + this.arenaSize + ", " + maxSizeVal);
        }
        this.maxSize = maxSizeVal;
        memoryManager.updateMaxSize(this.maxSize);
        this.minAllocLog2 = 31 - Integer.numberOfLeadingZeros(this.minAllocation);
        this.maxAllocLog2 = 31 - Integer.numberOfLeadingZeros(this.maxAllocation);
        this.arenaSizeLog2 = 63 - Long.numberOfLeadingZeros(this.arenaSize);
        this.maxArenas = (int)(this.maxSize / (long)this.arenaSize);
        this.arenas = new Arena[this.maxArenas];
        for (int i = 0; i < this.maxArenas; ++i) {
            this.arenas[i] = new Arena();
        }
        this.arenas[0].init();
        this.allocatedArenas.set(1);
        this.memoryManager = memoryManager;
        this.metrics = metrics;
        metrics.incrAllocatedArena();
    }

    public void allocateMultiple(MemoryBuffer[] dest, int size) throws Allocator.AllocatorOutOfMemoryException {
        int startArenaIx;
        assert (size > 0) : "size is " + size;
        if (size > this.maxAllocation) {
            throw new RuntimeException("Trying to allocate " + size + "; max is " + this.maxAllocation);
        }
        int freeListIx = 31 - Integer.numberOfLeadingZeros(size);
        if (size != 1 << freeListIx) {
            ++freeListIx;
        }
        freeListIx = Math.max(freeListIx - this.minAllocLog2, 0);
        int allocLog2 = freeListIx + this.minAllocLog2;
        int allocationSize = 1 << allocLog2;
        this.memoryManager.reserveMemory(dest.length << allocLog2, true);
        int destAllocIx = 0;
        for (int i = 0; i < dest.length; ++i) {
            if (dest[i] != null) continue;
            dest[i] = this.createUnallocated();
        }
        int arenaCount = this.allocatedArenas.get();
        if (arenaCount < 0) {
            arenaCount = -arenaCount - 1;
        }
        long threadId = arenaCount > 1 ? Thread.currentThread().getId() : 0L;
        int index = startArenaIx = (int)(threadId % (long)arenaCount);
        do {
            int newDestIx;
            if ((newDestIx = this.arenas[index].allocateFast(index, freeListIx, dest, destAllocIx, allocationSize)) == dest.length) {
                return;
            }
            assert (newDestIx != -1);
            destAllocIx = newDestIx;
            if (++index != arenaCount) continue;
            index = 0;
        } while (index != startArenaIx);
        for (int attempt = 0; attempt < 5; ++attempt) {
            int startArenaIx2;
            int arenaIx = startArenaIx2 = (int)((threadId + (long)attempt) % (long)arenaCount);
            do {
                int newDestIx;
                if ((newDestIx = this.arenas[arenaIx].allocateWithSplit(arenaIx, freeListIx, dest, destAllocIx, allocationSize)) == dest.length) {
                    return;
                }
                assert (newDestIx != -1);
                destAllocIx = newDestIx;
                if (++arenaIx != arenaCount) continue;
                arenaIx = 0;
            } while (arenaIx != startArenaIx2);
            if (attempt == 0) {
                for (int arenaIx2 = arenaCount; arenaIx2 < this.arenas.length; ++arenaIx2) {
                    if ((destAllocIx = this.arenas[arenaIx2].allocateWithExpand(arenaIx2, freeListIx, dest, destAllocIx, allocationSize)) != dest.length) continue;
                    return;
                }
            }
            this.memoryManager.forceReservedMemory(allocationSize, dest.length - destAllocIx);
            LlapIoImpl.LOG.warn("Failed to allocate despite reserved memory; will retry " + attempt);
        }
        String msg = "Failed to allocate " + size + "; at " + destAllocIx + " out of " + dest.length;
        LlapIoImpl.LOG.error(msg + "\nALLOCATOR STATE:\n" + this.debugDump() + "\nPARENT STATE:\n" + this.memoryManager.debugDumpForOom());
        throw new Allocator.AllocatorOutOfMemoryException(msg);
    }

    public void deallocate(MemoryBuffer buffer) {
        this.deallocateInternal(buffer, true);
    }

    @Override
    public void deallocateEvicted(MemoryBuffer buffer) {
        this.deallocateInternal(buffer, false);
    }

    private void deallocateInternal(MemoryBuffer buffer, boolean doReleaseMemory) {
        LlapDataBuffer buf = (LlapDataBuffer)buffer;
        long memUsage = buf.getMemoryUsage();
        this.arenas[buf.arenaIndex].deallocate(buf);
        if (doReleaseMemory) {
            this.memoryManager.releaseMemory(memUsage);
        }
    }

    public boolean isDirectAlloc() {
        return this.isDirect;
    }

    public String debugDump() {
        StringBuilder result = new StringBuilder("NOTE: with multiple threads the dump is not guaranteed to be consistent");
        for (Arena arena : this.arenas) {
            arena.debugDump(result);
        }
        return result.toString();
    }

    @Override
    public boolean getIsDirect() {
        return this.isDirect;
    }

    @Override
    public int getMinAllocation() {
        return this.minAllocation;
    }

    @Override
    public int getMaxAllocation() {
        return this.maxAllocation;
    }

    @Override
    public int getArenaSize() {
        return this.arenaSize;
    }

    @Override
    public long getMaxCacheSize() {
        return this.maxSize;
    }

    private ByteBuffer preallocate(int arenaSize) {
        if (this.isMapped) {
            Preconditions.checkArgument((boolean)this.isDirect, (Object)"All memory mapped allocations have to be direct buffers");
            try {
                File rf = File.createTempFile("arena-", ".cache", this.cacheDir.toFile());
                RandomAccessFile rwf = new RandomAccessFile(rf, "rw");
                rwf.setLength(arenaSize);
                MappedByteBuffer rwbuf = rwf.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, arenaSize);
                rwf.close();
                rf.delete();
                return rwbuf;
            }
            catch (IOException ioe) {
                LlapIoImpl.LOG.warn("Failed trying to allocate memory mapped arena", (Throwable)ioe);
                throw new OutOfMemoryError("Failed trying to allocate memory mapped arena: " + ioe.getMessage());
            }
        }
        return this.isDirect ? ByteBuffer.allocateDirect(arenaSize) : ByteBuffer.allocate(arenaSize);
    }

    public MemoryBuffer createUnallocated() {
        return new LlapDataBuffer();
    }

    private static class FreeList {
        ReentrantLock lock = new ReentrantLock(false);
        int listHead = -1;

        private FreeList() {
        }
    }

    private class Arena {
        private ByteBuffer data;
        private byte[] headers;
        private FreeList[] freeLists;

        private Arena() {
        }

        void init() {
            try {
                this.data = BuddyAllocator.this.preallocate(BuddyAllocator.this.arenaSize);
            }
            catch (OutOfMemoryError oom) {
                throw new OutOfMemoryError("Cannot allocate " + BuddyAllocator.this.arenaSize + " bytes: " + oom.getMessage() + "; make sure your xmx and process size are set correctly.");
            }
            int maxMinAllocs = 1 << BuddyAllocator.this.arenaSizeLog2 - BuddyAllocator.this.minAllocLog2;
            this.headers = new byte[maxMinAllocs];
            int allocLog2Diff = BuddyAllocator.this.maxAllocLog2 - BuddyAllocator.this.minAllocLog2;
            int freeListCount = allocLog2Diff + 1;
            this.freeLists = new FreeList[freeListCount];
            for (int i = 0; i < freeListCount; ++i) {
                this.freeLists[i] = new FreeList();
            }
            int maxMaxAllocs = 1 << BuddyAllocator.this.arenaSizeLog2 - BuddyAllocator.this.maxAllocLog2;
            int headerIndex = 0;
            int headerStep = 1 << allocLog2Diff;
            this.freeLists[allocLog2Diff].listHead = 0;
            int i = 0;
            int offset = 0;
            while (i < maxMaxAllocs) {
                this.headers[headerIndex] = this.makeHeader(allocLog2Diff, false);
                this.data.putInt(offset, i == 0 ? -1 : headerIndex - headerStep);
                this.data.putInt(offset + 4, i == maxMaxAllocs - 1 ? -1 : headerIndex + headerStep);
                headerIndex += headerStep;
                ++i;
                offset += BuddyAllocator.this.maxAllocation;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void debugDump(StringBuilder result) {
            result.append("\nArena: ");
            if (this.data == null) {
                result.append(" not allocated");
                return;
            }
            byte[] headers = new byte[this.headers.length];
            System.arraycopy(this.headers, 0, headers, 0, headers.length);
            int allocSize = BuddyAllocator.this.minAllocation;
            int i = 0;
            while (i < this.freeLists.length) {
                result.append("\n  free list for size " + allocSize + ": ");
                FreeList freeList = this.freeLists[i];
                freeList.lock.lock();
                try {
                    int nextHeaderIx = freeList.listHead;
                    while (nextHeaderIx >= 0) {
                        result.append(nextHeaderIx + ", ");
                        nextHeaderIx = this.getNextFreeListItem(this.offsetFromHeaderIndex(nextHeaderIx));
                    }
                }
                finally {
                    freeList.lock.unlock();
                }
                ++i;
                allocSize <<= 1;
            }
            for (i = 0; i < headers.length; ++i) {
                byte header = headers[i];
                if (header == 0) continue;
                int freeListIx = this.freeListFromHeader(header);
                int offset = this.offsetFromHeaderIndex(i);
                boolean isFree = (header & 1) == 0;
                result.append("\n  block " + i + " at " + offset + ": size " + (1 << freeListIx + BuddyAllocator.this.minAllocLog2) + ", " + (isFree ? "free" : "allocated"));
            }
        }

        private int freeListFromHeader(byte header) {
            return (header >> 1) - 1;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private int allocateFast(int arenaIx, int freeListIx, MemoryBuffer[] dest, int ix, int size) {
            if (this.data == null) {
                return -1;
            }
            FreeList freeList = this.freeLists[freeListIx];
            if (!freeList.lock.tryLock()) {
                return ix;
            }
            try {
                int n = this.allocateFromFreeListUnderLock(arenaIx, freeList, freeListIx, dest, ix, size);
                return n;
            }
            finally {
                freeList.lock.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private int allocateWithSplit(int arenaIx, int freeListIx, MemoryBuffer[] dest, int ix, int allocationSize) {
            if (this.data == null) {
                return -1;
            }
            FreeList freeList = this.freeLists[freeListIx];
            int remaining = -1;
            freeList.lock.lock();
            try {
                ix = this.allocateFromFreeListUnderLock(arenaIx, freeList, freeListIx, dest, ix, allocationSize);
                remaining = dest.length - ix;
                if (remaining == 0) {
                    int n = ix;
                    return n;
                }
            }
            finally {
                freeList.lock.unlock();
            }
            byte headerData = this.makeHeader(freeListIx, true);
            int headerStep = 1 << freeListIx;
            for (int splitListIx = freeListIx + 1; remaining > 0 && splitListIx < this.freeLists.length; ++splitListIx) {
                int splitWaysLog2 = splitListIx - freeListIx;
                assert (splitWaysLog2 > 0);
                int splitWays = 1 << splitWaysLog2;
                int lastSplitBlocksRemaining = -1;
                int lastSplitNextHeader = -1;
                FreeList splitList = this.freeLists[splitListIx];
                splitList.lock.lock();
                try {
                    int headerIx = splitList.listHead;
                    while (headerIx >= 0 && remaining > 0) {
                        int origOffset;
                        int offset = origOffset = this.offsetFromHeaderIndex(headerIx);
                        int toTake = Math.min(splitWays, remaining);
                        remaining -= toTake;
                        lastSplitBlocksRemaining = splitWays - toTake;
                        while (toTake > 0) {
                            this.headers[headerIx] = headerData;
                            ((LlapDataBuffer)dest[ix]).initialize(arenaIx, this.data, offset, allocationSize);
                            ++ix;
                            --toTake;
                            headerIx += headerStep;
                            offset += allocationSize;
                        }
                        lastSplitNextHeader = headerIx;
                        headerIx = this.getNextFreeListItem(origOffset);
                    }
                    this.replaceListHeadUnderLock(splitList, headerIx);
                }
                finally {
                    splitList.lock.unlock();
                }
                if (remaining != 0) continue;
                int newListIndex = freeListIx;
                while (lastSplitBlocksRemaining > 0) {
                    if ((lastSplitBlocksRemaining & 1) == 1) {
                        FreeList newFreeList = this.freeLists[newListIndex];
                        newFreeList.lock.lock();
                        this.headers[lastSplitNextHeader] = this.makeHeader(newListIndex, false);
                        try {
                            this.addBlockToFreeListUnderLock(newFreeList, lastSplitNextHeader);
                        }
                        finally {
                            newFreeList.lock.unlock();
                        }
                        lastSplitNextHeader += 1 << newListIndex;
                    }
                    lastSplitBlocksRemaining >>>= 1;
                    ++newListIndex;
                }
            }
            return ix;
        }

        private void replaceListHeadUnderLock(FreeList freeList, int headerIx) {
            if (headerIx == freeList.listHead) {
                return;
            }
            if (headerIx >= 0) {
                int newHeadOffset = this.offsetFromHeaderIndex(headerIx);
                this.data.putInt(newHeadOffset, -1);
            }
            freeList.listHead = headerIx;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private int allocateWithExpand(int arenaIx, int freeListIx, MemoryBuffer[] dest, int ix, int size) {
            int arenaCount;
            while (true) {
                int allocArenaCount = arenaCount = BuddyAllocator.this.allocatedArenas.get();
                if (arenaCount < 0) {
                    allocArenaCount = -arenaCount - 1;
                }
                if (allocArenaCount > arenaIx) {
                    return this.allocateWithSplit(arenaIx, freeListIx, dest, ix, size);
                }
                if (arenaIx + 1 == -arenaCount) {
                    try {
                        Arena arena = this;
                        synchronized (arena) {
                            this.wait(100L);
                        }
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                    continue;
                }
                assert (arenaCount == arenaIx) : "Arena count " + arenaCount + " but " + arenaIx + " is not being allocated";
                if (BuddyAllocator.this.allocatedArenas.compareAndSet(arenaCount, -arenaCount - 1)) break;
            }
            assert (this.data == null);
            this.init();
            boolean isCommited = BuddyAllocator.this.allocatedArenas.compareAndSet(-arenaCount - 1, arenaCount + 1);
            assert (isCommited);
            Arena arena = this;
            synchronized (arena) {
                this.notifyAll();
            }
            BuddyAllocator.this.metrics.incrAllocatedArena();
            return this.allocateWithSplit(arenaIx, freeListIx, dest, ix, size);
        }

        public int offsetFromHeaderIndex(int lastSplitNextHeader) {
            return lastSplitNextHeader << BuddyAllocator.this.minAllocLog2;
        }

        public int allocateFromFreeListUnderLock(int arenaIx, FreeList freeList, int freeListIx, MemoryBuffer[] dest, int ix, int size) {
            int current = freeList.listHead;
            while (current >= 0 && ix < dest.length) {
                int offset = this.offsetFromHeaderIndex(current);
                this.headers[current] = this.makeHeader(freeListIx, true);
                current = this.getNextFreeListItem(offset);
                ((LlapDataBuffer)dest[ix]).initialize(arenaIx, this.data, offset, size);
                ++ix;
            }
            this.replaceListHeadUnderLock(freeList, current);
            return ix;
        }

        private int getPrevFreeListItem(int offset) {
            return this.data.getInt(offset);
        }

        private int getNextFreeListItem(int offset) {
            return this.data.getInt(offset + 4);
        }

        private byte makeHeader(int freeListIx, boolean isInUse) {
            return (byte)(freeListIx + 1 << 1 | (isInUse ? 1 : 0));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void deallocate(LlapDataBuffer buffer) {
            assert (this.data != null);
            int headerIx = buffer.byteBuffer.position() >>> BuddyAllocator.this.minAllocLog2;
            int freeListIx = this.freeListFromHeader(this.headers[headerIx]);
            assert (freeListIx == 31 - Integer.numberOfLeadingZeros(buffer.allocSize) - BuddyAllocator.this.minAllocLog2) : buffer.allocSize + " " + freeListIx;
            while (true) {
                FreeList freeList = this.freeLists[freeListIx];
                int bHeaderIx = headerIx ^ 1 << freeListIx;
                freeList.lock.lock();
                try {
                    if (freeListIx == this.freeLists.length - 1 || this.headers[bHeaderIx] != this.makeHeader(freeListIx, false)) {
                        this.addBlockToFreeListUnderLock(freeList, headerIx);
                        this.headers[headerIx] = this.makeHeader(freeListIx, false);
                        break;
                    }
                    this.removeBlockFromFreeList(freeList, bHeaderIx);
                    this.headers[headerIx] = 0;
                    this.headers[bHeaderIx] = 0;
                }
                finally {
                    freeList.lock.unlock();
                }
                ++freeListIx;
                headerIx = Math.min(headerIx, bHeaderIx);
            }
        }

        private void addBlockToFreeListUnderLock(FreeList freeList, int headerIx) {
            if (freeList.listHead >= 0) {
                int oldHeadOffset = this.offsetFromHeaderIndex(freeList.listHead);
                assert (this.getPrevFreeListItem(oldHeadOffset) == -1);
                this.data.putInt(oldHeadOffset, headerIx);
            }
            int offset = this.offsetFromHeaderIndex(headerIx);
            this.data.putInt(offset, -1);
            this.data.putInt(offset + 4, freeList.listHead);
            freeList.listHead = headerIx;
        }

        private void removeBlockFromFreeList(FreeList freeList, int headerIx) {
            int bOffset = this.offsetFromHeaderIndex(headerIx);
            int bpHeaderIx = this.getPrevFreeListItem(bOffset);
            int bnHeaderIx = this.getNextFreeListItem(bOffset);
            if (freeList.listHead == headerIx) {
                assert (bpHeaderIx == -1);
                freeList.listHead = bnHeaderIx;
            }
            if (bpHeaderIx != -1) {
                this.data.putInt(this.offsetFromHeaderIndex(bpHeaderIx) + 4, bnHeaderIx);
            }
            if (bnHeaderIx != -1) {
                this.data.putInt(this.offsetFromHeaderIndex(bnHeaderIx), bpHeaderIx);
            }
        }
    }
}

