/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hive.orc.impl;

import com.google.common.annotations.VisibleForTesting;
import com.google.protobuf.CodedOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hive.orc.CompressionCodec;
import org.apache.hive.orc.CompressionKind;
import org.apache.hive.orc.OrcFile;
import org.apache.hive.orc.OrcProto;
import org.apache.hive.orc.impl.OutStream;
import org.apache.hive.orc.impl.PhysicalWriter;
import org.apache.hive.orc.impl.SnappyCodec;
import org.apache.hive.orc.impl.StreamName;
import org.apache.hive.orc.impl.WriterImpl;
import org.apache.hive.orc.impl.ZlibCodec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PhysicalFsWriter
implements PhysicalWriter {
    private static final Logger LOG = LoggerFactory.getLogger(PhysicalFsWriter.class);
    private static final int HDFS_BUFFER_SIZE = 262144;
    private FSDataOutputStream rawWriter = null;
    private OutStream writer = null;
    private CodedOutputStream protobufWriter = null;
    private final FileSystem fs;
    private final Path path;
    private final long blockSize;
    private final int bufferSize;
    private final CompressionCodec codec;
    private final double paddingTolerance;
    private final long defaultStripeSize;
    private final CompressionKind compress;
    private final boolean addBlockPadding;
    private final OrcFile.CompressionStrategy compressionStrategy;
    private final Map<StreamName, BufferedStream> streams = new TreeMap<StreamName, BufferedStream>();
    private long adjustedStripeSize;
    private long headerLength;
    private long stripeStart;
    private int metadataLength;
    private int footerLength;

    public PhysicalFsWriter(FileSystem fs, Path path, int numColumns, OrcFile.WriterOptions opts) {
        this.fs = fs;
        this.path = path;
        this.defaultStripeSize = this.adjustedStripeSize = opts.getStripeSize();
        this.addBlockPadding = opts.getBlockPadding();
        this.bufferSize = opts.isEnforceBufferSize() ? opts.getBufferSize() : PhysicalFsWriter.getEstimatedBufferSize(this.defaultStripeSize, numColumns, opts.getBufferSize());
        this.compress = opts.getCompress();
        this.compressionStrategy = opts.getCompressionStrategy();
        this.codec = PhysicalFsWriter.createCodec(this.compress);
        this.paddingTolerance = opts.getPaddingTolerance();
        this.blockSize = opts.getBlockSize();
        LOG.info("ORC writer created for path: {} with stripeSize: {} blockSize: {} compression: {} bufferSize: {}", new Object[]{path, this.defaultStripeSize, this.blockSize, this.compress, this.bufferSize});
    }

    @Override
    public void initialize() throws IOException {
        if (this.rawWriter != null) {
            return;
        }
        this.rawWriter = this.fs.create(this.path, false, 262144, this.fs.getDefaultReplication(this.path), this.blockSize);
        this.rawWriter.writeBytes("ORC");
        this.headerLength = this.rawWriter.getPos();
        this.writer = new OutStream("metadata", this.bufferSize, this.codec, new DirectStream(this.rawWriter));
        this.protobufWriter = CodedOutputStream.newInstance((OutputStream)this.writer);
    }

    private void padStripe(long indexSize, long dataSize, int footerSize) throws IOException {
        this.stripeStart = this.rawWriter.getPos();
        long currentStripeSize = indexSize + dataSize + (long)footerSize;
        long available = this.blockSize - this.stripeStart % this.blockSize;
        long overflow = currentStripeSize - this.adjustedStripeSize;
        float availRatio = (float)available / (float)this.defaultStripeSize;
        if (availRatio > 0.0f && availRatio < 1.0f && (double)availRatio > this.paddingTolerance) {
            double correction = overflow > 0L ? (double)overflow / (double)this.adjustedStripeSize : 0.0;
            correction = correction > this.paddingTolerance ? this.paddingTolerance : correction;
            this.adjustedStripeSize = (long)((1.0 - correction) * (double)(availRatio * (float)this.defaultStripeSize));
        } else if ((double)availRatio >= 1.0) {
            this.adjustedStripeSize = this.defaultStripeSize;
        }
        if ((double)availRatio < this.paddingTolerance && this.addBlockPadding) {
            int writeLen;
            long padding;
            byte[] pad = new byte[(int)Math.min(262144L, padding)];
            LOG.info(String.format("Padding ORC by %d bytes (<=  %.2f * %d)", padding, Float.valueOf(availRatio), this.defaultStripeSize));
            this.stripeStart += padding;
            for (padding = this.blockSize - this.stripeStart % this.blockSize; padding > 0L; padding -= (long)writeLen) {
                writeLen = (int)Math.min(padding, (long)pad.length);
                this.rawWriter.write(pad, 0, writeLen);
            }
            this.adjustedStripeSize = this.defaultStripeSize;
        } else if (currentStripeSize < this.blockSize && this.stripeStart % this.blockSize + currentStripeSize > this.blockSize) {
            this.adjustedStripeSize = this.defaultStripeSize;
        }
    }

    @Override
    public long getPhysicalStripeSize() {
        return this.adjustedStripeSize;
    }

    @Override
    public boolean isCompressed() {
        return this.codec != null;
    }

    public static CompressionCodec createCodec(CompressionKind kind) {
        switch (kind) {
            case NONE: {
                return null;
            }
            case ZLIB: {
                return new ZlibCodec();
            }
            case SNAPPY: {
                return new SnappyCodec();
            }
            case LZO: {
                try {
                    ClassLoader loader = Thread.currentThread().getContextClassLoader();
                    if (loader == null) {
                        loader = WriterImpl.class.getClassLoader();
                    }
                    Class<?> lzo = loader.loadClass("org.apache.hadoop.hive.ql.io.orc.LzoCodec");
                    return (CompressionCodec)lzo.newInstance();
                }
                catch (ClassNotFoundException e) {
                    throw new IllegalArgumentException("LZO is not available.", e);
                }
                catch (InstantiationException e) {
                    throw new IllegalArgumentException("Problem initializing LZO", e);
                }
                catch (IllegalAccessException e) {
                    throw new IllegalArgumentException("Insufficient access to LZO", e);
                }
            }
        }
        throw new IllegalArgumentException("Unknown compression codec: " + (Object)((Object)kind));
    }

    private void writeStripeFooter(OrcProto.StripeFooter footer, long dataSize, long indexSize, OrcProto.StripeInformation.Builder dirEntry) throws IOException {
        footer.writeTo(this.protobufWriter);
        this.protobufWriter.flush();
        this.writer.flush();
        dirEntry.setOffset(this.stripeStart);
        dirEntry.setFooterLength(this.rawWriter.getPos() - this.stripeStart - dataSize - indexSize);
    }

    @VisibleForTesting
    public static int getEstimatedBufferSize(long stripeSize, int numColumns, int bs) {
        int estBufferSize = (int)(stripeSize / (long)(20 * numColumns));
        return (estBufferSize = PhysicalFsWriter.getClosestBufferSize(estBufferSize)) > bs ? bs : estBufferSize;
    }

    private static int getClosestBufferSize(int estBufferSize) {
        int kb4 = 4096;
        int kb8 = 8192;
        int kb16 = 16384;
        int kb32 = 32768;
        int kb64 = 65536;
        int kb128 = 131072;
        int kb256 = 262144;
        if (estBufferSize <= 4096) {
            return 4096;
        }
        if (estBufferSize > 4096 && estBufferSize <= 8192) {
            return 8192;
        }
        if (estBufferSize > 8192 && estBufferSize <= 16384) {
            return 16384;
        }
        if (estBufferSize > 16384 && estBufferSize <= 32768) {
            return 32768;
        }
        if (estBufferSize > 32768 && estBufferSize <= 65536) {
            return 65536;
        }
        if (estBufferSize > 65536 && estBufferSize <= 131072) {
            return 131072;
        }
        return 262144;
    }

    @Override
    public void writeFileMetadata(OrcProto.Metadata.Builder builder) throws IOException {
        long startPosn = this.rawWriter.getPos();
        OrcProto.Metadata metadata = builder.build();
        metadata.writeTo(this.protobufWriter);
        this.protobufWriter.flush();
        this.writer.flush();
        this.metadataLength = (int)(this.rawWriter.getPos() - startPosn);
    }

    @Override
    public void writeFileFooter(OrcProto.Footer.Builder builder) throws IOException {
        long bodyLength = this.rawWriter.getPos() - (long)this.metadataLength;
        builder.setContentLength(bodyLength);
        builder.setHeaderLength(this.headerLength);
        long startPosn = this.rawWriter.getPos();
        OrcProto.Footer footer = builder.build();
        footer.writeTo(this.protobufWriter);
        this.protobufWriter.flush();
        this.writer.flush();
        this.footerLength = (int)(this.rawWriter.getPos() - startPosn);
    }

    @Override
    public void writePostScript(OrcProto.PostScript.Builder builder) throws IOException {
        builder.setCompression(this.writeCompressionKind(this.compress));
        builder.setFooterLength(this.footerLength);
        builder.setMetadataLength(this.metadataLength);
        if (this.compress != CompressionKind.NONE) {
            builder.setCompressionBlockSize(this.bufferSize);
        }
        OrcProto.PostScript ps = builder.build();
        long startPosn = this.rawWriter.getPos();
        ps.writeTo((OutputStream)this.rawWriter);
        long length = this.rawWriter.getPos() - startPosn;
        if (length > 255L) {
            throw new IllegalArgumentException("PostScript too large at " + length);
        }
        this.rawWriter.writeByte((int)length);
    }

    @Override
    public void close() throws IOException {
        this.rawWriter.close();
    }

    private OrcProto.CompressionKind writeCompressionKind(CompressionKind kind) {
        switch (kind) {
            case NONE: {
                return OrcProto.CompressionKind.NONE;
            }
            case ZLIB: {
                return OrcProto.CompressionKind.ZLIB;
            }
            case SNAPPY: {
                return OrcProto.CompressionKind.SNAPPY;
            }
            case LZO: {
                return OrcProto.CompressionKind.LZO;
            }
        }
        throw new IllegalArgumentException("Unknown compression " + (Object)((Object)kind));
    }

    @Override
    public void flush() throws IOException {
        this.rawWriter.hflush();
    }

    @Override
    public long getRawWriterPosition() throws IOException {
        return this.rawWriter.getPos();
    }

    @Override
    public void appendRawStripe(byte[] stripe, int offset, int length, OrcProto.StripeInformation.Builder dirEntry) throws IOException {
        long start = this.rawWriter.getPos();
        long availBlockSpace = this.blockSize - start % this.blockSize;
        if ((long)length < this.blockSize && (long)length > availBlockSpace && this.addBlockPadding) {
            byte[] pad = new byte[(int)Math.min(262144L, availBlockSpace)];
            LOG.info(String.format("Padding ORC by %d bytes while merging..", availBlockSpace));
            start += availBlockSpace;
            while (availBlockSpace > 0L) {
                int writeLen = (int)Math.min(availBlockSpace, (long)pad.length);
                this.rawWriter.write(pad, 0, writeLen);
                availBlockSpace -= (long)writeLen;
            }
        }
        this.rawWriter.write(stripe);
        dirEntry.setOffset(start);
    }

    @Override
    public OutStream getOrCreatePhysicalStream(StreamName name) throws IOException {
        BufferedStream result = this.streams.get(name);
        if (result == null) {
            EnumSet<CompressionCodec.Modifier> modifiers = this.createCompressionModifiers(name.getKind());
            result = new BufferedStream(name.toString(), this.bufferSize, this.codec == null ? null : this.codec.modify(modifiers));
            this.streams.put(name, result);
        }
        return result.outStream;
    }

    private EnumSet<CompressionCodec.Modifier> createCompressionModifiers(OrcProto.Stream.Kind kind) {
        switch (kind) {
            case BLOOM_FILTER: 
            case DATA: 
            case DICTIONARY_DATA: {
                return EnumSet.of(CompressionCodec.Modifier.TEXT, this.compressionStrategy == OrcFile.CompressionStrategy.SPEED ? CompressionCodec.Modifier.FAST : CompressionCodec.Modifier.DEFAULT);
            }
            case LENGTH: 
            case DICTIONARY_COUNT: 
            case PRESENT: 
            case ROW_INDEX: 
            case SECONDARY: {
                return EnumSet.of(CompressionCodec.Modifier.FASTEST, CompressionCodec.Modifier.BINARY);
            }
        }
        LOG.warn("Missing ORC compression modifiers for " + (Object)((Object)kind));
        return null;
    }

    @Override
    public void finalizeStripe(OrcProto.StripeFooter.Builder footerBuilder, OrcProto.StripeInformation.Builder dirEntry) throws IOException {
        long indexSize = 0L;
        long dataSize = 0L;
        for (Map.Entry<StreamName, BufferedStream> pair : this.streams.entrySet()) {
            BufferedStream receiver = pair.getValue();
            OutStream outStream = receiver.outStream;
            if (outStream.isSuppressed()) continue;
            outStream.flush();
            long streamSize = receiver.getOutputSize();
            StreamName name = pair.getKey();
            footerBuilder.addStreams(OrcProto.Stream.newBuilder().setColumn(name.getColumn()).setKind(name.getKind()).setLength(streamSize));
            if (StreamName.Area.INDEX == name.getArea()) {
                indexSize += streamSize;
                continue;
            }
            dataSize += streamSize;
        }
        dirEntry.setIndexLength(indexSize).setDataLength(dataSize);
        OrcProto.StripeFooter footer = footerBuilder.build();
        this.padStripe(indexSize, dataSize, footer.getSerializedSize());
        for (Map.Entry<StreamName, BufferedStream> pair : this.streams.entrySet()) {
            pair.getValue().spillToDiskAndClear();
        }
        this.writeStripeFooter(footer, dataSize, indexSize, dirEntry);
    }

    @Override
    public long estimateMemory() {
        long result = 0L;
        for (BufferedStream stream : this.streams.values()) {
            result += stream.getBufferSize();
        }
        return result;
    }

    @Override
    public void writeIndexStream(StreamName name, OrcProto.RowIndex.Builder rowIndex) throws IOException {
        OutStream stream = this.getOrCreatePhysicalStream(name);
        rowIndex.build().writeTo(stream);
        stream.flush();
    }

    @Override
    public void writeBloomFilterStream(StreamName name, OrcProto.BloomFilterIndex.Builder bloomFilterIndex) throws IOException {
        OutStream stream = this.getOrCreatePhysicalStream(name);
        bloomFilterIndex.build().writeTo(stream);
        stream.flush();
    }

    @VisibleForTesting
    public OutputStream getStream() throws IOException {
        this.initialize();
        return this.rawWriter;
    }

    private class BufferedStream
    implements OutStream.OutputReceiver {
        private final OutStream outStream;
        private final List<ByteBuffer> output = new ArrayList<ByteBuffer>();

        BufferedStream(String name, int bufferSize, CompressionCodec codec) throws IOException {
            this.outStream = new OutStream(name, bufferSize, codec, this);
        }

        @Override
        public void output(ByteBuffer buffer) {
            this.output.add(buffer);
        }

        public long getBufferSize() {
            long result = 0L;
            for (ByteBuffer buf : this.output) {
                result += (long)buf.capacity();
            }
            return this.outStream.getBufferSize() + result;
        }

        public void spillToDiskAndClear() throws IOException {
            if (!this.outStream.isSuppressed()) {
                for (ByteBuffer buffer : this.output) {
                    PhysicalFsWriter.this.rawWriter.write(buffer.array(), buffer.arrayOffset() + buffer.position(), buffer.remaining());
                }
            }
            this.outStream.clear();
            this.output.clear();
        }

        public long getOutputSize() {
            long result = 0L;
            for (ByteBuffer buffer : this.output) {
                result += (long)buffer.remaining();
            }
            return result;
        }

        public String toString() {
            return this.outStream.toString();
        }
    }

    private class DirectStream
    implements OutStream.OutputReceiver {
        private final FSDataOutputStream output;

        DirectStream(FSDataOutputStream output) {
            this.output = output;
        }

        @Override
        public void output(ByteBuffer buffer) throws IOException {
            this.output.write(buffer.array(), buffer.arrayOffset() + buffer.position(), buffer.remaining());
        }
    }
}

