/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package hivemall.utils.lang;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

public final class StringUtils {

    private StringUtils() {}

    @Nonnull
    public static byte[] getBytes(@Nonnull final String s) {
        final int len = s.length();
        final byte[] b = new byte[len * 2];
        for (int i = 0; i < len; i++) {
            Primitives.putChar(b, i * 2, s.charAt(i));
        }
        return b;
    }

    @Nonnull
    public static String toString(@Nonnull final byte[] b) {
        return toString(b, 0, b.length);
    }

    @Nonnull
    public static String toString(@Nonnull final byte[] b, final int off, final int len) {
        final int clen = len >>> 1;
        final char[] c = new char[clen];
        for (int i = 0; i < clen; i++) {
            final int j = off + (i << 1);
            c[i] = (char) ((b[j + 1] & 0xFF) + ((b[j + 0]) << 8));
        }
        return new String(c);
    }

    /**
     * Checks whether the String a valid Java number. this code is ported from jakarta commons lang.
     *
     * @link http://jakarta.apache.org/commons/lang/apidocs/org/apache/commons/lang
     *       /math/NumberUtils.html
     */
    public static boolean isNumber(@Nullable final String str) {
        if (str == null || str.length() == 0) {
            return false;
        }
        char[] chars = str.toCharArray();
        int sz = chars.length;
        boolean hasExp = false;
        boolean hasDecPoint = false;
        boolean allowSigns = false;
        boolean foundDigit = false;
        // deal with any possible sign up front
        int start = (chars[0] == '-') ? 1 : 0;
        if (sz > start + 1) {
            if (chars[start] == '0' && chars[start + 1] == 'x') {
                int i = start + 2;
                if (i == sz) {
                    return false; // str == "0x"
                }
                // checking hex (it can't be anything else)
                for (; i < chars.length; i++) {
                    if ((chars[i] < '0' || chars[i] > '9') && (chars[i] < 'a' || chars[i] > 'f')
                            && (chars[i] < 'A' || chars[i] > 'F')) {
                        return false;
                    }
                }
                return true;
            }
        }
        sz--; // don't want to loop to the last char, check it afterwords
        // for type qualifiers
        int i = start;
        // loop to the next to last char or to the last char if we need another digit to
        // make a valid number (e.g. chars[0..5] = "1234E")
        while (i < sz || (i < sz + 1 && allowSigns && !foundDigit)) {
            if (chars[i] >= '0' && chars[i] <= '9') {
                foundDigit = true;
                allowSigns = false;

            } else if (chars[i] == '.') {
                if (hasDecPoint || hasExp) {
                    // two decimal points or dec in exponent
                    return false;
                }
                hasDecPoint = true;
            } else if (chars[i] == 'e' || chars[i] == 'E') {
                // we've already taken care of hex.
                if (hasExp) {
                    // two E's
                    return false;
                }
                if (!foundDigit) {
                    return false;
                }
                hasExp = true;
                allowSigns = true;
            } else if (chars[i] == '+' || chars[i] == '-') {
                if (!allowSigns) {
                    return false;
                }
                allowSigns = false;
                foundDigit = false; // we need a digit after the E
            } else {
                return false;
            }
            i++;
        }
        if (i < chars.length) {
            if (chars[i] >= '0' && chars[i] <= '9') {
                // no type qualifier, OK
                return true;
            }
            if (chars[i] == 'e' || chars[i] == 'E') {
                // can't have an E at the last byte
                return false;
            }
            if (!allowSigns
                    && (chars[i] == 'd' || chars[i] == 'D' || chars[i] == 'f' || chars[i] == 'F')) {
                return foundDigit;
            }
            if (chars[i] == 'l' || chars[i] == 'L') {
                // not allowing L with an exponent
                return foundDigit && !hasExp;
            }
            // last character is illegal
            return false;
        }
        // allowSigns is true iff the val ends in 'E'
        // found digit it to make sure weird stuff like '.' and '1E-' doesn't pass
        return !allowSigns && foundDigit;
    }

    public static boolean isInt(@Nonnull final String i) {
        try {
            Integer.parseInt(i);
            return true;
        } catch (NumberFormatException nfe) {
            return false;
        }
    }

    public static boolean isDouble(@Nonnull final String i) {
        try {
            Double.parseDouble(i);
            return true;
        } catch (NumberFormatException nfe) {
            return false;
        }
    }

    public static void clear(@Nonnull final StringBuilder buf) {
        buf.setLength(0);
    }

    @Nonnull
    public static String concat(@Nonnull final List<String> list, @Nonnull final String sep) {
        final StringBuilder buf = new StringBuilder(128);
        for (String s : list) {
            if (s == null) {
                continue;
            }
            buf.append(s);
            buf.append(sep);
        }
        return buf.toString();
    }

    @Nonnull
    public static String join(@Nonnull final List<String> list, @Nonnull final String sep) {
        final StringBuilder buf = new StringBuilder(128);
        for (int i = 0, size = list.size(); i < size; i++) {
            if (i > 0) { // append separator before each element, except for the head element
                buf.append(sep);
            }
            final String s = list.get(i);
            if (s != null) {
                buf.append(s);
            }
        }
        return buf.toString();
    }

    @Nonnull
    public static String join(@Nonnull final Object[] list, @Nonnull final char sep) {
        final StringBuilder buf = new StringBuilder(128);
        for (int i = 0; i < list.length; i++) {
            if (i > 0) { // append separator before each element, except for the head element
                buf.append(sep);
            }
            final Object s = list[i];
            if (s != null) {
                buf.append(s);
            }
        }
        return buf.toString();
    }

    @Nullable
    public static String[] split(@Nullable final String str, final char separatorChar) {
        return split(str, separatorChar, false);
    }

    @Nullable
    public static String[] split(@Nullable final String str, final char separatorChar,
            final boolean preserveAllTokens) {
        if (str == null) {
            return null;
        }
        final int len = str.length();
        if (len == 0) {
            return new String[0];
        }
        final List<String> list = new ArrayList<String>();
        int i = 0, start = 0;
        boolean match = false;
        boolean lastMatch = false;
        while (i < len) {
            if (str.charAt(i) == separatorChar) {
                if (match || preserveAllTokens) {
                    list.add(str.substring(start, i));
                    match = false;
                    lastMatch = true;
                }
                start = ++i;
                continue;
            }
            lastMatch = false;
            match = true;
            i++;
        }
        if (match || preserveAllTokens && lastMatch) {
            list.add(str.substring(start, i));
        }
        return list.toArray(new String[list.size()]);
    }

    @Nonnull
    public static String format(@Nullable String template, @Nullable Object... args) {
        template = String.valueOf(template);

        StringBuilder builder = new StringBuilder(template.length() + 16 * args.length);
        int templateStart = 0;
        int i = 0;
        while (i < args.length) {
            int placeholderStart = template.indexOf("%s", templateStart);
            if (placeholderStart == -1) {
                break;
            }
            builder.append(template, templateStart, placeholderStart);
            builder.append(args[i++]);
            templateStart = placeholderStart + 2;
        }
        builder.append(template, templateStart, template.length());

        if (i < args.length) {
            builder.append(" [");
            builder.append(args[i++]);
            while (i < args.length) {
                builder.append(", ");
                builder.append(args[i++]);
            }
            builder.append(']');
        }

        return builder.toString();
    }

    public static int compare(@Nullable final String o1, @Nullable final String o2) {
        return compare(o1, o2, true);
    }

    public static int compare(@Nullable final String o1, @Nullable final String o2,
            final boolean nullIsLess) {
        if (o1 == o2) {
            return 0;
        }
        if (o1 == null) {
            return nullIsLess ? -1 : 1;
        }
        if (o2 == null) {
            return nullIsLess ? 1 : -1;
        }
        return o1.compareTo(o2);
    }

}
