/*
 * Decompiled with CFR 0.152.
 */
package hivemall.utils.buffer;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

public final class DynamicByteArray {
    static final int DEFAULT_CHUNKSIZE = 32768;
    static final int DEFAULT_NUM_CHUNKS = 128;
    private final int chunkSize;
    private byte[][] data;
    private int length;
    private int initializedChunks = 0;

    public DynamicByteArray() {
        this(128, 32768);
    }

    public DynamicByteArray(int numChunks, int chunkSize) {
        if (chunkSize == 0) {
            throw new IllegalArgumentException("bad chunksize");
        }
        this.chunkSize = chunkSize;
        this.data = new byte[numChunks][];
    }

    private void grow(int chunkIndex) {
        if (chunkIndex >= this.initializedChunks) {
            if (chunkIndex >= this.data.length) {
                int newSize = Math.max(chunkIndex + 1, 2 * this.data.length);
                byte[][] newChunk = new byte[newSize][];
                System.arraycopy(this.data, 0, newChunk, 0, this.data.length);
                this.data = newChunk;
            }
            for (int i = this.initializedChunks; i <= chunkIndex; ++i) {
                this.data[i] = new byte[this.chunkSize];
            }
            this.initializedChunks = chunkIndex + 1;
        }
    }

    public byte get(int index) {
        if (index >= this.length) {
            throw new IndexOutOfBoundsException("Index " + index + " is outside of 0.." + (this.length - 1));
        }
        int i = index / this.chunkSize;
        int j = index % this.chunkSize;
        return this.data[i][j];
    }

    public void get(@Nonnull ByteBuffer result, int offset, int length) {
        result.clear();
        int currentChunk = offset / this.chunkSize;
        int currentOffset = offset % this.chunkSize;
        int currentLength = Math.min(length, this.chunkSize - currentOffset);
        while (length > 0) {
            result.put(this.data[currentChunk], currentOffset, currentLength);
            ++currentChunk;
            currentOffset = 0;
            currentLength = Math.min(length -= currentLength, this.chunkSize - currentOffset);
        }
    }

    public int getInt(int index) {
        return DynamicByteArray.toInt(this.get(index), this.get(index + 1), this.get(index + 2), this.get(index + 3));
    }

    public long getLong(int index) {
        return DynamicByteArray.toLong(this.get(index), this.get(index + 1), this.get(index + 2), this.get(index + 3), this.get(index + 4), this.get(index + 5), this.get(index + 6), this.get(index + 7));
    }

    public float getFloat(int index) {
        return Float.intBitsToFloat(this.getInt(index));
    }

    public double getDouble(int index) {
        return Double.longBitsToDouble(this.getLong(index));
    }

    private static int toInt(byte b3, byte b2, byte b1, byte b0) {
        return b3 << 24 | (b2 & 0xFF) << 16 | (b1 & 0xFF) << 8 | b0 & 0xFF;
    }

    private static long toLong(byte b7, byte b6, byte b5, byte b4, byte b3, byte b2, byte b1, byte b0) {
        return (long)b7 << 56 | ((long)b6 & 0xFFL) << 48 | ((long)b5 & 0xFFL) << 40 | ((long)b4 & 0xFFL) << 32 | ((long)b3 & 0xFFL) << 24 | ((long)b2 & 0xFFL) << 16 | ((long)b1 & 0xFFL) << 8 | (long)b0 & 0xFFL;
    }

    public void setInt(int index, int value) {
        this.set(index, DynamicByteArray.int3(value));
        this.set(index + 1, DynamicByteArray.int2(value));
        this.set(index + 2, DynamicByteArray.int1(value));
        this.set(index + 3, DynamicByteArray.int0(value));
    }

    public void setLong(int index, long value) {
        this.set(index, DynamicByteArray.long7(value));
        this.set(index + 1, DynamicByteArray.long6(value));
        this.set(index + 2, DynamicByteArray.long5(value));
        this.set(index + 3, DynamicByteArray.long4(value));
        this.set(index + 4, DynamicByteArray.long3(value));
        this.set(index + 5, DynamicByteArray.long2(value));
        this.set(index + 6, DynamicByteArray.long1(value));
        this.set(index + 7, DynamicByteArray.long0(value));
    }

    public void setFloat(int index, float value) {
        this.setInt(index, Float.floatToIntBits(value));
    }

    public void setDouble(int index, double value) {
        this.setLong(index, Double.doubleToLongBits(value));
    }

    private static byte int3(int x) {
        return (byte)(x >> 24);
    }

    private static byte int2(int x) {
        return (byte)(x >> 16);
    }

    private static byte int1(int x) {
        return (byte)(x >> 8);
    }

    private static byte int0(int x) {
        return (byte)x;
    }

    private static byte long7(long x) {
        return (byte)(x >> 56);
    }

    private static byte long6(long x) {
        return (byte)(x >> 48);
    }

    private static byte long5(long x) {
        return (byte)(x >> 40);
    }

    private static byte long4(long x) {
        return (byte)(x >> 32);
    }

    private static byte long3(long x) {
        return (byte)(x >> 24);
    }

    private static byte long2(long x) {
        return (byte)(x >> 16);
    }

    private static byte long1(long x) {
        return (byte)(x >> 8);
    }

    private static byte long0(long x) {
        return (byte)x;
    }

    public void set(int index, byte value) {
        int i = index / this.chunkSize;
        int j = index % this.chunkSize;
        this.grow(i);
        if (index >= this.length) {
            this.length = index + 1;
        }
        this.data[i][j] = value;
    }

    public int add(byte value) {
        int i = this.length / this.chunkSize;
        int j = this.length % this.chunkSize;
        this.grow(i);
        this.data[i][j] = value;
        int result = this.length++;
        return result;
    }

    public int add(@Nonnull byte[] value, int valueOffset, int valueLength) {
        int i = this.length / this.chunkSize;
        int j = this.length % this.chunkSize;
        this.grow((this.length + valueLength) / this.chunkSize);
        int remaining = valueLength;
        while (remaining > 0) {
            int size = Math.min(remaining, this.chunkSize - j);
            System.arraycopy(value, valueOffset, this.data[i], j, size);
            remaining -= size;
            valueOffset += size;
            ++i;
            j = 0;
        }
        int result = this.length;
        this.length += valueLength;
        return result;
    }

    public int add(@Nonnull ByteBuffer src, int valueOffset, int valueLength) {
        int i = this.length / this.chunkSize;
        int j = this.length % this.chunkSize;
        this.grow((this.length + valueLength) / this.chunkSize);
        int remaining = valueLength;
        while (remaining > 0) {
            int size = Math.min(remaining, this.chunkSize - j);
            src.get(this.data[i], j, size);
            remaining -= size;
            valueOffset += size;
            ++i;
            j = 0;
        }
        int result = this.length;
        this.length += valueLength;
        return result;
    }

    public int compare(byte[] other, int otherOffset, int otherLength, int ourOffset, int ourLength) {
        int maxLength;
        int currentChunk = ourOffset / this.chunkSize;
        int currentOffset = ourOffset % this.chunkSize;
        for (maxLength = Math.min(otherLength, ourLength); maxLength > 0 && other[otherOffset] == this.data[currentChunk][currentOffset]; --maxLength) {
            ++otherOffset;
            if (++currentOffset != this.chunkSize) continue;
            ++currentChunk;
            currentOffset = 0;
        }
        if (maxLength == 0) {
            return otherLength - ourLength;
        }
        int otherByte = 0xFF & other[otherOffset];
        int ourByte = 0xFF & this.data[currentChunk][currentOffset];
        return otherByte > ourByte ? 1 : -1;
    }

    public int size() {
        return this.length;
    }

    public void clear() {
        this.length = 0;
        for (int i = 0; i < this.data.length; ++i) {
            this.data[i] = null;
        }
        this.initializedChunks = 0;
    }

    public void read(@Nonnull InputStream in) throws IOException {
        int currentChunk = this.length / this.chunkSize;
        int currentOffset = this.length % this.chunkSize;
        this.grow(currentChunk);
        int currentLength = in.read(this.data[currentChunk], currentOffset, this.chunkSize - currentOffset);
        while (currentLength > 0) {
            this.length += currentLength;
            currentOffset = this.length % this.chunkSize;
            if (currentOffset == 0) {
                currentChunk = this.length / this.chunkSize;
                this.grow(currentChunk);
            }
            currentLength = in.read(this.data[currentChunk], currentOffset, this.chunkSize - currentOffset);
        }
    }

    public void write(@Nonnull OutputStream out, int offset, int length) throws IOException {
        int currentChunk = offset / this.chunkSize;
        int currentOffset = offset % this.chunkSize;
        while (length > 0) {
            int currentLength = Math.min(length, this.chunkSize - currentOffset);
            out.write(this.data[currentChunk], currentOffset, currentLength);
            length -= currentLength;
            ++currentChunk;
            currentOffset = 0;
        }
    }

    public String toString() {
        int i;
        StringBuilder sb = new StringBuilder(this.length * 3);
        sb.append('{');
        int l = this.length - 1;
        for (i = 0; i < l; ++i) {
            sb.append(Integer.toHexString(this.get(i)));
            sb.append(',');
        }
        sb.append(Integer.toHexString(this.get(i)));
        sb.append('}');
        return sb.toString();
    }

    @Nullable
    public byte[] get() {
        byte[] result = null;
        if (this.length > 0) {
            int currentChunk = 0;
            int currentOffset = 0;
            int currentLength = Math.min(this.length, this.chunkSize);
            int destOffset = 0;
            result = new byte[this.length];
            int totalLength = this.length;
            while (totalLength > 0) {
                System.arraycopy(this.data[currentChunk], currentOffset, result, destOffset, currentLength);
                destOffset += currentLength;
                ++currentChunk;
                currentOffset = 0;
                currentLength = Math.min(totalLength -= currentLength, this.chunkSize - currentOffset);
            }
        }
        return result;
    }

    public long getSizeInBytes() {
        return this.initializedChunks * this.chunkSize;
    }
}

