/*
 * Decompiled with CFR 0.152.
 */
package hivemall.utils.collections.arrays;

import hivemall.utils.collections.arrays.IntArray;
import hivemall.utils.function.Consumer;
import hivemall.utils.lang.ArrayUtils;
import hivemall.utils.lang.Preconditions;
import hivemall.utils.math.MathUtils;
import java.util.Arrays;
import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;

public final class SparseIntArray
implements IntArray {
    private static final long serialVersionUID = -2814248784231540118L;
    private int[] mKeys;
    private int[] mValues;
    private int mSize;

    public SparseIntArray() {
    }

    public SparseIntArray(@Nonnegative int initialCapacity) {
        this.mKeys = new int[initialCapacity];
        this.mValues = new int[initialCapacity];
        this.mSize = 0;
    }

    public SparseIntArray(@Nonnull int[] values) {
        this.mKeys = MathUtils.permutation(values.length);
        this.mValues = values;
        this.mSize = values.length;
    }

    public SparseIntArray(@Nonnull int[] mKeys, @Nonnull int[] mValues, @Nonnegative int mSize) {
        this.mKeys = mKeys;
        this.mValues = mValues;
        this.mSize = mSize;
    }

    public void init(@Nonnull int[] keys, @Nonnull int[] values) {
        this.init(keys, values, keys.length);
    }

    public void init(@Nonnull int[] keys, @Nonnull int[] values, int size) {
        if (size > keys.length || size > values.length) {
            throw new IllegalArgumentException(String.format("Illegal size was specified... size = %d, keys.length = %d, values.length", size, keys.length, values.length));
        }
        this.mKeys = keys;
        this.mValues = values;
        this.mSize = size;
    }

    public int[] keys() {
        return this.mKeys;
    }

    public int[] values() {
        return this.mValues;
    }

    public IntArray deepCopy() {
        int[] newKeys = new int[this.mSize];
        int[] newValues = new int[this.mSize];
        System.arraycopy(this.mKeys, 0, newKeys, 0, this.mSize);
        System.arraycopy(this.mValues, 0, newValues, 0, this.mSize);
        return new SparseIntArray(newKeys, newValues, this.mSize);
    }

    @Override
    public int get(int key) {
        return this.get(key, 0);
    }

    @Override
    public int get(int key, int valueIfKeyNotFound) {
        int i = Arrays.binarySearch(this.mKeys, 0, this.mSize, key);
        if (i < 0) {
            return valueIfKeyNotFound;
        }
        return this.mValues[i];
    }

    public void delete(int key) {
        int i = Arrays.binarySearch(this.mKeys, 0, this.mSize, key);
        if (i >= 0) {
            this.removeAt(i);
        }
    }

    public void removeAt(int index) {
        System.arraycopy(this.mKeys, index + 1, this.mKeys, index, this.mSize - (index + 1));
        System.arraycopy(this.mValues, index + 1, this.mValues, index, this.mSize - (index + 1));
        --this.mSize;
    }

    public void removeRange(@Nonnegative int start2, @Nonnegative int end) {
        int sizeToRemove;
        int endPos;
        Preconditions.checkArgument(start2 <= end);
        int startPos = this.indexOfKey(start2);
        if (startPos < 0) {
            startPos ^= 0xFFFFFFFF;
        }
        if ((endPos = this.indexOfKey(end)) < 0) {
            endPos ^= 0xFFFFFFFF;
        }
        if ((sizeToRemove = endPos - startPos) <= 0) {
            return;
        }
        ArrayUtils.clearRange(this.mKeys, startPos, endPos, 0);
        ArrayUtils.clearRange(this.mValues, startPos, endPos, 0);
        this.mSize -= sizeToRemove;
    }

    @Override
    public void put(int key, int value) {
        int i = Arrays.binarySearch(this.mKeys, 0, this.mSize, key);
        if (i >= 0) {
            this.mValues[i] = value;
        } else {
            this.mKeys = ArrayUtils.insert(this.mKeys, this.mSize, i ^= 0xFFFFFFFF, key);
            this.mValues = ArrayUtils.insert(this.mValues, this.mSize, i, value);
            ++this.mSize;
        }
    }

    @Override
    public void increment(int key, int value) {
        int i = Arrays.binarySearch(this.mKeys, 0, this.mSize, key);
        if (i >= 0) {
            int n = i;
            this.mValues[n] = this.mValues[n] + value;
        } else {
            this.mKeys = ArrayUtils.insert(this.mKeys, this.mSize, i ^= 0xFFFFFFFF, key);
            this.mValues = ArrayUtils.insert(this.mValues, this.mSize, i, value);
            ++this.mSize;
        }
    }

    @Override
    public int size() {
        return this.mSize;
    }

    public int firstKey() {
        if (this.mSize == 0) {
            return -1;
        }
        return this.mKeys[0];
    }

    public int lastKey() {
        if (this.mSize == 0) {
            return -1;
        }
        return this.mKeys[this.mSize - 1];
    }

    @Override
    public int keyAt(int index) {
        return this.mKeys[index];
    }

    public int valueAt(int index) {
        return this.mValues[index];
    }

    public void setKeyAt(int index, int key) {
        this.mKeys[index] = key;
    }

    public void setValueAt(int index, int value) {
        this.mValues[index] = value;
    }

    public int indexOfKey(int key) {
        return Arrays.binarySearch(this.mKeys, 0, this.mSize, key);
    }

    public int indexOfValue(int value) {
        for (int i = 0; i < this.mSize; ++i) {
            if (this.mValues[i] != value) continue;
            return i;
        }
        return -1;
    }

    public void clear() {
        this.clear(true);
    }

    public void clear(boolean zeroFill) {
        this.mSize = 0;
        if (zeroFill) {
            Arrays.fill(this.mKeys, 0);
            Arrays.fill(this.mValues, 0);
        }
    }

    public void append(int key, int value) {
        if (this.mSize != 0 && key <= this.mKeys[this.mSize - 1]) {
            this.put(key, value);
            return;
        }
        this.mKeys = ArrayUtils.append(this.mKeys, this.mSize, key);
        this.mValues = ArrayUtils.append(this.mValues, this.mSize, value);
        ++this.mSize;
    }

    public void append(@Nonnegative int dstPos, @Nonnull int[] values) {
        if (this.mSize == 0) {
            this.mKeys = MathUtils.permutation(dstPos, values.length);
            this.mValues = (int[])values.clone();
            this.mSize = values.length;
            return;
        }
        int lastKey = this.mKeys[this.mSize - 1];
        for (int i = 0; i < values.length; ++i) {
            int key = dstPos + i;
            if (key > lastKey) {
                int size = values.length - i;
                int[] appendKeys = MathUtils.permutation(key, size);
                this.mKeys = ArrayUtils.concat(this.mKeys, 0, this.mSize, appendKeys, 0, appendKeys.length);
                this.mValues = ArrayUtils.concat(this.mValues, 0, this.mSize, values, i, size);
                this.mSize += size;
                break;
            }
            this.put(key, values[i]);
        }
    }

    public void append(@Nonnegative int dstPos, @Nonnull int[] values, int offset, int length) {
        if (this.mSize == 0) {
            this.mKeys = MathUtils.permutation(dstPos, length);
            this.mValues = Arrays.copyOfRange(values, offset, length);
            this.mSize = length;
            return;
        }
        int lastKey = this.mKeys[this.mSize - 1];
        for (int i = 0; i < length; ++i) {
            int valuePos = offset + i;
            int key = dstPos + i;
            if (key > lastKey) {
                int size = length - i;
                int[] appendKeys = MathUtils.permutation(key, size);
                this.mKeys = ArrayUtils.concat(this.mKeys, 0, this.mSize, appendKeys, 0, appendKeys.length);
                this.mValues = ArrayUtils.concat(this.mValues, 0, this.mSize, values, valuePos, size);
                this.mSize += size;
                break;
            }
            this.put(key, values[valuePos]);
        }
    }

    public void forEach(@Nonnegative int start2, @Nonnegative int end, @Nonnull Consumer consumer) {
        int endPos;
        int startPos = this.indexOfKey(start2);
        if (startPos < 0) {
            startPos ^= 0xFFFFFFFF;
        }
        if ((endPos = this.indexOfKey(end)) < 0) {
            endPos ^= 0xFFFFFFFF;
        }
        int[] keys = this.mKeys;
        int[] values = this.mValues;
        for (int i = startPos; i < endPos; ++i) {
            int k = keys[i];
            int v = values[i];
            consumer.accept(k, v);
        }
    }

    @Override
    public void forEach(@Nonnull Consumer consumer) {
        int size = this.mSize;
        int[] keys = this.mKeys;
        int[] values = this.mValues;
        for (int i = 0; i < size; ++i) {
            int k = keys[i];
            int v = values[i];
            consumer.accept(k, v);
        }
    }

    @Override
    @Nonnull
    public int[] toArray() {
        return this.toArray(true);
    }

    @Override
    public int[] toArray(boolean copy) {
        if (this.mSize == 0) {
            return new int[0];
        }
        int last = this.mKeys[this.mSize - 1];
        int[] array = new int[last + 1];
        for (int i = 0; i < this.mSize; ++i) {
            int k = this.mKeys[i];
            int v = this.mValues[i];
            Preconditions.checkArgument(k >= 0, "Negative key is not allowed for toArray(): " + k);
            array[k] = v;
        }
        return array;
    }

    public String toString() {
        if (this.mSize == 0) {
            return "{}";
        }
        StringBuilder buffer = new StringBuilder(this.mSize * 28);
        buffer.append('{');
        for (int i = 0; i < this.mSize; ++i) {
            if (i > 0) {
                buffer.append(", ");
            }
            int key = this.keyAt(i);
            buffer.append(key);
            buffer.append('=');
            int value = this.valueAt(i);
            buffer.append(value);
        }
        buffer.append('}');
        return buffer.toString();
    }
}

