/*
 * Decompiled with CFR 0.152.
 */
package hivemall.utils.collections.maps;

import hivemall.utils.collections.IMapIterator;
import hivemall.utils.lang.Copyable;
import hivemall.utils.lang.Preconditions;
import hivemall.utils.math.Primes;
import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Arrays;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

public final class OpenHashTable<K, V>
implements Externalizable {
    public static final float DEFAULT_LOAD_FACTOR = 0.75f;
    public static final float DEFAULT_GROW_FACTOR = 2.0f;
    private static final float SHRINK_FACTOR = 0.1f;
    private static final float GROW_FACTOR_AT_SHRINK = 1.7f;
    protected static final byte FREE = 0;
    protected static final byte FULL = 1;
    protected static final byte REMOVED = 2;
    protected float _loadFactor;
    protected float _growFactor;
    protected int _used;
    protected int _freeEntries;
    protected int _growThreshold;
    protected int _shrinkThreshold;
    protected K[] _keys;
    protected V[] _values;
    protected byte[] _states;

    public OpenHashTable() {
    }

    public OpenHashTable(int size) {
        this(size, 0.75f, 2.0f);
    }

    public OpenHashTable(int size, float loadFactor, float growFactor) {
        if (size < 1) {
            throw new IllegalArgumentException();
        }
        this._loadFactor = loadFactor;
        this._growFactor = growFactor;
        int actualSize = Primes.findLeastPrimeNumber(size);
        this._keys = new Object[actualSize];
        this._values = new Object[actualSize];
        this._states = new byte[actualSize];
        this._used = 0;
        this._freeEntries = actualSize;
        this._growThreshold = Math.round((float)actualSize * this._loadFactor);
        this._shrinkThreshold = Math.round((float)actualSize * 0.1f);
    }

    public Object[] getKeys() {
        return this._keys;
    }

    public Object[] getValues() {
        return this._values;
    }

    public byte[] getStates() {
        return this._states;
    }

    public boolean containsKey(@CheckForNull K key) {
        return this.findKey(key) >= 0;
    }

    public V get(@CheckForNull K key) {
        int i = this.findKey(key);
        if (i == -1) {
            return null;
        }
        return this._values[i];
    }

    public V put(@CheckForNull K key, @Nullable V value) {
        byte state;
        K[] keys;
        block10: {
            Preconditions.checkNotNull(key);
            int hash = OpenHashTable.keyHash(key);
            int keyLength = this._keys.length;
            int keyIdx = hash % keyLength;
            boolean expanded = this.preAddEntry(keyIdx);
            if (expanded) {
                keyLength = this._keys.length;
                keyIdx = hash % keyLength;
            }
            keys = this._keys;
            V[] values = this._values;
            byte[] states = this._states;
            state = states[keyIdx];
            if (state == 1) {
                if (OpenHashTable.equals(keys[keyIdx], key)) {
                    V old = values[keyIdx];
                    values[keyIdx] = value;
                    return old;
                }
                int loopIndex = keyIdx;
                int decr = 1 + hash % (keyLength - 2);
                do {
                    if ((keyIdx -= decr) < 0) {
                        keyIdx += keyLength;
                    }
                    if (keyIdx == loopIndex) {
                        throw new IllegalStateException("Detected infinite loop where key=" + key + ", keyIdx=" + keyIdx);
                    }
                    state = states[keyIdx];
                    if (state == 0) break block10;
                } while (!OpenHashTable.equals(keys[keyIdx], key));
                if (state == 1) {
                    V old = values[keyIdx];
                    values[keyIdx] = value;
                    return old;
                }
                assert (state == 2);
            }
        }
        keys[keyIdx] = key;
        values[keyIdx] = value;
        states[keyIdx] = 1;
        ++this._used;
        if (state == 0) {
            --this._freeEntries;
            if (this._freeEntries < this._shrinkThreshold) {
                int newCapacity = Math.max(keys.length, Math.round((float)this._used * 1.7f));
                this.ensureCapacity(newCapacity);
            }
        }
        return null;
    }

    private static boolean equals(@Nonnull Object k1, @Nonnull Object k2) {
        return k1 == k2 || k1.equals(k2);
    }

    protected boolean preAddEntry(int index) {
        if (this._used + 1 >= this._growThreshold) {
            int newCapacity = Math.round((float)this._keys.length * this._growFactor);
            this.ensureCapacity(newCapacity);
            return true;
        }
        return false;
    }

    protected int findKey(@CheckForNull K key) {
        int startIndex;
        Preconditions.checkNotNull(key);
        K[] keys = this._keys;
        byte[] states = this._states;
        int keyLength = keys.length;
        int hash = OpenHashTable.keyHash(key);
        int decr = 1 + hash % (keyLength - 2);
        int keyIdx = startIndex = hash % keyLength;
        do {
            byte state;
            if ((state = states[keyIdx]) == 0) {
                return -1;
            }
            if (OpenHashTable.equals(keys[keyIdx], key)) {
                if (state == 1) {
                    return keyIdx;
                }
                assert (state == 2);
                return -1;
            }
            if ((keyIdx -= decr) >= 0) continue;
            keyIdx += keyLength;
        } while (keyIdx != startIndex);
        throw new IllegalStateException("Detected infinite loop where key=" + key + ", keyIdx=" + keyIdx);
    }

    public V remove(@CheckForNull K key) {
        int keyIdx = this.findKey(key);
        if (keyIdx == -1) {
            return null;
        }
        V old = this._values[keyIdx];
        this._states[keyIdx] = 2;
        --this._used;
        return old;
    }

    public int size() {
        return this._used;
    }

    public void clear() {
        Arrays.fill(this._states, (byte)0);
        this._used = 0;
        this._freeEntries = this._states.length;
    }

    public IMapIterator<K, V> entries() {
        return new MapIterator(false);
    }

    public IMapIterator<K, V> entries(boolean releaseSeen) {
        return new MapIterator(releaseSeen);
    }

    public String toString() {
        int len = this.size() * 10 + 2;
        StringBuilder buf = new StringBuilder(len);
        buf.append('{');
        IMapIterator<K, V> i = this.entries();
        while (i.next() != -1) {
            String key = i.getKey().toString();
            buf.append(key);
            buf.append('=');
            buf.append(i.getValue());
            if (!i.hasNext()) continue;
            buf.append(',');
        }
        buf.append('}');
        return buf.toString();
    }

    protected void ensureCapacity(@Nonnegative int newCapacity) {
        int prime = Primes.findLeastPrimeNumber(newCapacity);
        this.rehash(prime);
    }

    private void rehash(@Nonnegative int newCapacity) {
        K[] oldKeys = this._keys;
        V[] oldValues = this._values;
        byte[] oldStates = this._states;
        int oldCapacity = oldKeys.length;
        Object[] newkeys = new Object[newCapacity];
        Object[] newValues = new Object[newCapacity];
        byte[] newStates = new byte[newCapacity];
        int used = 0;
        for (int i = 0; i < oldCapacity; ++i) {
            if (oldStates[i] != 1) continue;
            K k = oldKeys[i];
            V v = oldValues[i];
            int hash = OpenHashTable.keyHash(k);
            int keyIdx = hash % newCapacity;
            if (newStates[keyIdx] == 1) {
                int decr = 1 + hash % (newCapacity - 2);
                int loopIndex = keyIdx;
                do {
                    if ((keyIdx -= decr) < 0) {
                        keyIdx += newCapacity;
                    }
                    if (keyIdx != loopIndex) continue;
                    throw new IllegalStateException("Detected infinite loop where key=" + k + ", keyIdx=" + keyIdx);
                } while (newStates[keyIdx] != 0);
            }
            newkeys[keyIdx] = k;
            newValues[keyIdx] = v;
            newStates[keyIdx] = 1;
            ++used;
        }
        this._keys = newkeys;
        this._values = newValues;
        this._states = newStates;
        this._used = used;
        this._freeEntries = newCapacity - used;
        this._growThreshold = Math.round((float)newCapacity * this._loadFactor);
        this._shrinkThreshold = Math.round((float)newCapacity * 0.1f);
    }

    private static int keyHash(@Nonnull Object key) {
        int hash = key.hashCode();
        return hash & Integer.MAX_VALUE;
    }

    @Override
    public void writeExternal(ObjectOutput out) throws IOException {
        out.writeFloat(this._loadFactor);
        out.writeFloat(this._growFactor);
        out.writeInt(this._used);
        IMapIterator<K, V> itor = this.entries();
        while (itor.next() != -1) {
            out.writeObject(itor.getKey());
            out.writeObject(itor.getValue());
        }
    }

    @Override
    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
        this._loadFactor = in.readFloat();
        this._growFactor = in.readFloat();
        int used = in.readInt();
        int newCapacity = Primes.findLeastPrimeNumber(Math.round((float)used * 1.7f));
        Object[] keys = new Object[newCapacity];
        Object[] values = new Object[newCapacity];
        byte[] states = new byte[newCapacity];
        for (int i = 0; i < used; ++i) {
            Object k = in.readObject();
            Object v = in.readObject();
            int hash = OpenHashTable.keyHash(k);
            int keyIdx = hash % newCapacity;
            if (states[keyIdx] == 1) {
                int decr = 1 + hash % (newCapacity - 2);
                int loopIndex = keyIdx;
                do {
                    if ((keyIdx -= decr) < 0) {
                        keyIdx += newCapacity;
                    }
                    if (keyIdx != loopIndex) continue;
                    throw new IllegalStateException("Detected infinite loop where key=" + k + ", keyIdx=" + keyIdx);
                } while (states[keyIdx] != 0);
            }
            keys[keyIdx] = k;
            values[keyIdx] = v;
            states[keyIdx] = 1;
        }
        this._keys = keys;
        this._values = values;
        this._states = states;
        this._used = used;
        this._freeEntries = newCapacity - used;
        this._growThreshold = Math.round((float)newCapacity * this._loadFactor);
        this._shrinkThreshold = Math.round((float)newCapacity * 0.1f);
    }

    private final class MapIterator
    implements IMapIterator<K, V> {
        final boolean releaseSeen;
        int nextEntry;
        int lastEntry = -1;

        MapIterator(boolean releaseSeen) {
            this.releaseSeen = releaseSeen;
            this.nextEntry = this.nextEntry(0);
        }

        int nextEntry(int index) {
            while (index < OpenHashTable.this._keys.length && OpenHashTable.this._states[index] != 1) {
                ++index;
            }
            return index;
        }

        @Override
        public boolean hasNext() {
            return this.nextEntry < OpenHashTable.this._keys.length;
        }

        @Override
        public int next() {
            int curEntry;
            if (this.releaseSeen) {
                this.free(this.lastEntry);
            }
            if (!this.hasNext()) {
                return -1;
            }
            this.lastEntry = curEntry = this.nextEntry;
            this.nextEntry = this.nextEntry(curEntry + 1);
            return curEntry;
        }

        @Override
        public K getKey() {
            if (this.lastEntry == -1) {
                throw new IllegalStateException();
            }
            return OpenHashTable.this._keys[this.lastEntry];
        }

        @Override
        public V getValue() {
            if (this.lastEntry == -1) {
                throw new IllegalStateException();
            }
            return OpenHashTable.this._values[this.lastEntry];
        }

        @Override
        public <T extends Copyable<V>> void getValue(T probe) {
            probe.copyFrom(this.getValue());
        }

        private void free(int index) {
            if (index < 0) {
                return;
            }
            OpenHashTable.this._keys[index] = null;
            OpenHashTable.this._values[index] = null;
            OpenHashTable.this._states[index] = 0;
        }
    }
}

