/*
 * Decompiled with CFR 0.152.
 */
package hivemall.utils.math;

import hivemall.utils.collections.lists.DoubleArrayList;
import hivemall.utils.lang.Preconditions;
import hivemall.utils.math.MathUtils;
import java.util.Arrays;
import javax.annotation.Nonnull;
import org.apache.commons.math3.linear.Array2DRowRealMatrix;
import org.apache.commons.math3.linear.ArrayRealVector;
import org.apache.commons.math3.linear.BlockRealMatrix;
import org.apache.commons.math3.linear.DecompositionSolver;
import org.apache.commons.math3.linear.DefaultRealMatrixPreservingVisitor;
import org.apache.commons.math3.linear.LUDecomposition;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.linear.RealVector;
import org.apache.commons.math3.linear.SingularMatrixException;
import org.apache.commons.math3.linear.SingularValueDecomposition;

public final class MatrixUtils {
    private MatrixUtils() {
    }

    @Nonnull
    public static double[] aryule(@Nonnull double[] R, @Nonnull double[] A, int order) {
        Preconditions.checkArgument(R.length > order, "|R| MUST be greater than or equals to " + order + ": " + R.length);
        Preconditions.checkArgument(A.length >= order + 1, "|A| MUST be greater than or equals to " + (order + 1) + ": " + A.length);
        double[] E = new double[order + 1];
        A[0] = 1.0;
        E[0] = R[0];
        for (int k = 0; k < order; ++k) {
            double lambda = 0.0;
            for (int j = 0; j <= k; ++j) {
                lambda -= A[j] * R[k + 1 - j];
            }
            double Ek = E[k];
            lambda = Ek == 0.0 ? 0.0 : (lambda /= Ek);
            int n = 0;
            int last = (k + 1) / 2;
            while (n <= last) {
                int i = k + 1 - n;
                double tmp = A[i] + lambda * A[n];
                int n2 = n++;
                A[n2] = A[n2] + lambda * A[i];
                A[i] = tmp;
            }
            E[k + 1] = Ek * (1.0 - lambda * lambda);
        }
        for (int i = 0; i < order + 1; ++i) {
            A[i] = -A[i];
        }
        return E;
    }

    @Deprecated
    @Nonnull
    public static double[] aryule2(@Nonnull double[] R, @Nonnull double[] A, int order) {
        Preconditions.checkArgument(R.length > order, "|C| MUST be greater than or equals to " + order + ": " + R.length);
        Preconditions.checkArgument(A.length >= order + 1, "|A| MUST be greater than or equals to " + (order + 1) + ": " + A.length);
        double[] E = new double[order + 1];
        E[0] = 1.0;
        A[0] = 1.0;
        A[1] = -R[1] / R[0];
        E[1] = R[0] + R[1] * A[1];
        for (int k = 1; k < order; ++k) {
            int i;
            double lambda = 0.0;
            for (int j = 0; j <= k; ++j) {
                lambda -= A[j] * R[k + 1 - j];
            }
            lambda /= E[k];
            double[] U = new double[k + 2];
            double[] V = new double[k + 2];
            U[0] = 1.0;
            for (i = 1; i <= k; ++i) {
                U[i] = A[i];
                V[k + 1 - i] = A[i];
            }
            V[k + 1] = 1.0;
            int threshold = k + 2;
            for (i = 0; i < threshold; ++i) {
                A[i] = U[i] + lambda * V[i];
            }
            E[k + 1] = E[k] * (1.0 - lambda * lambda);
        }
        for (int i = 0; i < order + 1; ++i) {
            A[i] = -A[i];
        }
        return E;
    }

    @Nonnull
    public static double[] arburg(@Nonnull double[] X, @Nonnull double[] A, int order) {
        int i;
        Preconditions.checkArgument(X.length > order, "|X| MUST be greater than or equals to " + order + ": " + X.length);
        Preconditions.checkArgument(A.length >= order + 1, "|A| MUST be greater than or equals to " + (order + 1) + ": " + A.length);
        int nDataPoints = X.length;
        double[] E = new double[order + 1];
        E[0] = 0.0;
        for (int i2 = 0; i2 < nDataPoints; ++i2) {
            E[0] = E[0] + X[i2] * X[i2];
        }
        int currentErrorSequenceSize = nDataPoints - 1;
        double[] F = new double[currentErrorSequenceSize];
        double[] B = new double[currentErrorSequenceSize];
        for (i = 0; i < currentErrorSequenceSize; ++i) {
            F[i] = X[i + 1];
            B[i] = X[i];
        }
        A[0] = 1.0;
        for (i = 0; i < order; ++i) {
            int j;
            double numerator = 0.0;
            double denominator = 0.0;
            for (int j2 = 0; j2 < currentErrorSequenceSize; ++j2) {
                numerator += F[j2] * B[j2];
                denominator += F[j2] * F[j2] + B[j2] * B[j2];
            }
            numerator *= 2.0;
            double g = 0.0;
            if (denominator != 0.0) {
                g = numerator / denominator;
            }
            double[] prevA = new double[i];
            for (j = 0; j < i; ++j) {
                prevA[j] = A[j + 1];
            }
            A[1] = g;
            for (j = 1; j < i + 1; ++j) {
                A[j + 1] = prevA[j - 1] - g * prevA[i - j];
            }
            E[i + 1] = E[i] * (1.0 - g * g);
            double[] prevF = new double[currentErrorSequenceSize];
            for (int j3 = 0; j3 < currentErrorSequenceSize; ++j3) {
                prevF[j3] = F[j3];
            }
            int nextErrorSequenceSize = nDataPoints - i - 2;
            for (int j4 = 0; j4 < nextErrorSequenceSize; ++j4) {
                F[j4] = prevF[j4 + 1] - g * B[j4 + 1];
                B[j4] = B[j4] - g * prevF[j4];
            }
            currentErrorSequenceSize = nextErrorSequenceSize;
        }
        int mid = order / 2 + 1;
        for (i = 1; i < mid; ++i) {
            double tmp = A[i];
            A[i] = A[order + 1 - i];
            A[order + 1 - i] = tmp;
        }
        for (i = 0; i < order + 1; ++i) {
            A[i] = -A[i];
        }
        return E;
    }

    @Nonnull
    public static RealMatrix[][] toeplitz(@Nonnull RealMatrix[] c, int dim) {
        Preconditions.checkArgument(dim >= 1, "Invalid dimension: " + dim);
        Preconditions.checkArgument(c.length >= dim, "|c| must be greater than " + dim + ": " + c.length);
        RealMatrix c0 = c[0];
        RealMatrix[][] toeplitz = new RealMatrix[dim][dim];
        for (int row = 0; row < dim; ++row) {
            toeplitz[row][row] = c0;
            for (int col = 0; col < dim; ++col) {
                if (row < col) {
                    toeplitz[row][col] = c[col - row].transpose();
                    continue;
                }
                if (row <= col) continue;
                toeplitz[row][col] = c[row - col];
            }
        }
        return toeplitz;
    }

    @Nonnull
    public static double[][] toeplitz(@Nonnull double[] c) {
        return MatrixUtils.toeplitz(c, c.length);
    }

    @Nonnull
    public static double[][] toeplitz(@Nonnull double[] c, int dim) {
        Preconditions.checkArgument(dim >= 1, "Invalid dimension: " + dim);
        Preconditions.checkArgument(c.length >= dim, "|c| must be greater than " + dim + ": " + c.length);
        double c0 = c[0];
        double[][] toeplitz = new double[dim][dim];
        for (int row = 0; row < dim; ++row) {
            toeplitz[row][row] = c0;
            for (int col = 0; col < dim; ++col) {
                if (row < col) {
                    toeplitz[row][col] = c[col - row];
                    continue;
                }
                if (row <= col) continue;
                toeplitz[row][col] = c[row - col];
            }
        }
        return toeplitz;
    }

    @Nonnull
    public static double[] flatten(@Nonnull RealMatrix[][] grid) {
        Preconditions.checkArgument(grid.length >= 1, "The number of rows must be greater than 1");
        Preconditions.checkArgument(grid[0].length >= 1, "The number of cols must be greater than 1");
        int rows = grid.length;
        int cols = grid[0].length;
        RealMatrix grid00 = grid[0][0];
        Preconditions.checkNotNull(grid00);
        int cellRows = grid00.getRowDimension();
        int cellCols = grid00.getColumnDimension();
        final DoubleArrayList list = new DoubleArrayList(rows * cols * cellRows * cellCols);
        DefaultRealMatrixPreservingVisitor visitor = new DefaultRealMatrixPreservingVisitor(){

            @Override
            public void visit(int row, int column, double value) {
                list.add(value);
            }
        };
        for (int row = 0; row < rows; ++row) {
            for (int col = 0; col < cols; ++col) {
                RealMatrix cell = grid[row][col];
                cell.walkInRowOrder(visitor);
            }
        }
        return list.toArray();
    }

    @Nonnull
    public static double[] flatten(@Nonnull RealMatrix[] grid) {
        Preconditions.checkArgument(grid.length >= 1, "The number of rows must be greater than 1");
        int rows = grid.length;
        RealMatrix grid0 = grid[0];
        Preconditions.checkNotNull(grid0);
        int cellRows = grid0.getRowDimension();
        int cellCols = grid0.getColumnDimension();
        final DoubleArrayList list = new DoubleArrayList(rows * cellRows * cellCols);
        DefaultRealMatrixPreservingVisitor visitor = new DefaultRealMatrixPreservingVisitor(){

            @Override
            public void visit(int row, int column, double value) {
                list.add(value);
            }
        };
        for (int row = 0; row < rows; ++row) {
            RealMatrix cell = grid[row];
            cell.walkInRowOrder(visitor);
        }
        return list.toArray();
    }

    @Nonnull
    public static RealMatrix[] unflatten(@Nonnull double[] data, int rows, int cols, int len) {
        RealMatrix[] grid = new RealMatrix[len];
        int offset = 0;
        for (int k = 0; k < len; ++k) {
            BlockRealMatrix cell = new BlockRealMatrix(rows, cols);
            grid[k] = cell;
            for (int i = 0; i < rows; ++i) {
                for (int j = 0; j < cols; ++j) {
                    if (offset >= data.length) {
                        throw new IndexOutOfBoundsException("Offset " + offset + " exceeded data.length " + data.length);
                    }
                    double value = data[offset];
                    cell.setEntry(i, j, value);
                    ++offset;
                }
            }
        }
        if (offset != data.length) {
            throw new IllegalArgumentException("Invalid data for unflatten");
        }
        return grid;
    }

    @Nonnull
    public static RealMatrix combinedMatrices(@Nonnull RealMatrix[][] grid, int dimensions) {
        Preconditions.checkArgument(grid.length >= 1, "The number of rows must be greater than 1");
        Preconditions.checkArgument(grid[0].length >= 1, "The number of cols must be greater than 1");
        Preconditions.checkArgument(dimensions > 0, "Dimension should be more than 0: ", dimensions);
        int rows = grid.length;
        int cols = grid[0].length;
        BlockRealMatrix combined = new BlockRealMatrix(rows * dimensions, cols * dimensions);
        for (int row = 0; row < grid.length; ++row) {
            for (int col = 0; col < grid[row].length; ++col) {
                combined.setSubMatrix(grid[row][col].getData(), row * dimensions, col * dimensions);
            }
        }
        return combined;
    }

    @Nonnull
    public static RealMatrix combinedMatrices(@Nonnull RealMatrix[] grid) {
        Preconditions.checkArgument(grid.length >= 1, "The number of rows must be greater than 0: " + grid.length);
        int rows = grid.length;
        int rowDims = grid[0].getRowDimension();
        int colDims = grid[0].getColumnDimension();
        BlockRealMatrix combined = new BlockRealMatrix(rows * rowDims, colDims);
        for (int row = 0; row < grid.length; ++row) {
            RealMatrix cell = grid[row];
            Preconditions.checkArgument(cell.getRowDimension() == rowDims, "Mismatch in row dimensions at row ", row);
            Preconditions.checkArgument(cell.getColumnDimension() == colDims, "Mismatch in col dimensions at row ", row);
            combined.setSubMatrix(cell.getData(), row * rowDims, 0);
        }
        return combined;
    }

    @Nonnull
    public static RealMatrix inverse(@Nonnull RealMatrix m) throws SingularMatrixException {
        return MatrixUtils.inverse(m, true);
    }

    @Nonnull
    public static RealMatrix inverse(@Nonnull RealMatrix m, boolean exact) throws SingularMatrixException {
        RealMatrix inv;
        LUDecomposition LU = new LUDecomposition(m);
        DecompositionSolver solver = LU.getSolver();
        if (exact || solver.isNonSingular()) {
            inv = solver.getInverse();
        } else {
            SingularValueDecomposition SVD = new SingularValueDecomposition(m);
            inv = SVD.getSolver().getInverse();
        }
        return inv;
    }

    public static double det(@Nonnull RealMatrix m) {
        LUDecomposition LU = new LUDecomposition(m);
        return LU.getDeterminant();
    }

    @Nonnull
    public static double[][] eye(int n) {
        double[][] eye = new double[n][n];
        for (int i = 0; i < n; ++i) {
            eye[i][i] = 1.0;
        }
        return eye;
    }

    @Nonnull
    public static RealMatrix solve(@Nonnull RealMatrix L, @Nonnull RealMatrix R) throws SingularMatrixException {
        return MatrixUtils.solve(L, R, true);
    }

    @Nonnull
    public static RealMatrix solve(@Nonnull RealMatrix L, @Nonnull RealMatrix R, boolean exact) throws SingularMatrixException {
        RealMatrix A;
        LUDecomposition LU = new LUDecomposition(R);
        DecompositionSolver solver = LU.getSolver();
        if (exact || solver.isNonSingular()) {
            A = LU.getSolver().solve(L);
        } else {
            SingularValueDecomposition SVD = new SingularValueDecomposition(R);
            A = SVD.getSolver().solve(L);
        }
        return A;
    }

    public static double power1(@Nonnull RealMatrix A, @Nonnull double[] x0, int nIter, @Nonnull double[] u, @Nonnull double[] v) {
        Preconditions.checkArgument(A.getColumnDimension() == x0.length, "Column size of A and length of x should be same");
        Preconditions.checkArgument(A.getRowDimension() == u.length, "Row size of A and length of u should be same");
        Preconditions.checkArgument(x0.length == v.length, "Length of x and u should be same");
        Preconditions.checkArgument(nIter >= 1, "Invalid number of iterations: " + nIter);
        RealMatrix AtA = A.transpose().multiply(A);
        RealVector x = new ArrayRealVector(x0);
        for (int i = 0; i < nIter; ++i) {
            x = AtA.operate(x);
        }
        double xNorm = x.getNorm();
        int n = v.length;
        for (int i = 0; i < n; ++i) {
            v[i] = x.getEntry(i) / xNorm;
        }
        ArrayRealVector Av = new ArrayRealVector(A.operate(v));
        double s = ((RealVector)Av).getNorm();
        int n2 = u.length;
        for (int i = 0; i < n2; ++i) {
            u[i] = ((RealVector)Av).getEntry(i) / s;
        }
        return s;
    }

    public static void lanczosTridiagonalization(@Nonnull RealMatrix C, @Nonnull double[] a, @Nonnull RealMatrix T) {
        Preconditions.checkArgument(Arrays.deepEquals((Object[])C.getData(), (Object[])C.transpose().getData()), "Target matrix C must be a symmetric matrix");
        Preconditions.checkArgument(C.getColumnDimension() == a.length, "Column size of A and length of a should be same");
        Preconditions.checkArgument(T.getRowDimension() == T.getColumnDimension(), "T must be a square matrix");
        int s = T.getRowDimension();
        T.setSubMatrix(new double[s][s], 0, 0);
        RealVector a0 = new ArrayRealVector(a.length);
        RealVector r = new ArrayRealVector(a);
        double beta0 = 1.0;
        for (int i = 0; i < s; ++i) {
            RealVector a1 = r.mapDivide(beta0);
            RealVector Ca1 = C.operate(a1);
            double alpha1 = a1.dotProduct(Ca1);
            r = Ca1.add(a1.mapMultiply(-1.0 * alpha1)).add(a0.mapMultiply(-1.0 * beta0));
            double beta1 = r.getNorm();
            T.setEntry(i, i, alpha1);
            if (i - 1 >= 0) {
                T.setEntry(i, i - 1, beta0);
            }
            if (i + 1 < s) {
                T.setEntry(i, i + 1, beta1);
            }
            a0 = a1.copy();
            beta0 = beta1;
        }
    }

    public static void tridiagonalQR(@Nonnull RealMatrix T, @Nonnull RealMatrix R, @Nonnull RealMatrix Qt) {
        int n = T.getRowDimension();
        Preconditions.checkArgument(n == R.getRowDimension() && n == R.getColumnDimension(), "T and R must be the same shape");
        Preconditions.checkArgument(n == Qt.getRowDimension() && n == Qt.getColumnDimension(), "T and Qt must be the same shape");
        R.setSubMatrix(T.getData(), 0, 0);
        Qt.setSubMatrix(MatrixUtils.eye(n), 0, 0);
        for (int i = 0; i < n - 1; ++i) {
            RealVector x = T.getSubMatrix(i, i + 1, i, i).getColumnVector(0);
            x = MatrixUtils.unitL2norm(x);
            RealMatrix subR = R.getSubMatrix(i, i + 1, 0, n - 1);
            R.setSubMatrix(subR.subtract(x.outerProduct(subR.preMultiply(x)).scalarMultiply(2.0)).getData(), i, 0);
            RealMatrix subQt = Qt.getSubMatrix(i, i + 1, 0, n - 1);
            Qt.setSubMatrix(subQt.subtract(x.outerProduct(subQt.preMultiply(x)).scalarMultiply(2.0)).getData(), i, 0);
        }
    }

    @Nonnull
    static RealVector unitL2norm(@Nonnull RealVector x) {
        double x0 = x.getEntry(0);
        double sign = MathUtils.sign(x0);
        x.setEntry(0, x0 + sign * x.getNorm());
        return x.unitVector();
    }

    public static void tridiagonalEigen(@Nonnull RealMatrix T, @Nonnull int nIter, @Nonnull double[] eigvals, @Nonnull RealMatrix eigvecs) {
        int i;
        Preconditions.checkArgument(Arrays.deepEquals((Object[])T.getData(), (Object[])T.transpose().getData()), "Target matrix T must be a symmetric (tridiagonal) matrix");
        Preconditions.checkArgument(eigvecs.getRowDimension() == eigvecs.getColumnDimension(), "eigvecs must be a square matrix");
        Preconditions.checkArgument(T.getRowDimension() == eigvecs.getRowDimension(), "T and eigvecs must be the same shape");
        Preconditions.checkArgument(eigvals.length == eigvecs.getRowDimension(), "Number of eigenvalues and eigenvectors must be same");
        int nEig = eigvals.length;
        eigvecs.setSubMatrix(MatrixUtils.eye(nEig), 0, 0);
        RealMatrix T_ = T.copy();
        for (i = 0; i < nIter; ++i) {
            Array2DRowRealMatrix R = new Array2DRowRealMatrix(nEig, nEig);
            Array2DRowRealMatrix Qt = new Array2DRowRealMatrix(nEig, nEig);
            MatrixUtils.tridiagonalQR(T_, R, Qt);
            RealMatrix Q = Qt.transpose();
            T_ = R.multiply(Q);
            eigvecs.setSubMatrix(eigvecs.multiply(Q).getData(), 0, 0);
        }
        for (i = 0; i < nEig; ++i) {
            eigvals[i] = T_.getEntry(i, i);
        }
    }
}

