/*
 * Decompiled with CFR 0.152.
 */
package smile.clustering;

import smile.clustering.KMeans;
import smile.math.Math;
import smile.math.matrix.EigenValueDecomposition;

public class SpectralClustering {
    private int k;
    private int[] y;
    private int[] size;
    private double sigma;
    private double distortion;

    public SpectralClustering(double[][] W, int k) {
        if (k < 2) {
            throw new IllegalArgumentException("Invalid number of clusters: " + k);
        }
        this.k = k;
        int n = W.length;
        for (int i = 0; i < n; ++i) {
            if (W[i].length != n) {
                throw new IllegalArgumentException("The adjacency matrix is not square.");
            }
            if (W[i][i] != 0.0) {
                throw new IllegalArgumentException(String.format("Vertex %d has self loop: ", i));
            }
            for (int j = 0; j < i; ++j) {
                if (W[i][j] != W[j][i]) {
                    throw new IllegalArgumentException("The adjacency matrix is not symmetric.");
                }
                if (!(W[i][j] < 0.0)) continue;
                throw new IllegalArgumentException("Negative entry of adjacency matrix: " + W[i][j]);
            }
        }
        double[] D = new double[n];
        for (int i = 0; i < n; ++i) {
            for (int j = 0; j < n; ++j) {
                int n2 = i;
                D[n2] = D[n2] + W[i][j];
            }
            if (D[i] == 0.0) {
                throw new IllegalArgumentException("Isolated vertex: " + i);
            }
            D[i] = 1.0 / Math.sqrt(D[i]);
        }
        double[][] L = new double[n][n];
        for (int i = 0; i < n; ++i) {
            for (int j = 0; j < i; ++j) {
                L[i][j] = D[i] * W[i][j] * D[j];
                L[j][i] = L[i][j];
            }
        }
        EigenValueDecomposition eigen = Math.eigen(L, k);
        double[][] Y = eigen.getEigenVectors();
        for (int i = 0; i < n; ++i) {
            Math.unitize2(Y[i]);
        }
        KMeans kmeans = new KMeans(Y, k);
        this.distortion = kmeans.distortion;
        this.y = kmeans.getClusterLabel();
        this.size = kmeans.getClusterSize();
    }

    public SpectralClustering(double[][] data, int k, double sigma) {
        if (k < 2) {
            throw new IllegalArgumentException("Invalid number of clusters: " + k);
        }
        if (sigma <= 0.0) {
            throw new IllegalArgumentException("Invalid standard deviation of Gaussian kernel: " + sigma);
        }
        this.k = k;
        this.sigma = sigma;
        int n = data.length;
        double gamma = -0.5 / (sigma * sigma);
        double[][] W = new double[n][n];
        for (int i = 0; i < n; ++i) {
            for (int j = 0; j < i; ++j) {
                W[i][j] = Math.exp(gamma * Math.squaredDistance(data[i], data[j]));
                W[j][i] = W[i][j];
            }
        }
        double[] D = new double[n];
        for (int i = 0; i < n; ++i) {
            for (int j = 0; j < n; ++j) {
                int n2 = i;
                D[n2] = D[n2] + W[i][j];
            }
            if (D[i] < 1.0E-5) {
                System.err.format("Small D[%d] = %f. The data may contain outliers.\n", i, D[i]);
            }
            D[i] = 1.0 / Math.sqrt(D[i]);
        }
        double[][] L = W;
        for (int i = 0; i < n; ++i) {
            for (int j = 0; j < i; ++j) {
                L[i][j] = D[i] * W[i][j] * D[j];
                L[j][i] = L[i][j];
            }
        }
        EigenValueDecomposition eigen = Math.eigen(L, k);
        double[][] Y = eigen.getEigenVectors();
        for (int i = 0; i < n; ++i) {
            Math.unitize2(Y[i]);
        }
        KMeans kmeans = new KMeans(Y, k);
        this.distortion = kmeans.distortion;
        this.y = kmeans.getClusterLabel();
        this.size = kmeans.getClusterSize();
    }

    public SpectralClustering(double[][] data, int l, int k, double sigma) {
        int i;
        if (l < k || l >= ((double[][])data).length) {
            throw new IllegalArgumentException("Invalid number of random samples: " + l);
        }
        if (k < 2) {
            throw new IllegalArgumentException("Invalid number of clusters: " + k);
        }
        if (sigma <= 0.0) {
            throw new IllegalArgumentException("Invalid standard deviation of Gaussian kernel: " + sigma);
        }
        this.k = k;
        this.sigma = sigma;
        int n = ((double[][])data).length;
        double gamma = -0.5 / (sigma * sigma);
        int[] index = Math.permutate(n);
        double[][] x = new double[n][];
        for (int i2 = 0; i2 < n; ++i2) {
            x[i2] = data[index[i2]];
        }
        data = x;
        double[][] C = new double[n][l];
        double[] D = new double[n];
        for (i = 0; i < n; ++i) {
            double sum = 0.0;
            for (int j = 0; j < n; ++j) {
                if (i == j) continue;
                double w = Math.exp(gamma * Math.squaredDistance(data[i], data[j]));
                sum += w;
                if (j >= l) continue;
                C[i][j] = w;
            }
            if (sum < 1.0E-5) {
                System.err.format("Small D[%d] = %f. The data may contain outliers.\n", i, sum);
            }
            D[i] = 1.0 / Math.sqrt(sum);
        }
        for (i = 0; i < n; ++i) {
            for (int j = 0; j < l; ++j) {
                C[i][j] = D[i] * C[i][j] * D[j];
            }
        }
        double[][] W = new double[l][l];
        for (int i3 = 0; i3 < l; ++i3) {
            System.arraycopy(C[i3], 0, W[i3], 0, l);
        }
        EigenValueDecomposition eigen = Math.eigen(W, k);
        double[] e = eigen.getEigenValues();
        double scale = Math.sqrt((double)l / (double)n);
        for (int i4 = 0; i4 < k; ++i4) {
            if (e[i4] <= 0.0) {
                throw new IllegalStateException("Non-positive eigen value: " + e[i4]);
            }
            e[i4] = scale / e[i4];
        }
        double[][] U = eigen.getEigenVectors();
        for (int i5 = 0; i5 < l; ++i5) {
            for (int j = 0; j < k; ++j) {
                double[] dArray = U[i5];
                int n2 = j;
                dArray[n2] = dArray[n2] * e[j];
            }
        }
        double[][] Y = Math.abmm(C, U);
        for (int i6 = 0; i6 < n; ++i6) {
            Math.unitize2(Y[i6]);
        }
        KMeans kmeans = new KMeans(Y, k);
        this.distortion = kmeans.distortion;
        this.size = kmeans.getClusterSize();
        int[] label = kmeans.getClusterLabel();
        this.y = new int[n];
        for (int i7 = 0; i7 < n; ++i7) {
            this.y[index[i7]] = label[i7];
        }
    }

    public int getNumClusters() {
        return this.k;
    }

    public int[] getClusterLabel() {
        return this.y;
    }

    public int[] getClusterSize() {
        return this.size;
    }

    public double getGaussianKernelWidth() {
        return this.sigma;
    }

    public double distortion() {
        return this.distortion;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(String.format("Spectral Clustering distortion in feature space: %.5f\n", this.distortion));
        sb.append(String.format("Clusters of %d data points:\n", this.y.length));
        for (int i = 0; i < this.k; ++i) {
            int r = (int)Math.round(1000.0 * (double)this.size[i] / (double)this.y.length);
            sb.append(String.format("%3d\t%5d (%2d.%1d%%)\n", i, this.size[i], r / 10, r % 10));
        }
        return sb.toString();
    }
}

