/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */
package org.apache.http.impl.client.cache;

import java.io.File;

import org.apache.http.client.cache.HttpCacheStorage;
import org.apache.http.client.cache.ResourceFactory;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.execchain.ClientExecChain;

/**
 * @since (4.3)
 */
public class CachingHttpClientBuilder extends HttpClientBuilder {

    private ResourceFactory resourceFactory;
    private HttpCacheStorage storage;
    private File cacheDir;
    private CacheConfig cacheConfig;
    private SchedulingStrategy schedulingStrategy;

    public static CachingHttpClientBuilder create() {
        return new CachingHttpClientBuilder();
    }

    protected CachingHttpClientBuilder() {
        super();
    }

    public final CachingHttpClientBuilder setResourceFactory(
            final ResourceFactory resourceFactory) {
        this.resourceFactory = resourceFactory;
        return this;
    }

    public final CachingHttpClientBuilder setHttpCacheStorage(
            final HttpCacheStorage storage) {
        this.storage = storage;
        return this;
    }

    public final CachingHttpClientBuilder setCacheDir(
            final File cacheDir) {
        this.cacheDir = cacheDir;
        return this;
    }

    public final CachingHttpClientBuilder setCacheConfig(
            final CacheConfig cacheConfig) {
        this.cacheConfig = cacheConfig;
        return this;
    }

    public final CachingHttpClientBuilder setSchedulingStrategy(
            final SchedulingStrategy schedulingStrategy) {
        this.schedulingStrategy = schedulingStrategy;
        return this;
    }

    @Override
    protected ClientExecChain decorateMainExec(final ClientExecChain mainExec) {
        final CacheConfig config = this.cacheConfig != null ? this.cacheConfig : CacheConfig.DEFAULT;
        ResourceFactory resourceFactory = this.resourceFactory;
        if (resourceFactory == null) {
            if (this.cacheDir == null) {
                resourceFactory = new HeapResourceFactory();
            } else {
                resourceFactory = new FileResourceFactory(cacheDir);
            }
        }
        HttpCacheStorage storage = this.storage;
        if (storage == null) {
            if (this.cacheDir == null) {
                storage = new BasicHttpCacheStorage(cacheConfig);
            } else {
                final ManagedHttpCacheStorage managedStorage = new ManagedHttpCacheStorage(cacheConfig);
                addCloseable(managedStorage);
                storage = managedStorage;
            }
            storage = new BasicHttpCacheStorage(cacheConfig);
        }
        final AsynchronousValidator revalidator = createAsynchronousRevalidator(config);
        return new CachingExec(mainExec,
                new BasicHttpCache(resourceFactory, storage, config), config, revalidator);
    }

    private AsynchronousValidator createAsynchronousRevalidator(final CacheConfig config) {
        if (config.getAsynchronousWorkersMax() > 0) {
            final SchedulingStrategy configuredSchedulingStrategy = createSchedulingStrategy(config);
            final AsynchronousValidator revalidator = new AsynchronousValidator(configuredSchedulingStrategy);
            addCloseable(revalidator);
            return revalidator;
        }
        return null;
    }

    private SchedulingStrategy createSchedulingStrategy(final CacheConfig config) {
        return schedulingStrategy != null ? schedulingStrategy : new ImmediateSchedulingStrategy(config);
    }

}
