/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.table.action.rollback;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.function.BiPredicate;
import java.util.stream.Collectors;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hudi.avro.model.HoodieRollbackMetadata;
import org.apache.hudi.avro.model.HoodieRollbackPlan;
import org.apache.hudi.client.heartbeat.HoodieHeartbeatClient;
import org.apache.hudi.client.transaction.TransactionManager;
import org.apache.hudi.common.HoodieRollbackStat;
import org.apache.hudi.common.bootstrap.index.BootstrapIndex;
import org.apache.hudi.common.engine.HoodieEngineContext;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.table.timeline.HoodieActiveTimeline;
import org.apache.hudi.common.table.timeline.HoodieInstant;
import org.apache.hudi.common.table.timeline.HoodieTimeline;
import org.apache.hudi.common.table.timeline.TimelineMetadataUtils;
import org.apache.hudi.common.util.ClusteringUtils;
import org.apache.hudi.common.util.HoodieTimer;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.config.HoodieWriteConfig;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.exception.HoodieRollbackException;
import org.apache.hudi.metadata.HoodieTableMetadata;
import org.apache.hudi.table.HoodieTable;
import org.apache.hudi.table.action.BaseActionExecutor;
import org.apache.hudi.table.action.rollback.BaseRollbackHelper;
import org.apache.hudi.table.action.rollback.RollbackUtils;
import org.apache.hudi.table.marker.WriteMarkersFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class BaseRollbackActionExecutor<T, I, K, O>
extends BaseActionExecutor<T, I, K, O, HoodieRollbackMetadata> {
    private static final Logger LOG = LoggerFactory.getLogger(BaseRollbackActionExecutor.class);
    protected final HoodieInstant instantToRollback;
    protected final boolean deleteInstants;
    protected final boolean skipTimelinePublish;
    private final TransactionManager txnManager;
    private final boolean skipLocking;
    protected HoodieInstant resolvedInstant;

    public BaseRollbackActionExecutor(HoodieEngineContext context, HoodieWriteConfig config, HoodieTable<T, I, K, O> table, String instantTime, HoodieInstant instantToRollback, boolean deleteInstants, boolean skipLocking, boolean isRestore) {
        this(context, config, table, instantTime, instantToRollback, deleteInstants, false, skipLocking, isRestore);
    }

    public BaseRollbackActionExecutor(HoodieEngineContext context, HoodieWriteConfig config, HoodieTable<T, I, K, O> table, String instantTime, HoodieInstant instantToRollback, boolean deleteInstants, boolean skipTimelinePublish, boolean skipLocking, boolean isRestore) {
        super(context, config, table, instantTime);
        this.instantToRollback = instantToRollback;
        this.resolvedInstant = instantToRollback;
        this.deleteInstants = deleteInstants;
        this.skipTimelinePublish = skipTimelinePublish;
        this.skipLocking = skipLocking;
        this.txnManager = new TransactionManager(config, (FileSystem)table.getMetaClient().getFs());
    }

    protected abstract List<HoodieRollbackStat> executeRollback(HoodieRollbackPlan var1) throws IOException;

    private HoodieRollbackMetadata runRollback(HoodieTable<T, I, K, O> table, HoodieInstant rollbackInstant, HoodieRollbackPlan rollbackPlan) {
        ValidationUtils.checkArgument((rollbackInstant.getState().equals((Object)HoodieInstant.State.REQUESTED) || rollbackInstant.getState().equals((Object)HoodieInstant.State.INFLIGHT) ? 1 : 0) != 0);
        HoodieInstant inflightInstant = rollbackInstant.isRequested() ? table.getActiveTimeline().transitionRollbackRequestedToInflight(rollbackInstant) : rollbackInstant;
        HoodieTimer rollbackTimer = HoodieTimer.start();
        List<HoodieRollbackStat> stats = this.doRollbackAndGetStats(rollbackPlan);
        HoodieRollbackMetadata rollbackMetadata = TimelineMetadataUtils.convertRollbackMetadata((String)this.instantTime, (Option)Option.of((Object)rollbackTimer.endTimer()), Collections.singletonList(this.instantToRollback), stats);
        this.finishRollback(inflightInstant, rollbackMetadata);
        WriteMarkersFactory.get(this.config.getMarkersType(), table, this.instantToRollback.getTimestamp()).quietDeleteMarkerDir(this.context, this.config.getMarkersDeleteParallelism());
        return rollbackMetadata;
    }

    @Override
    public HoodieRollbackMetadata execute() {
        this.table.getMetaClient().reloadActiveTimeline();
        Option rollbackInstant = this.table.getRollbackTimeline().filterInflightsAndRequested().filter(instant -> instant.getTimestamp().equals(this.instantTime)).firstInstant();
        if (!rollbackInstant.isPresent()) {
            throw new HoodieRollbackException("No pending rollback instants found to execute rollback");
        }
        try {
            HoodieRollbackPlan rollbackPlan = RollbackUtils.getRollbackPlan(this.table.getMetaClient(), (HoodieInstant)rollbackInstant.get());
            return this.runRollback(this.table, (HoodieInstant)rollbackInstant.get(), rollbackPlan);
        }
        catch (IOException e) {
            throw new HoodieIOException("Failed to fetch rollback plan for commit " + this.instantTime, e);
        }
    }

    private void validateSavepointRollbacks() {
        List<String> savepoints = this.table.getCompletedSavepointTimeline().getInstantsAsStream().map(HoodieInstant::getTimestamp).collect(Collectors.toList());
        savepoints.forEach(s -> {
            if (s.contains(this.instantToRollback.getTimestamp())) {
                throw new HoodieRollbackException("Could not rollback a savepointed commit. Delete savepoint first before rolling back" + s);
            }
        });
    }

    private void validateRollbackCommitSequence() {
        if (this.config.getFailedWritesCleanPolicy().isEager() && !HoodieTableMetadata.isMetadataTable((String)this.config.getBasePath())) {
            String instantTimeToRollback = this.instantToRollback.getTimestamp();
            HoodieTimeline commitTimeline = this.table.getCompletedCommitsTimeline();
            HoodieTimeline inflightAndRequestedCommitTimeline = this.table.getPendingCommitTimeline();
            if (instantTimeToRollback != null && !commitTimeline.empty() && !commitTimeline.findInstantsAfter(instantTimeToRollback, Integer.MAX_VALUE).empty()) {
                try {
                    if (!HoodieHeartbeatClient.heartbeatExists((FileSystem)this.table.getMetaClient().getFs(), this.config.getBasePath(), instantTimeToRollback).booleanValue()) {
                        throw new HoodieRollbackException("Found commits after time :" + instantTimeToRollback + ", please rollback greater commits first");
                    }
                }
                catch (IOException io) {
                    throw new HoodieRollbackException("Unable to rollback commits ", io);
                }
            }
            List inflights = inflightAndRequestedCommitTimeline.getInstantsAsStream().filter(instant -> {
                if (!instant.getAction().equals("replacecommit")) {
                    return true;
                }
                return !ClusteringUtils.isPendingClusteringInstant((HoodieTableMetaClient)this.table.getMetaClient(), (HoodieInstant)instant);
            }).map(HoodieInstant::getTimestamp).collect(Collectors.toList());
            if (instantTimeToRollback != null && !inflights.isEmpty() && inflights.indexOf(instantTimeToRollback) != inflights.size() - 1) {
                throw new HoodieRollbackException("Found in-flight commits after time :" + instantTimeToRollback + ", please rollback greater commits first");
            }
        }
    }

    private void rollBackIndex() {
        if (!this.table.getIndex().rollbackCommit(this.instantToRollback.getTimestamp())) {
            throw new HoodieRollbackException("Rollback index changes failed, for time :" + this.instantToRollback);
        }
        LOG.info("Index rolled back for commits " + this.instantToRollback);
    }

    public List<HoodieRollbackStat> doRollbackAndGetStats(HoodieRollbackPlan hoodieRollbackPlan) {
        String instantTimeToRollback = this.instantToRollback.getTimestamp();
        boolean isPendingCompaction = Objects.equals("compaction", this.instantToRollback.getAction()) && !this.instantToRollback.isCompleted();
        boolean isPendingClustering = Objects.equals("replacecommit", this.instantToRollback.getAction()) && !this.instantToRollback.isCompleted() && ClusteringUtils.getClusteringPlan((HoodieTableMetaClient)this.table.getMetaClient(), (HoodieInstant)this.instantToRollback).isPresent();
        this.validateSavepointRollbacks();
        if (!isPendingCompaction && !isPendingClustering) {
            this.validateRollbackCommitSequence();
        }
        this.backupRollbackInstantsIfNeeded();
        try {
            List<HoodieRollbackStat> stats = this.executeRollback(hoodieRollbackPlan);
            LOG.info("Rolled back inflight instant " + instantTimeToRollback);
            if (!isPendingCompaction) {
                this.rollBackIndex();
            }
            return stats;
        }
        catch (IOException e) {
            throw new HoodieIOException("Unable to execute rollback ", e);
        }
    }

    protected List<HoodieRollbackStat> executeRollback(HoodieInstant instantToRollback, HoodieRollbackPlan rollbackPlan) {
        return new BaseRollbackHelper(this.table.getMetaClient(), this.config).performRollback(this.context, instantToRollback, rollbackPlan.getRollbackRequests());
    }

    protected void finishRollback(HoodieInstant inflightInstant, HoodieRollbackMetadata rollbackMetadata) throws HoodieIOException {
        boolean enableLocking = !this.skipLocking && !this.skipTimelinePublish;
        try {
            if (enableLocking) {
                this.txnManager.beginTransaction((Option<HoodieInstant>)Option.of((Object)inflightInstant), (Option<HoodieInstant>)Option.empty());
            }
            if (!this.skipTimelinePublish) {
                this.writeTableMetadata(rollbackMetadata);
            }
            this.deleteInflightAndRequestedInstant(this.deleteInstants, this.table.getActiveTimeline(), this.resolvedInstant);
            if (!this.skipTimelinePublish) {
                this.table.getActiveTimeline().transitionRollbackInflightToComplete(inflightInstant, TimelineMetadataUtils.serializeRollbackMetadata((HoodieRollbackMetadata)rollbackMetadata));
                LOG.info("Rollback of Commits " + rollbackMetadata.getCommitsRollback() + " is complete");
            }
        }
        catch (IOException e) {
            throw new HoodieIOException("Error executing rollback at instant " + this.instantTime, e);
        }
        finally {
            if (enableLocking) {
                this.txnManager.endTransaction((Option<HoodieInstant>)Option.of((Object)inflightInstant));
            }
        }
    }

    protected void deleteInflightAndRequestedInstant(boolean deleteInstant, HoodieActiveTimeline activeTimeline, HoodieInstant instantToBeDeleted) {
        if (deleteInstant) {
            LOG.info("Deleting instant=" + instantToBeDeleted);
            activeTimeline.deletePending(instantToBeDeleted);
            if (instantToBeDeleted.isInflight() && !this.table.getMetaClient().getTimelineLayoutVersion().isNullVersion()) {
                instantToBeDeleted = new HoodieInstant(HoodieInstant.State.REQUESTED, instantToBeDeleted.getAction(), instantToBeDeleted.getTimestamp());
                activeTimeline.deletePending(instantToBeDeleted);
            }
            LOG.info("Deleted pending commit " + instantToBeDeleted);
        } else {
            LOG.warn("Rollback finished without deleting inflight instant file. Instant=" + instantToBeDeleted);
        }
    }

    protected void dropBootstrapIndexIfNeeded(HoodieInstant instantToRollback) {
        if (HoodieTimeline.compareTimestamps((String)instantToRollback.getTimestamp(), (BiPredicate)HoodieTimeline.EQUALS, (String)"00000000000001")) {
            LOG.info("Dropping bootstrap index as metadata bootstrap commit is getting rolled back !!");
            BootstrapIndex.getBootstrapIndex((HoodieTableMetaClient)this.table.getMetaClient()).dropIndex();
        }
    }

    private void backupRollbackInstantsIfNeeded() {
        if (!this.config.shouldBackupRollbacks()) {
            return;
        }
        Path backupDir = new Path(this.config.getRollbackBackupDirectory());
        if (!backupDir.isAbsolute()) {
            backupDir = new Path(this.table.getMetaClient().getMetaPath(), this.config.getRollbackBackupDirectory());
        }
        HoodieActiveTimeline activeTimeline = this.table.getActiveTimeline();
        ArrayList<HoodieInstant> instantsToBackup = new ArrayList<HoodieInstant>(3);
        instantsToBackup.add(this.instantToRollback);
        if (this.instantToRollback.isCompleted()) {
            instantsToBackup.add(HoodieTimeline.getInflightInstant((HoodieInstant)this.instantToRollback, (HoodieTableMetaClient)this.table.getMetaClient()));
            instantsToBackup.add(HoodieTimeline.getRequestedInstant((HoodieInstant)this.instantToRollback));
        }
        if (this.instantToRollback.isInflight()) {
            instantsToBackup.add(HoodieTimeline.getRequestedInstant((HoodieInstant)this.instantToRollback));
        }
        for (HoodieInstant instant : instantsToBackup) {
            try {
                activeTimeline.copyInstant(instant, backupDir);
                LOG.info(String.format("Copied instant %s to backup dir %s during rollback at %s", instant, backupDir, this.instantTime));
            }
            catch (HoodieIOException e) {
                LOG.warn("Failed to backup rollback instant: " + e.getMessage());
            }
        }
    }
}

