/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.client.heartbeat;

import java.io.IOException;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.hudi.client.heartbeat.HeartbeatUtils;
import org.apache.hudi.common.heartbeat.HoodieHeartbeatUtils;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieHeartbeatException;
import org.apache.hudi.storage.HoodieStorage;
import org.apache.hudi.storage.StoragePath;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class HoodieHeartbeatClient
implements AutoCloseable,
Serializable {
    private static final Logger LOG = LoggerFactory.getLogger(HoodieHeartbeatClient.class);
    private final transient HoodieStorage storage;
    private final String basePath;
    private final String heartbeatFolderPath;
    private final Long heartbeatIntervalInMs;
    private final Long maxAllowableHeartbeatIntervalInMs;
    private final Map<String, Heartbeat> instantToHeartbeatMap;

    public HoodieHeartbeatClient(HoodieStorage storage, String basePath, Long heartbeatIntervalInMs, Integer numTolerableHeartbeatMisses) {
        ValidationUtils.checkArgument((heartbeatIntervalInMs >= 1000L ? 1 : 0) != 0, (String)"Cannot set heartbeat lower than 1 second");
        this.storage = storage;
        this.basePath = basePath;
        this.heartbeatFolderPath = HoodieTableMetaClient.getHeartbeatFolderPath((String)basePath);
        this.heartbeatIntervalInMs = heartbeatIntervalInMs;
        this.maxAllowableHeartbeatIntervalInMs = this.heartbeatIntervalInMs * (long)numTolerableHeartbeatMisses.intValue();
        this.instantToHeartbeatMap = new ConcurrentHashMap<String, Heartbeat>();
    }

    public void start(String instantTime) {
        LOG.info("Received request to start heartbeat for instant time {}", (Object)instantTime);
        Heartbeat heartbeat = this.instantToHeartbeatMap.get(instantTime);
        ValidationUtils.checkArgument((heartbeat == null || heartbeat.isHeartbeatStopped() == false ? 1 : 0) != 0, (String)("Cannot restart a stopped heartbeat for " + instantTime));
        if (heartbeat == null || !heartbeat.isHeartbeatStarted().booleanValue()) {
            Heartbeat newHeartbeat = new Heartbeat();
            newHeartbeat.setHeartbeatStarted(true);
            this.instantToHeartbeatMap.put(instantTime, newHeartbeat);
            this.updateHeartbeat(instantTime);
            newHeartbeat.getTimer().scheduleAtFixedRate((TimerTask)new HeartbeatTask(instantTime), this.heartbeatIntervalInMs, (long)this.heartbeatIntervalInMs);
        }
    }

    public Heartbeat stop(String instantTime) throws HoodieException {
        Heartbeat heartbeat = this.instantToHeartbeatMap.remove(instantTime);
        if (this.isHeartbeatStarted(heartbeat)) {
            this.stopHeartbeatTimer(heartbeat);
            HeartbeatUtils.deleteHeartbeatFile(this.storage, this.basePath, instantTime);
            LOG.info("Deleted heartbeat file for instant {}", (Object)instantTime);
        }
        return heartbeat;
    }

    public void stopHeartbeatTimers() throws HoodieException {
        this.instantToHeartbeatMap.values().stream().filter(this::isHeartbeatStarted).forEach(this::stopHeartbeatTimer);
    }

    private boolean isHeartbeatStarted(Heartbeat heartbeat) {
        return heartbeat != null && heartbeat.isHeartbeatStarted() != false && heartbeat.isHeartbeatStopped() == false;
    }

    private void stopHeartbeatTimer(Heartbeat heartbeat) {
        LOG.info("Stopping heartbeat for instant {}", (Object)heartbeat.getInstantTime());
        heartbeat.getTimer().cancel();
        heartbeat.setHeartbeatStopped(true);
        LOG.info("Stopped heartbeat for instant {}", (Object)heartbeat.getInstantTime());
    }

    public static Boolean heartbeatExists(HoodieStorage storage, String basePath, String instantTime) throws IOException {
        StoragePath heartbeatFilePath = new StoragePath(HoodieTableMetaClient.getHeartbeatFolderPath((String)basePath), instantTime);
        return storage.exists(heartbeatFilePath);
    }

    public boolean isHeartbeatExpired(String instantTime) throws IOException {
        Long currentTime = System.currentTimeMillis();
        Heartbeat lastHeartbeatForWriter = this.instantToHeartbeatMap.get(instantTime);
        if (lastHeartbeatForWriter == null) {
            LOG.info("Heartbeat not found in internal map, falling back to reading from DFS");
            long lastHeartbeatForWriterTime = HoodieHeartbeatUtils.getLastHeartbeatTime((HoodieStorage)this.storage, (String)this.basePath, (String)instantTime);
            lastHeartbeatForWriter = new Heartbeat();
            lastHeartbeatForWriter.setLastHeartbeatTime(lastHeartbeatForWriterTime);
            lastHeartbeatForWriter.setInstantTime(instantTime);
            lastHeartbeatForWriter.getTimer().cancel();
        }
        if (currentTime - lastHeartbeatForWriter.getLastHeartbeatTime() > this.maxAllowableHeartbeatIntervalInMs) {
            LOG.warn("Heartbeat expired, currentTime = {}, last heartbeat = {}, heartbeat interval = {}", new Object[]{currentTime, lastHeartbeatForWriter, this.heartbeatIntervalInMs});
            return true;
        }
        return false;
    }

    private void updateHeartbeat(String instantTime) throws HoodieHeartbeatException {
        try {
            Long newHeartbeatTime = System.currentTimeMillis();
            OutputStream outputStream = this.storage.create(new StoragePath(this.heartbeatFolderPath, instantTime), true);
            outputStream.close();
            Heartbeat heartbeat = this.instantToHeartbeatMap.get(instantTime);
            if (heartbeat.getLastHeartbeatTime() != null && this.isHeartbeatExpired(instantTime)) {
                LOG.error("Aborting, missed generating heartbeat within allowable interval {} ms", (Object)this.maxAllowableHeartbeatIntervalInMs);
                Thread.currentThread().interrupt();
            }
            heartbeat.setInstantTime(instantTime);
            heartbeat.setLastHeartbeatTime(newHeartbeatTime);
            heartbeat.setNumHeartbeats(heartbeat.getNumHeartbeats() + 1);
        }
        catch (IOException io) {
            Boolean isHeartbeatStopped = this.instantToHeartbeatMap.get(instantTime).isHeartbeatStopped;
            if (isHeartbeatStopped.booleanValue()) {
                LOG.info("update heart beat failed, because the instant time {} was stopped", (Object)instantTime);
                return;
            }
            throw new HoodieHeartbeatException("Unable to generate heartbeat for instant " + instantTime, (Throwable)io);
        }
    }

    public String getHeartbeatFolderPath() {
        return this.heartbeatFolderPath;
    }

    public Heartbeat getHeartbeat(String instantTime) {
        return this.instantToHeartbeatMap.get(instantTime);
    }

    @Override
    public void close() {
        this.stopHeartbeatTimers();
        this.instantToHeartbeatMap.clear();
    }

    class HeartbeatTask
    extends TimerTask {
        private final String instantTime;

        HeartbeatTask(String instantTime) {
            this.instantTime = instantTime;
        }

        @Override
        public void run() {
            HoodieHeartbeatClient.this.updateHeartbeat(this.instantTime);
        }
    }

    static class Heartbeat {
        private String instantTime;
        private Boolean isHeartbeatStarted = false;
        private Boolean isHeartbeatStopped = false;
        private Long lastHeartbeatTime;
        private Integer numHeartbeats = 0;
        private Timer timer = new Timer(true);

        Heartbeat() {
        }

        public String getInstantTime() {
            return this.instantTime;
        }

        public void setInstantTime(String instantTime) {
            this.instantTime = instantTime;
        }

        public Boolean isHeartbeatStarted() {
            return this.isHeartbeatStarted;
        }

        public void setHeartbeatStarted(Boolean heartbeatStarted) {
            this.isHeartbeatStarted = heartbeatStarted;
        }

        public Boolean isHeartbeatStopped() {
            return this.isHeartbeatStopped;
        }

        public void setHeartbeatStopped(Boolean heartbeatStopped) {
            this.isHeartbeatStopped = heartbeatStopped;
        }

        public Long getLastHeartbeatTime() {
            return this.lastHeartbeatTime;
        }

        public void setLastHeartbeatTime(Long lastHeartbeatTime) {
            this.lastHeartbeatTime = lastHeartbeatTime;
        }

        public Integer getNumHeartbeats() {
            return this.numHeartbeats;
        }

        public void setNumHeartbeats(Integer numHeartbeats) {
            this.numHeartbeats = numHeartbeats;
        }

        public Timer getTimer() {
            return this.timer;
        }

        public void setTimer(Timer timer) {
            this.timer = timer;
        }

        public String toString() {
            return "Heartbeat{instantTime='" + this.instantTime + '\'' + ", isHeartbeatStarted=" + this.isHeartbeatStarted + ", isHeartbeatStopped=" + this.isHeartbeatStopped + ", lastHeartbeatTime=" + this.lastHeartbeatTime + ", numHeartbeats=" + this.numHeartbeats + ", timer=" + this.timer + '}';
        }
    }
}

