/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.config;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.Instant;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.function.BiConsumer;
import javax.annotation.concurrent.Immutable;
import org.apache.hudi.common.config.ConfigClassProperty;
import org.apache.hudi.common.config.ConfigGroups;
import org.apache.hudi.common.config.ConfigProperty;
import org.apache.hudi.common.config.HoodieConfig;
import org.apache.hudi.common.config.OrderedProperties;
import org.apache.hudi.common.config.TypedProperties;
import org.apache.hudi.common.model.HoodieIndexDefinition;
import org.apache.hudi.common.util.BinaryUtil;
import org.apache.hudi.common.util.ConfigUtils;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.metadata.MetadataPartitionType;
import org.apache.hudi.storage.HoodieStorage;
import org.apache.hudi.storage.StoragePath;

@ConfigClassProperty(name="Common Index Configs", groupName=ConfigGroups.Names.WRITE_CLIENT, subGroupName=ConfigGroups.SubGroupNames.INDEX, areCommonConfigs=true, description="")
@Immutable
public class HoodieIndexingConfig
extends HoodieConfig {
    public static final String INDEX_DEFINITION_FILE = "index.properties";
    public static final String INDEX_DEFINITION_FILE_BACKUP = "index.properties.backup";
    public static final ConfigProperty<String> INDEX_NAME = ConfigProperty.key("hoodie.index.name").noDefaultValue().sinceVersion("1.0.0").withDocumentation("Name of the expression index. This is also used for the partition name in the metadata table.");
    public static final ConfigProperty<String> INDEX_TYPE = ConfigProperty.key("hoodie.expression.index.type").defaultValue(MetadataPartitionType.COLUMN_STATS.name()).withValidValues(MetadataPartitionType.COLUMN_STATS.name(), MetadataPartitionType.BLOOM_FILTERS.name(), MetadataPartitionType.SECONDARY_INDEX.name()).sinceVersion("1.0.0").withDocumentation("Type of the expression index. Default is `column_stats` if there are no functions and expressions in the command. Valid options could be BITMAP, COLUMN_STATS, LUCENE, etc. If index_type is not provided, and there are functions or expressions in the command then a expression index using column stats will be created.");
    public static final ConfigProperty<String> INDEX_FUNCTION = ConfigProperty.key("hoodie.expression.index.function").noDefaultValue().sinceVersion("1.0.0").withDocumentation("Function to be used for building the expression index.");
    public static final ConfigProperty<String> INDEX_DEFINITION_CHECKSUM = ConfigProperty.key("hoodie.table.checksum").noDefaultValue().sinceVersion("1.0.0").withDocumentation("Index definition checksum is used to guard against partial writes in HDFS. It is added as the last entry in index.properties and then used to validate while reading table config.");
    private static final String INDEX_DEFINITION_CHECKSUM_FORMAT = "%s.%s";

    public static void create(HoodieStorage storage, StoragePath metadataFolder, Properties properties) throws IOException {
        if (!storage.exists(metadataFolder)) {
            storage.createDirectory(metadataFolder);
        }
        HoodieConfig hoodieConfig = new HoodieConfig(properties);
        StoragePath propertyPath = new StoragePath(metadataFolder, INDEX_DEFINITION_FILE);
        try (OutputStream outputStream = storage.create(propertyPath);){
            if (!hoodieConfig.contains(INDEX_NAME)) {
                throw new IllegalArgumentException(INDEX_NAME.key() + " property needs to be specified");
            }
            hoodieConfig.setDefaultValue(INDEX_TYPE);
            HoodieIndexingConfig.storeProperties(hoodieConfig.getProps(), outputStream);
        }
    }

    public static void update(HoodieStorage storage, StoragePath metadataFolder, Properties updatedProps) {
        HoodieIndexingConfig.modify(storage, metadataFolder, updatedProps, ConfigUtils::upsertProperties);
    }

    public static void delete(HoodieStorage storage, StoragePath metadataFolder, Set<String> deletedProps) {
        Properties props = new Properties();
        deletedProps.forEach(p -> props.setProperty((String)p, ""));
        HoodieIndexingConfig.modify(storage, metadataFolder, props, ConfigUtils::deleteProperties);
    }

    public static void recover(HoodieStorage storage, StoragePath metadataFolder) throws IOException {
        StoragePath cfgPath = new StoragePath(metadataFolder, INDEX_DEFINITION_FILE);
        StoragePath backupCfgPath = new StoragePath(metadataFolder, INDEX_DEFINITION_FILE_BACKUP);
        ConfigUtils.recoverIfNeeded(storage, cfgPath, backupCfgPath);
    }

    private static void modify(HoodieStorage storage, StoragePath metadataFolder, Properties modifyProps, BiConsumer<Properties, Properties> modifyFn) {
        StoragePath cfgPath = new StoragePath(metadataFolder, INDEX_DEFINITION_FILE);
        StoragePath backupCfgPath = new StoragePath(metadataFolder, INDEX_DEFINITION_FILE_BACKUP);
        try {
            String checksum;
            ConfigUtils.recoverIfNeeded(storage, cfgPath, backupCfgPath);
            TypedProperties props = ConfigUtils.fetchConfigs(storage, metadataFolder, INDEX_DEFINITION_FILE, INDEX_DEFINITION_FILE_BACKUP, 5, 1000);
            try (OutputStream out = storage.create(backupCfgPath, false);){
                HoodieIndexingConfig.storeProperties(props, out);
            }
            storage.deleteFile(cfgPath);
            try (OutputStream out = storage.create(cfgPath, true);){
                modifyFn.accept(props, modifyProps);
                checksum = HoodieIndexingConfig.storeProperties(props, out);
            }
            var9_12 = null;
            try (InputStream in = storage.open(cfgPath);){
                props.clear();
                props.load(in);
                if (!props.containsKey(INDEX_DEFINITION_CHECKSUM.key()) || !props.getProperty(INDEX_DEFINITION_CHECKSUM.key()).equals(checksum)) {
                    storage.deleteFile(cfgPath);
                    throw new HoodieIOException("Checksum property missing or does not match.");
                }
            }
            catch (Throwable throwable) {
                var9_12 = throwable;
                throw throwable;
            }
            storage.deleteFile(backupCfgPath);
        }
        catch (IOException e) {
            throw new HoodieIOException("Error updating table configs.", e);
        }
    }

    private static Properties getOrderedPropertiesWithTableChecksum(Properties props) {
        OrderedProperties orderedProps = new OrderedProperties(props);
        ((Properties)orderedProps).put(INDEX_DEFINITION_CHECKSUM.key(), String.valueOf(HoodieIndexingConfig.generateChecksum(props)));
        return orderedProps;
    }

    private static String storeProperties(Properties props, OutputStream outputStream) throws IOException {
        String checksum;
        if (HoodieIndexingConfig.isValidChecksum(props)) {
            checksum = props.getProperty(INDEX_DEFINITION_CHECKSUM.key());
            props.store(outputStream, "Updated at " + Instant.now());
        } else {
            Properties propsWithChecksum = HoodieIndexingConfig.getOrderedPropertiesWithTableChecksum(props);
            propsWithChecksum.store(outputStream, "Properties saved on " + Instant.now());
            checksum = propsWithChecksum.getProperty(INDEX_DEFINITION_CHECKSUM.key());
            props.setProperty(INDEX_DEFINITION_CHECKSUM.key(), checksum);
        }
        return checksum;
    }

    private static boolean isValidChecksum(Properties props) {
        return props.containsKey(INDEX_DEFINITION_CHECKSUM.key()) && HoodieIndexingConfig.validateChecksum(props);
    }

    public static long generateChecksum(Properties props) {
        if (!props.containsKey(INDEX_NAME.key())) {
            throw new IllegalArgumentException(INDEX_NAME.key() + " property needs to be specified");
        }
        String table = props.getProperty(INDEX_NAME.key());
        String database = props.getProperty(INDEX_TYPE.key(), "");
        return BinaryUtil.generateChecksum(StringUtils.getUTF8Bytes((String)String.format(INDEX_DEFINITION_CHECKSUM_FORMAT, database, table)));
    }

    public static boolean validateChecksum(Properties props) {
        return Long.parseLong(props.getProperty(INDEX_DEFINITION_CHECKSUM.key())) == HoodieIndexingConfig.generateChecksum(props);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static HoodieIndexingConfig copy(HoodieIndexingConfig expressionIndexConfig) {
        return HoodieIndexingConfig.newBuilder().fromIndexConfig(expressionIndexConfig).build();
    }

    public static HoodieIndexingConfig merge(HoodieIndexingConfig expressionIndexConfig1, HoodieIndexingConfig expressionIndexConfig2) {
        return HoodieIndexingConfig.newBuilder().fromIndexConfig(expressionIndexConfig1).fromIndexConfig(expressionIndexConfig2).build();
    }

    public static HoodieIndexingConfig fromIndexDefinition(HoodieIndexDefinition indexDefinition) {
        return HoodieIndexingConfig.newBuilder().withIndexName(indexDefinition.getIndexName()).withIndexType(indexDefinition.getIndexType()).withIndexFunction(indexDefinition.getIndexFunction()).build();
    }

    public String getIndexName() {
        return this.getString(INDEX_NAME);
    }

    public String getIndexType() {
        return this.getString(INDEX_TYPE);
    }

    public String getIndexFunction() {
        return this.getString(INDEX_FUNCTION);
    }

    public boolean isIndexUsingBloomFilter() {
        return this.getIndexType().equalsIgnoreCase(MetadataPartitionType.BLOOM_FILTERS.name());
    }

    public boolean isIndexUsingColumnStats() {
        return this.getIndexType().equalsIgnoreCase(MetadataPartitionType.COLUMN_STATS.name());
    }

    public boolean isIndexUsingRecordIndex() {
        return this.getIndexType().equalsIgnoreCase(MetadataPartitionType.RECORD_INDEX.name());
    }

    public static class Builder {
        private final HoodieIndexingConfig expressionIndexConfig = new HoodieIndexingConfig();

        public Builder fromFile(File propertiesFile) throws IOException {
            try (FileReader reader = new FileReader(propertiesFile);){
                this.expressionIndexConfig.getProps().load(reader);
                Builder builder = this;
                return builder;
            }
        }

        public Builder fromProperties(Properties props) {
            this.expressionIndexConfig.getProps().putAll((Map<?, ?>)props);
            return this;
        }

        public Builder fromIndexConfig(HoodieIndexingConfig expressionIndexConfig) {
            this.expressionIndexConfig.getProps().putAll((Map<?, ?>)expressionIndexConfig.getProps());
            return this;
        }

        public Builder withIndexName(String indexName) {
            this.expressionIndexConfig.setValue(INDEX_NAME, indexName);
            return this;
        }

        public Builder withIndexType(String indexType) {
            this.expressionIndexConfig.setValue(INDEX_TYPE, indexType);
            return this;
        }

        public Builder withIndexFunction(String indexFunction) {
            this.expressionIndexConfig.setValue(INDEX_FUNCTION, indexFunction);
            return this;
        }

        public HoodieIndexingConfig build() {
            this.expressionIndexConfig.setDefaults(HoodieIndexingConfig.class.getName());
            return this.expressionIndexConfig;
        }
    }
}

