/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.util.collection;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hudi.common.serialization.CustomSerializer;
import org.apache.hudi.common.serialization.DefaultSerializer;
import org.apache.hudi.common.util.FileIOUtils;
import org.apache.hudi.common.util.HoodieTimer;
import org.apache.hudi.common.util.SerializationUtils;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.rocksdb.AbstractImmutableNativeReference;
import org.rocksdb.ColumnFamilyDescriptor;
import org.rocksdb.ColumnFamilyHandle;
import org.rocksdb.ColumnFamilyOptions;
import org.rocksdb.DBOptions;
import org.rocksdb.InfoLogLevel;
import org.rocksdb.Options;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.rocksdb.Statistics;
import org.rocksdb.WriteBatch;
import org.rocksdb.WriteOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RocksDBDAO {
    private static final Logger LOG = LoggerFactory.getLogger(RocksDBDAO.class);
    private transient ConcurrentHashMap<String, ColumnFamilyHandle> managedHandlesMap;
    private transient ConcurrentHashMap<String, ColumnFamilyDescriptor> managedDescriptorMap;
    private transient RocksDB rocksDB;
    private boolean closed = false;
    private final String rocksDBBasePath;
    private final transient Map<String, CustomSerializer<?>> columnFamilySerializers;
    private long totalBytesWritten;

    public RocksDBDAO(String basePath, String rocksDBBasePath) {
        this(basePath, rocksDBBasePath, new HashMap());
    }

    public RocksDBDAO(String basePath, String rocksDBBasePath, Map<String, CustomSerializer<?>> columnFamilySerializers) {
        this.rocksDBBasePath = String.format("%s/%s/%s", rocksDBBasePath, URI.create(basePath).getPath().replace(":", "").replace("/", "_"), UUID.randomUUID());
        this.columnFamilySerializers = columnFamilySerializers;
        this.init();
        this.totalBytesWritten = 0L;
    }

    private RocksDB getRocksDB() {
        return this.rocksDB;
    }

    private void init() {
        try {
            LOG.info("DELETING RocksDB persisted at " + this.rocksDBBasePath);
            FileIOUtils.deleteDirectory((File)new File(this.rocksDBBasePath));
            this.managedHandlesMap = new ConcurrentHashMap();
            this.managedDescriptorMap = new ConcurrentHashMap();
            DBOptions dbOptions = new DBOptions().setCreateIfMissing(true).setCreateMissingColumnFamilies(true).setWalDir(this.rocksDBBasePath).setStatsDumpPeriodSec(300).setStatistics(new Statistics());
            dbOptions.setLogger(new org.rocksdb.Logger(dbOptions){

                protected void log(InfoLogLevel infoLogLevel, String logMsg) {
                    switch (infoLogLevel) {
                        case DEBUG_LEVEL: {
                            LOG.debug("From Rocks DB : {}", (Object)logMsg);
                            break;
                        }
                        case WARN_LEVEL: {
                            LOG.warn("From Rocks DB : {}", (Object)logMsg);
                            break;
                        }
                        case ERROR_LEVEL: 
                        case FATAL_LEVEL: {
                            LOG.error("From Rocks DB : {}", (Object)logMsg);
                            break;
                        }
                        default: {
                            LOG.info("From Rocks DB : {}", (Object)logMsg);
                        }
                    }
                }
            });
            List<ColumnFamilyDescriptor> managedColumnFamilies = this.loadManagedColumnFamilies(dbOptions);
            ArrayList managedHandles = new ArrayList(managedColumnFamilies.size());
            FileIOUtils.mkdir((File)new File(this.rocksDBBasePath));
            this.rocksDB = RocksDB.open((DBOptions)dbOptions, (String)this.rocksDBBasePath, managedColumnFamilies, managedHandles);
            ValidationUtils.checkArgument((managedHandles.size() == managedColumnFamilies.size() ? 1 : 0) != 0, (String)"Unexpected number of handles are returned");
            for (int index = 0; index < managedHandles.size(); ++index) {
                ColumnFamilyHandle handle = (ColumnFamilyHandle)managedHandles.get(index);
                ColumnFamilyDescriptor descriptor = managedColumnFamilies.get(index);
                String familyNameFromHandle = new String(handle.getName());
                String familyNameFromDescriptor = new String(descriptor.getName());
                ValidationUtils.checkArgument((boolean)familyNameFromDescriptor.equals(familyNameFromHandle), (String)"Family Handles not in order with descriptors");
                this.managedHandlesMap.put(familyNameFromHandle, handle);
                this.managedDescriptorMap.put(familyNameFromDescriptor, descriptor);
            }
        }
        catch (IOException | RocksDBException re) {
            LOG.error("Got exception opening Rocks DB instance ", re);
            throw new HoodieException(re);
        }
    }

    private List<ColumnFamilyDescriptor> loadManagedColumnFamilies(DBOptions dbOptions) throws RocksDBException {
        ArrayList<ColumnFamilyDescriptor> managedColumnFamilies = new ArrayList<ColumnFamilyDescriptor>();
        Options options = new Options(dbOptions, new ColumnFamilyOptions());
        List existing = RocksDB.listColumnFamilies((Options)options, (String)this.rocksDBBasePath);
        if (existing.isEmpty()) {
            LOG.info("No column family found. Loading default");
            managedColumnFamilies.add(RocksDBDAO.getColumnFamilyDescriptor(RocksDB.DEFAULT_COLUMN_FAMILY));
        } else {
            LOG.info("Loading column families :" + existing.stream().map(String::new).collect(Collectors.toList()));
            managedColumnFamilies.addAll(existing.stream().map(RocksDBDAO::getColumnFamilyDescriptor).collect(Collectors.toList()));
        }
        return managedColumnFamilies;
    }

    private static ColumnFamilyDescriptor getColumnFamilyDescriptor(byte[] columnFamilyName) {
        return new ColumnFamilyDescriptor(columnFamilyName, new ColumnFamilyOptions());
    }

    public void writeBatch(BatchHandler handler) {
        try (WriteBatch batch = new WriteBatch();){
            handler.apply(batch);
            this.getRocksDB().write(new WriteOptions(), batch);
        }
        catch (RocksDBException re) {
            throw new HoodieException((Throwable)re);
        }
    }

    public <T extends Serializable> void putInBatch(WriteBatch batch, String columnFamilyName, String key, T value) {
        try {
            byte[] payload = this.serializePayload(columnFamilyName, value);
            batch.put(this.managedHandlesMap.get(columnFamilyName), StringUtils.getUTF8Bytes((String)key), payload);
        }
        catch (Exception e) {
            throw new HoodieException((Throwable)e);
        }
    }

    public <K extends Serializable, T> void putInBatch(WriteBatch batch, String columnFamilyName, K key, T value) {
        try {
            byte[] keyBytes = SerializationUtils.serialize(key);
            byte[] payload = this.serializePayload(columnFamilyName, value);
            batch.put(this.managedHandlesMap.get(columnFamilyName), keyBytes, payload);
        }
        catch (Exception e) {
            throw new HoodieException((Throwable)e);
        }
    }

    public <T extends Serializable> void put(String columnFamilyName, String key, T value) {
        try {
            byte[] payload = this.serializePayload(columnFamilyName, value);
            this.getRocksDB().put(this.managedHandlesMap.get(columnFamilyName), StringUtils.getUTF8Bytes((String)key), payload);
        }
        catch (Exception e) {
            throw new HoodieException((Throwable)e);
        }
    }

    public <K extends Serializable, T> void put(String columnFamilyName, K key, T value) {
        try {
            byte[] payload = this.serializePayload(columnFamilyName, value);
            this.getRocksDB().put(this.managedHandlesMap.get(columnFamilyName), SerializationUtils.serialize(key), payload);
        }
        catch (Exception e) {
            throw new HoodieException((Throwable)e);
        }
    }

    public void deleteInBatch(WriteBatch batch, String columnFamilyName, String key) {
        try {
            batch.delete(this.managedHandlesMap.get(columnFamilyName), StringUtils.getUTF8Bytes((String)key));
        }
        catch (RocksDBException e) {
            throw new HoodieException((Throwable)e);
        }
    }

    public <K extends Serializable> void deleteInBatch(WriteBatch batch, String columnFamilyName, K key) {
        try {
            batch.delete(this.managedHandlesMap.get(columnFamilyName), SerializationUtils.serialize(key));
        }
        catch (Exception e) {
            throw new HoodieException((Throwable)e);
        }
    }

    public void delete(String columnFamilyName, String key) {
        try {
            this.getRocksDB().delete(this.managedHandlesMap.get(columnFamilyName), StringUtils.getUTF8Bytes((String)key));
        }
        catch (RocksDBException e) {
            throw new HoodieException((Throwable)e);
        }
    }

    public <K extends Serializable> void delete(String columnFamilyName, K key) {
        try {
            this.getRocksDB().delete(this.managedHandlesMap.get(columnFamilyName), SerializationUtils.serialize(key));
        }
        catch (Exception e) {
            throw new HoodieException((Throwable)e);
        }
    }

    public <T extends Serializable> T get(String columnFamilyName, String key) {
        return this.get(columnFamilyName, this.getKeyBytes(key));
    }

    public <K extends Serializable, T extends Serializable> T get(String columnFamilyName, K key) {
        return this.get(columnFamilyName, this.getKeyBytes(key));
    }

    public <K extends Serializable, T extends Serializable> T get(String columnFamilyName, byte[] key) {
        ValidationUtils.checkArgument((!this.closed ? 1 : 0) != 0);
        try {
            byte[] val = this.getRocksDB().get(this.managedHandlesMap.get(columnFamilyName), key);
            return (T)((Serializable)this.deserializePayload(columnFamilyName, val));
        }
        catch (Exception e) {
            throw new HoodieException((Throwable)e);
        }
    }

    public <T extends Serializable> Stream<Pair<String, T>> prefixSearch(String columnFamilyName, String prefix) {
        ValidationUtils.checkArgument((!this.closed ? 1 : 0) != 0);
        HoodieTimer timer = HoodieTimer.start();
        long timeTakenMicro = 0L;
        LinkedList<Pair<String, Serializable>> results = new LinkedList<Pair<String, Serializable>>();
        try (RocksIterator it = this.getRocksDB().newIterator(this.managedHandlesMap.get(columnFamilyName));){
            it.seek(StringUtils.getUTF8Bytes((String)prefix));
            while (it.isValid() && new String(it.key()).startsWith(prefix)) {
                long beginTs = System.nanoTime();
                Serializable val = (Serializable)this.deserializePayload(columnFamilyName, it.value());
                timeTakenMicro += (System.nanoTime() - beginTs) / 1000L;
                results.add(Pair.of(new String(it.key()), val));
                it.next();
            }
        }
        LOG.info("Prefix Search for (query=" + prefix + ") on " + columnFamilyName + ". Total Time Taken (msec)=" + timer.endTimer() + ". Serialization Time taken(micro)=" + timeTakenMicro + ", num entries=" + results.size());
        return results.stream();
    }

    public <T extends Serializable, R> Iterator<Pair<T, R>> iterator(String columnFamilyName) {
        return new IteratorWrapper(this.getRocksDB().newIterator(this.managedHandlesMap.get(columnFamilyName)), this.getSerializerForColumnFamily(columnFamilyName));
    }

    public <T extends Serializable> void prefixDelete(String columnFamilyName, String prefix) {
        ValidationUtils.checkArgument((!this.closed ? 1 : 0) != 0);
        LOG.info("Prefix DELETE (query=" + prefix + ") on " + columnFamilyName);
        RocksIterator it = this.getRocksDB().newIterator(this.managedHandlesMap.get(columnFamilyName));
        it.seek(StringUtils.getUTF8Bytes((String)prefix));
        String firstEntry = null;
        String lastEntry = null;
        while (it.isValid() && new String(it.key()).startsWith(prefix)) {
            String result = new String(it.key());
            it.next();
            if (firstEntry == null) {
                firstEntry = result;
            }
            lastEntry = result;
        }
        it.close();
        if (null != firstEntry) {
            try {
                this.getRocksDB().deleteRange(this.managedHandlesMap.get(columnFamilyName), StringUtils.getUTF8Bytes(firstEntry), StringUtils.getUTF8Bytes(lastEntry));
                this.getRocksDB().delete(this.managedHandlesMap.get(columnFamilyName), StringUtils.getUTF8Bytes((String)lastEntry));
            }
            catch (RocksDBException e) {
                LOG.error("Got exception performing range delete");
                throw new HoodieException((Throwable)e);
            }
        }
    }

    public void addColumnFamily(String columnFamilyName) {
        ValidationUtils.checkArgument((!this.closed ? 1 : 0) != 0);
        this.managedDescriptorMap.computeIfAbsent(columnFamilyName, colFamilyName -> {
            try {
                ColumnFamilyDescriptor descriptor = RocksDBDAO.getColumnFamilyDescriptor(StringUtils.getUTF8Bytes((String)colFamilyName));
                ColumnFamilyHandle handle = this.getRocksDB().createColumnFamily(descriptor);
                this.managedHandlesMap.put((String)colFamilyName, handle);
                return descriptor;
            }
            catch (RocksDBException e) {
                throw new HoodieException((Throwable)e);
            }
        });
    }

    public void dropColumnFamily(String columnFamilyName) {
        ValidationUtils.checkArgument((!this.closed ? 1 : 0) != 0);
        this.managedDescriptorMap.computeIfPresent(columnFamilyName, (colFamilyName, descriptor) -> {
            ColumnFamilyHandle handle = this.managedHandlesMap.get(colFamilyName);
            try {
                this.getRocksDB().dropColumnFamily(handle);
                handle.close();
            }
            catch (RocksDBException e) {
                throw new HoodieException((Throwable)e);
            }
            this.managedHandlesMap.remove(columnFamilyName);
            return null;
        });
    }

    public synchronized void close() {
        if (!this.closed) {
            this.closed = true;
            this.managedHandlesMap.values().forEach(AbstractImmutableNativeReference::close);
            this.managedHandlesMap.clear();
            this.managedDescriptorMap.clear();
            this.getRocksDB().close();
            try {
                FileIOUtils.deleteDirectory((File)new File(this.rocksDBBasePath));
            }
            catch (IOException e) {
                throw new HoodieIOException(e.getMessage(), e);
            }
        }
    }

    public long getTotalBytesWritten() {
        return this.totalBytesWritten;
    }

    private <T> byte[] serializePayload(String columnFamily, T value) throws IOException {
        CustomSerializer<T> serializer = this.getSerializerForColumnFamily(columnFamily);
        byte[] payload = serializer.serialize(value);
        this.totalBytesWritten += (long)payload.length;
        return payload;
    }

    private <T> T deserializePayload(String columnFamily, byte[] value) {
        CustomSerializer<T> serializer = this.getSerializerForColumnFamily(columnFamily);
        if (value == null) {
            return null;
        }
        return serializer.deserialize(value);
    }

    private <T> CustomSerializer<T> getSerializerForColumnFamily(String columnFamily) {
        return this.columnFamilySerializers.computeIfAbsent(columnFamily, cf -> new DefaultSerializer());
    }

    private byte[] getKeyBytes(String key) {
        return StringUtils.getUTF8Bytes((String)key);
    }

    private <K extends Serializable> byte[] getKeyBytes(K key) {
        try {
            return SerializationUtils.serialize(key);
        }
        catch (IOException e) {
            throw new HoodieException((Throwable)e);
        }
    }

    String getRocksDBBasePath() {
        return this.rocksDBBasePath;
    }

    public static interface BatchHandler {
        public void apply(WriteBatch var1);
    }

    private static class IteratorWrapper<T, R>
    implements Iterator<Pair<T, R>> {
        private final RocksIterator iterator;
        private final CustomSerializer<R> deserializer;

        public IteratorWrapper(RocksIterator iterator, CustomSerializer<R> deserializer) {
            this.iterator = iterator;
            this.deserializer = deserializer;
            iterator.seekToFirst();
        }

        @Override
        public boolean hasNext() {
            return this.iterator.isValid();
        }

        @Override
        public Pair<T, R> next() {
            if (!this.hasNext()) {
                throw new IllegalStateException("next() called on rocksDB with no more valid entries");
            }
            Object key = SerializationUtils.deserialize(this.iterator.key());
            R val = this.deserializer.deserialize(this.iterator.value());
            this.iterator.next();
            return Pair.of(key, val);
        }
    }
}

