/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.io.storage;

import java.io.IOException;
import org.apache.hudi.common.config.HoodieMetadataConfig;
import org.apache.hudi.common.config.HoodieReaderConfig;
import org.apache.hudi.common.config.TypedProperties;
import org.apache.hudi.common.util.ConfigUtils;
import org.apache.hudi.common.util.Either;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.ValidationUtils;
import org.apache.hudi.common.util.hash.MurmurHash;
import org.apache.hudi.common.util.io.ByteBufferBackedInputStream;
import org.apache.hudi.io.ByteArraySeekableDataInputStream;
import org.apache.hudi.io.SeekableDataInputStream;
import org.apache.hudi.io.hfile.CachingHFileReaderImpl;
import org.apache.hudi.io.hfile.HFileReader;
import org.apache.hudi.io.hfile.HFileReaderImpl;
import org.apache.hudi.storage.HoodieStorage;
import org.apache.hudi.storage.StoragePath;

public class HFileReaderFactory {
    private final HoodieStorage storage;
    private final HoodieMetadataConfig metadataConfig;
    private final TypedProperties properties;
    private final Either<StoragePath, byte[]> fileSource;
    private Option<Long> fileSizeOpt;

    private HFileReaderFactory(HoodieStorage storage, TypedProperties properties, Either<StoragePath, byte[]> fileSource, Option<Long> fileSizeOpt) {
        this.storage = storage;
        this.metadataConfig = HoodieMetadataConfig.newBuilder().withProperties(properties).build();
        this.properties = properties;
        this.fileSource = fileSource;
        this.fileSizeOpt = fileSizeOpt;
    }

    public HFileReader createHFileReader() throws IOException {
        if (this.fileSizeOpt.isEmpty()) {
            this.fileSizeOpt = Option.of((Object)this.determineFileSize());
        }
        long fileSize = (Long)this.fileSizeOpt.get();
        SeekableDataInputStream inputStream = this.createInputStream(fileSize);
        if (this.shouldEnableBlockCaching()) {
            int blockCacheSize = ConfigUtils.getIntWithAltKeys(this.properties, HoodieReaderConfig.HFILE_BLOCK_CACHE_SIZE);
            int cacheTtlMinutes = ConfigUtils.getIntWithAltKeys(this.properties, HoodieReaderConfig.HFILE_BLOCK_CACHE_TTL_MINUTES);
            String filePath = this.getFilePath();
            return new CachingHFileReaderImpl(inputStream, fileSize, filePath, blockCacheSize, cacheTtlMinutes);
        }
        return new HFileReaderImpl(inputStream, fileSize);
    }

    private boolean shouldEnableBlockCaching() {
        return ConfigUtils.getBooleanWithAltKeys(this.properties, HoodieReaderConfig.HFILE_BLOCK_CACHE_ENABLED);
    }

    private String getFilePath() {
        if (this.fileSource.isLeft()) {
            return this.fileSource.asLeft().toString();
        }
        int murmurHash = MurmurHash.getInstance().hash(this.fileSource.asRight());
        return String.valueOf(murmurHash);
    }

    private long determineFileSize() throws IOException {
        if (this.fileSource.isLeft()) {
            return this.storage.getPathInfo(this.fileSource.asLeft()).getLength();
        }
        return this.fileSource.asRight().length;
    }

    private SeekableDataInputStream createInputStream(long fileSize) throws IOException {
        if (this.fileSource.isLeft()) {
            if (fileSize <= (long)this.metadataConfig.getFileCacheMaxSizeMB() * 1024L * 1024L) {
                byte[] buffer;
                StoragePath path = this.fileSource.asLeft();
                try (SeekableDataInputStream stream = this.storage.openSeekable(path, false);){
                    buffer = new byte[(int)this.storage.getPathInfo(path).getLength()];
                    stream.readFully(buffer);
                }
                return new ByteArraySeekableDataInputStream(new ByteBufferBackedInputStream(buffer));
            }
            return this.storage.openSeekable(this.fileSource.asLeft(), false);
        }
        return new ByteArraySeekableDataInputStream(new ByteBufferBackedInputStream(this.fileSource.asRight()));
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private HoodieStorage storage;
        private Option<TypedProperties> properties = Option.empty();
        private Either<StoragePath, byte[]> fileSource;
        private Option<Long> fileSizeOpt = Option.empty();

        public Builder withStorage(HoodieStorage storage) {
            this.storage = storage;
            return this;
        }

        public Builder withProps(TypedProperties props) {
            this.properties = Option.of((Object)props);
            return this;
        }

        public Builder withPath(StoragePath path) {
            ValidationUtils.checkState((this.fileSource == null ? 1 : 0) != 0, (String)"HFile source already set, cannot set path");
            this.fileSource = Either.left(path);
            return this;
        }

        public Builder withContent(byte[] bytesContent) {
            ValidationUtils.checkState((this.fileSource == null ? 1 : 0) != 0, (String)"HFile source already set, cannot set bytes content");
            this.fileSource = Either.right(bytesContent);
            return this;
        }

        public Builder withFileSize(long fileSize) {
            ValidationUtils.checkState((fileSize >= 0L ? 1 : 0) != 0, (String)"file size is invalid, should be greater than or equal to zero");
            this.fileSizeOpt = Option.of((Object)fileSize);
            return this;
        }

        public HFileReaderFactory build() {
            ValidationUtils.checkArgument((this.storage != null ? 1 : 0) != 0, (String)"Storage cannot be null");
            ValidationUtils.checkArgument((this.fileSource != null ? 1 : 0) != 0, (String)"HFile source cannot be null");
            TypedProperties props = this.properties.isPresent() ? (TypedProperties)this.properties.get() : new TypedProperties();
            return new HFileReaderFactory(this.storage, props, this.fileSource, this.fileSizeOpt);
        }
    }
}

