/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.avro;

import java.io.IOException;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.hudi.avro.HoodieAvroUtils;
import org.apache.hudi.com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.org.apache.avro.Schema;
import org.apache.hudi.org.apache.avro.generic.GenericData;
import org.apache.hudi.org.apache.avro.generic.GenericRecord;

public class MercifulJsonConverter {
    private static final Map<Schema.Type, JsonToAvroFieldProcessor> FIELD_TYPE_PROCESSORS = MercifulJsonConverter.getFieldTypeProcessors();
    private static final Map<String, Map<String, String>> SANITIZED_FIELD_MAPPINGS = new ConcurrentHashMap<String, Map<String, String>>();
    private final ObjectMapper mapper;
    private final String invalidCharMask;
    private final boolean shouldSanitize;

    private static Map<Schema.Type, JsonToAvroFieldProcessor> getFieldTypeProcessors() {
        return Collections.unmodifiableMap(new HashMap<Schema.Type, JsonToAvroFieldProcessor>(){
            {
                this.put(Schema.Type.STRING, MercifulJsonConverter.generateStringTypeHandler());
                this.put(Schema.Type.BOOLEAN, MercifulJsonConverter.generateBooleanTypeHandler());
                this.put(Schema.Type.DOUBLE, MercifulJsonConverter.generateDoubleTypeHandler());
                this.put(Schema.Type.FLOAT, MercifulJsonConverter.generateFloatTypeHandler());
                this.put(Schema.Type.INT, MercifulJsonConverter.generateIntTypeHandler());
                this.put(Schema.Type.LONG, MercifulJsonConverter.generateLongTypeHandler());
                this.put(Schema.Type.ARRAY, MercifulJsonConverter.generateArrayTypeHandler());
                this.put(Schema.Type.RECORD, MercifulJsonConverter.generateRecordTypeHandler());
                this.put(Schema.Type.ENUM, MercifulJsonConverter.generateEnumTypeHandler());
                this.put(Schema.Type.MAP, MercifulJsonConverter.generateMapTypeHandler());
                this.put(Schema.Type.BYTES, MercifulJsonConverter.generateBytesTypeHandler());
                this.put(Schema.Type.FIXED, MercifulJsonConverter.generateFixedTypeHandler());
            }
        });
    }

    public MercifulJsonConverter() {
        this(false, "__");
    }

    public MercifulJsonConverter(boolean shouldSanitize, String invalidCharMask) {
        this(new ObjectMapper(), shouldSanitize, invalidCharMask);
    }

    public MercifulJsonConverter(ObjectMapper mapper, boolean shouldSanitize, String invalidCharMask) {
        this.mapper = mapper;
        this.shouldSanitize = shouldSanitize;
        this.invalidCharMask = invalidCharMask;
    }

    public GenericRecord convert(String json, Schema schema) {
        try {
            Map jsonObjectMap = this.mapper.readValue(json, Map.class);
            return MercifulJsonConverter.convertJsonToAvro(jsonObjectMap, schema, this.shouldSanitize, this.invalidCharMask);
        }
        catch (IOException e) {
            throw new HoodieIOException(e.getMessage(), e);
        }
    }

    public static void clearCache(String schemaFullName) {
        SANITIZED_FIELD_MAPPINGS.remove(schemaFullName);
    }

    private static GenericRecord convertJsonToAvro(Map<String, Object> inputJson, Schema schema, boolean shouldSanitize, String invalidCharMask) {
        GenericData.Record avroRecord = new GenericData.Record(schema);
        for (Schema.Field f : schema.getFields()) {
            Object val = shouldSanitize ? MercifulJsonConverter.getFieldFromJson(f, inputJson, schema.getFullName(), invalidCharMask) : inputJson.get(f.name());
            if (val == null) continue;
            avroRecord.put(f.pos(), MercifulJsonConverter.convertJsonToAvroField(val, f.name(), f.schema(), shouldSanitize, invalidCharMask));
        }
        return avroRecord;
    }

    private static Object getFieldFromJson(Schema.Field fieldSchema, Map<String, Object> inputJson, String schemaFullName, String invalidCharMask) {
        Object match;
        Map schemaToJsonFieldNames = SANITIZED_FIELD_MAPPINGS.computeIfAbsent(schemaFullName, unused -> new ConcurrentHashMap());
        if (!schemaToJsonFieldNames.containsKey(fieldSchema.name())) {
            for (String inputFieldName : inputJson.keySet()) {
                if (schemaToJsonFieldNames.containsKey(inputFieldName)) continue;
                String sanitizedJsonFieldName = HoodieAvroUtils.sanitizeName(inputFieldName, invalidCharMask);
                schemaToJsonFieldNames.putIfAbsent(sanitizedJsonFieldName, inputFieldName);
            }
        }
        if ((match = inputJson.get(schemaToJsonFieldNames.getOrDefault(fieldSchema.name(), fieldSchema.name()))) != null) {
            return match;
        }
        for (String alias : fieldSchema.aliases()) {
            if (!inputJson.containsKey(alias)) continue;
            return inputJson.get(alias);
        }
        return null;
    }

    private static Schema getNonNull(Schema schema) {
        List<Schema> types = schema.getTypes();
        Schema.Type firstType = types.get(0).getType();
        return firstType.equals((Object)Schema.Type.NULL) ? types.get(1) : types.get(0);
    }

    private static boolean isOptional(Schema schema) {
        return schema.getType().equals((Object)Schema.Type.UNION) && schema.getTypes().size() == 2 && (schema.getTypes().get(0).getType().equals((Object)Schema.Type.NULL) || schema.getTypes().get(1).getType().equals((Object)Schema.Type.NULL));
    }

    private static Object convertJsonToAvroField(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
        if (MercifulJsonConverter.isOptional(schema)) {
            if (value == null) {
                return null;
            }
            schema = MercifulJsonConverter.getNonNull(schema);
        } else if (value == null) {
            throw new HoodieJsonToAvroConversionException(null, name, schema, shouldSanitize, invalidCharMask);
        }
        JsonToAvroFieldProcessor processor = FIELD_TYPE_PROCESSORS.get((Object)schema.getType());
        if (null != processor) {
            return processor.convertToAvro(value, name, schema, shouldSanitize, invalidCharMask);
        }
        throw new IllegalArgumentException("JsonConverter cannot handle type: " + (Object)((Object)schema.getType()));
    }

    private static JsonToAvroFieldProcessor generateBooleanTypeHandler() {
        return new JsonToAvroFieldProcessor(){

            @Override
            public Pair<Boolean, Object> convert(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
                if (value instanceof Boolean) {
                    return Pair.of(true, value);
                }
                return Pair.of(false, null);
            }
        };
    }

    private static JsonToAvroFieldProcessor generateIntTypeHandler() {
        return new JsonToAvroFieldProcessor(){

            @Override
            public Pair<Boolean, Object> convert(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
                if (value instanceof Number) {
                    return Pair.of(true, ((Number)value).intValue());
                }
                if (value instanceof String) {
                    return Pair.of(true, Integer.valueOf((String)value));
                }
                return Pair.of(false, null);
            }
        };
    }

    private static JsonToAvroFieldProcessor generateDoubleTypeHandler() {
        return new JsonToAvroFieldProcessor(){

            @Override
            public Pair<Boolean, Object> convert(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
                if (value instanceof Number) {
                    return Pair.of(true, ((Number)value).doubleValue());
                }
                if (value instanceof String) {
                    return Pair.of(true, Double.valueOf((String)value));
                }
                return Pair.of(false, null);
            }
        };
    }

    private static JsonToAvroFieldProcessor generateFloatTypeHandler() {
        return new JsonToAvroFieldProcessor(){

            @Override
            public Pair<Boolean, Object> convert(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
                if (value instanceof Number) {
                    return Pair.of(true, Float.valueOf(((Number)value).floatValue()));
                }
                if (value instanceof String) {
                    return Pair.of(true, Float.valueOf((String)value));
                }
                return Pair.of(false, null);
            }
        };
    }

    private static JsonToAvroFieldProcessor generateLongTypeHandler() {
        return new JsonToAvroFieldProcessor(){

            @Override
            public Pair<Boolean, Object> convert(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
                if (value instanceof Number) {
                    return Pair.of(true, ((Number)value).longValue());
                }
                if (value instanceof String) {
                    return Pair.of(true, Long.valueOf((String)value));
                }
                return Pair.of(false, null);
            }
        };
    }

    private static JsonToAvroFieldProcessor generateStringTypeHandler() {
        return new JsonToAvroFieldProcessor(){

            @Override
            public Pair<Boolean, Object> convert(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
                return Pair.of(true, value.toString());
            }
        };
    }

    private static JsonToAvroFieldProcessor generateBytesTypeHandler() {
        return new JsonToAvroFieldProcessor(){

            @Override
            public Pair<Boolean, Object> convert(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
                return Pair.of(true, ByteBuffer.wrap(StringUtils.getUTF8Bytes(value.toString())));
            }
        };
    }

    private static JsonToAvroFieldProcessor generateFixedTypeHandler() {
        return new JsonToAvroFieldProcessor(){

            @Override
            public Pair<Boolean, Object> convert(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
                List converval = (List)value;
                byte[] src = new byte[converval.size()];
                for (int i = 0; i < converval.size(); ++i) {
                    src[i] = ((Integer)converval.get(i)).byteValue();
                }
                byte[] dst = new byte[schema.getFixedSize()];
                System.arraycopy(src, 0, dst, 0, Math.min(schema.getFixedSize(), src.length));
                return Pair.of(true, new GenericData.Fixed(schema, dst));
            }
        };
    }

    private static JsonToAvroFieldProcessor generateEnumTypeHandler() {
        return new JsonToAvroFieldProcessor(){

            @Override
            public Pair<Boolean, Object> convert(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
                if (schema.getEnumSymbols().contains(value.toString())) {
                    return Pair.of(true, new GenericData.EnumSymbol(schema, value.toString()));
                }
                throw new HoodieJsonToAvroConversionException(String.format("Symbol %s not in enum", value.toString()), schema.getFullName(), schema, shouldSanitize, invalidCharMask);
            }
        };
    }

    private static JsonToAvroFieldProcessor generateRecordTypeHandler() {
        return new JsonToAvroFieldProcessor(){

            @Override
            public Pair<Boolean, Object> convert(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
                GenericData.Record result = new GenericData.Record(schema);
                return Pair.of(true, MercifulJsonConverter.convertJsonToAvro((Map)value, schema, shouldSanitize, invalidCharMask));
            }
        };
    }

    private static JsonToAvroFieldProcessor generateArrayTypeHandler() {
        return new JsonToAvroFieldProcessor(){

            @Override
            public Pair<Boolean, Object> convert(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
                Schema elementSchema = schema.getElementType();
                ArrayList<Object> listRes = new ArrayList<Object>();
                for (Object v : (List)value) {
                    listRes.add(MercifulJsonConverter.convertJsonToAvroField(v, name, elementSchema, shouldSanitize, invalidCharMask));
                }
                return Pair.of(true, new GenericData.Array(schema, listRes));
            }
        };
    }

    private static JsonToAvroFieldProcessor generateMapTypeHandler() {
        return new JsonToAvroFieldProcessor(){

            @Override
            public Pair<Boolean, Object> convert(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
                Schema valueSchema = schema.getValueType();
                HashMap mapRes = new HashMap();
                for (Map.Entry v : ((Map)value).entrySet()) {
                    mapRes.put(v.getKey(), MercifulJsonConverter.convertJsonToAvroField(v.getValue(), name, valueSchema, shouldSanitize, invalidCharMask));
                }
                return Pair.of(true, mapRes);
            }
        };
    }

    public static class HoodieJsonToAvroConversionException
    extends HoodieException {
        private Object value;
        private String fieldName;
        private Schema schema;
        private boolean shouldSanitize;
        private String invalidCharMask;

        public HoodieJsonToAvroConversionException(Object value, String fieldName, Schema schema, boolean shouldSanitize, String invalidCharMask) {
            this.value = value;
            this.fieldName = fieldName;
            this.schema = schema;
            this.shouldSanitize = shouldSanitize;
            this.invalidCharMask = invalidCharMask;
        }

        @Override
        public String toString() {
            if (this.shouldSanitize) {
                return String.format("Json to Avro Type conversion error for field %s, %s for %s. Field sanitization is enabled with a mask of %s.", this.fieldName, this.value, this.schema, this.invalidCharMask);
            }
            return String.format("Json to Avro Type conversion error for field %s, %s for %s", this.fieldName, this.value, this.schema);
        }
    }

    private static abstract class JsonToAvroFieldProcessor
    implements Serializable {
        private JsonToAvroFieldProcessor() {
        }

        public Object convertToAvro(Object value, String name, Schema schema, boolean shouldSanitize, String invalidCharMask) {
            Pair<Boolean, Object> res = this.convert(value, name, schema, shouldSanitize, invalidCharMask);
            if (!res.getLeft().booleanValue()) {
                throw new HoodieJsonToAvroConversionException(value, name, schema, shouldSanitize, invalidCharMask);
            }
            return res.getRight();
        }

        protected abstract Pair<Boolean, Object> convert(Object var1, String var2, Schema var3, boolean var4, String var5);
    }
}

